﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/capsrv/capsrv_ApplicationAlbumFileEntry.h>
#include <nn/capsrv/capsrv_ApplicationData.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include "capsrv_AlbumMovieReadStreamHandle.h"
#include "capsrv_Result.h"

// このヘッダには、アプリ向けに公開される nn::album の API の shim 実装向けの
// 関数を定義しています。アプリ開発者には公開されません。

namespace nn { namespace capsrv {

    //! @brief アプリ向けにアルバムへのアクセス機能を初期化します。
    //! @pre  アプリ向けのアルバムへのアクセス機能が初期化されていない
    //! @post アプリ向けのアルバムへのアクセス機能が初期化されている
    //! @details
    //! 本ヘッダの全ての API は事前に本関数を呼び出しておく必要があります。
    Result InitializeAlbumAccessForApplication() NN_NOEXCEPT;

    //! @brief アプリ向けのアルバムへのアクセス機能の終了処理を行います。
    //! @pre  アプリ向けのアルバムへのアクセス機能が初期化されている
    //! @post アプリ向けのアルバムへのアクセス機能が初期化されていない
    //! @details
    //! 本関数発行後にアルバムにアクセスする関数を呼び出してはいけません。
    void FinalizeAlbumAccessForApplication() NN_NOEXCEPT;


    //! @brief 指定されたファイル種別のファイルリストを取得します。
    //!
    //! @param[out] pOutCount     取得した対象ファイル数の格納先ポインタ
    //! @param[out] pBuffer       取得した対象ファイルリストを受け取る配列
    //! @param[in]  bufferLength  pBuffer で渡す配列の長さ（要素数）
    //! @param[in]  contents      対象とするファイル種別
    //! @param[in]  beginDateTime ファイル作成時間のフィルタ用先頭時間
    //! @param[in]  endDateTime   ファイル作成時間のフィルタ用末尾時間
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @pre bufferLength >= 0
    //! @post 成功した場合、*pOutCount に取得したファイル数が代入される
    //! @post 成功した場合、pBuffer にファイルリストが代入される
    //! @post 失敗した場合、*pOutCount に 0 が代入される
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //!
    //! @details
    //! contents で指定された対象ファイル種別のファイルリストを pBuffer が指す
    //! ApplicationAlbumFileEntry 配列に格納します。ファイルリストには、
    //! beginDateTime <= （対象ファイルの作成時間） < endDateTime を
    //! 満たすファイルのみを取得します。
    //!
    //! pBuffer に書き込まれる要素の順番は保証されません。
    //! また、bufferLength が取得可能なファイル数よりも大きな値だった場合、
    //! 0 から (*pOutCount - 1) 番目の要素に値が書き込まれます。
    //!
    //! bufferLength が取得可能なファイル数よりも小さな値だった場合、
    //! pBuffer には bufferLength 個の要素だけが書き込まれます。
    //! 対象ファイルの完全なリストを取得するためには、
    //! bufferLength >= GetAlbumContentsFileCountForApplication() のような
    //! 配列を渡す必要があります。
    //!
    Result GetAlbumContentsFileListForApplication(int* pOutCount, ApplicationAlbumFileEntry* pBuffer, int bufferLength, AlbumFileContentsType contents, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime) NN_NOEXCEPT;


    //! @brief 指定されたユーザ識別子を持つファイルリストを取得します。
    //!
    //! @param[out] pOutCount     取得した対象ファイル数の格納先ポインタ
    //! @param[out] pBuffer       取得した対象ファイルリストを受け取る配列
    //! @param[in]  bufferLength  pBuffer で渡す配列の長さ（要素数）
    //! @param[in]  contents      取得対象とするファイル種別
    //! @param[in]  uid           取得対象とするユーザ識別子
    //! @param[in]  beginDateTime ファイル作成時間のフィルタ用先頭時間
    //! @param[in]  endDateTime   ファイル作成時間のフィルタ用末尾時間
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @pre uid != nn::account::InvalidUid
    //! @pre bufferLength >= 0
    //! @post 成功した場合、*pOutCount に取得したファイル数が代入される
    //! @post 成功した場合、pBuffer にファイルリストが代入される
    //! @post 失敗した場合、*pOutCount に 0 が代入される
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //!
    //! @details
    //! contents で指定された対象ファイル種別のファイルリストを pBuffer が指す
    //! ApplicationAlbumFileEntry 配列に格納します。ファイルリストには、
    //! beginDateTime <= （対象ファイルの作成時間） < endDateTime を
    //! 満たすファイルのみを取得します。
    //!
    //! 本 API では uid で指定したユーザ識別子情報が埋め込まれたファイルのみを
    //! ファイルリストとして取得します。
    //!
    //! pBuffer に書き込まれる要素の順番は保証されません。
    //! また、bufferLength が取得可能なファイル数よりも大きな値だった場合、
    //! 0 から (*pOutCount - 1) 番目の要素に値が書き込まれます。
    //!
    //! bufferLength が取得可能なファイル数よりも小さな値だった場合、
    //! pBuffer には bufferLength 個の要素だけが書き込まれます。
    //! 対象ファイルの完全なリストを取得するためには、
    //! bufferLength >= GetAlbumContentsFileCountForApplication() のような
    //! 配列を渡す必要があります。
    //!
    Result GetAlbumContentsFileListForApplicationEx1(
        int* pOutCount,
        ApplicationAlbumFileEntry* pBuffer,
        int bufferLength,
        AlbumFileContentsType contents,
        const account::Uid& uid,
        AlbumFileDateTime beginDateTime,
        AlbumFileDateTime endDateTime
    ) NN_NOEXCEPT;


    //! @brief アプリ独自の動画ファイルを削除します。
    //!
    //! @param[in]  entry       削除対象のファイル識別子
    //! @param[in]  contents    対象とするファイル種別
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @post 成功した場合、指定したファイルは削除される
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //!
    //! @details
    //! entry で指定されたアプリ独自の動画ファイルを削除します。
    //!
    Result DeleteAlbumContentsFileForApplication(const ApplicationAlbumFileEntry& entry, AlbumFileContentsType contents) NN_NOEXCEPT;


    //! @brief 指定されたアルバムファイルを削除します。
    //!
    //! @param[in]  entry       削除対象のファイル識別子
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @post 成功した場合、指定したファイルは削除される
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumInvalidFileId
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //!
    //! @details
    //! entry で指定されたアルバムファイルを削除します。
    //! 本関数はデバッグ用途のため、ファイルの種別に限らず、
    //! entry で指定されたファイルを削除します。
    //!
    //! 製品機で本 API が呼ばれた場合には、内部でアボートします。
    //!
    Result DeleteAlbumFileForApplicationDebug(const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT;


    //! @brief 指定したファイルの大きさをバイトで取得します。
    //!
    //! @param[out] pOutSize    ファイルサイズの格納先
    //! @param[in]  entry       対象ファイルの識別子
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //!
    //! @details
    //! entry で指定されたファイルのファイルサイズ（バイト）を取得します。
    //!
    Result GetAlbumContentsFileSizeForApplication(uint64_t* pOutSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT;


    //! @brief 対象メディアファイルの静止画の画像データを読み込みます。
    //!
    //! @param[out] pOutWidth      読み込んだ静止画の幅（ピクセル）の格納先
    //! @param[out] pOutHeight     読み込んだ静止画の高さ（ピクセル）の格納先
    //! @param[out] pOutApplicationData 画像データに埋め込まれた任意データの格納先
    //! @param[out] pBuffer        画像データを受け取るバッファ
    //! @param[in]  bufferSize     画像データを受け取るバッファサイズ（バイト）
    //! @param[in]  pWorkBuffer    作業用バッファ
    //! @param[in]  workBufferSize 作業用バッファサイズ（バイト）
    //! @param[in]  entry          対象ファイルの識別子
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @post 成功失敗に関わらず pWorkBuffer は 0 埋めされる
    //! @post 成功した場合は *pOutWidth  == 1280
    //! @post 成功した場合は *pOutHeight == 720
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //!
    //! @details
    //! entry で指定された対象メディアファイルの静止画の画像データをデコードし、
    //! pBuffer と bufferSize で指定されたバッファ領域に読み込みます。
    //! 取得される画像データは sRGBA8 フォーマットで、 パディングなしの
    //! リニア画像 (stride = 4 * width, size = 4 * width * height) です。
    //! デコード時のオプションには既定値が使用されます。
    //!
    //! 本関数は対象ファイルに静止画ファイルだけでなく、動画ファイルを
    //! 指定することも可能です。その場合、動画ファイルに埋め込まれた
    //! 静止画の画像データに対して処理されます。
    //!
    //! pBuffer には画像データを受け取るのに十分な大きさのバッファを渡す
    //! 必要があります。現実装では 4 * 1280 * 720 バイト以上のバッファを
    //! 渡せば十分です。pBuffer には CPU キャッシュが有効な GPU の
    //! メモリプール上のメモリを渡すことができます。
    //!
    //! bufferSize が画像データサイズよりも大きかった場合、
    //! pBuffer の先頭から画像データを書き込み、以降は 0 埋めされます。
    //! bufferSize が画像データサイズよりも小さかった場合、
    //! ResultAlbumReadBufferShortage を返して失敗します。
    //!
    //! pWorkBuffer には GetAlbumContentsFileSizeForApplication() で
    //! 取得できるファイルサイズよりも大きいバッファを渡す必要があります。
    //! AlbumFileSizeLimit_ScreenShot バイト以上のバッファを渡せば十分です。
    //! pWorkBuffer に GPU のメモリプール上のメモリを渡すことはできません。
    //! workBufferSize が必要な大きさに満たなかった場合、
    //! ResultAlbumReadBufferShortage を返して失敗します。
    //!
    Result LoadAlbumContentsFileScreenShotImageForApplication(
        int* pOutWidth,
        int* pOutHeight,
        ApplicationData* pOutApplicationData,
        void* pBuffer,
        size_t bufferSize,
        void* pWorkBuffer,
        size_t workBufferSize,
        const ApplicationAlbumFileEntry& entry
        ) NN_NOEXCEPT;


    //! @brief 対象メディアファイル内のサムネイル画像データを読み込みます。
    //!
    //! @param[out] pOutWidth      読み込んだサムネイル画像の幅（ピクセル）の格納先
    //! @param[out] pOutHeight     読み込んだサムネイル画像の高さ（ピクセル）の格納先
    //! @param[out] pOutApplicationData 画像データに埋め込まれた任意データの格納先
    //! @param[out] pBuffer        サムネイル画像データを受け取るバッファ
    //! @param[in]  bufferSize     サムネイル画像データを受け取るバッファサイズ
    //! @param[in]  pWorkBuffer    作業用バッファです。
    //! @param[in]  workBufferSize 作業用バッファの大きさ（バイト）です。
    //! @param[in]  entry          対象ファイルの識別子
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @post 成功失敗に関わらず pWorkBuffer は 0 埋めされます。
    //! @post 成功した場合は *pOutWidth  == ViewerThumbnailImageSize_Width
    //! @post 成功した場合は *pOutHeight == ViewerThumbnailImageSize_Height
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::capsrv::ResultAlbumError
    //! @retval nn::capsrv::ResultAlbumIsNotMounted
    //! @retval nn::capsrv::ResultAlbumFileNotFound
    //! @retval nn::capsrv::ResultAlbumReadBufferShortage
    //! @retval nn::capsrv::ResultAlbumFileNoThumbnail
    //!
    //! @details
    //! entry で指定された対象メディアファイルのサムネイル画像をデコードし、
    //! pBuffer と bufferSize で指定されたバッファ領域に読み込みます。
    //! 取得される画像データは sRGBA8 フォーマットで、 パディングなしの
    //! リニア画像 (stride = 4 * width, size = 4 * width * height) です。
    //! デコード時のオプションには既定値が使用されます。
    //!
    //! 本関数は対象ファイルに静止画ファイルだけでなく、動画ファイルを
    //! 指定することも可能です。その場合、動画ファイルに埋め込まれた
    //! 静止画の画像データに対して処理されます。
    //!
    //! pBuffer にはサムネイル画像を受け取るのに十分な大きさのバッファを
    //! 渡す必要があります。これには、ViewerThumbnailImageDataSize_Raw バイト
    //! 以上のバッファを渡せば十分です。pBuffer には CPU キャッシュが有効な
    //! GPU のメモリプール上のメモリを渡すことができます。
    //!
    //! bufferSize がサムネイル画像のデータサイズよりも大きかった場合、
    //! pBuffer の先頭からサムネイル画像を書き込み、以降は 0 埋めされます。
    //! bufferSize がサムネイル画像のデータサイズよりも小さかった場合、
    //! ResultAlbumReadBufferShortage により失敗します。
    //!
    //! pWorkBuffer にはデコード前のサムネイルデータのサイズよりも大きい
    //! バッファを渡す必要があります。
    //! これには、ViewerThumbnailImageDataSize_EncodedLimit バイト以上の
    //! バッファを渡せば十分です。
    //! pWorkBuffer に GPU のメモリプール上のメモリを渡すことはできません。
    //!
    //! workBufferSize が必要な大きさに満たなかった場合、
    //! ResultAlbumReadBufferShortage を返して失敗します。
    //!
    //! 正規のファイルに対しても ResultAlbumFileNoThumbnail が返る場合があります。
    //!
    Result LoadAlbumContentsFileThumbnailImageForApplication(
        int* pOutWidth,
        int* pOutHeight,
        ApplicationData* pOutApplicationData,
        void* pBuffer,
        size_t bufferSize,
        void* pWorkBuffer,
        size_t workBufferSize,
        const ApplicationAlbumFileEntry& entry
        ) NN_NOEXCEPT;


//! @name 動画読込ストリーム
//! @{

    //! @brief 動画読込ストリームを開きます。
    //!
    //! @param[out] pOutHandle  動画読込ストリームのハンドルの格納先
    //! @param[in]  entry       対象動画ファイルの識別子
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @retval ResultAlbumResourceLimit 同時にオープンできるストリーム数の上限に達しています。
    //!
    //! @details
    //! 動画読込ストリームを開きます。
    //! 読込が終わった後は CloseAlbumMovieReadStreamForApplication() で
    //! ストリームを閉じる必要があります。
    //!
    Result OpenAlbumMovieReadStreamForApplication(AlbumMovieReadStreamHandle* pOutHandle, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT;


    //! @brief 動画読込ストリームを閉じます。
    //!
    //! @param[in] handle   動画読込ストリームのハンドル
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @details
    //! 動画読込ストリームを閉じます。
    //!
    void CloseAlbumMovieReadStreamForApplication(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;


    //! @brief 動画ファイルの動画データ部分のサイズを取得します。
    //!
    //! @param[out] pOutSize    動画データ部分のサイズ（バイト）の格納先
    //! @param[in]  handle      動画読込ストリームのハンドル
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @retval ResultAlbumNotFound handle が無効です。
    //!
    //! @details
    //! 動画ファイルの動画データ部分のサイズを取得します。
    //! nn::capsrv::ReadDataFromAlbumMovieReadStreamForApplication() では
    //! 先頭からこの API で取得されるサイズの範囲で有効なデータを取得できます。
    //!
    //! @see nn::capsrv::OpenAlbumMovieReadStreamForApplication()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    //!
    Result GetAlbumMovieReadStreamDataSizeForApplication(uint64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;


    //! @brief 動画ファイルの動画データ部分を読み込みます。
    //!
    //! @param[out] pOutReadSize 読込んだバイト数の格納先
    //! @param[in]  buffer       読込んだデータを受け取るバッファ
    //! @param[in]  size         読込むデータの大きさ（バイト）
    //! @param[in]  handle       動画読込ストリームのハンドル
    //! @param[in]  offset       データ読込の開始位置オフセット
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //! @pre size >= 0
    //! @pre size % nn::capsrv::AlbumMovieDataUnitSize == 0
    //! @pre offset >= 0
    //! @pre offset % nn::capsrv::AlbumMovieDataUnitSize == 0
    //!
    //! @retval ResultAlbumNotFound handle が不正です。
    //!
    //! @details
    //! 動画ファイルの動画データ部分を読み込みます。
    //! オフセット及びサイズはともに nn::capsrv::AlbumMovieDataUnitSize の
    //! 整数倍でなければなりません。動画データの大きさは
    //! nn::capsrv::GetAlbumMovieReadStreamDataSizeForApplication()
    //! で取得できます。読込範囲が動画データの大きさを超える場合、
    //! 範囲外の部分はゼロクリアされます。
    //!
    //! @see nn::capsrv::OpenAlbumMovieReadStreamForApplication()
    //! @see nn::capsrv::movie::MovieReaderFileSystem
    //!
    Result ReadDataFromAlbumMovieReadStreamForApplication(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT;


    //! @brief 動画読込ストリームの内部エラーの値を取得します。
    //!
    //! @param[in]  handle  動画読込ストリームのハンドル
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @details
    //! 動画読込ストリームの内部エラーの値を返します。
    //! ストリームが正常な場合は nn::ResultSuccess() が返ります。
    //!
    Result GetAlbumMovieReadStreamBrokenReasonForApplication(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT;

//! @}

//! @name アルバム使用状況の取得
//! @{

    //! @brief 指定コンテンツのファイル作成に対する事前チェックを行なう
    //!
    //! @param[in]  contents    対象とするファイル種別
    //! @param[in]  size        必要とする空き容量（バイト）
    //!
    //! @pre InitializeAlbumAccessForApplication() により初期化済みである
    //!
    //! @retval nn::ResultSuccess
    //! @retval nn::fs::ResultNotMounted
    //! @retval nn::capsrv::ResultAlbumFileCountLimit
    //! @retval nn::capsrv::ResultAlbumIsFull
    //! @retval nn::capstv::ResultAlbumError
    //!
    //! @details
    //! 指定された contents のファイル作成のための事前チェックとして
    //! アプリから見た使用状況をチェックし、その結果を Result で返します。
    //!
    nn::Result PrecheckToCreateContentsForApplication(
        AlbumFileContentsType contents,
        uint64_t size
        ) NN_NOEXCEPT;

//! @}

}}  // namespace nn::capsrv
