﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  ユーザ識別子を伴うアルバムファイルの作成や制御に関する定義
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>

#include "album_CommonApi.h"
#include "album_ImageSize.h"
#include "album_ImageOrientation.h"
#include "album_AlbumReportOption.h"
#include "album_AlbumFileEntry.private.h"

namespace nn { namespace album {

//! @brief  指定可能なユーザ識別子数の最大値を表す定数です。
//!
const int UidListCountMax = 8;

//! @name アルバムの静止画ファイル関連機能
//! @{

//----------------------------------------------------------------------------
//! @brief 指定されたユーザ識別子を持つアルバムの静止画ファイルのファイルリストを取得します。
//!
//! @param[out] pOutCount     取得した静止画ファイル数の格納先ポインタ
//! @param[out] buffer        取得した静止画ファイルリストを受け取る配列
//! @param[in]  bufferLength  buffer で渡す配列の長さ（要素数）
//! @param[in]  uid           リストアップ対象のユーザ識別子
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre uid != nn::account::InvalidUid
//! @pre bufferLength >= 0
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! アルバムに保存されている静止画ファイルのファイルリストを
//! buffer が指す AlbumFileEntry 配列に取得します。基本的には、
//! nn::album::GetAlbumScreenshotFileList(int*, AlbumFileEntry[], int)
//! 関数と同様です。
//!
//! 本 API では uid で指定されたユーザ識別子情報が埋め込まれている
//! ファイルのみをリストアップします。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでエラー内容を出画できます。
//!
nn::Result GetAlbumScreenshotFileList(
    int* pOutCount,
    AlbumFileEntry buffer[],
    int bufferLength,
    const nn::account::Uid& uid
) NN_NOEXCEPT;

//----------------------------------------------------------------------------
//! @brief  ユーザ識別子を埋め込んだ画像データを画面写真としてアルバムに保存します。
//!
//! @param[in] pImageData       画面写真として保存する画像データ
//! @param[in] imageDataSize    画像データの大きさ（バイト）
//! @param[in] screenshotSize   保存する画面写真のサイズ
//! @param[in] imageOrientation 画像の向き
//! @param[in] reportOption     保存の結果を自動的にユーザーに通知する設定
//! @param[in] pUidList         ユーザ識別子を納めた配列へのポインタ
//! @param[in] uidListCount     ユーザ識別子の数
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//! @pre pImageData != nullptr
//! @pre imageDataSize >= 4 * GetImageWidth(screenshotSize) * GetImageHeight(screenshotSize)
//! @pre uidListCount >= 0 && uidListCount <= nn::album::UidListCountMax
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 画像データを画面写真としてアルバムに保存します。
//!
//! pImageData には保存する画像データの先頭へのポインタを渡します。
//! 渡すべき画像のフォーマットは、sRGB 色空間で表現された各チャンネルが
//! 8 ビットの RGBA 画像フォーマットで、左上原点、パディングなしの
//! 線形タイリング画像 (stride = 4 * 横幅, size = 4 * 横幅 * 高さ) です。
//!
//! pImageData には CPU キャッシュが有効な GPU のメモリプール上のメモリを
//! 渡すことができます。
//!
//! screenshotSize には保存する画面写真の大きさを指定します。
//! 本 API では画像データのスケーリングを行わないため、pImageData が表現する
//! 画像の大きさは screenshotSize が示す画面写真の大きさと一致する必要があります。
//!
//! imageOrientation には列挙型 nn::album::ImageOrientation の中から
//! 画像の回転方向を指定します。
//! reportOption には列挙型 nn::album::AlbumReportOption の中から
//! 所望の設定を指定します。
//!
//! pUidList と uidListCount には画像データに埋め込むユーザ識別子を
//! nn::account::Uid 型の配列で指定します。uidListCount には 0 から
//! nn::album::UidListCountMax までが指定可能です。
//!
nn::Result SaveScreenshot(
    const void* pImageData,
    size_t imageDataSize,
    ImageSize screenshotSize,
    ImageOrientation imageOrientation,
    AlbumReportOption reportOption,
    const nn::account::Uid* pUidList,
    int uidListCount
) NN_NOEXCEPT;

//! @}

}}  // namespace nn::album

