﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#ifdef LOGREADER_EXPORTS
#define LOGREADER_API extern "C" __declspec(dllexport)
#else
#define LOGREADER_API extern "C" __declspec(dllimport)
#endif

#include <cstring>
#include <autoTestAssistTool/LogReaderTypes.h>
#include <autoTestAssistTool/Common/autoTestAssistTool_LibTypes.h>

/**
*   @file
*   @brief  ログ取得に関する API
*/


/**
* @brief        対象ターゲットからログの取得を開始します。
*
* @param[in]    serialNumber    対象ターゲットのシリアル番号
* @param[in]    logMaxLineCount   保持するログを最大行数
*
* @return       結果を返します
* @retval       LogReaderResult_Success            成功しました。
* @retval       LogReaderResult_ConnectFailed      接続に失敗しました。
* @retval       LogReaderResult_AlreadyStarted     既に記録が開始されています。
* @retval       LogReaderResult_InvalidArgument    引数に渡した値が不正です。
*
* @pre
*  - TargetManager で対象ターゲットに接続している
*  - logMaxLineCount >= 0
*
* @details      ログを対象ターゲットから取得し、1行ごとにメモリに保存します。
*               最新のログに改行コードが含まれていない場合、次に改行コードが出力された際に保存されます。
*               ログの保存時には内部でその行が保存された時間も保持します。@n
*               また、ライブラリ内で現在どのログのどの行を参照しているかをカーソル行として保持します。@n
*               取得開始時は、最初に取得されたログをカーソル行として設定します。@n
*               ログの取得や正規表現のマッチングの開始はカーソル行から行われます。@n
*               logMaxLineCount を超える行数読み込まれた場合は最古のログを破棄し、新たな行を取得します。
*               なお、破棄される際にカーソル行が最古のログを指していた場合は次の行のログを指すようになります。@n
*/
LOGREADER_API LogReaderResult StartLogStoring(const SerialNumberString& serialNumber, int logMaxLineCount);

/**
* @brief        カーソル行のログのサイズを取得します。
*
* @param[out]   pOutSize        取得したログのサイズ
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_NotFound        ログが取得されていません。
*
* @details      ログのサイズにはヌル文字も含みます。
*/
LOGREADER_API LogReaderResult GetLogLineSize(size_t* pOutSize);

/**
* @brief        カーソル行のログを取得します。
*
* @param[out]   buffer          取得した文字列を格納するバッファ
* @param[in]    bufferSize      buffer のサイズ
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_NotFound        ログが取得されていません。
*
* @details      取得したログはヌル終端された UTF-8 エンコードの文字列として格納されます。@n
*               bufferSize がログの長さに満たない場合、bufferSize 分のログを返します。@n
*/
LOGREADER_API LogReaderResult GetLogLine(char* buffer, size_t bufferSize);

/**
* @brief        現在のカーソル行の次の行が保存されるのを待機します。
*
* @param[in]    timeoutMs       タイムアウトまでの時間 [ms]
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_Timeout         タイムアウトしました。
*
* @details      現在のカーソル行の次に新たな行が保存されるのを待機します。@n
*               既に存在する場合はすぐに処理が返ります。@n
*               timoutMs に指定した時間経過でタイムアウトします。
*/
LOGREADER_API LogReaderResult WaitForNextLine(int timeoutMs);

/**
* @brief        指定した正規表現に一致するログが見つかるまでカーソル行を進めます。
*
* @param[in]    regexString         ヌル終端された正規表現の文字列
* @param[in]    timeoutMs           タイムアウトまでの時間 [ms]
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_Timeout         タイムアウトしました。
*
* @details      指定した正規表現に一致するログが見つかるまでカーソル行を進めます。@n
*               指定する正規表現文字列は UTF-8 エンコードの文字列として指定してください。@n
*               正規表現のマッチングには std::regex_match を使用しており、既定のロケール設定でマッチングが行われます。@n
*               最新のログまで到達しても一致しなかった場合は、次のログが取得されるのを待機します。@n
*               timeoutMs に指定した時間経過しても一致しなかった場合は LogReaderResult_Timeout のエラーを返します。@n
*               なお、一致しなかった場合、カーソル行は最新のログを指した状態になります。
*/
LOGREADER_API LogReaderResult WaitUntilMatched(const char* regexString, int timeoutMs);

/**
* @brief        指定した正規表現に一致するログが見つかるまでカーソル行を戻します。
*
* @param[in]    regexString         ヌル終端された正規表現の文字列
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_NotFound        一致するログが見つかりませんでした。
*
* @details      指定した正規表現に一致するログが見つかるまでカーソル行を戻します。@n
*               指定する正規表現文字列は UTF-8 エンコードの文字列として指定してください。@n
*               正規表現のマッチングには std::regex_match を使用しており、既定のロケール設定でマッチングが行われます。@n
*               保存されている最古のログに到達しても一致しなかった場合は LogReaderResult_NotFound のエラーを返します。@n
*               なお、一致しなかった場合、カーソル行は最古のログを指した状態になります。
*/
LOGREADER_API LogReaderResult SearchFormerLog(const char* regexString);

/**
* @brief        カーソル行を最新のログの位置に移動します。
*/
LOGREADER_API void MoveToNewestLine();

/**
* @brief        カーソル行の保存時間から指定時間遡った中で最も古いログを指すように移動します。
*
* @param[in]    timeMs          遡る時間 [ms]
*/
LOGREADER_API void MoveToFormerLineByTime(int timeMs);

/**
* @brief        カーソル行を1つ次の行に移動します。
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_CannotMoveLine  移動できませんでした。
*
* @details      既にカーソル行が最新の行だった場合は LogReaderResult_CannotMoveLine が返ります。
*/
LOGREADER_API LogReaderResult MoveToNextLine();

/**
* @brief        カーソル行を1つ前の行に移動します。
*
* @return       結果を返します。
* @retval       LogReaderResult_Success         成功しました。
* @retval       LogReaderResult_CannotMoveLine  移動できませんでした。
*
* @details      既にカーソル行が最古の行だった場合は LogReaderResult_CannotMoveLine が返ります。
*/
LOGREADER_API LogReaderResult MoveToPreviousLine();

/**
* @brief        対象ターゲットからのログ取得を終了し、保管しているログを破棄します。
*/
LOGREADER_API void StopLogStoring();
