﻿using System;
using System.IO;
using System.Runtime.InteropServices;
using CommandUtility;

namespace MakeRecoveryWriter
{
    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct RecoveryWriterParameter
    {
        public uint BctOffsetOnQspi;
        public uint BctSize;

        public uint BootLoaderOffsetOnQspi;
        public uint BootLoaderSize;

        public uint Package2OffsetOnQspi;
        public uint Package2Size;

        public uint SystemPartitionOffsetOnQspi;
        public uint SystemPartitionSize;
    }

    public class RecoveryWriterImage
    {
        public const long SectorSize = 512;
        public const long ProgramSize = 64 * 1024;
        public const long ParameterSize = 3 * 1024;
        public const long GptSize = 17 * 1024;

        public const long ProgramOffset = 0;
        public const long ParameterOffset = ProgramOffset + ProgramSize;
        public const long GptOffset = ParameterOffset + ParameterSize;
        public const long DataOffset = GptOffset + GptSize;

        public static RecoveryWriterParameter LoadParameter(Stream reader)
        {
            reader.Seek(ParameterOffset, SeekOrigin.Begin);
            return BinaryUtility.ReadBinary<RecoveryWriterParameter>(reader);
        }

        public static long AlignSector(long value)
        {
            return BinaryUtility.AlignUp(value, SectorSize);
        }

        public static void MakeImage(FileStream writer, FileInfo programInfo, FileInfo gptInfo, FileInfo bctInfo, FileInfo bootLoaderInfo, FileInfo package2Info, FileInfo systemPartitionInfo, long systemPartitionAlignment, long minSize, long maxSize)
        {
            ErrorHandling.VerifyFileSize(programInfo, ProgramSize);
            ErrorHandling.VerifyFileSize(gptInfo, GptSize);

            var parameter = MakeParameter(bctInfo, bootLoaderInfo, package2Info, systemPartitionInfo, systemPartitionAlignment);
            var parameterBytes = MakeParameterBytes(parameter);

            FileUtility.WriteToStream(writer, programInfo);
            FileUtility.WriteBytes(writer, parameterBytes);
            FileUtility.WriteToStream(writer, gptInfo);
            FileUtility.WriteToStream(writer, bctInfo);
            FileUtility.WritePaddingToAddress(writer, parameter.BootLoaderOffsetOnQspi);
            FileUtility.WriteToStream(writer, bootLoaderInfo);
            FileUtility.WritePaddingToAddress(writer, parameter.Package2OffsetOnQspi);
            FileUtility.WriteToStream(writer, package2Info);
            FileUtility.WritePaddingToAddress(writer, parameter.SystemPartitionOffsetOnQspi);
            FileUtility.WriteToStream(writer, systemPartitionInfo);
            FileUtility.WritePaddingToAddressIfNeed(writer, parameter.SystemPartitionOffsetOnQspi + systemPartitionAlignment);

            if(writer.Position < minSize)
            {
                FileUtility.WritePaddingToAddress(writer, minSize);
            }

            ErrorHandling.Verify(writer.Position <= maxSize, $"Exceeded image size. actual={writer.Position}, max={maxSize}");
        }

        private static byte[] MakeParameterBytes(RecoveryWriterParameter parameter)
        {
            using (var writer = new MemoryStream())
            {
                BinaryUtility.WriteBinary<RecoveryWriterParameter>(writer, parameter);
                FileUtility.WritePaddingToAddress(writer, ParameterSize);
                return writer.ToArray();
            }
        }

        public static RecoveryWriterParameter MakeParameter(FileInfo bctInfo, FileInfo bootLoaderInfo, FileInfo package2Info, FileInfo systemPartitionInfo, long systemPartitionAlignment)
        {
            var parameter = new RecoveryWriterParameter();

            parameter.BctOffsetOnQspi = (uint)DataOffset;
            parameter.BctSize = (uint)AlignSector(bctInfo.Length);
            parameter.BootLoaderOffsetOnQspi = (uint)(parameter.BctOffsetOnQspi + parameter.BctSize);
            parameter.BootLoaderSize = (uint)AlignSector(bootLoaderInfo.Length);
            parameter.Package2OffsetOnQspi = parameter.BootLoaderOffsetOnQspi + parameter.BootLoaderSize;
            parameter.Package2Size = (uint)AlignSector(package2Info.Length);
            parameter.SystemPartitionOffsetOnQspi = parameter.Package2OffsetOnQspi + parameter.Package2Size;
            parameter.SystemPartitionSize = (uint)Math.Max(AlignSector(systemPartitionInfo.Length), systemPartitionAlignment);

            return parameter;
        }
    }
}
