﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

namespace MakeInitialImage
{
    [Serializable()]
    [StructLayout(LayoutKind.Explicit, CharSet = CharSet.Ansi)]
    public struct InitialImageCommand
    {
        public enum CommandType
        {
            WriteGpt,
            WriteRawImage,
            UpdateGptHeader,
            WritePartitionImage,
            WriteSparsePartitionImage,
            WriteFsPartitionImage,
            WriteFsSparsePartitionImage,
            EnsureProtectProductionInfo,
            UpdateSecureInfo,
            WriteBatchedBootPartitions,
            EraseEmmcWithEscapingData,
            InitializeZero,
            EnableUpdatingFromBootImagePackage,
            WriteDatabase
        }

        [FieldOffset(0)]
        public CommandType Command;

        [FieldOffset(8)]
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 7)]
        public long[] Arguments;

        [FieldOffset(64)]
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = SizeOfStringArgument)]
        public string StringArgument;

        [FieldOffset(96)]
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = SizeOfStringArgument)]
        public string StringArgument2;

        public const int SizeOfStringArgument = 32;

        public static InitialImageCommand MakeWriteGpt(long protectiveMbr, long gptHeader, long partitionEntries, long partitionAddresses)
        {
            return MakeCommand(CommandType.WriteGpt, protectiveMbr, gptHeader, partitionEntries, partitionAddresses);
        }

        public static InitialImageCommand MakeWriteRawImageCommand(long destinationIndex, long sourceIndex)
        {
            return MakeCommand(CommandType.WriteRawImage, destinationIndex, sourceIndex);
        }

        public static InitialImageCommand MakeWritePartitionImageCommand(long partitionIndex, long sourceIndex, long offset)
        {
            return MakeCommand(CommandType.WritePartitionImage, partitionIndex, sourceIndex, offset);
        }

        public static InitialImageCommand MakeWriteSparsePartitionImageCommand(long partitionIndex, long sourceIndex, long offset)
        {
            return MakeCommand(CommandType.WriteSparsePartitionImage, partitionIndex, sourceIndex, offset);
        }

        public static InitialImageCommand MakeWriteFsPartitionImageCommand(string targetParitionName, long sourceIndex, long offset)
        {
            return MakeCommand(CommandType.WriteFsPartitionImage, targetParitionName, sourceIndex, offset);
        }

        public static InitialImageCommand MakeWriteFsSparsePartitionImageCommand(string targetParitionName, long sourceIndex, long offset)
        {
            return MakeCommand(CommandType.WriteFsSparsePartitionImage, targetParitionName, sourceIndex, offset);
        }

        public static InitialImageCommand MakeEnsureProtectProductionInfoCommand(bool allowOverwritingProductionInfo, long gptHeader, long partitionEntries, long partitionAddresses)
        {
            return MakeCommand(CommandType.EnsureProtectProductionInfo, allowOverwritingProductionInfo ? 1 : 0, gptHeader, partitionEntries, partitionAddresses);
        }

        public static InitialImageCommand MakeEraseEmmcWithEscapingDataCommand(long targetPartition1, long targetPartition2)
        {
            return MakeCommand(CommandType.EraseEmmcWithEscapingData, targetPartition1, targetPartition2);
        }

        public static InitialImageCommand MakeUpdateSecureInfoCommand(long secureInfoIndex)
        {
            return MakeCommand(CommandType.UpdateSecureInfo, secureInfoIndex);
        }

        public static InitialImageCommand MakeWriteBatchedBootPartitionsCommand(string bctParitionName, long srcBctIndex, long bctOffset, string blParitionName, long srcBlIndex, long blOffset, long secureInfoIndex)
        {
            return MakeCommand(CommandType.WriteBatchedBootPartitions, bctParitionName, blParitionName, srcBctIndex, bctOffset, srcBlIndex, blOffset, secureInfoIndex);
        }

        public static InitialImageCommand MakeInitializeZeroCmmand(string targetParitionName)
        {
            return MakeCommand(CommandType.InitializeZero, targetParitionName);
        }

        public static InitialImageCommand MakeEnableUpdatingFromBootImagePackage(long target)
        {
            return MakeCommand(CommandType.EnableUpdatingFromBootImagePackage, target);
        }

        public static InitialImageCommand MakeWriteDatabaseCommand(string targetParitionName, long sourcePartition)
        {
            return MakeCommand(CommandType.WriteDatabase, targetParitionName, sourcePartition);
        }

        private static InitialImageCommand MakeCommand(CommandType command, params long[] args)
        {
            return new InitialImageCommand
            {
                Command = command,
                Arguments = args.Concat(new long[7 - args.Count()]).ToArray()
            };
        }

        private static InitialImageCommand MakeCommand(CommandType command, string stringArgument, params long[] args)
        {
            return new InitialImageCommand
            {
                Command = command,
                Arguments = args.Concat(new long[7 - args.Count()]).ToArray(),
                StringArgument = stringArgument
            };
        }

        private static InitialImageCommand MakeCommand(CommandType command, string stringArgument, string stringArgument2, params long[] args)
        {
            return new InitialImageCommand
            {
                Command = command,
                Arguments = args.Concat(new long[7 - args.Count()]).ToArray(),
                StringArgument = stringArgument,
                StringArgument2 = stringArgument2
            };
        }
    }
}
