﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using CommandUtility;
using System.Text.RegularExpressions;

namespace CmacFileTool.Commands
{
    public class AppendCmacCommand
    {
        [CommandLineOption('i', "input",
            Description = "set raw intial image or initial image with cmac",
            IsRequired = true)]
        public string InputFile { get; set; }

        [CommandLineOption('o', "output",
            Description = "set output as initial image with cmac",
            IsRequired = true)]
        public string OutputFile { get; set; }

        [CommandLineOption("new-key",
            Description = "set new key.(binary-file-path or text-file-path or 32-hexchars)",
            IsRequired = true)]
        public string NewKey { get; set; }

        [CommandLineOption("cmac-key",
            Description = "set cmac key for test.(default: genarate key from random number)")]
        public string CmacKey { get; set; }

        [CommandLineOption("block-size",
            Description = "set block size (default = 1MB)",
            DefaultValue = 1 * 1024 * 1024)]
        public int BlockSize { get; set; }

        [CommandLineOption("signiture",
            Description = "set signiture",
            DefaultValue = "INITIMGM")]
        public string Signature { get; set; }

        public void Run()
        {
            FileInfo inputFile = new FileInfo(this.InputFile);
            FileInfo outputFile = new FileInfo(this.OutputFile);
            byte[] newKey = LoadKeyFromArgument(this.NewKey);
            byte[] cmacKey;
            if (this.CmacKey == null)
            {
                cmacKey = CryptUtility.GenerateKey();
            }
            else
            {
                cmacKey = LoadKeyFromArgument(this.CmacKey);
            }

            if (!inputFile.Exists)
            {
                throw new Exception(string.Format("found no file: {0}", inputFile.FullName));
            }

            AppendCmac(outputFile, inputFile, this.Signature, this.BlockSize, newKey, cmacKey);
        }

        public static void AppendCmac(FileInfo output, FileInfo input, string signature, int blockSize, byte[] cmacKek, byte[] cmacKey)
        {
            using (var reader = input.OpenRead())
            using (var writer = output.OpenWrite())
            {
                // ヘッダのデータを作る
                var header = new CmacFileHeader(input, signature, blockSize, cmacKek, cmacKey);

                // ヘッダを書く
                header.WriteToStream(writer);

                // ブロックごとにCMACを計算していく
                for (int blockIndex = 0; blockIndex < header.NumberOfBlocks; blockIndex++)
                {
                    var indexData = BinaryUtility.ToBinary<int>(blockIndex);
                    var buffer = new byte[header.BlockSize + indexData.Length];
                    indexData.CopyTo(buffer, 0);
                    int readSize = reader.Read(buffer, indexData.Length, header.BlockSize);

                    byte[] cmac = CmacUtility.CalculateCmac(buffer, header.CmacKey);

                    writer.Write(cmac, 0, cmac.Length);
                    writer.Write(buffer, indexData.Length, header.BlockSize);
                }
            }
        }

        public static byte[] LoadKeyFromArgument(string keyArgument)
        {
            var fileInfo = new FileInfo(keyArgument);
            var hexKeyPattern = new Regex("^[0-9A-Fa-f]{32}$");

            if (fileInfo.Exists)
            {
                if (fileInfo.Length == 16)
                {
                    using (var reader = fileInfo.OpenRead())
                    {
                        var ret = new byte[16];
                        reader.Read(ret, 0, 16);
                        return ret;
                    }
                }
                else
                {
                    using (var reader = fileInfo.OpenText())
                    {
                        var text = reader.ReadToEnd().Trim();
                        if (hexKeyPattern.IsMatch(text))
                        {
                            return BinaryUtility.FromHexString(text);
                        }
                        else
                        {
                            throw new Exception(string.Format("unknown key file format: {0}", keyArgument));
                        }
                    }
                }
            }
            else if (hexKeyPattern.IsMatch(keyArgument))
            {
                return BinaryUtility.FromHexString(keyArgument);
            }
            else
            {
                throw new Exception(string.Format("unknown key argument: {0}", keyArgument));
            }
        }
    }
}
