﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;
using CommandUtility;

namespace CmacFileTool
{
    [Serializable()]
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct CmacFileHeaderRaw
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = SizeOfSignature)]
        public string Signature;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfCmac)]
        public byte[] HeaderCmac;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SizeOfKey)]
        public byte[] EncryptedCmacKey;

        public long BlockSize;

        public long NumberOfBlocks;

        public const int SizeOfSignature = 16;
        public const int SizeOfKey = 16;
        public const int SizeOfCmac = 16;
    }

    class CmacFileHeader
    {
        public CmacFileHeader(FileInfo input, string signature, int blockSize, byte[] cmacKek, byte[] cmacKey)
        {
            this.Signature = signature;
            this.BlockSize = blockSize;
            this.NumberOfBlocks = CalculateNumberOfBlocks(input.Length, blockSize);
            this.CmacKek = cmacKek;
            this.CmacKey = cmacKey;

            var header = new CmacFileHeaderRaw()
            {
                Signature = this.Signature,
                HeaderCmac = new byte[16],
                EncryptedCmacKey = CryptUtility.EncryptedKey(this.CmacKey, this.CmacKek),
                BlockSize = this.BlockSize,
                NumberOfBlocks = this.NumberOfBlocks
            };

            header.HeaderCmac = CalculateHeaderCmac(header);

            this.HeaderRaw = header;
        }

        public CmacFileHeader(string signature, int blockSize, long numberOfBlocks, byte[] cmacKek, byte[] cmacKey)
        {
            this.Signature = signature;
            this.BlockSize = blockSize;
            this.NumberOfBlocks = numberOfBlocks;
            this.CmacKek = cmacKek;
            this.CmacKey = cmacKey;

            var header = new CmacFileHeaderRaw()
            {
                Signature = this.Signature,
                HeaderCmac = new byte[16],
                EncryptedCmacKey = CryptUtility.EncryptedKey(this.CmacKey, this.CmacKek),
                BlockSize = this.BlockSize,
                NumberOfBlocks = this.NumberOfBlocks
            };

            header.HeaderCmac = CalculateHeaderCmac(header);

            this.HeaderRaw = header;
        }

        public CmacFileHeader(CmacFileHeaderRaw headerRaw, byte[] cmacKek, bool ignoreVerify)
        {
            this.Signature = headerRaw.Signature;
            this.BlockSize = (int)headerRaw.BlockSize;
            this.NumberOfBlocks = headerRaw.NumberOfBlocks;
            this.CmacKek = cmacKek;
            this.CmacKey = CryptUtility.EncryptedKey(headerRaw.EncryptedCmacKey, cmacKek);

            var header = new CmacFileHeaderRaw()
            {
                Signature = this.Signature,
                HeaderCmac = new byte[16],
                EncryptedCmacKey = CryptUtility.EncryptedKey(this.CmacKey, this.CmacKek),
                BlockSize = this.BlockSize,
                NumberOfBlocks = this.NumberOfBlocks
            };

            header.HeaderCmac = CalculateHeaderCmac(header);

            if (!ignoreVerify)
            {
                if (!Enumerable.SequenceEqual(header.HeaderCmac, headerRaw.HeaderCmac))
                {
                    throw new Exception("Invalid cmac header. Failed to verify cmac of header.");
                }
            }

            this.HeaderRaw = header;
        }

        internal void WriteToStream(System.IO.FileStream writer)
        {
            BinaryUtility.WriteBinary<CmacFileHeaderRaw>(writer, HeaderRaw);
        }

        private byte[] CalculateHeaderCmac(CmacFileHeaderRaw header)
        {
            var binaryHeader = BinaryUtility.ToBinary<CmacFileHeaderRaw>(header);
            return CmacUtility.CalculateCmac(binaryHeader, this.CmacKey);
        }

        private long CalculateNumberOfBlocks(long fileLength, int blockSize)
        {
            return (long)((fileLength + (blockSize - 1)) / blockSize);
        }

        public string Signature { get; private set; }

        public long NumberOfBlocks { get; private set; }

        public int BlockSize { get; private set; }

        public byte[] CmacKey { get; private set; }

        public byte[] CmacKek { get; private set; }

        public CmacFileHeaderRaw HeaderRaw { get; private set; }

        public static CmacFileHeader Load(FileStream reader, byte[] cmacKek, bool ignoreVerify = false)
        {
            var headerRaw = BinaryUtility.ReadBinary<CmacFileHeaderRaw>(reader);

            return new CmacFileHeader(headerRaw, cmacKek, ignoreVerify);
        }
    }
}
