﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace Nintendo.Log
{
    class RtfEditor
    {
        private static readonly Regex RtfColorTablePattern = new Regex(@"{\\colortbl ;(\\red(?<Red>\d+)\\green(?<Green>\d+)\\blue(?<Blue>\d+);)+}", RegexOptions.Compiled);

        private static IEnumerable<Color> GetColors(string text)
        {
            var match = RtfColorTablePattern.Match(text);
            if (!match.Success)
            {
                yield break;
            }

            var reds = match.Groups["Red"].Captures.GetEnumerator();
            var greens = match.Groups["Green"].Captures.GetEnumerator();
            var blues = match.Groups["Blue"].Captures.GetEnumerator();

            while (reds.MoveNext() && greens.MoveNext() && blues.MoveNext())
            {
                yield return Color.FromArgb(
                    int.Parse(((Capture)reds.Current).Value),
                    int.Parse(((Capture)greens.Current).Value),
                    int.Parse(((Capture)blues.Current).Value));
            }
        }
        private static readonly Regex SgrCommandPattern = new Regex(@"\u001b\[(?<CommandList>\d+(;\d+)*)?m", RegexOptions.Compiled);
        private static readonly Dictionary<int, Color> AnsiColorTable = new Dictionary<int, Color>()
        {
            { 0, Color.Black },
            { 1, Color.Crimson },
            { 2, Color.LimeGreen },
            { 3, Color.Goldenrod },
            { 4, Color.RoyalBlue },
            { 5, Color.DarkMagenta },
            { 6, Color.DarkCyan },
            { 7, Color.LightGray },
        };
        private static readonly Dictionary<int, Color> AnsiiAccentColorTable = new Dictionary<int, Color>()
        {
            { 0, Color.DarkGray },
            { 1, Color.Red },
            { 2, Color.Lime },
            { 3, Color.Gold },
            { 4, Color.CornflowerBlue },
            { 5, Color.DarkViolet },
            { 6, Color.MediumTurquoise },
            { 7, Color.WhiteSmoke },
        };
        private static readonly Dictionary<int, FontStyle> FontStyleTable = new Dictionary<int, FontStyle>()
        {
            { 1, FontStyle.Bold },
            { 3, FontStyle.Italic },
            { 4, FontStyle.Underline },
            { 9, FontStyle.Strikeout }
        };

        private static string EscapeControlCharacters(string input)
        {
            return input.Replace(
                @"\", @"\\").Replace(
                "{", @"\{").Replace(
                "}", @"\}").Replace(
                "\n", @"\par ");
        }

        private static string EscapeUnicode(string input)
        {
            var sb = new StringBuilder();
            foreach (var c in input)
            {
                var code = Convert.ToInt32(c);
                if (code <= 127)
                {
                    sb.Append(c);
                }
                else
                {
                    sb.Append($@"\u{code}?");
                }
            }
            return sb.ToString();
        }

        public static string Sgr2Rtf(string input, Color foreColor, Color backColor, FontStyle fontStyle, bool force)
        {
            var colorTable = new Dictionary<Color, int>();
            colorTable[Color.Black] = 0;
            if (!colorTable.ContainsKey(DefaultForeColor))
            {
                colorTable[DefaultForeColor] = colorTable.Count;
            }
            if (!colorTable.ContainsKey(DefaultBackColor))
            {
                colorTable[DefaultBackColor] = colorTable.Count;
            }

            input = EscapeControlCharacters(input);

            input = EscapeUnicode(input);

            input = SgrCommandPattern.Replace(input, match =>
            {
                var commandList =
                    (from commandCode in match.Groups["CommandList"].Value.Split(';')
                     select string.IsNullOrEmpty(commandCode)
                         ? 0 // 空のコマンドコードはリセットコマンドとして解釈する。
                         : int.Parse(commandCode));

                return GetRtfCommand(commandList, ref foreColor, ref backColor, ref fontStyle, colorTable);
            });

            var sb = new StringBuilder(@"{\rtf ");

            sb.Append(@"{\colortbl ;");
            foreach (var color in colorTable.Keys.Skip(1))
            {
                sb.Append($@"\red{color.R}\green{color.G}\blue{color.B};");
            }
            sb.Append('}');
            if (force)
            {
                sb.Append($@"\cf{colorTable[DefaultForeColor]}\highlight{colorTable[DefaultBackColor]} ");
            }
            sb.Append(input);
            sb.Append('}');

            return sb.ToString();
        }

        public static Color DefaultForeColor = Color.Black;
        public static Color DefaultBackColor = Color.White;

        private static int? Enumerate8BitValue(IEnumerator<int> enumerator)
        {
            if (!enumerator.MoveNext())
            {
                return null;
            }

            if (!(0 <= enumerator.Current && enumerator.Current < 256))
            {
                return null;
            }

            return enumerator.Current;
        }

        private static Color? Enumerate24BitColor(IEnumerator<int> enumerator)
        {
            int? red = Enumerate8BitValue(enumerator);
            if (!red.HasValue)
            {
                return null;
            }

            int? green = Enumerate8BitValue(enumerator);
            if (!green.HasValue)
            {
                return null;
            }

            int? blue = Enumerate8BitValue(enumerator);
            if (!blue.HasValue)
            {
                return null;
            }

            return Color.FromArgb(red.Value, green.Value, blue.Value);
        }

        private static void InterpretCommand(
            IEnumerable<int> commands, ref Color foreColor, ref Color backColor, ref FontStyle fontStyle)
        {
            var enumerator = commands.GetEnumerator();
            while (enumerator.MoveNext())
            {
                var command = enumerator.Current;
                if (command == 0)
                {
                    foreColor = DefaultForeColor;
                    backColor = DefaultBackColor;
                    fontStyle = FontStyle.Regular;
                }
                else if (FontStyleTable.ContainsKey(command))
                {
                    fontStyle |= FontStyleTable[command];
                }
                else if (command == 7)
                {
                    var temporary = foreColor;
                    foreColor = backColor;
                    backColor = temporary;
                }
                else if (30 <= command && command <= 37)
                {
                    foreColor = AnsiColorTable[command - 30];
                }
                else if (command == 38)
                {
                    if (!enumerator.MoveNext())
                    {
                        return;
                    }
                    if (enumerator.Current == 5) // 前景色の 8 bit カラー指定。(未対応）
                    {
                        if (!enumerator.MoveNext()) // TORIAEZU: カラー指定を読み飛ばす。
                        {
                            return;
                        }
                    }
                    else if (enumerator.Current == 2) // 前景色の 24 bit カラー指定。
                    {
                        var color = Enumerate24BitColor(enumerator);
                        if (!color.HasValue)
                        {
                            return;
                        }
                        foreColor = color.Value;
                    }
                    else
                    {
                        return;
                    }
                }
                else if (command == 39)
                {
                    foreColor = DefaultForeColor;
                }
                else if (40 <= command && command <= 47)
                {
                    backColor = AnsiColorTable[command - 40];
                }
                else if (command == 48)
                {
                    if (!enumerator.MoveNext())
                    {
                        return;
                    }
                    if (enumerator.Current == 5) // 背景色の 8 bit カラー指定。(未対応）
                    {
                        if (!enumerator.MoveNext()) // TORIAEZU: カラー指定を読み飛ばす。
                        {
                            return;
                        }
                    }
                    else if (enumerator.Current == 2) // 背景色の 24 bit カラー指定。
                    {
                        var color = Enumerate24BitColor(enumerator);
                        if (!color.HasValue)
                        {
                            return;
                        }
                        backColor = color.Value;
                    }
                    else
                    {
                        return;
                    }
                }
                else if (command == 49)
                {
                    backColor = DefaultBackColor;
                }
                else if (90 <= command && command <= 97)
                {
                    foreColor = AnsiiAccentColorTable[command - 90];
                }
                else if (100 <= command && command <= 107)
                {
                    backColor = AnsiiAccentColorTable[command - 100];
                }
            }
        }

        private static string GetRtfCommand(
            IEnumerable<int> commands, ref Color foreColor, ref Color backColor, ref FontStyle fontStyle, Dictionary<Color, int> colorTable)
        {
            var currentForeColor = foreColor;
            var currentBackColor = backColor;
            var currentFontStyle = fontStyle;

            InterpretCommand(commands, ref foreColor, ref backColor, ref fontStyle);

            var sb = new StringBuilder();

            if (foreColor != currentForeColor)
            {
                if (!colorTable.ContainsKey(foreColor))
                {
                    colorTable[foreColor] = colorTable.Count;
                }
                sb.Append($@"\cf{colorTable[foreColor]}");
            }

            if (backColor != currentBackColor)
            {
                if (!colorTable.ContainsKey(backColor))
                {
                    colorTable[backColor] = colorTable.Count;
                }
                sb.Append($@"\highlight{colorTable[backColor]}");
            }

            if ((fontStyle & FontStyle.Bold) != (currentFontStyle & FontStyle.Bold))
            {
                sb.Append((fontStyle & FontStyle.Bold) != 0 ? @"\b" : @"\b0");
            }

            if ((fontStyle & FontStyle.Italic) != (currentFontStyle & FontStyle.Italic))
            {
                sb.Append((fontStyle & FontStyle.Italic) != 0 ? @"\i" : @"\i0");
            }

            if ((fontStyle & FontStyle.Underline) != (currentFontStyle & FontStyle.Underline))
            {
                sb.Append((fontStyle & FontStyle.Underline) != 0 ? @"\ul" : @"\ulnone");
            }

            if ((fontStyle & FontStyle.Strikeout) != (currentFontStyle & FontStyle.Strikeout))
            {
                sb.Append((fontStyle & FontStyle.Strikeout) != 0 ? @"\strike" : @"\strike0");
            }

            if (sb.Length > 0)
            {
                sb.Append(' ');
                return sb.ToString();
            }
            else
            {
                return string.Empty;
            }
        }
    }
}
