﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml.Linq;

namespace Nintendo.Log
{
    internal class LogViewerWindow : Form
    {
        public LogViewerWindow()
        {
            TabControl = new TabControl();
            TabControl.Dock = DockStyle.Fill;
            TabControl.Margin = new Padding(0);
            TabControl.Font = UIFont;
            TabControl.MouseClick += TabControl_MouseClick;
            TabControl.MouseDown += TabControl_MouseDown;
            TabControl.SelectedIndexChanged += TabControl_SelectedIndexChanged;

            var newTab = new TabPage("(new)");
            TabControl.Controls.Add(newTab);

            TabCloseMenuItem = new ToolStripMenuItem("Close");
            TabCloseMenuItem.Click += TabCloseMenuItem_Click;

            TabRenameMenuItem = new ToolStripMenuItem("Rename");
            TabRenameMenuItem.Click += TabRenameMenuItem_Click;

            SaveLogWithFilterMenuItem = new ToolStripMenuItem();
            SaveLogWithFilterMenuItem.Click += SaveLogWithFilter_Click;

            SaveJsonLogWithFilterMenuItem = new ToolStripMenuItem();
            SaveJsonLogWithFilterMenuItem.Click += SaveJsonLogWithFilter_Click;

            TabContextMenuStrip = new ContextMenuStrip();
            TabContextMenuStrip.Items.Add(TabCloseMenuItem);
            TabContextMenuStrip.Items.Add(TabRenameMenuItem);
            TabContextMenuStrip.Items.Add(SaveLogWithFilterMenuItem);
            TabContextMenuStrip.Items.Add(SaveJsonLogWithFilterMenuItem);

            if (WindowList.Count == 0)
            {
                var openLogMenuItem = new ToolStripMenuItem("Open log file");
                SetupOpenLogHandler(openLogMenuItem);

                var convertLogMenuItem = new ToolStripMenuItem("Convert NX binary log");
                SetupConvertLogHandler(convertLogMenuItem);

                var saveLogMenuItem = new ToolStripMenuItem("Save Log as plain text");
                SetupSaveLogHandler(saveLogMenuItem, "txt", LogClient.CreateSavePlainTextLogTask);

                var saveLogAsJsonMenuItem = new ToolStripMenuItem("Save Log as JSON");
                SetupSaveLogHandler(saveLogAsJsonMenuItem, "json", LogClient.CreateSaveJsonLogTask);

                var newWindowMenuItem = new ToolStripMenuItem("Open new window");
                newWindowMenuItem.Click += NewWindowMenuItem_Click;
                newWindowMenuItem.ShortcutKeys = Keys.Control | Keys.N;

                var findMenuItem = new ToolStripMenuItem("Find");
                findMenuItem.Click += FindMenuItem_Click;
                findMenuItem.ShortcutKeys = Keys.Control | Keys.F;

                var coloringMenuItem = new ToolStripMenuItem("Coloring by log severity");
                coloringMenuItem.Click += ColoringMenuItem_Click;

                var appearanceMenuItem = new ToolStripMenuItem("Appearance");
                appearanceMenuItem.Click += AppearanceMenuItem_Click;

                var menuItem = new ToolStripMenuItem("Menu");
                menuItem.DropDownItems.Add(openLogMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(convertLogMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(saveLogMenuItem);
                menuItem.DropDownItems.Add(saveLogAsJsonMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(newWindowMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(findMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(MakeMetaInfoMenuItem());
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(coloringMenuItem);
                menuItem.DropDownItems.Add("-");
                menuItem.DropDownItems.Add(appearanceMenuItem);

                var menuStrip = new MenuStrip();
                menuStrip.Dock = DockStyle.Fill;
                menuStrip.Items.Add(menuItem);

                StatusLabel = new ToolStripStatusLabel();

                var statusStrip = new StatusStrip();
                statusStrip.Dock = DockStyle.Fill;
                statusStrip.Height = statusStrip.PreferredSize.Height;
                statusStrip.Items.Add(StatusLabel);

                var tableLayoutPanel = new TableLayoutPanel();
                tableLayoutPanel.Dock = DockStyle.Fill;
                tableLayoutPanel.RowCount = 3;
                tableLayoutPanel.ColumnCount = 1;
                tableLayoutPanel.RowStyles.Add(new RowStyle(SizeType.Absolute, menuStrip.Height));
                tableLayoutPanel.RowStyles.Add(new RowStyle(SizeType.Percent, 100));
                tableLayoutPanel.RowStyles.Add(new RowStyle(SizeType.Absolute, statusStrip.Height));
                tableLayoutPanel.Controls.Add(menuStrip, 0, 0);
                tableLayoutPanel.Controls.Add(TabControl, 0, 1);
                tableLayoutPanel.Controls.Add(statusStrip, 0, 2);
                this.Controls.Add(tableLayoutPanel);

                SetupDragAndDropFileOpen(); // Drag & Drop 機能は STA でしか利用できないため。
            }
            else
            {
                this.Controls.Add(TabControl);
            }

            this.Text = "Nintendo Log Viewer";
            this.ShowIcon = false;
            this.Opacity = WindowOpacity / 100.0;
            LogViewerWindow.WindowOpacityChanged += LogViewerWindow_WindowOpacityChanged;
            this.FormClosed += LogViewerWindow_FormClosed;
            this.KeyPreview = true;
            this.KeyDown += LogViewerWindow_KeyDown;

            WindowList.Add(this);
        }

        public LogViewerWindow(LogViewerTab tab)
            : this()
        {
            AddTab(tab);
        }

        private ToolStripMenuItem MakeMetaInfoMenuItem()
        {
            var timestampMenu = new ToolStripMenuItem("Timestamp");
            timestampMenu.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsTimestampShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var userSystemClockMenu = new ToolStripMenuItem("UserSystemClock");
            userSystemClockMenu.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsUserSystemClockShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var peerNameItem = new ToolStripMenuItem("PeerName");
            peerNameItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsPeerNameShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var processNameItem = new ToolStripMenuItem("ProcessName");
            processNameItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsProcessNameShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var moduleNameItem = new ToolStripMenuItem("ModuleName");
            moduleNameItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsModuleNameShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var threadNameItem = new ToolStripMenuItem("ThreadName");
            threadNameItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsThreadNameShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var functionNameItem = new ToolStripMenuItem("FunctionName");
            functionNameItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsFunctionNameShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var severityItem = new ToolStripMenuItem("Severity");
            severityItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsSeverityShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var verbosityItem = new ToolStripMenuItem("Verbosity");
            verbosityItem.Click += (sender, e) =>
            {
                var item = (ToolStripMenuItem)sender;
                LogViewerTab.IsVerbosityShown = item.Checked = !item.Checked;
                NotifyStyleChanged();
            };

            var metaInfoItem = new ToolStripMenuItem("Meta info");
            metaInfoItem.DropDownItems.Add(timestampMenu);
            metaInfoItem.DropDownItems.Add(userSystemClockMenu);
            metaInfoItem.DropDownItems.Add(processNameItem);
            metaInfoItem.DropDownItems.Add(peerNameItem);
            metaInfoItem.DropDownItems.Add(moduleNameItem);
            metaInfoItem.DropDownItems.Add(threadNameItem);
            metaInfoItem.DropDownItems.Add(functionNameItem);
            metaInfoItem.DropDownItems.Add(severityItem);
            metaInfoItem.DropDownItems.Add(verbosityItem);

            return metaInfoItem;
        }

        private static List<LogViewerWindow> WindowList = new List<LogViewerWindow>();

        public static LogViewerWindow GetMainWindow()
        {
            return LogViewerWindow.WindowList.First();
        }

        private static string SettingsFilePath
        {
            get
            {
                return Path.Combine(Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), "settings.xml");
            }
        }

        public static void SaveLayout()
        {
            var settingsRoot = new XElement("LogViewerSettings");

            var appearanceRoot = new XElement("Appearance");
            settingsRoot.Add(appearanceRoot);
            appearanceRoot.SetAttributeValue("FontFace", LogViewerTab.LogTextFont.Name);
            appearanceRoot.SetAttributeValue("FontSize", Convert.ToInt32(LogViewerTab.LogTextFont.Size));
            appearanceRoot.SetAttributeValue("FontForeColor", ColorTranslator.ToHtml(Color.FromArgb(LogViewerTab.LogTextForeColor.ToArgb()))); // 色名で保存されないように、一旦 RGB に変換している
            appearanceRoot.SetAttributeValue("FontBackColor", ColorTranslator.ToHtml(Color.FromArgb(LogViewerTab.LogTextBackColor.ToArgb())));
            appearanceRoot.SetAttributeValue("WindowOpacity", LogViewerWindow.WindowOpacity);

            var layoutRoot = new XElement("Layout");
            settingsRoot.Add(layoutRoot);
            foreach (LogViewerWindow window in WindowList)
            {
                var windowRoot = new XElement("Window");
                layoutRoot.Add(windowRoot);
                var bounds = window.WindowState == FormWindowState.Normal ? window.Bounds : window.RestoreBounds;
                var boundsString = $"{bounds.X}, {bounds.Y}, {bounds.Width}, {bounds.Height}";
                windowRoot.SetAttributeValue("Bounds", boundsString);
                if (window.WindowState == FormWindowState.Maximized)
                {
                    windowRoot.SetAttributeValue("Maximized", "true");
                }
                foreach (TabPage tab in window.TabControl.TabPages)
                {
                    if (tab is LogViewerClientTab) // new タブをレイアウトに保存しないため
                    {
                        var viewerTab = (LogViewerClientTab)tab;
                        var tabRoot = new XElement("Tab");
                        windowRoot.Add(tabRoot);
                        tabRoot.SetAttributeValue("Name", viewerTab.Text);
                        tabRoot.SetAttributeValue("Filter", viewerTab.Filter);
                    }
                }
            }
            settingsRoot.Save(SettingsFilePath);
        }

        public static void LoadLayout()
        {
            var settingsRoot = XElement.Load(SettingsFilePath);

            var appearanceRoot = settingsRoot.Element("Appearance");
            if (appearanceRoot != null)
            {
                var fontFace = appearanceRoot.Attribute("FontFace").Value;
                var fontSize = float.Parse(appearanceRoot.Attribute("FontSize").Value);
                LogViewerTab.LogTextFont = new Font(fontFace, fontSize);
                LogViewerTab.LogTextForeColor = ColorTranslator.FromHtml(appearanceRoot.Attribute("FontForeColor").Value);
                LogViewerTab.LogTextBackColor = ColorTranslator.FromHtml(appearanceRoot.Attribute("FontBackColor").Value);
                LogViewerWindow.WindowOpacity = Convert.ToInt32(appearanceRoot.Attribute("WindowOpacity").Value);
            }

            var layoutRoot = settingsRoot.Element("Layout");
            foreach (var windowRoot in layoutRoot.Elements("Window"))
            {
                var boundsValue =
                    windowRoot.Attribute("Bounds").Value.Split(',').Select(str => int.Parse(str)).ToArray();
                var bounds = new Rectangle(boundsValue[0], boundsValue[1], boundsValue[2], boundsValue[3]);
                var window = new LogViewerWindow();
                window.StartPosition = FormStartPosition.Manual;
                CorrectWindowBounds(ref bounds);
                window.Bounds = bounds;
                if (windowRoot.Attributes().Where(e => e.Name == "Maximized" && e.Value == "true").Any())
                {
                    window.WindowState = FormWindowState.Maximized;
                }
                foreach (var tabRoot in windowRoot.Elements("Tab"))
                {
                    var name = tabRoot.Attribute("Name").Value;
                    var filter = tabRoot.Attribute("Filter").Value;
                    var tab = new LogViewerClientTab(name);
                    tab.Filter = filter;
                    window.AddTab(tab);
                }

                if (window != GetMainWindow())
                {
                    Task.Factory.StartNew(window.ShowDialog);
                }
            }
        }

        /// <summary>
        /// ウィンドウがスクリーンに収まるように、ウィンドウのレイアウトを補正します。
        ///
        /// ウィンドウサイズが、スクリーンサイズ以下のときは、位置のみ調整します。
        /// そうでないときは、ウィンドウの位置を調整したうえで、サイズをスクリーンに合わせます。
        /// </summary>
        private static void CorrectWindowBounds(ref Rectangle bounds, int threshold = 30)
        {
            var screenBounds = Screen.FromRectangle(bounds).WorkingArea;

            if (bounds.Width <= screenBounds.Width)
            {
                if (bounds.Right < screenBounds.Left + threshold)
                {
                    bounds.X = screenBounds.Left;
                }
                else if (bounds.Left > screenBounds.Right - threshold)
                {
                    bounds.X = screenBounds.Right - bounds.Width;
                }
            }
            else
            {
                bounds.X = screenBounds.Left;
                bounds.Width = screenBounds.Width;
            }

            if (bounds.Height <= screenBounds.Height)
            {
                if (bounds.Bottom < screenBounds.Top + threshold)
                {
                    bounds.Y = screenBounds.Top;
                }
                else if (bounds.Top > screenBounds.Bottom - threshold)
                {
                    bounds.Y = screenBounds.Bottom - bounds.Height;
                }
            }
            else
            {
                bounds.Y = screenBounds.Top;
                bounds.Height = screenBounds.Height;
            }
        }

        private void AddTab(LogViewerTab tab) // new タブの手前にタブを追加する
        {
            var lastTabIndex = TabControl.TabPages.Count - 1;
            var newTab = TabControl.TabPages[lastTabIndex];
            newTab.ContextMenuStrip = TabContextMenuStrip;
            TabControl.Controls.Remove(newTab);
            TabControl.Controls.Add(tab);
            TabControl.Controls.Add(newTab);
        }

        private void RemoveTab(LogViewerTab tab)
        {
            TabControl.TabPages.Remove(tab);
            tab.Dispose();
            if (TabControl.TabPages.Count == 1)
            {
                this.Close();
            }
        }

        private void LogViewerWindow_FormClosed(object sender, FormClosedEventArgs e)
        {
            if (this == WindowList.First())
            {
                SaveLayout();
            }

            foreach (TabPage tab in TabControl.TabPages)
            {
                if (tab is LogViewerTab)
                {
                    var viewerTab = (LogViewerTab)tab;
                    viewerTab.Dispose();
                }
            }
            TabControl.TabPages.Clear();

            WindowOpacityChanged -= LogViewerWindow_WindowOpacityChanged;
            WindowList.Remove(this);
        }

        private void LogViewerWindow_KeyDown(object sender, KeyEventArgs e)
        {
            // Ctrl + Tab で new タブが選択されないようにするため、自前でタブ移動を実装する
            if (e.Control && e.KeyCode == Keys.Tab)
            {
                // new タブは含めないため、最後のタブのインデックスは TabCount - 2
                TabControl.SelectedIndex = e.Shift
                    ? TabControl.SelectedIndex == 0
                        ? TabControl.TabCount - 2
                        : TabControl.SelectedIndex - 1
                    : TabControl.SelectedIndex == (TabControl.TabCount - 2)
                        ? 0
                        : TabControl.SelectedIndex + 1;

                e.Handled = true;
            }
        }

        private void OpenFindBox()
        {
            ((LogViewerTab)TabControl.SelectedTab).OpenFindBox();
        }

        private static string StripInvalidFileNameChars(string text)
        {
            var copyText = string.Copy(text);
            foreach (var c in System.IO.Path.GetInvalidFileNameChars())
            {
                copyText = copyText.Replace(c.ToString(), "");
            }
            return copyText;
        }

        internal static string GetPathWithOpenFileDialog(string filter)
        {
            using (var fileDialog = new OpenFileDialog())
            {
                fileDialog.Filter = filter;
                fileDialog.CheckFileExists = true;
                return
                    fileDialog.ShowDialog() == DialogResult.OK
                        ? fileDialog.FileName
                        : string.Empty;
            }
        }

        public static bool IsSupportedFormat(string path)
        {
            var extension = Path.GetExtension(path).ToLower();
            return false
                || extension == ".nxbinlog"
                || extension == ".json";
        }

        public static bool IsLogFilePath(string path)
        {
            return File.Exists(path) && IsSupportedFormat(path);
        }

        public void OpenLogFile(string path)
        {
            var lastTabIndex = TabControl.TabPages.Count - 1;
            var tab = new LogViewerFileTab(path, path);
            AddTab(tab);
            TabControl.SelectedIndex = lastTabIndex;
        }

        private void SetupOpenLogHandler(ToolStripMenuItem menuItem)
        {
            menuItem.Click += (sender, e) =>
            {
                var path = GetPathWithOpenFileDialog("Log File (*.nxbinlog, *.json)|*.nxbinlog;*.json");
                if (string.IsNullOrEmpty(path))
                {
                    return;
                }

                OpenLogFile(path);
            };
        }

        private void SetupDragAndDropFileOpen()
        {
            AllowDrop = true;

            DragEnter += (sender, e) =>
            {
                if (!e.Data.GetDataPresent(DataFormats.FileDrop))
                {
                    return;
                }

                if (!((string[])e.Data.GetData(DataFormats.FileDrop)).Where(path => IsLogFilePath(path)).Any())
                {
                    return;
                }

                e.Effect = DragDropEffects.Copy;
            };

            DragDrop += (sender, e) =>
            {
                foreach (var path in ((string[])e.Data.GetData(DataFormats.FileDrop)).Where(path => IsLogFilePath(path)))
                {
                    OpenLogFile(path);
                }
            };
        }

        private void SetupConvertLogHandler(ToolStripMenuItem menuItem)
        {
            menuItem.Click += (sender, e) =>
            {
                var path = GetPathWithOpenFileDialog("NX binary log (*.nxbinlog)|*.nxbinlog");
                if (string.IsNullOrEmpty(path))
                {
                    return;
                }

                var outPath = string.Empty;
                using (var dialog = new SaveFileDialog())
                {
                    dialog.FileName = Path.GetFileName(path) + ".txt";
                    dialog.Filter = "|*.txt||*.json";
                    dialog.OverwritePrompt = true;
                    if (dialog.ShowDialog() != DialogResult.OK)
                    {
                        return;
                    }
                    outPath = dialog.FileName;
                }

                var cs = new CancellationTokenSource();

                using (var progressBox = new ProgressBox("Converting...", $"{path}{Environment.NewLine}-> {outPath}"))
                {
                    int progressCallbackCallCount = 0;
                    const int progressBarUpdateRatio = 10000;
                    Action<long, long> progressbarUpdater = (totalProcessSize, fileSize) =>
                    {
                        try
                        {
                            progressCallbackCallCount++;
                            if (progressCallbackCallCount % progressBarUpdateRatio == 0)
                            {
                                progressBox.SetProgress((double)totalProcessSize / fileSize);
                            }
                        }
                        catch (ObjectDisposedException)
                        {
                            System.Diagnostics.Debug.Assert(cs.IsCancellationRequested);
                        }
                    };

                    var task = Path.GetExtension(outPath).ToLower() == ".json"
                        ? LogClient.CreateNxbinlogToJsonTask(outPath, path, cs.Token, progressbarUpdater)
                        : LogClient.CreateNxbinlogToTextTask(outPath, path, cs.Token, progressbarUpdater);
                    task.ContinueWith((t) =>
                    {
                        if (!cs.IsCancellationRequested)
                        {
                            Invoke((MethodInvoker)(() =>
                            {
                                progressBox.DialogResult = DialogResult.OK;
                            }));
                        }
                    });
                    task.Start();

                    if (progressBox.ShowDialog() == DialogResult.Cancel)
                    {
                        cs.Cancel();
                    }
                }
            };
        }

        internal static string GetPathWithSaveFileDialog(string defaultName, string defaultExtention, bool withTimestamp)
        {
            using (var fileDialog = new SaveFileDialog())
            {
                fileDialog.FileName =
                    StripInvalidFileNameChars(
                        withTimestamp
                            ? $"{defaultName}_{DateTime.Now.ToString("yyyy-MM-dd_HH-mm-ss")}.{defaultExtention}"
                            : $"{defaultName}.{defaultExtention}");
                fileDialog.CheckFileExists = false;
                return
                    fileDialog.ShowDialog() == DialogResult.OK
                        ? fileDialog.FileName
                        : string.Empty;
            }
        }

        private void SetupSaveLogHandler(
            ToolStripMenuItem menuItem, string extention,
            Func<string, CancellationToken, Predicate<Dictionary<string, object>>, Task> saveTaskCreator)
        {
            var cancellationTokenSource = new CancellationTokenSource();
            var saveLogTask = Task.CompletedTask;
            var originalText = string.Copy(menuItem.Text);

            menuItem.Click += (sender, e) =>
            {
                if (saveLogTask.IsCompleted)
                {
                    var path = GetPathWithSaveFileDialog("log", extention, true);
                    if (string.IsNullOrEmpty(path))
                    {
                        return;
                    }

                    saveLogTask = saveTaskCreator(path, cancellationTokenSource.Token, log => true);
                    saveLogTask.Start();
                    menuItem.Text = "Stop saving - " + path;
                }
                else
                {
                    cancellationTokenSource.Cancel(true);
                    cancellationTokenSource.Dispose(); // 一度使用したら作り直す必要がある
                    cancellationTokenSource = new CancellationTokenSource();
                    saveLogTask.Wait();
                    menuItem.Text = originalText;
                }
            };
        }

        private void NewWindowMenuItem_Click(object sender, EventArgs e)
        {
            var window = new LogViewerWindow(new LogViewerClientTab());
            Task.Factory.StartNew(window.ShowDialog);
        }

        private void FindMenuItem_Click(object sender, EventArgs e)
        {
            OpenFindBox();
        }

        private void TabCloseMenuItem_Click(object sender, EventArgs e)
        {
            var clickedTab = (LogViewerTab)((ToolStripMenuItem)sender).Tag;
            RemoveTab(clickedTab);
        }

        private void TabRenameMenuItem_Click(object sender, EventArgs e)
        {
            var clickedTab = (LogViewerTab)((ToolStripMenuItem)sender).Tag;
            var newName = clickedTab.Text;
            var dialogResult = InputBox("Rename", "Enter new tab name.", ref newName);
            if (dialogResult == DialogResult.OK)
            {
                clickedTab.Text = newName;
            }
        }

        void SaveLogWithFilter_Click(object sender, EventArgs e)
        {
            var tab = ((LogViewerTab)((ToolStripMenuItem)sender).Tag);
            tab.StartOrStopSavingLogWithFilter();
        }

        void SaveJsonLogWithFilter_Click(object sender, EventArgs e)
        {
            var tab = ((LogViewerTab)((ToolStripMenuItem)sender).Tag);
            tab.StartOrStopSavingJsonLogWithFilter();
        }

        private void ColoringMenuItem_Click(object sender, EventArgs e)
        {
            var item = (ToolStripMenuItem)sender;
            LogViewerTab.IsColoringBySeverityEnabled = item.Checked = !item.Checked;
            NotifyStyleChanged();
        }

        private void AppearanceMenuItem_Click(object sender, EventArgs e)
        {
            AppearanceBox(NotifyStyleChanged);
        }

        private LogViewerTab GetTabByPosition(Point position)
        {
            for (var i = 0; i < TabControl.TabCount - 1; i++) // new タブを除外するため -1 する
            {
                if (TabControl.GetTabRect(i).Contains(position))
                {
                    return (LogViewerTab)TabControl.TabPages[i];
                }
            }
            return null;
        }

        private void TabControl_SelectedIndexChanged(object sender, EventArgs e)
        {
            (TabControl.SelectedTab as LogViewerTab)?.UpdateFull();
        }

        private void NotifyStyleChanged()
        {
            (TabControl.SelectedTab as LogViewerTab)?.UpdateFull();
        }

        private void TabControl_MouseClick(object sender, MouseEventArgs e)
        {
            var clickedTab = GetTabByPosition(e.Location);
            if (clickedTab == null) // タブ以外の部分がクリックされた
            {
                return;
            }

            if (e.Button == MouseButtons.Middle)
            {
                if (this == WindowList.First() && TabControl.TabPages.Count == 2)
                {
                    // メインウィンドウに、クリックされたタブと new タブしか残っていないときは何もしない
                    return;
                }
                RemoveTab(clickedTab);
            }
            else if (e.Button == MouseButtons.Right)
            {
                foreach (ToolStripMenuItem menuItem in TabContextMenuStrip.Items)
                {
                    // クリックされたタブが分かるように、メニューアイテムのタグに格納する
                    menuItem.Tag = clickedTab;
                }

                // メインウィンドウに、クリックされたタブと new タブしか残っていないときは Close を無効化する
                TabCloseMenuItem.Enabled =
                    (this != WindowList.First() || TabControl.TabPages.Count > 2);

                SaveLogWithFilterMenuItem.Text =
                    clickedTab.IsLogSaving
                        ? "Stop saving - " + clickedTab.LogFilePath
                        : "Save Log as plain text with the filter";

                SaveJsonLogWithFilterMenuItem.Text =
                    clickedTab.IsJsonLogSaving
                        ? "Stop saving - " + clickedTab.JsonLogFilePath
                        : "Save Log as JSON with the filter";

                TabContextMenuStrip.Show(TabControl, e.Location);
            }
        }

        private void TabControl_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            var lastTabIndex = TabControl.TabPages.Count - 1;
            if (TabControl.GetTabRect(lastTabIndex).Contains(e.Location))
            {
                AddTab(new LogViewerClientTab());
                TabControl.SelectedIndex = lastTabIndex;
            }
        }

        public static int WindowOpacity
        {
            get { return WindowOpacityValue; }
            set
            {
                WindowOpacityValue = Math.Max(Math.Min(value, 100), 30);
                WindowOpacityChanged?.Invoke(WindowOpacityValue);
            }
        }
        private static int WindowOpacityValue = 100;

        private delegate void ChangeOpacityEvent(double opacity);
        private static ChangeOpacityEvent WindowOpacityChanged;

        private void LogViewerWindow_WindowOpacityChanged(double opacity)
        {
            Invoke((MethodInvoker)(() =>
            {
                this.Opacity = opacity / 100;
            }));
        }

        private static DialogResult InputBox(string title, string prompt, ref string value)
        {
            const int Margin = 10;

            using (var promptLabel = new Label())
            using (var textBox = new TextBox())
            using (var cancelButton = new Button())
            using (var okButton = new Button())
            using (var form = new Form())
            {
                promptLabel.Text = prompt;
                promptLabel.Location = new Point(Margin, Margin);
                promptLabel.AutoSize = true;

                textBox.Text = value;
                textBox.Location = new Point(Margin, promptLabel.Bottom + Margin);
                textBox.Width = 300;

                cancelButton.Text = "Cancel";
                cancelButton.DialogResult = DialogResult.Cancel;
                cancelButton.Location = new Point(textBox.Right - cancelButton.Width, textBox.Bottom + Margin);

                okButton.Text = "OK";
                okButton.DialogResult = DialogResult.OK;
                okButton.Location = new Point(cancelButton.Left - Margin - okButton.Width, cancelButton.Top);

                form.Text = title;
                form.ClientSize = new Size(textBox.Width + Margin * 2, okButton.Bottom + Margin);
                form.FormBorderStyle = FormBorderStyle.FixedDialog;
                form.StartPosition = FormStartPosition.CenterScreen;
                form.MinimizeBox = false;
                form.MaximizeBox = false;
                form.AcceptButton = okButton;
                form.CancelButton = cancelButton;
                form.Controls.AddRange(new Control[] { promptLabel, textBox, okButton, cancelButton });

                var dialogResult = form.ShowDialog();
                value = textBox.Text;
                return dialogResult;
            }
        }

        private static bool IsFixedFont(FontFamily family)
        {
            if (family.IsStyleAvailable(FontStyle.Regular))
            {
                using (var font = new Font(family, 10))
                {
                    var diff = TextRenderer.MeasureText("WWW", font).Width - TextRenderer.MeasureText("...", font).Width;
                    return (Math.Abs(diff) < float.Epsilon * 2);
                }
            }
            return false;
        }

        private static void AppearanceBox(Action updateFunc)
        {
            const int Margin = 10;
            const int FontSizeMin = 6;
            const int FontSizeMax = 24;

            var selectedFont = LogViewerTab.LogTextFont;
            var selectedForeColor = LogViewerTab.LogTextForeColor;
            var selectedBackColor = LogViewerTab.LogTextBackColor;
            var selectedOpacity = LogViewerWindow.WindowOpacity;

            using (var fontGroup = new GroupBox())
            using (var fontNameBox = new ComboBox())
            using (var fontSizeBox = new ComboBox())
            using (var foreColorButton = new Button())
            using (var backColorButton = new Button())
            using (var opacityGroup = new GroupBox())
            using (var opacityBar = new TrackBar())
            using (var opacityLabel = new Label())
            using (var previewGroup = new GroupBox())
            using (var previewLabel = new Label())
            using (var cancelButton = new Button())
            using (var okButton = new Button())
            using (var form = new Form())
            {
                fontGroup.Text = "Font";
                var groupTopMargin = Margin + (TextRenderer.MeasureText(fontGroup.Text, fontGroup.Font).Height / 2);

                var fixedFontNames = (from family in FontFamily.Families where IsFixedFont(family) select family.Name).ToArray();
                fontNameBox.Items.AddRange(fixedFontNames);
                var selectedFontNameIndex = Array.IndexOf(fixedFontNames, selectedFont.Name);
                fontNameBox.SelectedIndex = selectedFontNameIndex >= 0 ? selectedFontNameIndex : 0;
                fontNameBox.Location = new Point(Margin, groupTopMargin);
                fontNameBox.Width = 200;
                fontNameBox.DropDownStyle = ComboBoxStyle.DropDownList;
                fontNameBox.SelectedIndexChanged += (sender, e) =>
                {
                    previewLabel.Font = new Font(fontNameBox.Text, float.Parse(fontSizeBox.Text));
                    LogViewerTab.LogTextFont = new Font(fontNameBox.Text, float.Parse(fontSizeBox.Text));
                    updateFunc?.Invoke();
                };

                var fontSizes = (from size in Enumerable.Range(FontSizeMin, FontSizeMax) select size.ToString()).ToArray();
                fontSizeBox.Items.AddRange(fontSizes);
                var selectedFontSizeIndex = Array.IndexOf(fontSizes, Convert.ToInt32(selectedFont.Size).ToString());
                fontSizeBox.SelectedIndex = selectedFontSizeIndex >= 0 ? selectedFontSizeIndex : 0;
                fontSizeBox.Location = new Point(fontNameBox.Right + Margin, fontNameBox.Top);
                fontSizeBox.Width = 50;
                fontSizeBox.DropDownStyle = ComboBoxStyle.DropDownList;
                fontSizeBox.SelectedIndexChanged += (sender, e) =>
                {
                    previewLabel.Font = new Font(fontNameBox.Text, float.Parse(fontSizeBox.Text));
                    LogViewerTab.LogTextFont = new Font(fontNameBox.Text, float.Parse(fontSizeBox.Text));
                    updateFunc?.Invoke();
                };

                foreColorButton.Text = "Fore Color";
                foreColorButton.Location = new Point(fontNameBox.Left, fontNameBox.Bottom + Margin);
                foreColorButton.Size = new Size((fontNameBox.Width + fontSizeBox.Width) / 2, fontNameBox.Height);
                foreColorButton.Click += (sender, e) =>
                {
                    using (var colorDialog = new ColorDialog())
                    {
                        colorDialog.Color = previewLabel.ForeColor;
                        if (colorDialog.ShowDialog() == DialogResult.OK)
                        {
                            previewLabel.ForeColor = colorDialog.Color;
                            LogViewerTab.LogTextForeColor = colorDialog.Color;
                            updateFunc?.Invoke();
                        }
                    }
                };

                backColorButton.Text = "Back Color";
                backColorButton.Location = new Point(foreColorButton.Right + Margin, foreColorButton.Top);
                backColorButton.Size = new Size(foreColorButton.Width, foreColorButton.Height);
                backColorButton.Click += (sender, e) =>
                {
                    using (var colorDialog = new ColorDialog())
                    {
                        colorDialog.Color = previewLabel.BackColor;
                        if (colorDialog.ShowDialog() == DialogResult.OK)
                        {
                            previewLabel.BackColor = colorDialog.Color;
                            LogViewerTab.LogTextBackColor = colorDialog.Color;
                            updateFunc?.Invoke();
                        }
                    }
                };

                fontGroup.Location = new Point(Margin, Margin);
                fontGroup.Controls.AddRange(new Control[] { fontNameBox, fontSizeBox, foreColorButton, backColorButton });
                fontGroup.ClientSize = new Size(backColorButton.Right + Margin, backColorButton.Bottom + Margin);

                opacityBar.Location = new Point(Margin, groupTopMargin);
                opacityBar.Minimum = 30;
                opacityBar.Maximum = 100;
                opacityBar.TickFrequency = 5;
                opacityBar.SmallChange = 1;
                opacityBar.TickStyle = TickStyle.BottomRight;
                opacityBar.Value = selectedOpacity;
                opacityBar.Width = fontNameBox.Width;
                opacityBar.ValueChanged += (sender, e) =>
                {
                    opacityLabel.Text = opacityBar.Value + " %";
                    previewLabel.BackColor = Color.FromArgb(
                        Convert.ToInt32(255 * (opacityBar.Value / 100.0)),
                        previewLabel.BackColor.R,
                        previewLabel.BackColor.G,
                        previewLabel.BackColor.B);
                    LogViewerWindow.WindowOpacity = opacityBar.Value;
                };

                opacityLabel.Text = opacityBar.Value + " %";
                opacityLabel.TextAlign = ContentAlignment.MiddleRight;
                opacityLabel.Location = new Point(opacityBar.Width + Margin, groupTopMargin);
                opacityLabel.Width = fontSizeBox.Width;

                opacityGroup.Text = "Transparency";
                opacityGroup.Location = new Point(Margin, fontGroup.Bottom + Margin);
                opacityGroup.Controls.AddRange(new Control[] { opacityBar, opacityLabel });
                opacityGroup.ClientSize = new Size(opacityBar.Width + opacityLabel.Width + Margin * 3, opacityBar.Height + groupTopMargin + Margin);

                previewLabel.Text = "hello, world";
                previewLabel.Font = selectedFont;
                previewLabel.TextAlign = ContentAlignment.MiddleCenter;
                previewLabel.ForeColor = selectedForeColor;
                previewLabel.BackColor = selectedBackColor;
                previewLabel.Location = new Point(Margin, groupTopMargin);
                previewLabel.Size = new Size(fontNameBox.Width + fontSizeBox.Width + Margin, 100);

                previewGroup.Text = "Preview";
                previewGroup.Location = new Point(Margin, opacityGroup.Bottom + Margin);
                previewGroup.Controls.Add(previewLabel);
                previewGroup.ClientSize = new Size(previewLabel.Width + Margin * 2, previewLabel.Height + groupTopMargin + Margin);

                cancelButton.Text = "Cancel";
                cancelButton.DialogResult = DialogResult.Cancel;
                cancelButton.Location = new Point(fontGroup.Right - cancelButton.Width, previewGroup.Bottom + Margin);

                okButton.Text = "OK";
                okButton.DialogResult = DialogResult.OK;
                okButton.Location = new Point(cancelButton.Left - Margin - okButton.Width, cancelButton.Top);

                form.Text = "Appearance";
                form.ClientSize = new Size(cancelButton.Right + Margin, cancelButton.Bottom + Margin);
                form.FormBorderStyle = FormBorderStyle.FixedDialog;
                form.StartPosition = FormStartPosition.CenterScreen;
                form.MinimizeBox = false;
                form.MaximizeBox = false;
                form.AcceptButton = okButton;
                form.CancelButton = cancelButton;
                form.Controls.AddRange(new Control[] { fontGroup, opacityGroup, previewGroup, okButton, cancelButton });

                var dialogResult = form.ShowDialog();

                var changeAny =
                    LogViewerTab.LogTextFont != selectedFont ||
                    LogViewerTab.LogTextForeColor != selectedForeColor ||
                    LogViewerTab.LogTextBackColor != selectedBackColor ||
                    LogViewerWindow.WindowOpacity != selectedOpacity;

                if (dialogResult == DialogResult.Cancel && changeAny)
                {
                    // 表示を元の設定に戻す
                    LogViewerTab.LogTextFont = selectedFont;
                    LogViewerTab.LogTextForeColor = selectedForeColor;
                    LogViewerTab.LogTextBackColor = selectedBackColor;
                    LogViewerWindow.WindowOpacity = selectedOpacity;
                    updateFunc?.Invoke();
                }
            }
        }

        private static readonly Font UIFont = new Font("Consolas", 10);
        private TabControl TabControl;
        private ContextMenuStrip TabContextMenuStrip;
        private ToolStripMenuItem TabCloseMenuItem;
        private ToolStripMenuItem TabRenameMenuItem;
        private ToolStripMenuItem SaveLogWithFilterMenuItem;
        private ToolStripMenuItem SaveJsonLogWithFilterMenuItem;
        private ToolStripStatusLabel StatusLabel;
    }
}
