﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Numerics;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace Nintendo.Log
{
    internal class LogViewerTab : TabPage
    {
        internal class DisplayInfo
        {
            public DisplayInfo(BigInteger startIndex, int length, Dictionary<string, object> log)
            {
                StartIndex = startIndex;
                Length = length;
                Log = log;
            }

            public BigInteger StartIndex { get; set; }
            public int Length { get; set; }
            public Dictionary<string, object> Log { get; set; }
        }

        public LogViewerTab()
            : this("Log")
        {
        }

        public LogViewerTab(string text)
            : base(text)
        {
            FilterBox.Dock = DockStyle.Fill;
            FilterBox.Margin = new Padding(0);
            FilterBox.HandleCreated += (sender, e) =>
            {
                FilterBox.TextChanged += FilterBox_TextChanged;
                FilterBox_TextChanged(sender, e);
            };

            FilterErrorTip.ToolTipTitle = "Syntax Error";
            FilterErrorTip.ToolTipIcon = ToolTipIcon.Error;

            LogTextBox.Dock = DockStyle.Fill;
            LogTextBox.Margin = new Padding(0);
            LogTextBox.Font = LogTextFont;
            LogTextBox.LanguageOption = RichTextBoxLanguageOptions.AutoFontSizeAdjust | RichTextBoxLanguageOptions.UIFonts;
            LogTextBox.ScrollBars = RichTextBoxScrollBars.ForcedVertical;
            LogTextBox.ReadOnly = true;
            LogTextBox.DetectUrls = false;
            LogTextBox.ForeColor = LogTextForeColor;
            LogTextBox.BackColor = LogTextBackColor;
            LogTextBox.HideSelection = false;
            LogTextBox.MouseMove += LogTextBox_MouseMove;
            LogTextBox.MouseDown += LogTextBox_MouseDown;
            LogTextBox.MouseLeave += LogTextBox_MouseLeave;
            LogTextBox.ContextMenuStrip = CreateLogTextBoxContextMenu();

            LogTextForeColorChanged += LogViewerTab_LogTextForeColorChanged;
            LogTextBackColorChanged += LogViewerTab_LogTextBackColorChanged;
            LogTextFontChanged += LogViewerTab_LogTextFontChanged;

            var tableLayoutPanel = new TableLayoutPanel();
            tableLayoutPanel.Dock = DockStyle.Fill;
            tableLayoutPanel.Margin = new Padding(0);
            tableLayoutPanel.RowCount = 2;
            tableLayoutPanel.ColumnCount = 1;
            tableLayoutPanel.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            tableLayoutPanel.RowStyles.Add(new RowStyle(SizeType.AutoSize));
            tableLayoutPanel.Controls.Add(FilterBox, 0, 0);
            tableLayoutPanel.Controls.Add(LogTextBox, 0, 1);
            this.Controls.Add(tableLayoutPanel);

            ShowMetaInfoTimer = new System.Timers.Timer();
            ShowMetaInfoTimer.AutoReset = true;
            ShowMetaInfoTimer.Interval = 500;
            ShowMetaInfoTimer.Elapsed += ShowMetaInfo;

            FindBox = CreateFindBox();
        }

        public new void Dispose()
        {
            // TODO: Task の終了

            LogTextForeColorChanged -= LogViewerTab_LogTextForeColorChanged;
            LogTextBackColorChanged -= LogViewerTab_LogTextBackColorChanged;
            LogTextFontChanged -= LogViewerTab_LogTextFontChanged;
        }

        protected void FilterBox_TextChanged(object sender, EventArgs e)
        {
            bool valid = true;
            try
            {
                LogFilter.CheckSyntax(Filter); // 構文チェックに失敗すると例外が投げられる
                UpdateFull();
            }
            catch (FilterSyntaxErrorException exception)
            {
                valid = false;
                if (!string.IsNullOrEmpty(exception.Message)) // メッセージが用意されていない場合はツールチップを表示しない
                {
                    FilterErrorTip.Show(exception.Message, FilterBox, 0, FilterBox.Bottom, 3000);
                }
                //else // デバッグするときはここをコメントアウトして、例外の発生した箇所をツールチップに表示する
                //{
                //    var message = string.IsNullOrEmpty(exception.Message)
                //        ? exception.StackTrace.Split(Environment.NewLine.ToCharArray()).First()
                //        : exception.Message;
                //    FilterErrorTip.Show(exception.Message, FilterBox, 0, FilterBox.Bottom, 3000);
                //}
            }
            FilterBox.BackColor = valid ? Color.White : Color.MistyRose;
        }

        protected static bool CheckFilterSyntax(string text)
        {
            try
            {
                LogFilter.CheckSyntax(text); // 構文チェックに失敗すると例外が投げられる
            }
            catch (FilterSyntaxErrorException)
            {
                return false;
            }
            return true;
        }

        private DisplayInfo CurrentSelection;
        private Point CurrentSelectionPoint;

        protected void LogTextBox_MouseMove(object sender, MouseEventArgs e)
        {
            var charTopLeft =
                LogTextBox.GetPositionFromCharIndex(
                    LogTextBox.GetCharIndexFromPosition(new Point(e.X, e.Y)));
            if (e.X > charTopLeft.X + LogTextBox.Font.Height / 2 || // TORIAEZU: Font.Width がないため、幅は Height の半分としておく
                e.Y > charTopLeft.Y + LogTextBox.Font.Height)
            {
                // 空白領域がポイントされたとき
                HideMetaInfo();
                return;
            }

            var newPoint = new Point(e.X, e.Y);
            var newSelection = FindMetaInfo(LogTextBox.GetCharIndexFromPosition(newPoint));
            if (newSelection == null || newSelection == CurrentSelection)
            {
                return;
            }

            CurrentSelection = newSelection;
            CurrentSelectionPoint = newPoint;

            ClearSelectionBackColor();
            ToolTip.Hide(LogTextBox);
            ShowMetaInfoTimer.Stop();

            ShowMetaInfoTimer.Start();
        }

        protected void LogTextBox_MouseDown(object sender, MouseEventArgs e)
        {
            HideMetaInfo();
        }

        protected void LogTextBox_MouseLeave(object sender, EventArgs e)
        {
            HideMetaInfo();
        }

        private void HideMetaInfo()
        {
            CurrentSelection = null;
            ClearSelectionBackColor();
            ShowMetaInfoTimer.Stop();
            ToolTip.Hide(LogTextBox);
        }

        protected ContextMenuStrip CreateLogTextBoxContextMenu()
        {
            var copyMenuItem = new ToolStripMenuItem("Copy");
            copyMenuItem.Click += (sender, e) =>
            {
                LogTextBox.Copy();
            };

            var clearMenuItem = new ToolStripMenuItem("Clear");
            clearMenuItem.Click += (sender, e) =>
            {
                LogTextBox.Clear();
                lock (ForegroundLogBuffer)
                    lock (BackgroundLogBuffer)
                    {
                        ForegroundLogBuffer.Clear();
                        ForegroundLogBeginIndex = 0;
                        ForegroundLogEndIndex = 0;
                        BackgroundLogBuffer.Clear();
                        BackgroundLogBeginIndex = BackgroundLogEndIndex;
                    }
            };

            var contextMenuStrip = new ContextMenuStrip();
            contextMenuStrip.Items.Add(copyMenuItem);
            contextMenuStrip.Items.Add(clearMenuItem);

            return contextMenuStrip;
        }

        private DisplayInfo FindMetaInfo(int characterIndex)
        {
            lock (ForegroundLogBuffer)
            {
                if (ForegroundLogBuffer.Count == 0)
                {
                    return null;
                }

                var normalizedIndex = characterIndex + ForegroundLogBeginIndex;
                var begin = 0;
                var end = ForegroundLogBuffer.Count;
                while (end - begin > 1)
                {
                    var pivot = (end - begin) / 2 + begin;
                    var pivotOffset = ForegroundLogBuffer[pivot].StartIndex;
                    if (normalizedIndex >= pivotOffset)
                    {
                        begin = pivot;
                    }
                    else
                    {
                        end = pivot;
                    }
                }

                var info = ForegroundLogBuffer[begin];
                if (info.StartIndex <= normalizedIndex && normalizedIndex < info.StartIndex + info.Length)
                {
                    return info;
                }
                else
                {
                    return null;
                }
            }
        }

        protected void ShowMetaInfo(object sender, System.Timers.ElapsedEventArgs args)
        {
            ShowMetaInfoTimer.Stop();

            if (TextSelected)
            {
                return; // テキスト選択されているときはメタデータを表示しない
            }

            Invoke((MethodInvoker)(() =>
            {
                if (CurrentSelection != null) // ログがないか、ポインタがテキストボックスの外にあるときは null になる
                {
                    var lastCharPosition = LogTextBox.GetPositionFromCharIndex(
                        (int)(CurrentSelection.StartIndex - ForegroundLogBeginIndex) + CurrentSelection.Length);
                    ToolTip.Show(
                        MakeMetaDataMessage(CurrentSelection.Log), LogTextBox,
                            CurrentSelectionPoint.X, lastCharPosition.Y + LogTextBox.Font.Height);
                    var startIndex = (int)(CurrentSelection.StartIndex - ForegroundLogBeginIndex);
                    var length = CurrentSelection.Length;
                    SetSelectionBackColor(Color.Coral, startIndex, length);
                }
            }));
        }

        private string MakeMetaDataMessage(Dictionary<string, object> log)
        {
            var stringBuilder = new StringBuilder();

            foreach (var pair in log.Where(pair => pair.Key != "TextLog"))
            {
                var tabString = new string('\t',
                    (pair.Key == "ThreadId" || pair.Key == "Severity") ? 2 : 1); // TORIAEZU: 短いキーはタブを 2 つ挿入する
                stringBuilder.Append(pair.Key + tabString + ": " + pair.Value + Environment.NewLine);
            }

            if (stringBuilder.Length > 0)
            {
                return stringBuilder.ToString().TrimEnd(Environment.NewLine.ToCharArray());
            }
            else
            {
                return "no information";
            }
        }

        private struct SelectionColor
        {
            public int StartIndex;
            public Color Color;

            public SelectionColor(int startIndex, Color color)
            {
                StartIndex = startIndex;
                Color = color;
            }
        }

        private List<SelectionColor> HighlightSelectionColors = new List<SelectionColor>();

        private void SetSelectionBackColor(Color color, int start, int length)
        {
            if (!(start >= 0) || !(length > 0))
            {
                return;
            }

            var isLogTailShown = IsLogTailShown; // セレクションを移動する前に、ログの末尾が表示されていたか調べる

            HighlightSelectionStart = start;
            HighlightSelectionLength = length;
            HighlightSelectionColors.Clear();
            for (var i = 0; i < length; i++)
            {
                LogTextBox.Select(start + i + 1, 0);
                if (!HighlightSelectionColors.Any() || HighlightSelectionColors.Last().Color != LogTextBox.SelectionBackColor)
                {
                    HighlightSelectionColors.Add(new SelectionColor(i, LogTextBox.SelectionBackColor));
                }
            }

            LogTextBox.Select(start, length);
            LogTextBox.SelectionBackColor = color;

            if (isLogTailShown)
            {
                // ログの末尾が表示されていたときは、自動スクロールのためにキャレットを末尾まで移動させる
                LogTextBox.Select(Math.Max(0, LogTextBox.TextLength - 1), 0);
            }
            else
            {
                LogTextBox.SelectionLength = 0;
            }
        }

        private void ClearSelectionBackColor()
        {
            if (!(HighlightSelectionLength > 0))
            {
                return;
            }

            var isLogTailShown = IsLogTailShown; // セレクションを移動する前に、ログの末尾が表示されていたか調べる

            var endIndex = HighlightSelectionLength;
            foreach (var selectionColor in HighlightSelectionColors.Reverse<SelectionColor>())
            {
                LogTextBox.Select(
                    HighlightSelectionStart + selectionColor.StartIndex,
                    endIndex - selectionColor.StartIndex);
                LogTextBox.SelectionBackColor = selectionColor.Color;
                endIndex = selectionColor.StartIndex;
            }

            if (isLogTailShown)
            {
                // ログの末尾が表示されていたときは、自動スクロールのためにキャレットを末尾まで移動させる
                LogTextBox.Select(Math.Max(0, LogTextBox.TextLength - 1), 0);
            }
            else
            {
                LogTextBox.SelectionLength = 0;
            }

            HighlightSelectionLength = 0;
        }

        protected bool Selected
        {
            get
            {
                var selected = false;
                Invoke((MethodInvoker)(() =>
                {
                    selected = ((TabControl)this.Parent)?.SelectedTab == this;
                }));
                return selected;
            }
        }

        private bool TextSelected
        {
            get
            {
                var selected = false;
                Invoke((MethodInvoker)(() =>
                {
                    selected = LogTextBox.SelectionLength > 0;
                }));
                return selected;
            }
        }

        protected bool IsLogTailShown
        {
            get
            {
                var isShown = false;
                Invoke((MethodInvoker)(() =>
                {
                    var lastLineOfScreen =
                        LogTextBox.GetLineFromCharIndex(
                            LogTextBox.GetCharIndexFromPosition(
                                new Point(0, LogTextBox.ClientSize.Height)));
                    var lastLineOfBuffer =
                        LogTextBox.GetLineFromCharIndex(
                            LogTextBox.TextLength - 1);

                    isShown = (lastLineOfScreen == lastLineOfBuffer);
                }));
                return isShown;
            }
        }

        public void UpdatePartial()
        {
            UpdateMessageQueue.Enqueue(new UpdateMessage { isFullUpdateRequired = false, isAutoScrollRequired = true });
        }

        public void UpdateFull()
        {
            UpdateMessageQueue.Enqueue(new UpdateMessage { isFullUpdateRequired = true, isAutoScrollRequired = false });
        }

        private bool FilterLocked
        {
            get { return !FilterBox.Enabled; }
            set { FilterBox.Enabled = !value; }
        }

        internal bool IsLogSaving
        {
            get { return !SaveLogTask.IsCompleted; }
        }

        internal bool IsJsonLogSaving
        {
            get { return !SaveJsonLogTask.IsCompleted; }
        }

        protected static string Crlf2Lf(string input)
        {
            return input.Replace("\r\n", "\n"); ;
        }

        private static string MakeSaverityText(int severity)
        {
            var array = new string[] {"Trace", "Info", "Warn", "Error", "Fatal"};
            return 0 <= severity && severity < array.Count() ? array[severity] : "?";
        }

        private static string ToMetaText(Dictionary<string, object> log, bool isShown, string key, int width, Func<object, string> stringizer)
        {
            if (!isShown)
            {
                return string.Empty;
            }
            if (!log.ContainsKey(key))
            {
                return "N/A".PackLeft(width);
            }
            return stringizer(log[key]).PackLeft(width);
        }

        private static string MakeMetaText(Dictionary<string, object> log)
        {
            var metaTexts = new string[]
            {
                ToMetaText(log, IsTimestampShown,       "Timestamp",        19, v => v.ToString()),
                ToMetaText(log, IsUserSystemClockShown, "UserSystemClock",  19, v => v.ToString()),
                ToMetaText(log, IsPeerNameShown,        "PeerName",         14, v => v.ToString()),
                ToMetaText(log, IsProcessNameShown,     "ProcessName",      8,  v => v.ToString()),
                ToMetaText(log, IsModuleNameShown,      "ModuleName",       6,  v => v.ToString()),
                ToMetaText(log, IsThreadNameShown,      "ThreadName",       32, v => v.ToString()),
                ToMetaText(log, IsFunctionNameShown,    "FunctionName",     32, v => v.ToString()),
                ToMetaText(log, IsSeverityShown,        "Severity",         5,  v => MakeSaverityText((int)v)),
                ToMetaText(log, IsVerbosityShown,       "Verbosity",        2,  v => $"v{(int)v}")
            };

            var showables = metaTexts.Where(s => !string.IsNullOrEmpty(s));

            return showables.Any()
                ? " " + string.Join(" | ", showables) + " | "
                : string.Empty;
        }

        private static string ColoringBySeverity(Dictionary<string, object> log)
        {
            var original = (string)log["TextLog"];

            if (!IsColoringBySeverityEnabled)
            {
                return original;
            }

            var prefixes = new string[]
            {
                "\u001b[90m",       // Trace: Dark gray,
                "",                 // Info:  White (terminal default),
                "\u001b[33m",       // Warn:  Yellow,
                "\u001b[31m",       // Error: Red,
                "\u001b[41;37m",    // Fatal: White with Red background
            };
            var suffix = "\u001b[0m";

            var severity = (int)log["Severity"];
            if (!(0 <= severity && severity < prefixes.Length))
            {
                return original;
            }

            var prefix = prefixes[severity];
            if (string.IsNullOrWhiteSpace(prefix))
            {
                return original;
            }

            return prefix + original + suffix;
        }

        protected void UpdateFunc()
        {
            var previousFilter = string.Empty;
            var prevBackgoundLogEndIndex = new BigInteger(0);

            while (true)
            {
                var message = UpdateMessageQueue.DequeueSync();
                var isFullUpdateRequired = message.isFullUpdateRequired;
                var isAutoScrollRequired = message.isAutoScrollRequired;

                if (this.Parent == null)
                {
                    break; // タブが閉じられたので抜ける
                }

                if (!Selected)
                {
                    continue; // タブが選択されていないときは更新しない
                }

                var currentFilter = string.Copy(Filter);
                previousFilter = currentFilter;

                // バッファをロックしてから参照すると、GUI スレッドからバッファにアクセスしているときにデッドロックするため、
                // ここで取り出しておく。
                var isTailShown = IsLogTailShown;

                var appendString = string.Empty;
                var removeTextLength = 0;
                lock (ForegroundLogBuffer)
                    lock (BackgroundLogBuffer)
                    {
                        var isAnyNewLogReceived = BackgroundLogEndIndex > prevBackgoundLogEndIndex;
                        var isUpdateRequired = isTailShown && isAnyNewLogReceived;
                        if (!(isFullUpdateRequired || isUpdateRequired))
                        {
                            continue;
                        }

                        var bufferAllOuted = BackgroundLogBeginIndex >= prevBackgoundLogEndIndex;

                        IEnumerable<Dictionary<string, object>> logs;

                        if (CheckFilterSyntax(currentFilter))
                        {
                            var filteredLogs =
                                ((isFullUpdateRequired || bufferAllOuted)
                                    ? BackgroundLogBuffer
                                    : BackgroundLogBuffer.GetRange(
                                        (int)(prevBackgoundLogEndIndex - BackgroundLogBeginIndex),
                                        (int)(BackgroundLogEndIndex - prevBackgoundLogEndIndex))).Where(
                                            log => LogFilter.Evaluate(currentFilter, log));
                            logs = filteredLogs.Skip(Math.Max(0, filteredLogs.Count() - ForegroundLogCountMax));
                        }
                        else
                        {
                            logs = Enumerable.Empty<Dictionary<string, object>>();
                        }

                        if (isFullUpdateRequired || bufferAllOuted || logs.Count() == ForegroundLogCountMax)
                        {
                            removeTextLength = (int)(ForegroundLogEndIndex - ForegroundLogBeginIndex);
                            ForegroundLogBuffer.Clear();
                            ForegroundLogBeginIndex = 0;
                            ForegroundLogEndIndex = 0;
                        }
                        else
                        {
                            var overflow = (ForegroundLogBuffer.Count + logs.Count()) > ForegroundLogCountMax;
                            if (overflow)
                            {
                                removeTextLength =
                                    (from log in ForegroundLogBuffer.Take(CleanupLogCount)
                                     select log.Length).Sum();
                                ForegroundLogBuffer.RemoveRange(0, CleanupLogCount);
                                ForegroundLogBeginIndex += removeTextLength;
                            }
                        }

                        prevBackgoundLogEndIndex = BackgroundLogEndIndex;

                        if (logs.Count() == 0 && removeTextLength == 0)
                        {
                            continue;
                        }

                        var stringBuilder = new StringBuilder();
                        foreach (var log in logs)
                        {
                            var textLog = MakeMetaText(log) + Crlf2Lf(ColoringBySeverity(log));
                            stringBuilder.Append(textLog);

                            var displayLength = SgrCommandProcessor.CalculateDisplayLength(textLog);
                            ForegroundLogBuffer.Add(new DisplayInfo(ForegroundLogEndIndex, displayLength, log));
                            ForegroundLogEndIndex += displayLength;
                        }
                        appendString = stringBuilder.ToString();
                    }

                Invoke((MethodInvoker)(() =>
                {
                    if (!isAutoScrollRequired)
                    {
                        LogTextBox.SuspendScroll();
                    }
                    try
                    {
                        if (HighlightSelectionLength > 0)
                        {
                            // ログをハイライトしているときは、ハイライト位置をずらす
                            HighlightSelectionStart -= removeTextLength;
                            if (HighlightSelectionStart < 0)
                            {
                                HighlightSelectionLength += HighlightSelectionStart; // マイナスなのでプラスで引く
                                if (HighlightSelectionLength < 0)
                                {
                                    HighlightSelectionLength = 0;
                                }
                            }
                        }
                        LogTextBox.FastRemoveText(0, removeTextLength);

                        var rtf = RtfEditor.Sgr2Rtf(appendString, LogTextBox.ForeColor, LogTextBox.BackColor, LogTextBox.Font.Style, isFullUpdateRequired);
                        LogTextBox.SelectionStart = LogTextBox.TextLength;
                        LogTextBox.SelectedRtf = rtf;

                        LogTextBox.ClearUndo();
                    }
                    finally
                    {
                        if (!isAutoScrollRequired)
                        {
                            LogTextBox.ResumeScroll();
                        }
                    }
                }));
            }
        }

        public void StartOrStopSavingLogWithFilter()
        {
            if (SaveLogTask.IsCompleted)
            {
                var currentFilter = string.Copy(Filter);
                if (!CheckFilterSyntax(currentFilter))
                {
                    MessageBox.Show(this, "Filter syntax is invalid.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                var path = LogViewerWindow.GetPathWithSaveFileDialog(Text, "txt", true);
                if (string.IsNullOrEmpty(path))
                {
                    return;
                }
                FilterLocked = true;
                LogFilePath = path;
                SaveLogCancellationTokenSource = new CancellationTokenSource();
                SaveLogTask = LogClient.CreateSavePlainTextLogTask(path, SaveLogCancellationTokenSource.Token, log => LogFilter.Evaluate(currentFilter, log));
                SaveLogTask.Start();
            }
            else
            {
                SaveLogCancellationTokenSource.Cancel(true);
                SaveLogCancellationTokenSource.Dispose();
                SaveLogTask.Wait();
                LogFilePath = string.Empty;
                if (SaveJsonLogTask.IsCompleted) // JSON ログも保存中でなければ、フィルタのロックを解除する
                {
                    FilterLocked = false;
                }
            }
        }

        public void StartOrStopSavingJsonLogWithFilter()
        {
            if (SaveJsonLogTask.IsCompleted)
            {
                var currentFilter = string.Copy(Filter);
                if (!CheckFilterSyntax(currentFilter))
                {
                    MessageBox.Show(this, "Filter syntax is invalid.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                var path = LogViewerWindow.GetPathWithSaveFileDialog(Text, "json", true);
                if (string.IsNullOrEmpty(path))
                {
                    return;
                }

                FilterLocked = true;
                JsonLogFilePath = path;
                SaveJsonLogCancellationTokenSource = new CancellationTokenSource();
                SaveJsonLogTask = LogClient.CreateSaveJsonLogTask(path, SaveJsonLogCancellationTokenSource.Token, log => LogFilter.Evaluate(currentFilter, log));
                SaveJsonLogTask.Start();
            }
            else
            {
                SaveJsonLogCancellationTokenSource.Cancel(true);
                SaveJsonLogCancellationTokenSource.Dispose();
                SaveJsonLogTask.Wait();
                JsonLogFilePath = string.Empty;
                if (SaveLogTask.IsCompleted) // プレーンテキストログも保存中でなければ、フィルタのロックを解除する
                {
                    FilterLocked = false;
                }
            }
        }

        private int Find(string text, int startIndex, bool findForward, bool useRegex, bool caseSensitive)
        {
            if (string.IsNullOrEmpty(text))
            {
                return -1;
            }

            var index = -1;

            if (useRegex)
            {
                try
                {
                    var options = RegexOptions.None
                        | (findForward ? RegexOptions.None : RegexOptions.RightToLeft)
                        | (caseSensitive ? RegexOptions.None : RegexOptions.IgnoreCase);
                    var pattern = new Regex(text, options);

                    var match = pattern.Match(LogTextBox.Text, startIndex);
                    if (!match.Success)
                    {
                        return -1;
                    }

                    LogTextBox.Select(match.Index, match.Length);
                    index = match.Index;
                }
                catch (System.ArgumentException)
                {
                    return -1; // 正規表現に誤りがある
                }
            }
            else
            {
                var options = RichTextBoxFinds.None
                    | (findForward ? RichTextBoxFinds.None : RichTextBoxFinds.Reverse)
                    | (caseSensitive ? RichTextBoxFinds.MatchCase : RichTextBoxFinds.None);

                index = findForward
                    ? LogTextBox.Find(text, startIndex, options)
                    : LogTextBox.Find(text, 0, startIndex, options);
                if (index < 0)
                {
                    return -1;
                }
            }

            return index;
        }

        private int FindWithRotate(string text, bool findForward, bool useRegex, bool caseSensitive)
        {
            var currentSelectionStart = LogTextBox.SelectionStart;
            var currentSelectionLength = LogTextBox.SelectionLength;

            var index = findForward
                ? Find(text, LogTextBox.SelectionStart + LogTextBox.SelectionLength, true, useRegex, caseSensitive)
                : Find(text, LogTextBox.SelectionStart, false, useRegex, caseSensitive);

            if (index >= 0)
            {
                return index;
            }

            // 見つからなかったら、先頭 or 末尾から検索しなおす
            index = findForward
                ? Find(text, 0, true, useRegex, caseSensitive)
                : Find(text, LogTextBox.TextLength - 1, false, useRegex, caseSensitive);

            if (index < 0)
            {
                LogTextBox.SelectionLength = 0;
                return -1;
            }

            return index;
        }

        internal void OpenFindBox()
        {
            FindBox.ShowDialog(this);
        }

        protected Form CreateFindBox()
        {
            const int Margin = 10;

            var str = string.Empty;
            var textBox = new TextBox();
            var regularExpression = new CheckBox();
            var caseSensitive = new CheckBox();
            var nextButton = new Button();
            var prevButton = new Button();
            var form = new Form();

            textBox.Location = new Point(Margin, Margin);
            textBox.Width = 300;
            textBox.TextChanged += (sender, e) =>
            {
                str = string.Copy(textBox.Text);

                if (Find(str, 0, true, regularExpression.Checked, caseSensitive.Checked) < 0)
                {
                    LogTextBox.SelectionLength = 0;
                }
            };

            textBox.KeyDown += (sender, e) =>
            {
                if (e.KeyCode == Keys.Enter)
                {
                    FindWithRotate(str, !e.Shift, regularExpression.Checked, caseSensitive.Checked);
                }
            };

            regularExpression.Text = "Regular expression (R)";
            regularExpression.AutoSize = true;
            regularExpression.Location = new Point(textBox.Left, textBox.Bottom + Margin);
            regularExpression.CheckedChanged += (sender, e) =>
            {
                if (Find(str, 0, true, regularExpression.Checked, caseSensitive.Checked) < 0)
                {
                    LogTextBox.SelectionLength = 0;
                }
            };

            caseSensitive.Text = "Case sensitive (C)";
            caseSensitive.AutoSize = true;
            caseSensitive.Location = new Point(regularExpression.Left, regularExpression.Bottom);
            caseSensitive.CheckedChanged += (sender, e) =>
            {
                if (Find(str, 0, true, regularExpression.Checked, caseSensitive.Checked) < 0)
                {
                    LogTextBox.SelectionLength = 0;
                }
            };

            nextButton.Text = "Next";
            nextButton.Location = new Point(textBox.Right - nextButton.Width, caseSensitive.Top);
            nextButton.KeyDown += (sender, e) =>
            {
                if (e.KeyCode == Keys.Enter)
                {
                    FindWithRotate(str, true, regularExpression.Checked, caseSensitive.Checked);
                }
            };
            nextButton.Click += (sender, e) =>
            {
                FindWithRotate(str, true, regularExpression.Checked, caseSensitive.Checked);
            };

            prevButton.Text = "Prev";
            prevButton.Location = new Point(nextButton.Left - Margin - prevButton.Width, nextButton.Top);
            prevButton.KeyDown += (sender, e) =>
            {
                if (e.KeyCode == Keys.Enter)
                {
                    FindWithRotate(str, false, regularExpression.Checked, caseSensitive.Checked);
                }
            };
            prevButton.Click += (sender, e) =>
            {
                FindWithRotate(str, false, regularExpression.Checked, caseSensitive.Checked);
            };

            form.Text = "Find";
            form.ClientSize = new Size(textBox.Width + Margin * 2, prevButton.Bottom + Margin);
            form.FormBorderStyle = FormBorderStyle.FixedDialog;
            form.StartPosition = FormStartPosition.CenterParent;
            form.MinimizeBox = false;
            form.MaximizeBox = false;
            form.ShowInTaskbar = false;
            form.KeyPreview = true;
            form.Shown += (sender, e) =>
            {
                textBox.Focus();
            };
            form.KeyDown += (sender, e) =>
            {
                if (e.Alt && e.KeyCode == Keys.R)
                {
                    regularExpression.Checked = !regularExpression.Checked;
                }
                else if (e.Alt && e.KeyCode == Keys.C)
                {
                    caseSensitive.Checked = !caseSensitive.Checked;
                }
                else if (e.KeyCode == Keys.Escape)
                {
                    form.Close();
                }
            };
            form.Controls.AddRange(new Control[] { textBox, prevButton, nextButton, regularExpression, caseSensitive });

            return form;
        }

        public static Color LogTextForeColor
        {
            get { return LogTextForeColorValue; }
            set
            {
                SgrCommandProcessor.DefaultForeColor = value;
                RtfEditor.DefaultForeColor = value;
                LogTextForeColorValue = value;
                LogTextForeColorChanged?.Invoke(LogTextForeColorValue);
            }
        }
        private static Color LogTextForeColorValue = Color.Black;
        protected static ChangeColorEvent LogTextForeColorChanged;
        protected void LogViewerTab_LogTextForeColorChanged(Color color)
        {
            Invoke((MethodInvoker)(() =>
            {
                LogTextBox.ForeColor = color;
            }));
        }

        public static Color LogTextBackColor
        {
            get { return LogTextBackColorValue; }
            set
            {
                SgrCommandProcessor.DefaultBackColor = value;
                RtfEditor.DefaultBackColor = value;
                LogTextBackColorValue = value;
                LogTextBackColorChanged?.Invoke(LogTextBackColorValue);
            }
        }
        protected static Color LogTextBackColorValue = Color.White;
        protected static ChangeColorEvent LogTextBackColorChanged;
        protected void LogViewerTab_LogTextBackColorChanged(Color color)
        {
            Invoke((MethodInvoker)(() =>
            {
                LogTextBox.BackColor = color;
            }));
        }

        public static Font LogTextFont
        {
            get { return LogTextFontValue; }
            set
            {
                LogTextFontValue = value;
                LogTextFontChanged?.Invoke(LogTextFontValue);
            }
        }
        protected static Font LogTextFontValue = new Font("MS Gothic", 10);
        protected static ChangeFontEvent LogTextFontChanged;
        protected void LogViewerTab_LogTextFontChanged(Font font)
        {
            Invoke((MethodInvoker)(() =>
            {
                LogTextBox.Font = font;
            }));
        }

        protected delegate void ChangeColorEvent(Color color);
        protected delegate void ChangeFontEvent(Font font);

        protected TextBox FilterBox = new TextBox();
        protected RichTextBoxEx LogTextBox = new RichTextBoxEx();
        protected ToolTip FilterErrorTip = new ToolTip();
        protected Form FindBox;

        public string Filter { get { return FilterBox.Text; } set { FilterBox.Text = value; } }

        public static readonly int CleanupLogCount = 1000;
        public static readonly int ForegroundLogCountMax = 20000;
        public static readonly int BackgroundLogCountMax = 100000;

        protected List<DisplayInfo> ForegroundLogBuffer = new List<DisplayInfo>();
        protected List<Dictionary<string, object>> BackgroundLogBuffer = new List<Dictionary<string, object>>();

        protected BigInteger ForegroundLogBeginIndex = new BigInteger(0);
        protected BigInteger ForegroundLogEndIndex = new BigInteger(0);
        protected BigInteger BackgroundLogBeginIndex = new BigInteger(0);
        protected BigInteger BackgroundLogEndIndex = new BigInteger(0);

        protected struct UpdateMessage
        {
            public bool isFullUpdateRequired;
            public bool isAutoScrollRequired;
        }

        protected SyncQueue<UpdateMessage> UpdateMessageQueue = new SyncQueue<UpdateMessage>();

        private Dictionary<string, object> LastLog = new Dictionary<string, object>();
        protected ToolTip ToolTip = new ToolTip();
        protected System.Timers.Timer ShowMetaInfoTimer;
        protected int HighlightSelectionStart;
        protected int HighlightSelectionLength = 0;

        protected Task SaveLogTask = Task.CompletedTask;
        protected Task SaveJsonLogTask = Task.CompletedTask;

        private CancellationTokenSource SaveLogCancellationTokenSource;
        private CancellationTokenSource SaveJsonLogCancellationTokenSource;

        public string LogFilePath { get; private set; } = string.Empty;
        public string JsonLogFilePath { get; private set; } = string.Empty;

        protected CancellationTokenSource ReceiveCancellationTokenSource = new CancellationTokenSource();

        public static bool IsTimestampShown { get; set; }
        public static bool IsUserSystemClockShown { get; set; }
        public static bool IsProcessNameShown { get; set; }
        public static bool IsPeerNameShown { get; set; }
        public static bool IsModuleNameShown { get; set; }
        public static bool IsThreadNameShown { get; set; }
        public static bool IsFunctionNameShown { get; set; }
        public static bool IsSeverityShown { get; set; }
        public static bool IsVerbosityShown { get; set; }
        public static bool IsColoringBySeverityEnabled { get; set; }
    }
}
