﻿using System;
using System.Diagnostics;
using System.IO;
using System.Net.Sockets;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks.Dataflow;
using Nintendo.Htcs;

namespace Nintendo.Log
{
    internal class LogPortEmulator : IDisposable
    {
        public LogPortEmulator(string peerName, string portName, ISourceBlock<string> sourceBlock)
        {
            var portNumber = GetReservationPort(peerName, portName);
            HtcsListener = new HtcsListener(new HtcsEndPoint(peerName, portName), portNumber);
            try
            {
                HtcsListener.Start();
            }
            catch (SocketException)
            {
                // 指定されたポートが既に使われているときは、空いているポートを使う
                HtcsListener = new HtcsListener(new HtcsEndPoint(peerName, portName));
                HtcsListener.Start();
            }

            ServerThread = new Thread(() => ServerFunc(sourceBlock));
            ServerThread.Start();
        }

        public void Dispose()
        {
            HtcsListener.Stop();
            ServerThread.Join();
        }

        private static int GetReservationPort(string peerName, string portName)
        {
            var reservationTextPath = Path.Combine(
                Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
                    @"Nintendo\Oasis\TargetManager\HTCSReservations.txt");
            try
            {
                var text = string.Empty;
                using (var sr = new StreamReader(reservationTextPath))
                {
                    while (!sr.EndOfStream)
                    {
                        var match = Regex.Match(
                            StripComment(sr.ReadLine()),
                                $@"^\s*({peerName})?\s*,\s*(?<PortName>[^\s,]+)\s*,\s*(?<TcpPortNumber>\d+)\s*$");
                        if (match.Success)
                        {
                            if (match.Groups["PortName"].Value == portName)
                            {
                                return int.Parse(match.Groups["TcpPortNumber"].Value);
                            }
                        }
                    }
                }
            }
            catch (FileNotFoundException)
            {
                // 設定ファイルが見つからない
            }

            return 0; // 有効な設定が見つからないときは、空いているポートを使う

        }

        private static string StripComment(string text)
        {
            return Regex.Match(text, "^[^#]*").Value;
        }

        private void ServerFunc(ISourceBlock<string> sourceBlock)
        {
            var broadcastBlock = new BroadcastBlock<string>(input => input);

            using (sourceBlock.LinkTo(broadcastBlock))
            using (var cts = new CancellationTokenSource())
            using (var rwl = new ReaderWriterLockSlim())
            {
                while (true)
                {
                    TcpClient client;
                    try
                    {
                        client = HtcsListener.AcceptTcpClient();
                    }
                    catch (Exception exception) when (
                        exception is SocketException ||
                        exception is InvalidOperationException)
                    {
                        break; // Stop() が呼ばれたので抜ける。
                    }

                    var thread = new Thread(() =>
                    {
                        rwl.EnterReadLock();
                        try
                        {
                            ClientFunc(client, broadcastBlock, cts.Token);
                        }
                        finally
                        {
                            rwl.ExitReadLock();
                        }
                    });
                    thread.Start();
                }

                cts.Cancel();

                if (rwl.TryEnterWriteLock(TimeoutToStopServerFunc))
                {
                    // WriteLock が取れる = すべての子スレッドが終了している。
                    rwl.ExitWriteLock();
                }
                else
                {
                    Debug.Assert(false);
                }
            }

            broadcastBlock.Complete();
            broadcastBlock.Completion.Wait();
        }

        private void ClientFunc(TcpClient client, ISourceBlock<string> sourceBlock, CancellationToken cancellationToken)
        {
            var disconnectCancellationTokenSource = new CancellationTokenSource();

            var writerThread = new Thread(() =>
            {
                try
                {
                    var bufferBlock = new BufferBlock<string>();
                    using (sourceBlock.LinkTo(bufferBlock))
                    using (var sw = new StreamWriter(client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        string preBufferedLog;
                        bufferBlock.TryReceive(log => true, out preBufferedLog); // 前回のログがバッファリングされていたら捨てる。

                        sw.AutoFlush = true;

                        while (true)
                        {
                            try
                            {
                                sw.Write(bufferBlock.Receive(disconnectCancellationTokenSource.Token));
                            }
                            catch (Exception exception) when (
                                exception is IOException ||
                                exception is OperationCanceledException ||
                                exception is InvalidOperationException)
                            {
                                break; // クライアントとの接続が切れたので抜ける。
                            }
                        }
                        bufferBlock.Complete();
                        bufferBlock.Completion.Wait();
                    }
                }
                catch (Exception exception) when (
                    exception is ArgumentException ||
                    exception is InvalidOperationException ||
                    exception is IOException)
                {
                    // StreamWriter を作る前に、クライアントとの接続が切れたとき。
                }
            });
            writerThread.Start();

            try
            {
                using (var ns = client.GetStream())
                using (var br = new BinaryReader(ns, new UTF8Encoding(false), true))
                using (cancellationToken.Register(() => ns.Close()))
                {
                    while (true)
                    {
                        try
                        {
                            br.ReadChar(); // 切断を検知するために Read し続ける。
                        }
                        catch (Exception exception) when (
                            exception is IOException ||
                            exception is ObjectDisposedException)
                        {
                            disconnectCancellationTokenSource.Cancel(); // 切断を検知したら書き込みタスクをキャンセルする。
                            break;
                        }
                    }
                }
            }
            catch (Exception exception) when (
                exception is InvalidOperationException ||
                exception is IOException)
            {
                // BinaryReader を作る前に、クライアントとの接続が切れたとき。
            }

            Debug.Assert(!client.Connected);
            writerThread.Join();
            client.Close();
        }

        private HtcsListener HtcsListener;
        private Thread ServerThread;
        private readonly TimeSpan TimeoutToStopServerFunc = TimeSpan.FromSeconds(10);
    }
}
