﻿using System;
using System.Diagnostics;
using System.Globalization;
using System.Threading;
using Nintendo.Foundation.IO;
using Nintendo.Log;

namespace Nintendo.ConvertNxbinlog
{
    internal class CommandArgument
    {
        [CommandLineOption('i', "input", IsRequired = true, Description = "Input file name.")]
        public string Intput { get; set; }

        [CommandLineOption('o', "output", IsRequired = true, Description = "Output file name.")]
        public string Output { get; set; }

        [CommandLineOption("json", Description = "Convert to json file.")]
        public bool Json { get; set; }
    }

    internal class CommandInterface
    {
        public static int Run(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                var settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = "Convert nxbinlog to ohter format file."
                };

                var parser = new CommandLineParser(settings);

                CommandArgument parsed;
                if (!parser.ParseArgs(args, out parsed))
                {
                    return 1;
                }

                ConvertNxbinlog(parsed.Output, parsed.Intput, parsed.Json);
                return 0;
            }
            catch (Exception exception)
            {
                printException(exception);
                return 1;
            }
        }

        private static void ConvertNxbinlog(string outPath, string path, bool useJson)
        {
            var cs = new CancellationTokenSource();
            ConsoleCancelEventHandler canceller = (sender, e) =>
            {
                cs.Cancel();
                e.Cancel = true;
            };
            Console.CancelKeyPress += canceller;
            try
            {
                int progressCallbackCallCount = 0;
                const int progressBarUpdateRatio = 10000;
                Action<long, long> progressbarUpdater = (totalProcessSize, fileSize) =>
                {
                    progressCallbackCallCount++;
                    if (progressCallbackCallCount % progressBarUpdateRatio == 0)
                    {
                        printProgress((double)totalProcessSize / fileSize);
                    }
                };

                var task = useJson
                    ? LogClient.CreateNxbinlogToJsonTask(outPath, path, cs.Token, progressbarUpdater)
                    : LogClient.CreateNxbinlogToTextTask(outPath, path, cs.Token, progressbarUpdater);
                task.Start();
                task.Wait();
                if (cs.IsCancellationRequested)
                {
                    Console.WriteLine();
                    Console.WriteLine("Cancelled.");
                }
                else
                {
                    printProgress(1.0);
                    Console.WriteLine();
                    Console.WriteLine("Completed.");
                }
            }
            finally
            {
                Console.CancelKeyPress -= canceller;
            }
        }


        private static void printException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Trace.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }

        private static void printProgress(double progress)
        {
            if (!(0.0 <= progress && progress <= 1.0))
            {
                throw new ArgumentException();
            }
            const int maxBarCount = 40;
            int barCount = (int)(maxBarCount * progress);
            Console.Write($"\r[{new string('|', barCount),-maxBarCount}] {progress,7:P2}");
        }
    }
}
