-- ----------------------------------------------------------------------------
-- Siglo htc protocol dessector
--

-- ----------------------------------------------------------------------------
-- Constants
--

-- Protocol numbers
ProtocolTm = 0
ProtocolHtcfs = 1
ProtocolHtcs = 2
ProtocolHtcenv = 3
ProtocolHtclow = 256

-- Packet category
PacketCategory = {}
PacketCategory[0x00] = "Request"
PacketCategory[0x01] = "Response"

-- Htclow packet type
PacketType = {}
PacketType[0x00] = "Data"
PacketType[0x01] = "DataAck"
PacketType[0x02] = "Syn"
PacketType[0x03] = "SynAck"
PacketType[0x04] = "Fin"
PacketType[0x05] = "FinAck"

-- Htcfs packet type
HtcfsPacketType = {}
HtcfsPacketType[0x00] = "GetMaxProtocolVersion"
HtcfsPacketType[0x01] = "SetProtocolVersion"
HtcfsPacketType[0x10] = "GetEntryType"
HtcfsPacketType[0x20] = "OpenFile"
HtcfsPacketType[0x21] = "CloseFile"
HtcfsPacketType[0x22] = "GetPriorityForFile"
HtcfsPacketType[0x23] = "SetPriorityForFile"
HtcfsPacketType[0x24] = "CreateFile"
HtcfsPacketType[0x25] = "DeleteFile"
HtcfsPacketType[0x26] = "RenameFile"
HtcfsPacketType[0x27] = "FileExists"
HtcfsPacketType[0x28] = "ReadFile"
HtcfsPacketType[0x29] = "WriteFile"
HtcfsPacketType[0x2a] = "FlushFile"
HtcfsPacketType[0x2b] = "GetFileTimeStamp"
HtcfsPacketType[0x2c] = "GetFileSize"
HtcfsPacketType[0x2d] = "SetFileSize"
HtcfsPacketType[0x30] = "OpenDirectory"
HtcfsPacketType[0x31] = "CloseDirectory"
HtcfsPacketType[0x32] = "GetPriorityForDirectory"
HtcfsPacketType[0x33] = "SetPriorityForDirectory"
HtcfsPacketType[0x34] = "CreateDirectory"
HtcfsPacketType[0x35] = "DeleteDirectory"
HtcfsPacketType[0x36] = "RenameDirectory"
HtcfsPacketType[0x37] = "DirectoryExists"
HtcfsPacketType[0x38] = "ReadDirectory"
HtcfsPacketType[0x39] = "GetEntryCount"

-- ----------------------------------------------------------------------------
-- Entry Point
--
htclow_proto = Proto("htclow", "HTCLOW", "HTCLOW Protocol");
htcfs_proto = Proto("htcfs", "HTCFS", "HTCFS Protocol");

function htclow_proto.dissector(buffer, pinfo, tree)
    pinfo.cols.protocol = "HTCLOW"

    local offset = 0
    local HeaderSize = 24

    --  Header がすべて到着するのを待つ
    if buffer:len() < HeaderSize then
        pinfo.desegment_len = DESEGMENT_ONE_MORE_SEGMENT
        return length - buffer:len()
    end

    while offset < buffer:len() do
        -- Htclow header
        local protocol = buffer(offset + 0, 2):le_uint()
        local version = buffer(offset + 2, 2):le_uint()
        local reserved = buffer(offset + 4, 2):le_uint()
        local packetType = buffer(offset + 6, 2):le_uint()
        local moduleId = buffer(offset + 8, 1):le_uint()
        local reservedId = buffer(offset + 9, 1):le_uint()
        local channelId = buffer(offset + 10, 2):le_uint()
        local bodySize = buffer(offset + 12, 4):le_uint()
        local sequenceId = buffer(offset + 16, 8):le_uint64()

        -- パケットが全て到着するのを待つ
        local packetSize = HeaderSize + bodySize
        local packetTail = offset + packetSize
        if buffer:len() < packetTail then
            pinfo.desegment_len = DESEGMENT_ONE_MORE_SEGMENT
            return packetTail - buffer:len()
        end

        -- Print htclow header
        local packetDescription = "Htclow Packet (" .. PacketType[packetType] .. " " .. sequenceId .. ")"
        local subTree = tree:add(htclow_proto, buffer(offset, packetSize), packetDescription)

        subTree:add(buffer(offset + 0, 2), "Protocol: " .. protocol)
        subTree:add(buffer(offset + 2, 2), "Version: " .. version)
        subTree:add(buffer(offset + 4, 2), "Reserved: " .. reserved)
        subTree:add(buffer(offset + 6, 2), "Packet type: " .. PacketType[packetType])
        subTree:add(buffer(offset + 8, 4), "Channel: " .. moduleId .. ":" .. reservedId .. ":" .. channelId)
        subTree:add(buffer(offset + 12, 4), "Body size: " .. bodySize)
        subTree:add(buffer(offset + 16, 8), "Sequence id: " .. sequenceId)

        if buffer:len() > offset + HeaderSize then
            local internalOffset = offset + HeaderSize
            local internalPacketSize = packetSize - HeaderSize
            local internalProtocol = buffer(internalOffset, 2):le_uint()

            -- Htcfs
            if internalProtocol == ProtocolHtcfs then

                -- Htcfs header
                local htcfsVersion = buffer(internalOffset + 2, 2):le_uint()
                local htcfsPacketCategory = buffer(internalOffset + 4, 2):le_uint()
                local htcfsPacketType = buffer(internalOffset + 6, 2):le_uint()
                local htcfsBodySize = buffer(internalOffset + 8, 8):le_int64()
                local htcfsParam0 = buffer(internalOffset + 16, 8):le_int64()
                local htcfsParam1 = buffer(internalOffset + 24, 8):le_int64()
                local htcfsParam2 = buffer(internalOffset + 32, 8):le_int64()
                local htcfsParam3 = buffer(internalOffset + 40, 8):le_int64()
                local htcfsParam4 = buffer(internalOffset + 48, 8):le_int64()

                -- Print htcfs header
                local htcfsDescription = "Htcfs Packet (" .. HtcfsPacketType[htcfsPacketType] .. ")"
                local htcfsTree = tree:add(htcfs_proto, buffer(htcfsOffset, htcfsPacketSize), htcfsDescription)

                htcfsTree:add(buffer(internalOffset + 2, 2), "Version: " .. htcfsVersion)
                htcfsTree:add(buffer(internalOffset + 4, 2), "Packet category: " .. PacketCategory[htcfsPacketCategory])
                htcfsTree:add(buffer(internalOffset + 6, 2), "Packet type: " .. HtcfsPacketType[htcfsPacketType])
                htcfsTree:add(buffer(internalOffset + 8, 8), "Body size: " .. htcfsBodySize)
                htcfsTree:add(buffer(internalOffset + 16, 8), "Param 0: " .. htcfsParam0)
                htcfsTree:add(buffer(internalOffset + 24, 8), "Param 1: " .. htcfsParam1)
                htcfsTree:add(buffer(internalOffset + 32, 8), "Param 2: " .. htcfsParam2)
                htcfsTree:add(buffer(internalOffset + 40, 8), "Param 3: " .. htcfsParam3)
                htcfsTree:add(buffer(internalOffset + 48, 8), "Param 4: " .. htcfsParam3)

                if htcfsBodySize > 0 then
                    -- Htcfs body
                    local htcfsBody = buffer(internalOffset + 64, bodySize):string()

                    -- Print htcfs body
                    htcfsTree:add(buffer(internalOffset + 64, htcfsBodySize), "Body: " .. htcfsBody)
                end

            end
        end

        -- Update offset
        offset = offset + packetSize
    end
end

tcp_table = DissectorTable.get("tcp.port")
tcp_table:add(20180, htclow_proto)
