﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "BridgeCallbackHandler.h"

BridgeEnumCallbackHandler::BridgeEnumCallbackHandler() : bridgeName( NULL ), totalBridgesFound( 0 ), isBridgeFound( false ){};
BridgeEnumCallbackHandler::BridgeEnumCallbackHandler( const char* name ) : totalBridgesFound( 0 ), isBridgeFound( false )
{
    bridgeName = reinterpret_cast<char*>( portability::mem::malloc( strlen( name ) + 1 ) );
    bridgeName = strcpy( bridgeName, name );
}

BridgeEnumCallbackHandler::~BridgeEnumCallbackHandler()
{
    if( bridgeName )
    {
        portability::mem::free( bridgeName );
    }

    discoveredBridgeNames.clear();
    discoveredBridgeIPs.clear();
    discoveredBridgePorts.clear();
    discoveredBridgeMacAddrs.clear();
}

// Callbacks
int BridgeEnumCallbackHandler::BridgeEnumCallback( void* ctx, int operationResult, BridgeHandle hOneBridge )
{
    BridgeEnumCallbackHandler* handler = (BridgeEnumCallbackHandler*)ctx;

    // On error, return immediately
    if( operationResult < 0 )
    {
        return operationResult;
    }

    // If bridge found, print info
    if( hOneBridge )
    {
        const char* found_bridge_name = BridgeHandleGetName( hOneBridge );
        const char* found_bridge_ip = BridgeHandleGetIP( hOneBridge );
        int found_bridge_port = GetBridgeHandleIpPort( hOneBridge );
        const char* found_bridge_mac_addr = GetBridgeHandleMacAddr( hOneBridge );

        // If we're searching for a specific bridge, print only if that's found
        if( handler->bridgeName != NULL )
        {
            if( strcmp( handler->bridgeName, found_bridge_name ) == 0 || strcmp( handler->bridgeName, found_bridge_ip ) == 0 )
            {
                printf( "%s, %s, %d, %s\n", found_bridge_name, found_bridge_ip, found_bridge_port, found_bridge_mac_addr );
                printf( "Found bridge.\n" );
                handler->isBridgeFound = true;

                // Request search stop
                return 0;
            }
        }
        // Else print every bridge we find (name, ip, port, mac)
        else
        {
            printf( "%s, %s, %d, %s\n", found_bridge_name, found_bridge_ip, found_bridge_port, found_bridge_mac_addr );
            handler->discoveredBridgeNames.push_back( portability::stl::string( found_bridge_name ) );
            handler->discoveredBridgeIPs.push_back( portability::stl::string( found_bridge_ip ) );
            handler->discoveredBridgePorts.push_back( found_bridge_port );
            handler->discoveredBridgeMacAddrs.push_back( found_bridge_mac_addr );
        }

        // Ask for more bridges
        return 1;
    }
    // If hOneBridge is NULL, there are no more bridges to be found.
    else
    {
        handler->totalBridgesFound = operationResult;
        return 0;
    }
}


BridgeCallbackHandlerBase::BridgeCallbackHandlerBase()
    : lastOperationResources( NULL ), lastOperationResult( BRIDGE_OPERATION_FAILED_BIT )
{
    pAutoResetEvent = new AutoResetEvent();
}

BridgeCallbackHandlerBase::~BridgeCallbackHandlerBase()
{
    delete lastOperationResources;
    delete pAutoResetEvent;
}

void BridgeCallbackHandlerBase::Reset()
{
    lastOperationResult = BRIDGE_OPERATION_FAILED_BIT;
    delete lastOperationResources;
    lastOperationResources = NULL;
    pAutoResetEvent->Reset();
}

AutoResetEvent* BridgeCallbackHandlerBase::GetAutoResetEvent() const
{
    return pAutoResetEvent;
}
void* BridgeCallbackHandlerBase::GetLastOperationResources() const
{
    return lastOperationResources;
}
int BridgeCallbackHandlerBase::GetLastOperationResult() const
{
    return lastOperationResult;
}

int BridgeCallbackHandlerBase::OnOperationComplete( void* ctx, int operationResult, void* operationResources )
{
    TICS_UNUSED(operationResources);

    BridgeCallbackHandlerBase* handler = (BridgeCallbackHandlerBase*)ctx;
    handler->lastOperationResult = operationResult;
    handler->GetAutoResetEvent()->Set();

    return operationResult;

}

int GetConfigSectionCallbackHandler::OnGetConfigSection( void* ctx, int operationResult, void* operationResources )
{
    GetConfigSectionCallbackHandler* handler = (GetConfigSectionCallbackHandler*)ctx;

    if( handler->lastOperationResources != NULL )
    {
        portability::mem::free( handler->lastOperationResources );
        handler->lastOperationResources = NULL;
    }

    if( operationResult != BRIDGE_ERROR_SUCCESS )
    {
        handler->lastOperationResult = operationResult;
        handler->pAutoResetEvent->Set();
        return operationResult;
    }

    if( operationResources == NULL )
    {
        handler->lastOperationResult = operationResult;
        handler->GetAutoResetEvent()->Set();
        return operationResult;
    }

    BridgeCallbackInfo* info = static_cast<BridgeCallbackInfo*>( operationResources );
    char* config = const_cast<char*>( info->zstrInfo );

    int configLength = strlen( config );

    if( config && strlen( config ) > 0 )
    {
        if( config[strlen( config ) - 1] != '\n' )
        {
        }
        printf(config);
    }

    handler->lastOperationResources = portability::mem::malloc( configLength + 1 );
    strncpy( (char*)( handler->lastOperationResources ), config, configLength );

    handler->lastOperationResult = operationResult;
    handler->GetAutoResetEvent()->Set();

    return operationResult;
}

int GetBridgeFirmwareVersionCallbackHandler::OnGetBridgeFirmwareVersion( void* ctx, int operationResult, void* bridgeFirmwareVersion )
{
    GetBridgeFirmwareVersionCallbackHandler* handler = (GetBridgeFirmwareVersionCallbackHandler*)ctx;

    if(operationResult == BRIDGE_ERROR_SUCCESS)
    {
        BridgeFirmwareVersion* versionInfo = (BridgeFirmwareVersion*)bridgeFirmwareVersion;
        printf("Package Version : %d\n", versionInfo->packageVersion);
        printf("Firmware Branch : %s\n", versionInfo->firmwareBranch);
        printf("Firmware Hash   : ");
        for(int i = 0; i < 20; i++)
        {
            printf("%x", versionInfo->firmwareHash[i]);
        }
        printf("\n");
        printf("Tics Branch     : %s\n", versionInfo->ticsBranch);
        printf("Tics Hash       : ");
        for(int i = 0; i < 20; i++)
        {
            printf("%x", versionInfo->ticsHash[i]);
        }
        printf("\n");

        delete handler->lastOperationResources;
        handler->lastOperationResources = portability::mem::malloc( sizeof(BridgeFirmwareVersion) );
        memcpy( handler->lastOperationResources, bridgeFirmwareVersion, sizeof( BridgeFirmwareVersion ) );
    }
    else
    {
        printf( "BridgeFirmwareGetVersion Failed: %d.\n", operationResult );
    }

    handler->lastOperationResult = operationResult;
    handler->GetAutoResetEvent()->Set();

    return operationResult;
}

int BridgeTargetStateCallbackHandler::OnGetTargetStatus(void *ctx, int operationResult, TargetStatusFlags flags, TargetStatusFlags validityMask, BridgeCallbackInfo *resources)
{
    TICS_UNUSED(resources);

    BridgeTargetStateCallbackHandler* handler = (BridgeTargetStateCallbackHandler*)ctx;

    if(operationResult == BRIDGE_ERROR_SUCCESS)
    {
        // TODO: Parse and show the result cleanly (?)
        printf("Flags  : 0x%x\n", flags);
        printf("Mask   : 0x%x\n", validityMask);
    }
    else
    {
        printf("Failed to get TargetStatus : %d.\n", operationResult);
    }

    handler->lastOperationResult = operationResult;
    handler->GetAutoResetEvent()->Set();

    return operationResult;
}
