﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using Nintendo.FsAccessLogAnalysis;

namespace Nintendo.FsAccessLogChecker
{
    internal class Logger
    {
        internal enum Tag
        {
            Pass,
            Fail,
            WarnForAccessLog,
            WarnForInputFile,
            Info,
        }

        internal static bool IsOutputAccessSummary { get; set; } = false;
        internal static int OutputLine { get; set; } = 0;

        internal static Tag ResultToTag(bool isSuccess)
        {
            return isSuccess ? Tag.Pass : Tag.Fail;
        }

        internal static Tag ResultToTag(FsGuideline.CheckResult result, FsGuideline.CheckResult.ErrorType mask)
        {
            if ((result.ErrorFlags & mask) != 0)
            {
                return Tag.Fail;
            }
            else if ((result.WarningFlags & mask) != 0)
            {
                return Tag.WarnForAccessLog;
            }
            return Tag.Pass;
        }

        internal static void WriteWithFile<T>(T value)
        {
            ++OutputLine;
            Console.Write(value);
            FileOutputStreamWriter?.Write(value);
        }
        internal static void WriteLineWithFile<T>(T value)
        {
            ++OutputLine;
            Console.WriteLine(value);
            FileOutputStreamWriter?.WriteLine(value);
        }
        internal static void OutputTag(Tag tag)
        {
            switch (tag)
            {
                case Tag.Pass:
                    Console.ForegroundColor = ConsoleColor.Green;
                    WriteWithFile("[PASS]");
                    break;
                case Tag.Fail:
                    Console.ForegroundColor = ConsoleColor.Red;
                    WriteWithFile("[FAIL]");
                    break;
                case Tag.WarnForAccessLog:
                    Console.ForegroundColor = ConsoleColor.Yellow;
                    WriteWithFile("[WARN]");
                    break;
                case Tag.WarnForInputFile:
                    Console.ForegroundColor = ConsoleColor.Yellow;
                    WriteWithFile("[WARNING]");
                    break;
                case Tag.Info:
                    WriteWithFile("[INFO]");
                    break;
            }
            Console.ResetColor();
        }

        internal static void OutputFail(string message)
        {
            Logger.OutputTag(Logger.Tag.Fail);
            Logger.WriteLineWithFile(" " + message.Trim());
        }

        internal static void OutputWarning(string message)
        {
            Logger.OutputTag(Logger.Tag.WarnForInputFile);
            Logger.WriteLineWithFile(" " + message.Trim());
        }

        internal static void OutputInfo(string message)
        {
            Logger.OutputTag(Logger.Tag.Info);
            Logger.WriteLineWithFile(" " + message.Trim());
        }

        internal static StreamWriter FileOutputStreamWriter { get; set; } = null;
    }

    internal class FsGuidelineChecker : FsGuideline
    {
        public FsGuidelineChecker()
        {
            IsValid = true;
        }

        public bool IsValid { get; private set; }

        public void SetUp(FsAccessLogAnalyzer analyzer)
        {
            Analyzer = analyzer;
        }

        public void OnParseLine(string line, FsAccessLog log, bool isVerbose, bool isQuiet)
        {
            // log がなかった場合はユーザー出力のログ
            if (log == null)
            {
                if (isVerbose)
                {
                    Logger.WriteLineWithFile(line);
                }
                return;
            }
            if (!log.IsAnalyzable())
            {
                if (line != string.Empty)
                {
                    WriteLine(line, log, isQuiet);
                }
                return;
            }

            // log の解析対応を行う
            if (IsAccumulatedLog(CheckingList, log))
            {
                // リストの末尾から追加したいログへの時間差を取得
                long distanceMilliseconds = CheckingList.GetDistanceMilliseconds(log);
                long listTotalMilliseconds = CheckingList.GetTotalMilliseconds();
                if (listTotalMilliseconds + distanceMilliseconds >= CheckMillisecondsStep)
                {
                    // log の Start 時点でチェック時間間隔を超えていた場合
                    UpdateAccessSummary(null, true);
                    Check();
                    UpdateAccessSummary(log, false);

                    // 解析後、ログを追加
                    AddCheckingLog(log);
                    // 次のログが遠くにある場合の、スライドする時間を計算
                    long slideMillisecondsDiff = listTotalMilliseconds - (CheckMillisecondsStep - distanceMilliseconds);
                    long slideMillisceonds = ((slideMillisecondsDiff / LogListSlideMilliseconds) + 1) * LogListSlideMilliseconds;
                    SlideLogList(slideMillisceonds);
                }
                else
                {
                    AddCheckingLog(log);
                    UpdateAccessSummary(log, true);
                    Check();
                    SlideLogList(LogListSlideMilliseconds);
                }
            }
            else
            {
                AddCheckingLog(log);
                UpdateAccessSummary(log, false);
            }
            if (line != string.Empty)
            {
                WriteLine(line, log, isQuiet);
            }
        }

        private void WriteLine(string line, FsAccessLog log, bool isQuiet)
        {
            // ログにエラーが有った場合は、エラー情報をつけて常に出力する
            if (log.HasLogError())
            {
                Console.ForegroundColor = ConsoleColor.DarkRed;
                Logger.WriteLineWithFile(line);
                if (log.AnalyzerError != null)
                {
                    Logger.WriteWithFile("[Log Error] ");
                    Logger.WriteLineWithFile(log.AnalyzerError);
                }
                if (log.ParserError != null)
                {
                    Logger.WriteWithFile("[Log Error] ");
                    Logger.WriteLineWithFile(log.ParserError);
                }
                if (log.OutputStateError != null)
                {
                    Logger.WriteWithFile("[Log Error] ");
                    Logger.WriteLineWithFile(log.OutputStateError);
                }
                Console.ResetColor();
            }
            else if (!isQuiet)
            {
                Logger.WriteLineWithFile(line);
            }
        }

        public void TearDown()
        {
            UpdateAccessSummary(null, true);
            if (Dirty && CheckingList.Any())
            {
                Check();
            }

            CheckPrecondition();
            OutputOverallResult();
        }

        private void Check()
        {
            Logger.WriteLineWithFile(MessageFormat.GetSeparateMessageCheckLogStart(CheckMillisecondsStep));
            GuidelineCheck();
            Logger.WriteLineWithFile(MessageFormat.SeparateMessageCheckLogEnd);
        }

        private void CheckPrecondition()
        {
            if (!Analyzer.IsExpectedSdkVersion(FsGuideline.RequiredSdkVersion))
            {
                IsValid = false;
            }
            if (!Analyzer.IsSpecNX())
            {
                IsValid = false;
            }
            if (!IsEnoughLogTime(Analyzer))
            {
                IsValid = false;
            }
            if (Analyzer.ErrorLogList.Any())
            {
                IsValid = false;
            }
            if (!Analyzer.IsCompleteLog)
            {
                IsValid = false;
            }
        }

        private void AddCheckingLog(FsAccessLog log)
        {
            CheckingList.Add(log);
            Dirty = true;
        }

        private class CheckResultSummary
        {
            public CheckResultSummary(CheckResult.ErrorType flag, Logger.Tag tag, string summary, List<string> details)
            {
                ErrorFlag = flag;
                Tag = tag;
                Summary = summary;
                Details = details;
            }
            public CheckResult.ErrorType ErrorFlag { get; set; }
            public Logger.Tag Tag { get; set; }
            public string Summary { get; set; }
            public List<string> Details { get; set; }
            public string Title { get { return MessageFormat.GetCheckLogTitle(ErrorFlag); } }
        }

        private CheckResult GetGuidelineCheckResult(List<FsAccessLog> list)
        {
#if FSACCESSLOGCHECKER_FOR_SYSTEM
            return FsGuideline.CheckForSystem(list);
#else
            if (Analyzer.ForSystem)
            {
                return FsGuideline.CheckForSystem(list);
            }
            return FsGuideline.Check(list);
#endif
        }

        private void GuidelineCheck()
        {
            var splitList = Analyzer.SplitWithMountTarget(CheckingList, FsAccessLogChecker.CheckMountTargets.ToArray());
            if (splitList.Any())
            {
                foreach (var targetLogs in splitList)
                {
                    CheckResult result = GetGuidelineCheckResult(targetLogs.Value);
                    if (result.IsVerificationError)
                    {
                        IsValid = false;
                    }

                    CheckResultList.Add(Logger.OutputLine, result);

                    // 結果の出力
                    Logger.WriteLineWithFile(targetLogs.Key.ToString());
                    foreach (var summary in EnumerateCheckResultSummary(result))
                    {
                        OutputResult(summary.Tag, summary.Title, summary.Summary);
                        if (summary.Details != null)
                        {
                            if (summary.Details.Any())
                            {
                                foreach (var detail in summary.Details)
                                {
                                    Logger.WriteLineWithFile(detail);
                                }
                            }
                            else
                            {
                                Logger.WriteLineWithFile("    none");
                            }
                        }
                    }
                }
            }
            Dirty = false;
        }

        private IEnumerable<CheckResultSummary> EnumerateCheckResultSummary(CheckResult result)
        {
            yield return new CheckResultSummary(
                CheckResult.ErrorType.WriteSize,
                Logger.ResultToTag(result, CheckResult.ErrorType.WriteSize),
                Utility.ToMegaByteString(result.WriteSizePerMinutes) + "/min", null);
            yield return new CheckResultSummary(
                CheckResult.ErrorType.WriteAccessCount,
                Logger.ResultToTag(result, CheckResult.ErrorType.WriteAccessCount),
                MessageFormat.FormatAverageValue(result.WriteAccessCountPerMinutes) + " times/min", null);
            if (result.FailMountCountPerSecondsList.Count > 0)
            {
                yield return new CheckResultSummary(
                    CheckResult.ErrorType.MountCount,
                    Logger.Tag.Fail,
                    string.Empty,
                    GetCheckResultMountCountDetails(result.FailMountCountPerSecondsList, false));
            }
            if (result.WarnMountCountPerSecondsList.Count > 0)
            {
                yield return new CheckResultSummary(
                    CheckResult.ErrorType.MountCount,
                    Logger.Tag.WarnForAccessLog,
                    string.Empty,
                    GetCheckResultMountCountDetails(result.WarnMountCountPerSecondsList, false));
            }
            if ((result.FailMountCountPerSecondsList.Count == 0 && result.WarnMountCountPerSecondsList.Count == 0)
                || result.PassMountCountPerSecondsList.Count != 0)
            {
                yield return new CheckResultSummary(
                    CheckResult.ErrorType.MountCount,
                    Logger.Tag.Pass,
                    string.Empty,
                    GetCheckResultMountCountDetails(
                        result.PassMountCountPerSecondsList, true).Take(MountListMaxDisplayCount).ToList());
            }
            if (result.FailReadAccessList.Count > 0)
            {
                result.FailReadAccessList.Sort();
                yield return new CheckResultSummary(
                        CheckResult.ErrorType.SameAddressRead,
                        Logger.Tag.Fail,
                    result.ForSystem ? MessageFormat.ForSystemMarker : string.Empty,
                    GetCheckResultReadAccessDetails(result.FailReadAccessList, false));
            }
            if (result.WarnReadAccessList.Count > 0)
            {
                result.WarnReadAccessList.Sort();
                yield return new CheckResultSummary(
                    CheckResult.ErrorType.SameAddressRead,
                    Logger.Tag.WarnForAccessLog,
                    result.ForSystem ? MessageFormat.ForSystemMarker : string.Empty,
                    GetCheckResultReadAccessDetails(result.WarnReadAccessList, false));
            }
            if ((result.FailReadAccessList.Count == 0 && result.WarnReadAccessList.Count == 0)
               || result.PassReadAccessList.Count != 0)
            {
                result.PassReadAccessList.Sort();
                yield return new CheckResultSummary(
                    CheckResult.ErrorType.SameAddressRead,
                    Logger.Tag.Pass,
                    result.ForSystem ? MessageFormat.ForSystemMarker : string.Empty,
                    GetCheckResultReadAccessDetails(
                        result.PassReadAccessList.Take(ReadAccessListMaxDisplayCount), true));
            }
        }
        private List<string> GetCheckResultMountCountDetails(Dictionary<string, double> MountCountPerSecondsList, bool outputRank)
        {
            int i = 0;
            return MountCountPerSecondsList.OrderByDescending(info => info.Value).Select(info =>
            {
                return string.Format("    {0}{1}: {2} times/sec",
                    outputRank ? MessageFormat.IntToOrdinal(++i) : string.Empty,
                    info.Key,
                    MessageFormat.FormatAverageValue(info.Value));
            }).ToList();
        }

        private List<string> GetCheckResultReadAccessDetails(IEnumerable<SameAddressAccessInfo> resultList, bool outputRank)
        {
            int i = 0;
            List<string> messages = new List<string>();
            foreach (var info in resultList)
            {
                messages.Add(
                    string.Format(
                        "    {0}{1}",
                        outputRank ? MessageFormat.IntToOrdinal(++i) : string.Empty,
                        info));
            }
            return messages;
        }

        private void OutputResult(Logger.Tag tag, string name, string value)
        {
            OutputResult(tag, name, value, null);
        }
        private void OutputResult(Logger.Tag tag, string name, string value, string appendMessage)
        {
            Logger.OutputTag(tag);
            string message = string.Format(" {0,-16}", name) + ": ";
            if (value != null)
            {
                message += value;
            }
            if (appendMessage != null && (tag != Logger.Tag.Pass))
            {
                message += string.Format(" ({0})", appendMessage);
            }
            Logger.WriteLineWithFile(message);
        }

        private void OutputOverallResult()
        {
            Logger.WriteLineWithFile(MessageFormat.SeparateMessageCheckOverallResultStart);

            if (IsValid)
            {
                Logger.WriteWithFile("Result: ");
                Logger.OutputTag(Logger.Tag.Pass);
                if (Analyzer.ForSystem)
                {
                    // システム向け頻度チェックを行ったことを示す
                    Logger.WriteLineWithFile(" " + MessageFormat.ForSystemMarker);
                }
                else
                {
                    Logger.WriteLineWithFile(string.Empty);
                }
            }
            else
            {
                Logger.WriteWithFile("Result: ");
                Logger.OutputTag(Logger.Tag.Fail);
                Logger.WriteLineWithFile(string.Empty);
                Logger.WriteLineWithFile(string.Empty);

                OutputDetails();
                //OutputAllErrors();
            }

            // Host へのアクセスがあった場合、警告
            if (Analyzer.HasAnyAccess(FsMountTarget.Host))
            {
                Logger.OutputWarning("Access to the HOST was detected. This ROM cannot be used for master submission.");
            }

            // sdk 3.0.0 未満かつシステム向けログがあった場合、警告
            if (!Analyzer.IsExpectedSdkVersion(new Version(3, 0, 0)) && Analyzer.HasSystemAccess() && !Analyzer.ForSystem)
            {
                Logger.OutputWarning("System access log was detected, but sdk is old. Please use NintendoSDK 3.0.0 or later.");
            }

            Logger.WriteLineWithFile(MessageFormat.SeparateMessageCheckOverallResultEnd);
        }

        private void OutputDetails()
        {
            Logger.WriteLineWithFile("Details:");
            OutputResult(Logger.ResultToTag(Analyzer.IsExpectedSdkVersion(FsGuideline.RequiredSdkVersion)),
                "SdkVersion", Analyzer.LogSdkVersion,
                Analyzer.HasSdkVersion() ? ("expect: Greater than " + FsGuideline.RequiredSdkVersion) : null);
            OutputResult(Logger.ResultToTag(Analyzer.IsSpecNX()),
                "Spec      ", Analyzer.LogSpec);
            OutputResult(Logger.ResultToTag(IsEnoughLogTime(Analyzer)),
                "TotalTime ", MessageFormat.FormatTime(Analyzer.GetTotalMilliseconds()),
                "expect: More than " + MessageFormat.FormatTotalMinutes(FsGuideline.RequiredLogTimeMilliseconds));
            if (!Analyzer.IsCompleteLog)
            {
                OutputResult(Logger.Tag.Fail, "Complete Log", "False");
            }

            CheckResult.ErrorType[] checkErrorTypes = (CheckResult.ErrorType[])Enum.GetValues(typeof(CheckResult.ErrorType));
            foreach (CheckResult.ErrorType errorType in checkErrorTypes.Skip(1))
            {
                Dictionary<int, CheckResult> warns = CheckResultList.Where(result => result.Value.WarningFlags.HasFlag(errorType)).ToDictionary(a => a.Key, a => a.Value);
                Dictionary<int, CheckResult> errors = CheckResultList.Where(result => result.Value.ErrorFlags.HasFlag(errorType)).ToDictionary(a => a.Key, a => a.Value);
                int warnCount = warns.Count();
                int errorCount = errors.Count();
                string title = MessageFormat.GetCheckLogTitle(errorType);
                if (errorCount > 0)
                {
                    OutputResult(Logger.Tag.Fail, errorType.ToString(), errorCount + " fails",
                    string.Format("Please refer to \"[FAIL] {0}\" in \"{1}\"", title, MessageFormat.SeparateMessageCheckLogStartTag));
                }
                if (warnCount > 0)
                {
                    OutputResult(Logger.Tag.WarnForAccessLog, errorType.ToString(), warnCount + " fails",
                        string.Format("Please refer to \"[WARN] {0}\" in \"{1}\"", title, MessageFormat.SeparateMessageCheckLogStartTag));
                }
                if (warnCount == 0 && errorCount == 0)
                {
                    OutputResult(Logger.Tag.Pass, errorType.ToString(), "0 fails", null);
                }
            }

            if (Analyzer.ErrorLogList.Any())
            {
                OutputResult(Logger.Tag.Fail, "Log Error", "Please refer to \"[Log Error]\"");
            }
        }

        private void OutputAllErrors(string output)
        {
            Logger.WriteLineWithFile("ListUp:");
            if (CheckResultList.Any(result => result.Value.IsVerificationError))
            {
                string outputFilePath = output != null ? System.IO.Path.GetFullPath(output) : "@";
                foreach (var resultPair in CheckResultList)
                {
                    int line = resultPair.Key;
                    var result = resultPair.Value;
                    foreach (var summary in EnumerateCheckResultSummary(result))
                    {
                        if (result.ErrorFlags.HasFlag(summary.ErrorFlag))
                        {
                            string message = string.Format("{0}:{1}: {2}: {3}: {4}",
                                outputFilePath, line, result.MountTarget.ToString(), summary.Title, summary.Summary);
                            Logger.WriteLineWithFile(message);
                            foreach (var detail in summary.Details)
                            {
                                Logger.WriteLineWithFile(detail);
                            }
                        }
                    }
                }
            }
        }

        private void UpdateAccessSummary(FsAccessLog log, bool flush)
        {
            if (!Logger.IsOutputAccessSummary)
            {
                return;
            }
            if (log != null)
            {
                AccessSummaryCheckList.Add(log);
            }
            if (AccessSummaryCheckList.Any())
            {
                if (flush || FsAccessLogAnalyzer.GetStartPointElapsedMilliseconds(AccessSummaryCheckList) > AccessSummaryOutputMilliseconds)
                {
                    Logger.WriteLineWithFile(MessageFormat.SeparateMessageAccessSummaryStart);
                    OutputAccessSummary(AccessSummaryCheckList);
                    Logger.WriteLineWithFile(MessageFormat.SeparateMessageAccessSummaryEnd);
                    AccessSummaryCheckList.Clear();
                }
            }
        }
        private void OutputAccessSummary(List<FsAccessLog> list)
        {
            // WriteFile
            {
                var writefiles = list.Where(log => log.GetFunctionFullName() == "WriteFile");
                var groups = writefiles.ToLookup(log => log.GetHashCode());
                foreach (var group in groups)
                {
                    var logs = group.ToList();
                    string path = logs.First().Path;
                    Logger.WriteLineWithFile(string.Format("W: {0}: {1}", path, Utility.ToReadabilityByteString(FsAccessLogAnalyzer.GetTotalWriteSize(logs))));
                }
            }
            // ReadFile
            {
                var readfiles = list.Where(log => log.GetFunctionFullName() == "ReadFile");
                var groups = readfiles.ToLookup(log => log.GetHashCode());
                foreach (var group in groups)
                {
                    var logs = group.ToList();
                    string path = logs.First().Path;
                    Logger.WriteLineWithFile(string.Format("R: {0}: {1}", path, Utility.ToReadabilityByteString(FsAccessLogAnalyzer.GetTotalReadSize(logs))));
                }
            }
        }

        private void SlideLogList(long slideMilliseconds)
        {
            CheckingList.SubList(slideMilliseconds);
        }

        // Analyzer 単位での状態
        private FsAccessLogList CheckingList { get; set; } = new FsAccessLogList();
        private List<FsAccessLog> AccessSummaryCheckList { get; set; } = new List<FsAccessLog>();
        private FsAccessLogAnalyzer Analyzer { get; set; }
        private bool Dirty { get; set; } = false;
        private Dictionary<int, CheckResult> CheckResultList { get; set; } = new Dictionary<int, CheckResult>();

        // LogListSlideMilliseconds 間隔ごとにスライドしながら解析していきます
        private readonly long LogListSlideMilliseconds = 60 * 1000;
        private readonly long AccessSummaryOutputMilliseconds = 60 * 1000;
        // 結果出力において、Mount の詳細を表示する個数
        private readonly int MountListMaxDisplayCount = 3;
        // 結果出力において、ReadAccessList の詳細を表示する個数
        private readonly int ReadAccessListMaxDisplayCount = 3;
    }
}
