﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

namespace Nintendo.FsAccessLogAnalysis
{
    public enum FsMountTarget
    {
        Unknown,
        Rom,
        SdCard,
        Nand,
        Host,
        AddOnContent,
        ApplicationPackage,
        Code,
        Content,
        GameCard,
        Logo,
        UpdatePartition,
        Bcat,
        Nfp,
    }

    public class FsMountUtility
    {
        public static string GetMountName(string path)
        {
            int index = path.IndexOf(":/");
            if (index < 0)
            {
                return null;
            }
            return path.Substring(0, index);
        }

        public static string RemoveMountName(string path)
        {
            int index = path.IndexOf(":/");
            if (index < 0)
            {
                return path;
            }
            return path.Substring(index + 2);
        }

        public static FsMountTarget GetSystemMountTarget(string path)
        {
            // 現状は Unknown を返す
            // SystemMountName とマウントターゲットの関連付けを構築し、名前引きで解決する
            // また、MountLogInfo にも常駐ターゲットとして予め登録するように変更をすること
            return FsMountTarget.Unknown;
        }

        public static string GetMountArgumentsString(FsAccessLog log)
        {
            string result = string.Empty;

            Func<string, object> getDefaultArgumentString = (string name) =>
            {
                if (log.Function == "MountCacheStorage" && name == "index")
                {
                    return 0;
                }
                return null;
            };

            Action<string, object> appendString = (string name, object argument) =>
            {
                object item = argument ?? getDefaultArgumentString(name);

                if (item != null)
                {
                    if (result.Length > 0)
                    {
                        result += ", ";
                    }
                    result += name + ": " + item.ToString();
                }
            };
            appendString("UserId", log.UserId);
            appendString("DataId", log.DataId);
            if (FsGuideline.IsCacheStorageIndexUsed || !string.Equals(log.Function, "MountCacheStorage"))
            {
                appendString("index", log.Index);
            }
            appendString("ApplicationId", log.ApplicationId);
            appendString("BisPartitionId", log.BisPartitionId);
            appendString("ContentStorageId", log.ContentStorageId);
            appendString("SaveDataId", log.SaveDataId);
            appendString("SaveDataSpaceId", log.SaveDataSpaceId);
            appendString("ImageDirectoryId", log.ImageDirectoryId);
            return "(" + result + ")";
        }
    }

    public class FsMountInfo
    {
        public string Name { get; set; }
        public FsMountTarget Target { get; set; }
        public MillisecondsPeriod Period { get; set; } = new MillisecondsPeriod();
        public bool ForSystem { get; set; }
        // マウント先の識別用ハッシュ
        public int MountHash { get; set; }
        // マウントを行ったログ
        public FsAccessLog MountLog { get; set; }

        public override string ToString()
        {
            return Name + "(" + Target.ToString() + "): " + Period.ToString();
        }
        public int CompareTo(FsMountInfo other)
        {
            return Period.CompareTo(other.Period);
        }

        public class OverlapCompare : IComparer<FsMountInfo>
        {
            public int Compare(FsMountInfo target, FsMountInfo find)
            {
                var compare = new MillisecondsPeriod.OverlapCompare();
                return compare.Compare(target.Period, find.Period);
            }
        }
    }

    public class MountLogInfo
    {
        private static readonly Regex DriveNamePattern = new Regex("^[a-zA-Z]$");
        private bool IsDriveName(string mountName)
        {
            return (mountName != null) && (DriveNamePattern.IsMatch(mountName));
        }

        internal FsMountInfo GetMountInfo(string mountName, long time)
        {
            // name がドライブ名の場合、nn::fs::detail::HostRootFileSystemMountName に置換する
            string name = IsDriveName(mountName) ? "@Host" : mountName;
            if (name != null && MountList.ContainsKey(name))
            {
                var list = MountList[name];
                if (list.Any())
                {
                    if (time < 0)
                    {
                        return list.Last();
                    }
                    else
                    {
                        FsMountInfo search = new FsMountInfo();
                        search.Period.Start = time;
                        search.Period.End = time;
                        int find = list.BinarySearch(search, new FsMountInfo.OverlapCompare());
                        if (find >= 0)
                        {
                            return list[find];
                        }
                    }
                }
            }
            return null;
        }

        internal bool RecordLog(FsAccessLog log)
        {
            string functionFullName = log.GetFunctionFullName();
            if (FsFunction.IsMountFunction(functionFullName))
            {
                FsMountInfo mountInfo = AddMountInfo(log);
                RegisterMountInfo(log, mountInfo);
                log.Type = FsApiType.Mount;
                return true;
            }
            else if (FsFunction.IsUnmountFunction(functionFullName))
            {
                FsMountInfo mountInfo = AddUnmountInfo(log);
                RegisterMountInfo(log, mountInfo);
                log.Type = FsApiType.Mount;
                return true;
            }
            return false;
        }

        internal void RegisterMountInfo(FsAccessLog log, FsMountInfo mountInfo)
        {
            log.MountTarget = mountInfo.Target;
            log.MountForSystem = mountInfo.ForSystem;
            log.MountHash = mountInfo.MountHash;
        }

        internal FsMountInfo GetMountInfoByPath(FsAccessLog log)
        {
            string mountName = log.Path == null ? null : FsMountUtility.GetMountName(log.Path);
            if (mountName == null)
            {
                // マウント名が空の場合、システム内部でパスが付与されている可能性があるため、
                // そちらからの解決を試みる
                FsFunction.Names.OpenAccessMountNameSubstitute.TryGetValue(log.GetFunctionFullName(), out mountName);
            }
            return GetMountInfo(mountName, -1);
        }

        internal void RegisterMountInfoByGroupAccess(FsAccessLog log)
        {
            FsFunction.GroupAccess groupAccess = FsFunction.GetGroupAccess(log);
            if (groupAccess != null)
            {
                log.MountTarget = groupAccess.MountTarget;
                log.MountForSystem = false;
                log.MountHash = 0;
            }
        }

        private FsMountInfo AddMountInfo(FsAccessLog log)
        {
            return AddMountInfo(log, GetMountTarget(log));
        }
        private FsMountInfo AddMountInfo(FsAccessLog log, FsMountTarget target)
        {
            FsMountInfo mountInfo = new FsMountInfo();
            mountInfo.Name = GetMountName(log);
            mountInfo.Period.Start = log.Start;
            mountInfo.Period.End = -1;
            mountInfo.Target = target;
            mountInfo.ForSystem = FsFunction.IsSystemMountFunction(log);
            mountInfo.MountHash = CalcMountHash(log);
            mountInfo.MountLog = log;
            if (log.IsResultSuccess() || log.IsNfpAccess())
            {
                List<FsMountInfo> list;
                if (MountList.TryGetValue(mountInfo.Name, out list))
                {
                    var lastMountInfo = list.Last();
                    var lastPeriod = lastMountInfo.Period;
                    if (lastPeriod.End < 0)
                    {
                        // MountRom が内部的に MountHost を呼ぶようなことがある。
                        // ログ的には内部的に呼ばれた関数が先に出力され、呼び出し側後に出力される。
                        // さらに、時間が最初のログを後のログが内包する。
                        // ソートされている場合があるので、どちらかが一方内包していたら、内部呼び出しと判定する
                        if (lastMountInfo.MountLog.Period.IsInclusion(log.Period) || log.Period.IsInclusion(lastMountInfo.MountLog.Period))
                        {
                            // 内部的に呼ばれたと判断し、呼び出し側のターゲット設定ではなく、呼び出された側のターゲットを引続
                            lastPeriod.Start = log.Start;
                            return lastMountInfo;
                        }

                        // 上記条件に合わない場合は、ログから Unmount が抜けている可能性があるので、
                        // ここまでの時刻で期間として区切り、新しい mountInfo を登録する
                        Debug.Assert(false, "Call of Unmount could not be confirmed");
                        lastPeriod.End = log.Start;
                    }
                    list.Add(mountInfo);
                }
                else
                {
                    list = new List<FsMountInfo>();
                    list.Add(mountInfo);
                    MountList.Add(mountInfo.Name, list);
                }
            }
            return mountInfo;
        }

        private FsMountInfo AddUnmountInfo(FsAccessLog log)
        {
            string mountName = GetMountName(log);
            if (!MountList.ContainsKey(mountName))
            {
                FsMountInfo mountInfo = new FsMountInfo();
                mountInfo.Name = mountName;
                mountInfo.Period.Start = 0;
                mountInfo.Period.End = 0;
                mountInfo.Target = FsMountUtility.GetSystemMountTarget(mountName);
                List<FsMountInfo> list = new List<FsMountInfo>();
                list.Add(mountInfo);
                MountList.Add(mountInfo.Name, list);
            }
            var last = MountList[mountName].Last();
            if (last.Period.End >= 0)
            {
                Debug.Print(mountName + " is not Mount, after " + last.Period.End + "ms");
            }
            last.Period.End = log.End;
            return last;
        }

        private FsMountTarget GetMountTarget(FsAccessLog log)
        {
            if (log.GetFunctionFullName() != null)
            {
                FsMountTarget? target = FsFunction.GetMountTarget(log);
                if (target.HasValue)
                {
                    return target.Value;
                }
            }
            return FsMountUtility.GetSystemMountTarget(GetMountName(log));
        }

        private int CalcMountHash(FsAccessLog log)
        {
            string hashString = FsFunction.GetMountFunctionNameHashString(log.GetFunctionFullName());
            hashString += "-" + log.UserId?.ToString();
            hashString += "-" + log.DataId?.ToString();
            hashString += "-" + (log.Index ?? 0).ToString();
            hashString += "-" + log.ApplicationId?.ToString();
            hashString += "-" + log.BisPartitionId?.ToString();
            hashString += "-" + log.ContentStorageId?.ToString();
            hashString += "-" + log.SaveDataId?.ToString();
            hashString += "-" + log.SaveDataSpaceId?.ToString();
            return hashString.GetHashCode();
        }

        private string GetMountName(FsAccessLog log)
        {
            if (log.Name != null)
            {
                return log.Name;
            }
            string name;
            if (FsFunction.Names.MountNameSubstitute.TryGetValue(log.GetFunctionFullName(), out name))
            {
                return name;
            }
            return "Unknown";
        }

        private Dictionary<string, List<FsMountInfo>> MountList = new Dictionary<string, List<FsMountInfo>>();
    }
}
