﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

namespace Nintendo.FsAccessLogAnalysis
{
    public class FsFunction
    {
        // 複数のアクセスがまとまった API 用の情報クラス
        public class GroupAccess
        {
            public GroupAccess(ulong writeSize, int writeCount, ulong readSize, int readCount, FsMountTarget mountTarget)
            {
                WriteSize = writeSize;
                WriteCount = writeCount;
                ReadSize = readSize;
                ReadCount = readCount;
                MountTarget = mountTarget;
            }
            public ulong WriteSize { get; private set; }
            public int WriteCount { get; private set; }
            public ulong ReadSize { get; private set; }
            public int ReadCount { get; private set; }
            public FsMountTarget MountTarget { get; private set; }
        }

        public class Names
        {
            public static readonly string[] HandleOpenAccess = {
                "OpenFile",
                "OpenDirectory",
                "DeliveryCacheDirectory::Open",
                "DeliveryCacheFile::Open",
            };
            public static readonly string[] HandleCloseAccess = {
                "CloseFile",
                "CloseDirectory",
                "DeliveryCacheDirectory::Close",
                "DeliveryCacheFile::Close",
            };
            public static readonly string[] UnmountAccess = {
                "Unmount",
                "UnmountDeliveryCacheStorage",
            };

            // マウント関数名とマウントターゲットの関連付け
            public static readonly Dictionary<string, Func<FsAccessLog, FsMountTarget>> MountAssociation = new Dictionary<string, Func<FsAccessLog, FsMountTarget>>()
            {
                { "Mount",                      (FsAccessLog log) => { return FsMountTarget.Unknown; } },
                { "MountRom",                   (FsAccessLog log) => { return FsMountTarget.Rom; } },
                { "MountHost",                  (FsAccessLog log) => { return FsMountTarget.Host; } },
                { "MountHostRoot",              (FsAccessLog log) => { return FsMountTarget.Host; } },
                { "MountSaveData",              (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountSaveDataReadOnly",      (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountDeviceSaveData",        (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountSdCard",                (FsAccessLog log) => { return FsMountTarget.SdCard; } },
                { "MountAddOnContent",          (FsAccessLog log) => { return FsMountTarget.AddOnContent; } },
                { "MountTemporaryStorage",      (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountCacheStorage",          (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountDeliveryCacheStorage",  (FsAccessLog log) => { return FsMountTarget.Bcat; } },
            };
            public static readonly Dictionary<string, Func<FsAccessLog, FsMountTarget>> MountAssociationForSystem = new Dictionary<string, Func<FsAccessLog, FsMountTarget>>()
            {
                { "MountApplicationPackage", (FsAccessLog log) => { return FsMountTarget.ApplicationPackage; } },
                { "MountCode",               (FsAccessLog log) => { return FsMountTarget.Code; } },
                { "MountContent",            (FsAccessLog log) => { return FsMountTarget.Content; } },
                { "MountGameCardPartition",  (FsAccessLog log) => { return FsMountTarget.GameCard; } },
                { "MountLogo",               (FsAccessLog log) => { return FsMountTarget.Logo; } },
                { "MountRomOnFile",          (FsAccessLog log) => { return FsMountTarget.Rom; } },
                { "MountSystemSaveData",     (FsAccessLog log) => { return GetTargetFromMountSystemSaveData(log); } },
                { "MountBcatSaveData",       (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountBis",                (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountSystemData",         (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountCacheStorage",       (FsAccessLog log) => { return FsMountTarget.Nand; } },
                { "MountContentStorage",     (FsAccessLog log) => { return GetTargetFromMountContentStorage(log); } },
                { "MountImageDirectory",     (FsAccessLog log) => { return GetTargetFromMountImageDirectory(log); } },
                { "MountCloudBackupWorkStorage", (FsAccessLog log) => { return GetTargetFromMountCloudBackupWorkStorage(log); } },
                { "MountRegisteredUpdatePartition", (FsAccessLog log) => { return FsMountTarget.UpdatePartition; } },
                { "MountSystemBcatSaveData", (FsAccessLog log) => { return FsMountTarget.Nand; } },
            };

            // マウント関数名とハッシュの関連付け
            // マウント関数名が異なるが、同一の場所にアクセスするようなものを関連付けるためのものです。
            public static readonly Dictionary<string, string> MountFunctionNameAssociation = new Dictionary<string, string>()
            {
                { "MountSaveDataReadOnly",      "MountSaveData" },
                { "MountDeliveryCacheStorage",  "MountBcatSaveData" },
                { "MountHostRoot",              "MountHost" },
            };

            // 内部的にマウント名が付与されるマウント関数のマウント名解決をするための関連付け
            public static readonly string BcatDeliveryCacheMountName = "@bcat-dc";
            public static readonly Dictionary<string, string> MountNameSubstitute = new Dictionary<string, string>
            {
                { "MountDeliveryCacheStorage", BcatDeliveryCacheMountName },
                { "UnmountDeliveryCacheStorage", BcatDeliveryCacheMountName },
            };

            // 内部的にマウント名が付与される Open 関数のマウント名解決をするための関連付け
            public static readonly Dictionary<string, string> OpenAccessMountNameSubstitute = new Dictionary<string, string>
            {
                { "DeliveryCacheDirectory::Open",  BcatDeliveryCacheMountName },
                { "DeliveryCacheDirectory::Read",  BcatDeliveryCacheMountName },
                { "DeliveryCacheDirectory::Close",  BcatDeliveryCacheMountName },
                { "DeliveryCacheDirectory::~DeliveryCacheDirectory", BcatDeliveryCacheMountName },
                { "DeliveryCacheFile::Open",  BcatDeliveryCacheMountName },
                { "DeliveryCacheFile::Read",  BcatDeliveryCacheMountName },
                { "DeliveryCacheFile::Close",  BcatDeliveryCacheMountName },
                { "DeliveryCacheFile::~DeliveryCacheFile", BcatDeliveryCacheMountName },
            };

            public static readonly string[] WriteAccess = {
                "WriteFile",
                "FlushFile",
                "SetFileSize",
                "CreateFile",
                "DeleteFile",
                "CreateDirectory",
                "DeleteDirectory",
                "CleanDirectoryRecursively",
                "DeleteDirectoryRecursively",
                "RenameFile",
                "RenameDirectory",
                "Commit",
                "CommitSaveData",
            };

            public static readonly string[] ReadAccess = {
                "ReadFile",
                "DeliveryCacheFile::Read",
                "QueryApplicationPlayStatistics",
                "QueryApplicationPlayStatisticsForSystem",
                "EnableIndividualFileDataCache",
            };

            // 関数と GroupAccess の関連付け
            public static readonly Dictionary<string, Func<FsAccessLog, GroupAccess>> WriteGroupAccess = new Dictionary<string, Func<FsAccessLog, GroupAccess>>()
            {
                { "ExtendSaveData",             GetExtendSaveData },
                { "SaveScreenshot",             GetAlbumSaveScreenshotGroupAccess },
                { "SaveAndShareScreenshot",     GetAlbumSaveScreenshotGroupAccess },
                { "nfp::Mount",                 GetNfpMetaWriteGroupAccess },
                { "nfp::CreateApplicationArea", GetNfpMetaWriteGroupAccess },
                { "nfp::Flush",                 GetNfpMetaWriteGroupAccess },
                { "CreateCacheStorage",         GetCacheStorageWriteGroupAccess },
                { "DeleteCacheStorage",         GetCacheStorageWriteGroupAccess },
            };
            public static readonly Dictionary<string, Func<FsAccessLog, GroupAccess>> ReadGroupAccess = new Dictionary<string, Func<FsAccessLog, GroupAccess>>()
            {
                { "GetSaveDataSize",                 GetSaveDataSizeReadGroupAccess },
                { "EnumerateDeliveryCacheDirectory", GetBcatCacheDirectoryMetaReadGroupAccess },
                { "DeliveryCacheDirectory::Open",    GetBcatCacheFileMetaReadGroupAccess },
                { "DeliveryCacheDirectory::Read",    GetBcatCacheFileMetaReadGroupAccess },
                { "nfp::Mount",                      GetNfpMetaReadGroupAccess },
                { "nfp::CreateApplicationArea",      GetNfpMetaReadGroupAccess },
                { "nfp::Restore",                    GetNfpMetaReadGroupAccess },
                { "nfp::Flush",                      GetNfpMetaReadGroupAccess },
                { "GetCacheStorageSize",             GetCacheStorageReadGroupAccess },
                { "QueryApplicationPlayStatistics",  GetPdmReadGroupAccess },
                { "QueryApplicationPlayStatisticsForSystem", GetPdmReadGroupAccess },
            };

            private static readonly GroupAccess SaveDataSizeReadGroupAccess = new GroupAccess(0, 0, 0, 1, FsMountTarget.Nand);
            private static readonly GroupAccess SaveScreenshotGroupAccess1280x720 = new GroupAccess(500 * 1024, 6, 0, 0, FsMountTarget.Nand);
            private static readonly GroupAccess ExtendSaveDataGroupAccess = new GroupAccess(0, 1, 0, 0, FsMountTarget.Nand);
            private static readonly GroupAccess BcatCacheDirectoryMetaReadGroupAccess = new GroupAccess(0, 0, 646, 1, FsMountTarget.Bcat);
            private static readonly GroupAccess BcatCacheFileMetaReadGroupAccess = new GroupAccess(0, 0, 1284, 1, FsMountTarget.Bcat);
            private static readonly GroupAccess NfpMetaWriteGroupAccess = new GroupAccess(604, 8, 0, 0, FsMountTarget.Nand);
            private static readonly GroupAccess NfpMetaReadGroupAccess = new GroupAccess(0, 0, 0, 1, FsMountTarget.Nfp);
            private static readonly GroupAccess CacheStorageWriteGroupAccess = new GroupAccess(0, 1, 0, 0, FsMountTarget.Nand);
            private static readonly GroupAccess CacheStorageReadGroupAccess = new GroupAccess(0, 0, 0, 1, FsMountTarget.Nand);
            private static readonly GroupAccess PdmReadGroupAccess = new GroupAccess(0, 0, 0, 1, FsMountTarget.Nand);

            private static FsMountTarget GetTargetFromMountContentStorage(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "MountContentStorage", "mistake MountAssociation");
                if ((log.ContentStorageId != null) && (log.ContentStorageId.Id == ContentStorageId.SdCard))
                {
                    return FsMountTarget.SdCard;
                }
                return FsMountTarget.Nand;
            }
            private static FsMountTarget GetTargetFromMountImageDirectory(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "MountImageDirectory", "mistake MountAssociation");
                if ((log.ImageDirectoryId != null) && (log.ImageDirectoryId.Id == ImageDirectoryId.SdCard))
                {
                    return FsMountTarget.SdCard;
                }
                return FsMountTarget.Nand;
            }
            private static FsMountTarget GetTargetFromMountCloudBackupWorkStorage(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "MountCloudBackupWorkStorage", "mistake MountAssociation");
                if ((log.CloudBackupWorkStorageId != null) && (log.CloudBackupWorkStorageId.Id == CloudBackupWorkStorageId.SdCard))
                {
                    return FsMountTarget.SdCard;
                }
                return FsMountTarget.Nand;
            }
            private static FsMountTarget GetTargetFromMountSystemSaveData(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "MountSystemSaveData", "mistake MountAssociation");
                if ((log.SaveDataSpaceId != null) && (log.SaveDataSpaceId.Id == SaveDataSpaceId.SdSystem))
                {
                    return FsMountTarget.SdCard;
                }
                return FsMountTarget.Nand;
            }
            private static GroupAccess GetSaveDataSizeReadGroupAccess(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "GetSaveDataSize", "mistake WriteGroupAccess");
                log.AccessInfoName = string.Format("GetSaveDataSize, UserId: {0}", log.UserId);
                return SaveDataSizeReadGroupAccess;
            }

            private static GroupAccess GetAlbumSaveScreenshotGroupAccess(FsAccessLog log)
            {
                if ((log.Width == 1280) && (log.Height == 720))
                {
                    return SaveScreenshotGroupAccess1280x720;
                }
                // 非対応なログだった場合は AnalyzerError を throw
                throw new AnalyzerError("Unknown image size.");
            }

            private static GroupAccess GetExtendSaveData(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "ExtendSaveData", "mistake WriteGroupAccess");
                return ExtendSaveDataGroupAccess;
            }
            private static GroupAccess GetNfpMetaWriteGroupAccess(FsAccessLog log)
            {
                Debug.Assert(log.IsNfpAccess(), "mistake WriteGroupAccess");
                if (!log.AccessType.HasFlag(FsApiAccessType.Write))
                {
                    return null;
                }
                return NfpMetaWriteGroupAccess;
            }

            private static GroupAccess GetNfpMetaReadGroupAccess(FsAccessLog log)
            {
                Debug.Assert(log.IsNfpAccess(), "mistake ReadGroupAccess");
                if (!log.AccessType.HasFlag(FsApiAccessType.Read))
                {
                    return null;
                }
                log.AccessInfoName = "nfp";
                return NfpMetaReadGroupAccess;
            }

            private static GroupAccess GetCacheStorageWriteGroupAccess(FsAccessLog log)
            {
                Debug.Assert(
                    log.GetFunctionFullName() == "CreateCacheStorage" || log.GetFunctionFullName() == "DeleteCacheStorage",
                    "mistake WriteGroupAccess");
                return CacheStorageWriteGroupAccess;
            }

            private static GroupAccess GetCacheStorageReadGroupAccess(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "GetCacheStorageSize", "mistake ReadGroupAccess");

                if (FsGuideline.IsCacheStorageIndexUsed)
                {
                    log.AccessInfoName = string.Format("GetCacheStorageSize, index: {0}", log.Index ?? 0);
                }
                else
                {
                    log.AccessInfoName = string.Format("GetCacheStorageSize", log.Index);
                }
                return CacheStorageReadGroupAccess;
            }

            private static GroupAccess GetBcatCacheDirectoryMetaReadGroupAccess(FsAccessLog log)
            {
                Debug.Assert(log.GetFunctionFullName() == "EnumerateDeliveryCacheDirectory", "mistake ReadGroupAccess");
                return BcatCacheDirectoryMetaReadGroupAccess;
            }

            private static GroupAccess GetBcatCacheFileMetaReadGroupAccess(FsAccessLog log)
            {
                log.AccessInfoName = "DeliveryCacheDirectory";
                return BcatCacheFileMetaReadGroupAccess;
            }
            private static GroupAccess GetPdmReadGroupAccess(FsAccessLog log)
            {
                log.AccessInfoName = "QueryApplicationPlayStatistics";
                return PdmReadGroupAccess;
            }
        }

        [Flags]
        public enum WriteOptionFlag
        {
            Flush = 1 << 0
        }

        public enum ContentStorageId
        {
            System,
            User,
            SdCard
        }

        public enum ImageDirectoryId
        {
            Nand,
            SdCard
        }

        public enum CloudBackupWorkStorageId
        {
            Nand,
            SdCard
        }

        public enum SaveDataSpaceId
        {
            System,
            User,
            SdSystem,
            ProperSystem = 100
        }

        public enum GameCardPartition
        {
            Update,
            Normal,
            Secure,
        }

        public enum ContentType
        {
            Meta,
            Control,
            Manual,
            Logo,
            Data
        }

        public enum FileSystemProxyType
        {
            Code,
            Rom,
            Logo,
            Control,
            Manual,
            Meta,
            Data,
            Package,
        }

        public enum BisPartitionId
        {
            BootPartition1Root = 0,
            // reserved
            BootPartition2Root = 10,
            // reserved
            UserDataRoot = 20,
            BootConfigAndPackage2Part1,
            BootConfigAndPackage2Part2,
            BootConfigAndPackage2Part3,
            BootConfigAndPackage2Part4,
            BootConfigAndPackage2Part5,
            BootConfigAndPackage2Part6,
            CalibrationBinary,
            CalibrationFile,
            SafeMode,
            User,
            System,
            SystemProperEncryption,
            SystemProperPartition,

            Invalid,
        }

        public static bool IsUnmountFunction(string functionName)
        {
            if (functionName == null)
            {
                return false;
            }
            return Names.UnmountAccess.Contains(functionName);
        }

        public static bool IsMountFunction(string functionName)
        {
            if (functionName == null)
            {
                return false;
            }
            return Names.MountAssociation.ContainsKey(functionName) || Names.MountAssociationForSystem.ContainsKey(functionName);
        }

        public static bool IsSystemMountFunction(FsAccessLog log)
        {
            if (log.Function == null)
            {
                return false;
            }
            // MountCacheStorage は System 用とそうでないものの両方があるので、ApplicationId の有無で判別する
            if (log.Function == "MountCacheStorage")
            {
                return log.ApplicationId != null;
            }
            return Names.MountAssociationForSystem.ContainsKey(log.Function);
        }

        public static FsMountTarget? GetMountTarget(FsAccessLog log)
        {
            Func<FsAccessLog, FsMountTarget> func;
            string functionFullName = log.GetFunctionFullName();
            if (!IsSystemMountFunction(log))
            {
                if (FsFunction.Names.MountAssociation.TryGetValue(functionFullName, out func))
                {
                    return func(log);
                }
            }
            else
            {
                if (FsFunction.Names.MountAssociationForSystem.TryGetValue(functionFullName, out func))
                {
                    return func(log);
                }
            }
            return null;
        }

        public static bool IsWrite(string functionName)
        {
            if (functionName == null)
            {
                return false;
            }
            if (Names.WriteAccess.Contains(functionName))
            {
                return true;
            }
            return Names.WriteGroupAccess.ContainsKey(functionName);
        }

        public static bool IsWriteSizeSameAsWritten(string functionName)
        {
            return functionName == "WriteFile";
        }

        public static bool IsRead(string functionName)
        {
            if (functionName == null)
            {
                return false;
            }
            if (Names.ReadAccess.Contains(functionName))
            {
                return true;
            }
            if (IsMountFunction(functionName))
            {
                return true;
            }
            return Names.ReadGroupAccess.ContainsKey(functionName);
        }

        public static GroupAccess GetGroupAccess(FsAccessLog log)
        {
            Func<FsAccessLog, GroupAccess> getGroupAccess;
            if (Names.WriteGroupAccess.TryGetValue(log.GetFunctionFullName(), out getGroupAccess))
            {
                return getGroupAccess(log);
            }
            if (Names.ReadGroupAccess.TryGetValue(log.GetFunctionFullName(), out getGroupAccess))
            {
                return getGroupAccess(log);
            }
            return null;
        }

        public static bool TryGetWriteGroupAccess(FsAccessLog log, out GroupAccess groupAccess)
        {
            Func<FsAccessLog, GroupAccess> getGroupAccess;
            if (Names.WriteGroupAccess.TryGetValue(log.GetFunctionFullName(), out getGroupAccess))
            {
                try
                {
                    groupAccess = getGroupAccess(log);
                    return groupAccess != null;
                }
                catch
                {
                }
            }
            groupAccess = null;
            return false;
        }

        public static bool TryGetReadGroupAccess(FsAccessLog log, out GroupAccess groupAccess)
        {
            Func<FsAccessLog, GroupAccess> getGroupAccess;
            if (Names.ReadGroupAccess.TryGetValue(log.GetFunctionFullName(), out getGroupAccess))
            {
                try
                {
                    groupAccess = getGroupAccess(log);
                    return groupAccess != null;
                }
                catch
                {
                }
            }
            groupAccess = null;
            return false;
        }

        internal static string GetMountFunctionNameHashString(string functionName)
        {
            if (!IsMountFunction(functionName))
            {
                return string.Empty;
            }
            if (Names.MountFunctionNameAssociation.ContainsKey(functionName))
            {
                return Names.MountFunctionNameAssociation[functionName];
            }
            return functionName;
        }
    }
}
