﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using System.Text.RegularExpressions;

namespace ExtractNca
{
    class ExtractNcaArguments
    {
        [CommandLineOption("input", Description = "input nsp path", IsRequired = true)]
        public string InputNspPath { get; set; }

        [CommandLineOption("output", Description = "output program nca path", IsRequired = true)]
        public string OutputNcaPath { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            ExtractNcaArguments parameters = new ExtractNcaArguments();
            if (CommandLineParser.Default.ParseArgs<ExtractNcaArguments>(args, out parameters))
            {
                ValidateParameters(parameters);
                ExtractNca(new FileInfo(parameters.OutputNcaPath), new FileInfo(parameters.InputNspPath));
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[Error] {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void ExtractNca(FileInfo outputNca, FileInfo inputNsp)
        {
            using(var tempHolder = new TemporaryFileHolder("ExtractNca"))
            {
                var outputDirectory = tempHolder.CreateTemporaryDirectory("nspcontent");

                CommandUtility.SdkTool.Execute(
                    "AuthoringTool.exe",
                    "extractnsp",
                    "-o",
                    outputDirectory.FullName,
                    inputNsp.FullName);

                var files = outputDirectory.EnumerateFiles();

                var programNca = files.First(fileInfo => {
                    return Regex.IsMatch(fileInfo.Name, "[0-9a-zA-Z]{32}\\.nca");
                });

                programNca.CopyTo(outputNca.FullName);
            }
        }

        private static void ValidateParameters(ExtractNcaArguments parameters)
        {
            if (!File.Exists(parameters.InputNspPath))
            {
                throw new Exception(string.Format("not found input file: {0}", parameters.InputNspPath));
            }
        }
    }
}
