﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace ErrorMessageDatabaseConverter
{
    using static ErrorMessageDatabaseConverter.InputXmlDataModel;

    public class Program
    {
        private class ArgumentOption
        {
            public ArgumentOption(string longName, string shortName, string defaultValue = null)
            {
                LongName = longName;
                ShortName = shortName;
                IsFlagType = false;
                Value = defaultValue;
            }
            public ArgumentOption(string longName, string shortName, bool defaultValue)
            {
                LongName = longName;
                ShortName = shortName;
                IsFlagType = true;
                Value = defaultValue.ToString();
            }
            public string LongName { get; set; }
            public string ShortName { get; set; }
            public bool IsFlagType { get; set; }
            public string Value { get; set; }
            public bool FlagOn { get { return (IsFlagType && !string.IsNullOrEmpty(Value) && Value.Equals(true.ToString())); } }
        }

        private static void PrintUsage()
        {
            string baseName = Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location);
            Console.WriteLine("{0}: エラーメッセージをシステムデータ化するためのデータ構造を作成するツールです。", baseName);
            Console.WriteLine("使用方法: {0} --error_list <errorXml> [--message_list <msgXml>] [--url_list <urlXml>] [--database_info <dbXml>] [--output <outputDir>]", baseName);
            Console.WriteLine("\n使用可能な引数:");

            Console.WriteLine("  -e, --error_list <errorXml>  エラーリストが記載されたXMLファイルもしくはXMLファイルを含むディレクトリを指定します。");
            Console.WriteLine("  -m, --message_list <msgXml>  メッセージリストが記載されたXMLファイルを指定します。[デフォルト値：なし]");
            Console.WriteLine("  -u, --url_list <urlXml>      URLリストが記載されたXMLファイル指定します。[デフォルト値：なし（URL挿入タグの置換を行わない）]");
            Console.WriteLine("  --database_info <infoXml>    データベースの情報が記載されたXMLファイルを指定します。[デフォルト値：なし]");
            Console.WriteLine("  -o, --output <outputDir>     メッセージデータを出力するディレクトリを指定します。[デフォルト値: .\\data]");
            Console.WriteLine("  --output_version <path>      バージョン情報を出力するパスを指定します。[デフォルト値: .\\message_ver_errorMessage.txt]");
            Console.WriteLine("  --output_version_only        バージョン情報のみを出力します。メッセージデータを出力しません。");
            Console.WriteLine("  --no_overwrite               出力先に同名のファイルが存在する場合は上書きしないようにします。");
            Console.WriteLine("  -s, --silent                 進捗を示すログを出力しないようにします。");
            Console.WriteLine("  --version                    ツールのバージョンを表示します。");
        }

        private static void PrintVersion()
        {
            string baseName = Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location);
            string version = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
            Console.WriteLine("{0}: version.{1}", baseName, version);
        }

        private static bool ParseArgs(string []args, ref Dictionary<string, ArgumentOption> options)
        {
            for (var i = 0; i < args.Length; i++)
            {
                bool validArg = false;
                foreach(var opt in options.Values)
                {
                    if (args[i] == opt.ShortName || args[i] == opt.LongName)
                    {
                        if(opt.IsFlagType)
                        {
                            opt.Value = true.ToString();
                        }
                        else
                        {
                            if(i + 1 >= args.Length)
                            {
                                return false;
                            }
                            opt.Value = args[i + 1];
                            i++;
                        }
                        validArg = true;
                        break;
                    }
                }
                if (!validArg)
                {
                    return false;
                }
            }
            return true;
        }

        public static int Main(string[] args)
        {
            if(BitConverter.IsLittleEndian == false)
            {
                // メッセージスタジオ用のタグ変換でリトルエンディアンを前提にしているため
                Util.PrintError("The running environment must be little endian.");
                return -1;
            }

            var options = new Dictionary<string, ArgumentOption>()
            {
                {"error_list", new ArgumentOption("--error_list", "-e") },
                {"url_list",  new ArgumentOption("--url_list", "-u") },
                {"message_list",  new ArgumentOption("--message_list", "-m") },
                {"database_info", new ArgumentOption("--database_info", string.Empty) },
                {"output", new ArgumentOption("--output", "-o", @".\data") },
                {"output_version", new ArgumentOption("--output_version", string.Empty, @".\message_ver_errorMessage.txt") },
                {"output_version_only", new ArgumentOption("--output_version_only", string.Empty, false) },
                {"no_overwrite", new ArgumentOption("--no_overwrite", string.Empty, false) },
                {"silent", new ArgumentOption("--silent", "-s", false) },
                {"version", new ArgumentOption("--version", string.Empty, false) },
                {"help", new ArgumentOption("--help", "-h", false) }
            };

            if(!ParseArgs(args, ref options))
            {
                Console.WriteLine("不明なオプションが指定されました。--help を実行し、使用方法を確認して下さい。");
                return -1;
            }

            if(options["help"].FlagOn)
            {
                PrintUsage();
                return 0;
            }

            if(options["version"].FlagOn)
            {
                PrintVersion();
                return 0;
            }

            var stopWatch = System.Diagnostics.Stopwatch.StartNew();

            bool showProgress = !options["silent"].FlagOn;

            if (showProgress)
            {
                Console.WriteLine(">> 変換元ファイルの読み込み開始");
            }

            var errorListPath = options["error_list"].Value;

            if (string.IsNullOrEmpty(errorListPath))
            {
                Console.WriteLine("--error_list オプションを使用して変換元の xml ファイルもしくは xml ファイルのあるディレクトリを指定してください。");
                Console.WriteLine("--help オプションでツールのヘルプを表示します。");
                return 0;
            }

            ErrorListElement errorListElement = null;
            var metaInfoDict = new Dictionary<string, MetaInfo>();

            if (File.Exists(errorListPath))
            {
                errorListElement = Util.DeserializeXml<ErrorListElement>(errorListPath);
                metaInfoDict.Add(Path.GetFileNameWithoutExtension(errorListPath), errorListElement.MetaInfo);
            }
            else if(Directory.Exists(errorListPath))
            {
                errorListElement = new ErrorListElement();
                var files = Util.ListFilesRecursively(errorListPath);
                foreach(var f in files)
                {
                    var ele = Util.DeserializeXml<ErrorListElement>(f);
                    if (ele != null)
                    {
                        errorListElement.ErrorElements.AddRange(ele.ErrorElements);
                        metaInfoDict.Add(Path.GetFileNameWithoutExtension(f), ele.MetaInfo);
                    }
                }
            }
            else
            {
                Util.PrintError("エラーリストに指定されたパスが見つかりません。");
                return -1;
            }

            if (errorListElement == null || errorListElement.ErrorElements.Count == 0)
            {
                Util.PrintError("エラーメッセージデータがひとつも読み込まれていません。");
                return -1;
            }

            var urlListPath = options["url_list"].Value;
            var urlListElement = string.IsNullOrEmpty(urlListPath) ? null :  Util.DeserializeXml<UrlListElement>(urlListPath);

            var messageListPath = options["message_list"].Value;
            var messageList = string.IsNullOrEmpty(messageListPath) ? null : Util.DeserializeXml<MessageListElement>(messageListPath);

            var databaseInfoPath = options["database_info"].Value;
            var databaseInfoElement = string.IsNullOrEmpty(databaseInfoPath) ? null : Util.DeserializeXml<DatabaseInfoElement>(databaseInfoPath);

            if(showProgress)
            {
                Console.WriteLine($"<< 変換元ファイルの読み込み完了 ({stopWatch.ElapsedMilliseconds} ms)");
            }

            Converter.CreateVersionText(options["output_version"].Value, messageList, metaInfoDict);

            if(options["output_version_only"].FlagOn)
            {
                return 0;
            }

            var converter = new Converter();

            SystemDataModel.SystemData systemData;
            if (!converter.Convert(errorListElement.ErrorElements,
                                   (urlListElement == null) ? null : urlListElement.UrlElements,
                                   (messageList == null) ? null : messageList.MessageElements,
                                   databaseInfoElement,
                                   out systemData, showProgress))
            {
                return -1;
            }

            if (!systemData.CreateDirectoryForAuthoring(options["output"].Value, options["no_overwrite"].FlagOn, showProgress))
            {
                return -1;
            }

            if(showProgress)
            {
                Console.WriteLine($"完了 ({stopWatch.ElapsedMilliseconds} ms)");
            }

            return 0;
        }
    }
}
