﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include <algorithm>

#include <nn/nn_Macro.h>

#include "OpusCodecImpl.h"

#include "opus.h"
#include "opus_multistream.h"
#include "opus_private.h"

namespace Nintendo {
namespace CodecTool {
namespace detail {
namespace {

const int32_t DefaultComplexity = 10;
const int32_t DefaultBitDepth = 16;

int32_t GetOpusBitRateMin(int32_t totalStreamCount, int32_t stereoStreamCount)
{
    NN_UNUSED(totalStreamCount);
    NN_UNUSED(stereoStreamCount);
    return 6000;
}

int32_t GetOpusBitRateMax(int32_t totalStreamCount, int32_t stereoStreamCount)
{
    const auto monoStreamCount = totalStreamCount - stereoStreamCount;
    return 300000 * monoStreamCount + 510000 * stereoStreamCount;
}

OpusEncoderImpl::CodingMode GetCodingModeFromPacket(unsigned char head)
{
    const unsigned char config = head >> 3;

    if (config < 12)
    {
        return OpusEncoderImpl::CodingMode::Silk;
    }
    else if (config < 16)
    {
        return OpusEncoderImpl::CodingMode::Hybrid;
    }
    return OpusEncoderImpl::CodingMode::Celt;
}

}

OpusEncoderImpl::OpusEncoderImpl()
    : m_IsInitialized(false)
    , m_SampleRate(0)
    , m_ChannelCount(0)
    , m_TotalStreamCount(0)
    , m_StereoStreamCount(0)
    , m_BitRate(0)
    , m_ApplicationType(0)
    , m_BitRateControl(BitRateControl::Invalid)
    , m_CodingMode(CodingMode::Invalid)
    , m_ChannelMapping()
    , m_WorkBuffer(nullptr)
{
}

OpusEncoderImpl::~OpusEncoderImpl()
{
    Finalize();
}

OpusResult OpusEncoderImpl::Initialize(int32_t sampleRate, int32_t channelCount, int32_t totalStreamCount, int32_t stereoStreamCount, const uint8_t channelMapping[])
{
    if (!(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000))
    {
        return OpusResult_InvalidSampleRate;
    }
    if (!(channelCount >= 1 && channelCount <= ChannelCountMax))
    {
        return OpusResult_InvalidChannelCount;
    }
    if (!(totalStreamCount >= 1 && totalStreamCount <= ChannelCountMax - stereoStreamCount))
    {
        return OpusResult_InvalidStreamCount;
    }
    if (!(stereoStreamCount >= 0 && stereoStreamCount <= totalStreamCount))
    {
        return OpusResult_InvalidStreamCount;
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_TotalStreamCount = totalStreamCount;
    m_StereoStreamCount = stereoStreamCount;
    m_ApplicationType = OPUS_APPLICATION_RESTRICTED_LOWDELAY;  // fixed to Celt
    m_BitRateControl = BitRateControl::ConstrainedVariable;

    auto size = opus_multistream_encoder_get_size(totalStreamCount, stereoStreamCount);
    m_WorkBuffer = new char[size];
    if (m_WorkBuffer == nullptr)
    {
        return OpusResult_OutOfMemory;
    }
    std::memcpy(m_ChannelMapping, channelMapping, channelCount);
    auto ret = opus_multistream_encoder_init(
        static_cast<::OpusMSEncoder*>(m_WorkBuffer),
        sampleRate,
        channelCount,
        totalStreamCount,
        stereoStreamCount,
        channelMapping,
        m_ApplicationType);
    if (ret != OPUS_OK)
    {
        delete[] m_WorkBuffer;
        m_WorkBuffer = nullptr;
        return OpusResult_InternalError;
    }
    // opus_demo に合わせておく。
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_COMPLEXITY(DefaultComplexity));
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_LSB_DEPTH(DefaultBitDepth));

    int32_t bitRate;
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_GET_BITRATE(&bitRate));
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_BITRATE(bitRate));
    m_BitRate = bitRate;

    SetBitRateControl(m_BitRateControl);

    m_IsInitialized = true;

    return OpusResult_Success;
}

void OpusEncoderImpl::Finalize()
{
    if (m_IsInitialized && m_WorkBuffer)
    {
        delete[] m_WorkBuffer;
        m_WorkBuffer = nullptr;
    }
}

OpusResult OpusEncoderImpl::Encode(int32_t* pOutSize, unsigned char* output, int32_t outputSize, const int16_t* input, int32_t sampleCount)
{
    if (pOutSize == nullptr || output == nullptr || input == nullptr)
    {
        return OpusResult_InvalidPointer;
    }
    if (!(sampleCount == m_SampleRate / 400 ||      // 2.5ms
          sampleCount == m_SampleRate / 200 ||      // 5ms
          sampleCount == m_SampleRate / 100 ||      // 10ms
          sampleCount == m_SampleRate / 50))        // 20ms
    {
        return OpusResult_InvalidSampleCount;
    }
    if (outputSize <= 0)
    {
        return OpusResult_InvalidSize;
    }

    auto ret = opus_multistream_encode(static_cast<::OpusMSEncoder*>(m_WorkBuffer), input, sampleCount, output, outputSize);
    if (ret < 0)
    {
        return OpusResult_InternalError;
    }

    if (m_CodingMode != GetCodingModeFromPacket(output[0]))
    {
        return OpusResult_UnexpectedCodingMode;
    }

    *pOutSize = ret;
    return OpusResult_Success;
}

int32_t OpusEncoderImpl::GetSampleRate() const
{
    return m_SampleRate;
}

int32_t OpusEncoderImpl::GetChannelCount() const
{
    return m_ChannelCount;
}

void OpusEncoderImpl::SetBitRate(int32_t bitRate)
{
    bitRate = std::max(GetOpusBitRateMin(m_TotalStreamCount, m_StereoStreamCount), bitRate);  // same lower boundary as opus_demo
    bitRate = std::min(GetOpusBitRateMax(m_TotalStreamCount, m_StereoStreamCount), bitRate);
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_BITRATE(bitRate));
    m_BitRate = bitRate;
}

int32_t OpusEncoderImpl::GetBitRate() const
{
    return m_BitRate;
}

OpusEncoderImpl::BitRateControl OpusEncoderImpl::GetBitRateControl() const
{
    return m_BitRateControl;
}

void OpusEncoderImpl::SetBitRateControl(BitRateControl value)
{
    m_BitRateControl = value;
    int vbr = (value == BitRateControl::Constant ? 0 : 1);
    int cvbr = (value == BitRateControl::ConstrainedVariable ? 1 : 0);
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_VBR(vbr));
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_VBR_CONSTRAINT(cvbr));
}

void OpusEncoderImpl::BindCodingMode(CodingMode value)
{
    switch (value)
    {
    case CodingMode::Auto:
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_FORCE_MODE(OPUS_AUTO));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_APPLICATION(OPUS_APPLICATION_AUDIO));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_BANDWIDTH(OPUS_AUTO));
        m_CodingMode = value;
        break;
    case CodingMode::Silk:
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_FORCE_MODE(MODE_SILK_ONLY));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_APPLICATION(OPUS_APPLICATION_VOIP));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_BANDWIDTH(OPUS_BANDWIDTH_WIDEBAND));
        m_CodingMode = value;
        m_ApplicationType = OPUS_APPLICATION_VOIP;
        break;
    case CodingMode::Celt:
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_FORCE_MODE(OPUS_AUTO));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_APPLICATION(OPUS_APPLICATION_RESTRICTED_LOWDELAY));
        opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_SET_BANDWIDTH(OPUS_AUTO));
        m_CodingMode = value;
        m_ApplicationType = OPUS_APPLICATION_RESTRICTED_LOWDELAY;
        break;
    default:
        break;
        // 以下を使おうとしたが、実行時エラーが発生するので断念。
        // NN_UNEXPECTED_DEFAULT;
    }
}

OpusEncoderImpl::CodingMode OpusEncoderImpl::GetCodingMode() const
{
    return m_CodingMode;
}

void OpusEncoderImpl::GetFinalRanges(uint32_t finalRanges[]) const
{
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_GET_FINAL_RANGE(finalRanges));
}

int32_t OpusEncoderImpl::GetPreSkipSampleCount() const
{
    int preskip = 0;
    opus_multistream_encoder_ctl(static_cast<::OpusMSEncoder*>(m_WorkBuffer), OPUS_GET_LOOKAHEAD(&preskip));
    return preskip;
}

void OpusEncoderImpl::GetChannelMapping(uint8_t channelMapping[]) const
{
    std::memcpy(channelMapping, m_ChannelMapping, GetChannelCount());
}

}  // namespace detail
}  // namespace CodecTool
}  // namespace Nintendo

