﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Change meta version of nsp file.

    .DESCRIPTION
        This script changes meta version of nsp file.

    .PARAMETER InputDirectory
        Specifies the path of nsp's directory.

    .PARAMETER OutputDirectory
        Specifies the path of output directory.
        If this parameter is not set, chaged nsp file is placed to $InputDirectory.

    .PARAMETER ForceVersion
        Specifies the meta version.
        If this parameter is not set, version is set from SDK firmware version.

    .PARAMETER LeaveIntermediateFiles
        [For debug] If this parameter is set, inter mediate files are left.

    .EXAMPLE
        Change nsp file's meta version to SDK firmware version.

        > ChangeNspMetaVersion.ps1 -InputDirectory C:\nsp_directory

    .EXAMPLE
        Change nsp file's meta version and place to another directory.

        > ChangeNspMetaVersion.ps1 -InputDirectory C:\nsp_directory -OutputDirectory C:\output_directory

    .EXAMPLE
        Change nsp file's meta version to specified version.

        > ChangeNspMetaVersion.ps1 -InputDirectory C:\nsp_directory -ForceVersion 0x10000100

#>

param(
    [parameter(mandatory=$true)][string] $InputDirectory,
    [string] $OutputDirectory = "",
    [uint32] $ForceVersion = 0,
    [string] $KeyConfigPath = "",
    [switch] $LeaveIntermediateFiles
)

[uint32] $SetMetaVersion = 0
[Boolean] $DirectoriesAreIdentical = $FALSE

$OutputDirectoryPath
$UseKeyConfigPath

$FimwareVersionXmlPath
$GetFirmwareMetaVersionPath
$DefaultKeyConfigPath
$AuthoringToolPath

$FimwareVersionXmlRelPath = "Common\Versions\NXFirmwareVersion.xml"
$GetFirmwareMetaVersionRelPath = "Tools\ChangeNspMetaVersion\GetFirmwareMetaVersion.exe"
# $GetFirmwareMetaVersionRelPath = "Programs/Eris/Outputs/Win32/Tools/ChangeNspMetaVersion/GetFirmwareMetaVersion/Release/GetFirmwareMetaVersion.exe"
$AuthoringToolRelPath1 = "Programs\Chris\Outputs\x86\Tools\AuthoringTools\AuthoringTool\Release\AuthoringTool.exe"
$AuthoringToolRelPath2 = "Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
$DefaultKeyConfigRelPath1 = "Programs\Chris\Sources\Tools\AuthoringTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
$DefaultKeyConfigRelPath2 = "Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"

function SetSdkRoot()
{
    $scriptDirectoryPath = Split-Path $script:MyInvocation.MyCommand.Path -Parent
    $rootDirectory = Split-Path -Parent $scriptDirectoryPath | Split-Path -Parent
    if (Join-Path $rootDirectory NintendoSdkRootMark | Test-Path)
    {
        Set-Item env:NINTENDO_SDK_ROOT -value $rootDirectory
    } else {
        $rootDirectory = Split-Path -Parent $rootDirectory | Split-Path -Parent | Split-Path -Parent
        if (Join-Path $rootDirectory NintendoSdkRootMark | Test-Path)
        {
            Set-Item env:NINTENDO_SDK_ROOT -value $rootDirectory
        }
    }
    SetRelPath
}

function SetRelPath
{
    $script:FimwareVersionXmlPath = Join-Path $env:NINTENDO_SDK_ROOT $script:FimwareVersionXmlRelPath
    $script:GetFirmwareMetaVersionPath = Join-Path $env:NINTENDO_SDK_ROOT $script:GetFirmwareMetaVersionRelPath
    $script:DefaultKeyConfigPath = Join-Path $env:NINTENDO_SDK_ROOT $script:DefaultKeyConfigRelPath1
    if(-not(Test-Path $script:DefaultKeyConfigPath))
    {
        $script:DefaultKeyConfigPath = Join-Path $env:NINTENDO_SDK_ROOT $script:DefaultKeyConfigRelPath2
    }

    $script:AuthoringToolPath = Join-Path $env:NINTENDO_SDK_ROOT $script:AuthoringToolRelPath1
    if(-not(Test-Path $script:AuthoringToolPath))
    {
        $script:AuthoringToolPath = Join-Path $env:NINTENDO_SDK_ROOT $script:AuthoringToolRelPath2
        if(-not(Test-Path $script:AuthoringToolPath))
        {
            throw "AuthoringTool.exe is NOT found."
        }
    }
}

function ParseMetaVersion
{
    if($script:ForceVersion -ne 0)
    {
        $script:SetMetaVersion = $script:ForceVersion
    }

    if(Test-Path $FimwareVersionXmlPath)
    {
        $xml = [xml](Get-Content $FimwareVersionXmlPath)
        [uint32]$majorVersion = $xml.Product.Version.Major
        [uint32]$minorVersion = $xml.Product.Version.Minor
        [uint32]$microVersion = $xml.Product.Version.Micro
        [uint32]$majorRelstep = $xml.Product.Version.MajorRelstep
        [uint32]$minorRelstep = $xml.Product.Version.MinorRelstep

        $script:SetMetaVersion = GetVersionValue $majorVersion $minorVersion $microVersion $majorRelstep $minorRelstep
    }
    elseif(Test-Path $GetFirmwareMetaVersionPath)
    {
        $logString = ""
        ExecuteProcessWithLog $GetFirmwareMetaVersionPath "" ([ref]$logString)
        $script:SetMetaVersion = [uint32]($logString.Trim())
    }
    else
    {
        throw "Cannot parse firmware version."
    }

    Write-Verbose ("Meta Version : 0x{0}" -F ($script:SetMetaVersion).ToString("x")) -Verbose
}

function GetVersionValue
{
    Param
    (
        [Parameter(Mandatory=$true)][uint32]$majorVersion,
        [Parameter(Mandatory=$true)][uint32]$minorVersion,
        [Parameter(Mandatory=$true)][uint32]$microVersion,
        [Parameter(Mandatory=$true)][uint32]$majorRelstep,
        [Parameter(Mandatory=$true)][uint32]$minorRelstep
    )

    return ($majorVersion -shl 26) -bor ($minorVersion -shl 20) -bor ($microVersion -shl 16) -bor ($majorRelstep -shl 8) -bor $minorRelstep
}

function SetDirectories
{
    if($script:OutputDirectory -eq "")
    {
        $script:OutputDirectoryPath = $script:InputDirectory
    }
    else
    {
        $script:OutputDirectoryPath = $script:OutputDirectory
    }

    if($script:OutputDirectoryPath -eq $script:InputDirectory)
    {
        $script:DirectoriesAreIdentical = $TRUE
    }

    if(-Not(Test-Path $script:OutputDirectoryPath))
    {
        New-Item $script:OutputDirectoryPath -itemType Directory | Out-Null
    }
    else
    {
        if($script:DirectoriesAreIdentical -ne $TRUE)
        {
            # ディレクトリ下にnspファイルがあるならばエラーを返す
            $nspCount = (Get-ChildItem $script:OutputDirectoryPath | Where-Object {$_.Extension.ToLower() -eq ".nsp"}).Count
            if($nspCount -gt 0)
            {
                throw ("Nsp file is already found in `$OutputDirectory. Please delete it. ({0})" -f $script:OutputDirectoryPath)
            }
        }
    }

    if($script:KeyConfigPath -ne "")
    {
        $script:UseKeyConfigPath = $script:KeyConfigPath
    }
    else
    {
        $script:UseKeyConfigPath = $script:DefaultKeyConfigPath
    }
    if(-Not(Test-Path $script:UseKeyConfigPath))
    {
        throw ("Key Config file is not found. ({0})" -f $script:UseKeyConfigPath)
    }
}

function GetTemporaryFolder
{
    $tempPath = [IO.Path]::GetTempPath()
    $wkFolderPath = $null

    Do{
        $wkFolderPath = Join-Path $tempPath ([IO.Path]::GetRandomFileName())
    }while(Test-Path $wkFolderPath)
    $wkFolder = mkdir $wkFolderPath
    return $wkFolderPath
}

# MEMO: 呼び出しプロセスがログを大量に出力するため表示を抑制するようにしている。
#       ログが必要になった場合は引数 $putLog を $TRUE にして呼び出すようにすること

Set-Variable -Name ExecuteProcessDisableResult -Value -1 -Option constant

function ExecuteProcess
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $exePath,

        [Parameter(Mandatory=$true)][AllowEmptyString()][string]
        $exeArgs,

        [int]
        $returnValue = $ExecuteProcessDisableResult,

        [Boolean]
        $putLog = $FALSE
    )

    end
    {
        $proc = New-Object "System.Diagnostics.Process"
        $psi = New-Object "System.Diagnostics.ProcessStartInfo"

        $psi.FileName = $exePath
        $psi.Arguments = $exeArgs
        $psi.UseShellExecute = $FALSE
        $psi.CreateNoWindow = $TRUE
        $psi.RedirectStandardOutput = $TRUE
        $psi.RedirectStandardError = $TRUE

        $proc.StartInfo = $psi


        if($putLog -eq $TRUE)
        {
            $stdSb = New-Object -TypeName System.Text.StringBuilder
            $errorSb = New-Object -TypeName System.Text.StringBuilder
            $scripBlock =
            {
                $x = $Event.SourceEventArgs.Data
                if (-not [String]::IsNullOrEmpty($x))
                {
                    [System.Console]::WriteLine($x)
                    $Event.MessageData.AppendLine($x)
                }
            }
            $stdEvent = Register-ObjectEvent -InputObject $proc -EventName OutputDataReceived -Action $scripBlock -MessageData $stdSb
            $errorEvent = Register-ObjectEvent -InputObject $proc -EventName ErrorDataReceived -Action $scripBlock -MessageData $errorSb
        }


        $procOutput = $proc.Start()
        $proc.BeginOutputReadLine()
        $proc.BeginErrorReadLine()

        $proc.WaitForExit()
        $proc.CancelOutputRead()
        $proc.CancelErrorRead()

        if($putLog -eq $TRUE)
        {
            Unregister-Event -SourceIdentifier $stdEvent.Name
            Unregister-Event -SourceIdentifier $errorEvent.Name
        }

        if($returnValue -ne $ExecuteProcessDisableResult)
        {
            if($proc.ExitCode -ne $returnValue) {
                Write-Host "error: process execution failed. (path: $exePath, return: $proc.ExitCode)"
                exit
            }
        }
    }
    begin
    {
        filter VerboseOutput
        {
            $_ | Out-String -Stream | Write-Verbose
        }
    }
}

function ExecuteProcessWithLog
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $exePath,

        [Parameter(Mandatory=$true)][AllowEmptyString()][string]
        $exeArgs,

        [Parameter(Mandatory=$true)][ref][string]
        $logString,

        [int]
        $returnValue = $ExecuteProcessDisableResult
    )

    $proc = New-Object "System.Diagnostics.Process"
    $psi = New-Object "System.Diagnostics.ProcessStartInfo"

    $psi.FileName = $exePath
    $psi.Arguments = $exeArgs
    $psi.UseShellExecute = $FALSE
    $psi.WindowStyle = [System.Diagnostics.ProcessWindowStyle]::Hidden
    $psi.RedirectStandardOutput = $TRUE

    $proc.StartInfo = $psi

    $procOutput = $proc.Start()

    $proc.WaitForExit()

    if($returnValue -ne $ExecuteProcessDisableResult)
    {
        if($proc.ExitCode -ne $returnValue) {
            Write-Host "error: process execution failed. (path: $exePath, return: $proc.ExitCode)"
            exit
        }
    }

    $logString.value = $proc.StandardOutput.ReadToEnd()
}


function ExtractAndModifyNsps
{
    $script:tempDir = GetTemporaryFolder
    if($script:LeaveIntermediateFiles)
    {
        echo "temporary directory = $tempDir"
    }
    $script:extractNspDirectory = Join-Path $tempDir "extractnsp"
    New-Item $script:extractNspDirectory -itemType Directory | Out-Null
    $script:originalNspDirectory = Join-Path $script:OutputDirectoryPath "originalnsp"

    if($script:DirectoriesAreIdentical -eq $TRUE)
    {
        if(Test-Path $script:originalNspDirectory)
        {
            throw ("The nsp backup directory is already found. Please delete it. ({0})" -f $script:originalNspDirectory)
        }
        else
        {
            New-Item $script:originalNspDirectory -itemType Directory | Out-Null
        }
    }


    $nspFiles = Get-ChildItem -LiteralPath $InputDirectory | Where-Object Extension -like ".nsp"
    foreach($nspFile in $nspFiles)
    {
        $extarctNspDirectory = Join-Path $extractNspDirectory $nspFile.BaseName
        $nspPath = $nspFile.FullName
        $nspFileName = $nspFile.BaseName + $nspFile.Extension
        $outputNspPath = Join-Path $script:OutputDirectoryPath $nspFileName

        Write-Verbose ("Proceeding {0}." -f $nspFileName) -Verbose

        ExecuteProcess $AuthoringToolPath "extract -o $extarctNspDirectory $nspPath --keyconfig $script:UseKeyConfigPath"

        $checkPattern = Join-Path $extarctNspDirectory -ChildPath "*.cnmt.nca" | Join-Path -ChildPath "*" | Join-Path -ChildPath "*.cnmt"
        $cmntFiles = @((Get-Item -path $checkPattern).FullName)
        if($cmntFiles.Count -gt 1)
        {
            Write-Warning "warn: cmnt is more than 1 (path: $nspPath count : $cmntFiles.Count)"
        }
        elseif($cmntFiles.Count -le 0)
        {
            Write-Warning "warn: cmnt is not found (path: $nspPath count : $cmntFiles.Count)"
            continue
        }
        $cmntFile = $cmntFiles[0]

        # Write-Host ("cmndFile : {0}" -f $cmntFile)
        EditVersionField $cmntFile ([string]$script:SetMetaVersion)
        $cmntEntryString = GetCnmtEntryString $extarctNspDirectory $cmntFile

        ExecuteProcess $AuthoringToolPath "replace $nspPath $cmntEntryString $cmntFile -o $script:OutputDirectoryPath --keyconfig $script:UseKeyConfigPath"
        RenameReplacedNsp $outputNspPath $script:DirectoriesAreIdentical
    }

}

function GetCnmtEntryString
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspExtractPath,

        [Parameter(Mandatory=$true)][string]
        $cmntFilePath
    )

    if(-not($cmntFilePath.Contains($nspExtractPath)))
    {
        Write-Host "Cmnt file is not nsp entry. (nsp:{nspExtractPath}, cmnt{cmntFilePath})"
        return ""
    }

    $entryString = $cmntFilePath.SubString($nspExtractPath.Length + 1, $cmntFilePath.Length - $nspExtractPath.Length - 1)
    $entryString = $entryString.Replace("\", "/")

    return $entryString
}

function RenameReplacedNsp
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspPath,

        [Boolean]
        $needBackup = $FALSE
    )

    $nspDirecotry = Split-Path $nspPath -Parent
    $nspFileBaseName = [System.IO.Path]::GetFileNameWithoutExtension($nspPath)
    $nspFileExtension = [System.IO.Path]::GetExtension($nspPath)
    $replacedNspFileBaseName = $nspFileBaseName + "_replaced"

    $replacedNspPath = Join-Path $nspDirecotry -ChildPath ($replacedNspFileBaseName + $nspFileExtension)
    if($needBackup -eq $TRUE)
    {
        MoveNspToOriginalDirectory $nspPath
    }
    Move-Item $replacedNspPath $nspPath
}

function MoveNspToOriginalDirectory
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $nspPath
    )

    Move-Item $nspPath $script:originalNspDirectory
}

function EditVersionField
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $path,

        [Parameter(Mandatory=$true)][string]
        $versionString
    )

    if($versionString.ToLower().Contains("0x"))
    {
        $version = [System.Convert]::ToUint32($versionString, 16)
    }
    else
    {
        $version = [System.Convert]::ToUint32($versionString, 10)
    }

    $src = [System.IO.File]::ReadAllBytes($path)

    $src[0x08] = $version -band 0xFF
    $src[0x09] = ($version -shr 8) -band 0xFF
    $src[0x0a] = ($version -shr 16) -band 0xFF
    $src[0x0b] = ($version -shr 24) -band 0xFF

    [System.IO.File]::WriteAllBytes($path,$src)
}

function ForceDeleteFolder
{
    Param
    (
        [Parameter(Mandatory=$true)][string]
        $directorypath
    )

    Remove-Item -path $directorypath -recurse -force
}


#######################################################################
#
# Main Start
#
#######################################################################
$tempDir
$extractNspDirectory
$originalNspDirectory

# Setting NINTENDO_SDK_ROOT environment variable.
SetSdkRoot

ParseMetaVersion

SetDirectories

ExtractAndModifyNsps

if(-not $LeaveIntermediateFiles)
{
    ForceDeleteFolder($tempDir)
}

# Log about finishing this script.
Write-Host $MyInvocation.MyCommand.Name "FINISHED."
exit 0

