﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace BfttfMaker
{
    class Program
    {
        enum Platform
        {
            None = 0x00,
            Nx = 0x01,
            Cafe = 0x02,
            Win = Nx | Cafe,
        }

        // プラットフォームを切り出して取得するメソッド
        static Platform GetPlatform(string arg)
        {
            Platform platform = Platform.None;
            string[] platformNames = arg.Split(',');
            foreach (string platformName in platformNames)
            {
                switch (platformName.Trim().ToLower())
                {
                    case "win":
                        platform |= Platform.Win;
                        break;
                    case "nx":
                        platform |= Platform.Nx;
                        break;
                    case "cafe":
                        platform |= Platform.Cafe;
                        break;
                    default:
                        Console.WriteLine("unknown platform [" + platformName + "]");
                        return Platform.None;
                }
            }
            return platform;
        }

        static void Main(string[] args)
        {
            /*
             * args[0] input ttf path
             * args[1] output bfttf path
             * args[2] specs
             */

            if (args.Length != 3)
            {
                Console.WriteLine("usage: BfttfMaker [input ttf path] [output bfttf path] [platform name]");
                return;
            }

            // プラットフォームの取得
            Platform platform = GetPlatform(args[2]);

            // 暗号化キーとシグネチャの設定
            uint key = 0;
            const uint signature = 0x7f9a0218;
            switch (platform)
            {
                case Platform.Win:
                    key = 0xa6018502;
                    break;
                case Platform.Nx:
                    key = 0x49621806;
                    break;
                case Platform.Cafe:
                    key = 0x8cf2dcd9;
                    break;
                default:
                    Console.WriteLine("unknown platform");
                    return;
            }

            // ttf のロード
            byte[] ttf;
            try
            {
                ttf = System.IO.File.ReadAllBytes(args[0]);
            }
            catch
            {
                Console.WriteLine("cannot read file[" + args[0] + "]");
                return;
            }

            // 4byte の倍数のバッファにコピー
            uint[] bfttf = new uint[2 + (ttf.Length + 3) / 4]; // シグネチャ + サイズ + ttfバイナリ列
            bfttf[0] = signature;
            bfttf[1] = (uint)ttf.Length;
            for (int i = 2; i < bfttf.Length; i++)
            {
                uint n = 0;
                for (int j = 0; j < 4; j++)
                {
                    int idx = (i - 2) * 4 + j;
                    if (ttf.Length <= idx)
                    {
                        continue;
                    }
                    n |= (uint)ttf[idx] << (8 * (3 - j));
                }
                bfttf[i] = n;
            }

            // XOR でスクランブルしてエンディアンを変換
            const uint mask = 0xff00ff00;
            for (int i = 0; i < bfttf.Length; i++)
            {
                uint n = bfttf[i] ^ key;
                n = ((n & mask) >> 8) | ((n << 8) & mask);
                bfttf[i] = (n >> 16) | (n << 16);
            }

            // 出力
            try
            {
                System.IO.BinaryWriter file = new System.IO.BinaryWriter(new System.IO.FileStream(args[1], System.IO.FileMode.Create));
                foreach (uint n in bfttf)
                {
                    file.Write(n);
                }
                file.Close();
            }
            catch
            {
                Console.WriteLine("cannot write file[" + args[1] + "]");
                return;
            }
        }
    }
}
