﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Concurrent;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Reactive.Linq;
using System.Text;
using System.Windows;
using System.Windows.Threading;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.NspEntriesWindow
{
    [DebuggerDisplay("Name:{Name}, Directory:{Directory}, FileSize:{FileSize}, Children:{Children.Count}")]
    public class FileEntry : ViewModelBase
    {
        public string Name { get; set; }
        public string Directory { get; set; }

        #region FileSize

        private ulong _FileSize;

        public ulong FileSize
        {
            get { return _FileSize; }
            set { SetProperty(ref _FileSize, value); }
        }

        #endregion

        public string RawPath { get; set; }

        public NspFileModifiedType ModifiedType { get; set; }

        public ulong ModifiedFileSize { get; set; }

        public bool IsDirectory => FileSize == ulong.MaxValue;
        public string FullPath => Directory == "/" ? Name : $"{Directory}/{Name}";

        public FileEntryType Type => IsDirectory ? FileEntryType.Directory : FileEntryType.UnknownFile;

        public string StatutsText => IsDirectory
            ? null
            : string.Format(Resources.NspEntries_FileStatus,
                Resources.ResourceManager.GetString($"{nameof(NspFileModifiedType)}_{ModifiedType}_Caption"), Name);

        #region IsExpanded

        private bool _IsExpanded;

        public bool IsExpanded
        {
            get { return _IsExpanded; }
            set { SetProperty(ref _IsExpanded, value); }
        }

        #endregion

        private ObservableCollection<FileEntry> _Children;

        public ObservableCollection<FileEntry> Children
        {
            get { return _Children ?? (_Children = new ObservableCollection<FileEntry>()); }
            set { _Children = value; }
        }

        private readonly ConcurrentDictionary<string, FileEntry> _entryDict =
            new ConcurrentDictionary<string, FileEntry>();

        public void Clear()
        {
            Children.Clear();
            _entryDict.Clear();
        }

        public void AddEntry(FileEntry entry)
        {
            Debug.Assert(_entryDict.ContainsKey(entry.Name) == false);

            Action a = () => Children.Add(entry);
            Application.Current?.Dispatcher.Invoke(DispatcherPriority.Background, TimeSpan.FromMilliseconds(100), a);

            _entryDict[entry.Name] = entry;
        }

        public void AddEntry(FileEntry entry, string[] dir, int dirIndex)
        {
            if (dirIndex == dir.Length)
            {
                AddEntry(entry);
                return;
            }

            FileEntry target;
            if (_entryDict.TryGetValue(dir[dirIndex], out target))
                target.AddEntry(entry, dir, dirIndex + 1);

            else
            {
                target = new FileEntry
                {
                    Name = dir[dirIndex],
                    Directory = MakeDirectoryString(dir, dirIndex),
                    FileSize = ulong.MaxValue
                };

                AddEntry(target);

                target.AddEntry(entry, dir, dirIndex + 1);
            }
        }

        private static string MakeDirectoryString(string[] dir, int dirIndex)
        {
            var sb = new StringBuilder();

            for (var i = 0; i < dirIndex; ++i)
            {
                sb.Append("/");
                sb.Append(dir[i]);
            }

            return sb.Length == 0 ? "/" : sb.ToString();
        }

        private StringParamVm _FileSizeParam;

        public StringParamVm FileSizeParam
        {
            get
            {
                if (_FileSizeParam == null)
                {
                    var fileSizeRp = this.ObserveProperty(x => x.FileSize)
                        .Select(x =>
                        {
                            if (ModifiedType == NspFileModifiedType.Changed)
                            {
                                return $"{NumberHelper.ToReadableSizeString(x)} => {NumberHelper.ToReadableSizeString(ModifiedFileSize)}";
                            }
                            return NumberHelper.ToReadableSizeString(x);
                        })
                        .ToReactiveProperty();

                    _FileSizeParam = new StringParamVm(
                        nameof(Resources.FileSize),
                        null,
                        fileSizeRp
                    ).AddTo(CompositeDisposable);
                    _FileSizeParam.IsReadOnly.Value = true;
                }

                return _FileSizeParam;
            }

            set { _FileSizeParam = value; }
        }

        public FileEntry()
        {
            CompositeDisposable.Add(() =>
            {
                if (_Children == null)
                    return;

                foreach (var e in Children)
                    e.Dispose();
            });
        }
    }
}
