﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Reactive.Concurrency;
using System.Reactive.Linq;
using System.Threading.Tasks;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging.IO;
using Livet.Messaging.Windows;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MakeNspPatchWindow
{
    public class MakeNspPatchWindowVm : ViewModelBase
    {
        public ReactiveCommand CloseCommand { get; }

        public ReactiveCommand MakePatchCommand { get; }

        public FilePathStringParamVm OriginalNspPath { get; }

        public FilePathStringParamVm RevisedNspPath { get; }

        public FilePathStringParamVm PreviousNspPath { get; }

        public FilePathStringParamVm ApplicationDescPath { get; }

        public enum IncrementStatus
        {
            Ok,
            PreviousOrRevisedNotSet,
            NotIncremented,
        }

        public IncrementStatus NewPatchVersionIncrementStatus
        {
            get
            {
                if (PreviousNspPath.IsUse.Value == false)
                    return IncrementStatus.Ok;
                var previous = PreviousPatchContentMeta.Value;
                var revised = RevisedAppContentMeta.Value;
                if (previous == null || revised == null)
                    return IncrementStatus.PreviousOrRevisedNotSet;
                // Previous パッチと Current アプリのリリースバージョンの差異が 1 より大きい場合、意図した操作かを確認
                return previous.ReleaseVersion + 1 == revised.ReleaseVersion
                    ? IncrementStatus.Ok
                    : IncrementStatus.NotIncremented;
            }
        }

        public ParamVm[] Params { get; }

        public bool IsMakePatch { get; private set; }

        // ダミー。検索系コントロールのために用意
        public ReactiveProperty<string> DelaySearchWord { get; set; }

        public ReactiveProperty<string> ValidationMessage { get; }

        public ReactiveProperty<ApplicationContentMeta> RevisedAppContentMeta { get; }
        public ReactiveProperty<ApplicationContentMeta> OriginalAppContentMeta { get; }
        public ReactiveProperty<PatchContentMeta> PreviousPatchContentMeta { get; }

        private readonly Container _diContainer;

        public MakeNspPatchWindowVm(Container diContainer, App app)
        {
            _diContainer = diContainer;

            CloseCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseCommand.Subscribe(_ => Close()).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            ValidationMessage = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            ValidationMessage.SetValidateNotifyError(x => (string)x);

            RevisedAppContentMeta = new ReactiveProperty<ApplicationContentMeta>().AddTo(CompositeDisposable);
            OriginalAppContentMeta = new ReactiveProperty<ApplicationContentMeta>().AddTo(CompositeDisposable);
            PreviousPatchContentMeta = new ReactiveProperty<PatchContentMeta>().AddTo(CompositeDisposable);

            PreviousPatchContentMeta
                .CombineLatest(OriginalAppContentMeta, (Previous, Revised) => new {Previous, Revised})
                .Subscribe(x =>
                {
                    ValidationMessage.Value =
                        x.Previous != null &&
                        x.Revised != null &&
                        x.Previous.ReleaseVersion >= x.Revised.ReleaseVersion
                        ? string.Format(
                            Resources.NspPatch_PatchVersionIsLessOrEqualThanPrevious,
                            x.Previous,
                            x.Revised)
                        : null;
                })
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            Observable
                .Merge(RevisedAppContentMeta.ToUnit())
                .Merge(OriginalAppContentMeta.ToUnit())
                .Merge(PreviousPatchContentMeta.ToUnit())
                .Subscribe(_ =>
                {
                    var original = OriginalAppContentMeta.Value;
                    var revised = RevisedAppContentMeta.Value;
                    if (original == null || revised == null)
                    {
                        return;
                    }

                    if (original.Id != revised.Id)
                    {
                        ValidationMessage.Value =
                            Resources.NspHandleResult_NspIsNotSameAppIdBetweenOriginalAndRevisedError;
                        return;
                    }

                    var previous = PreviousPatchContentMeta.Value;
                    if (app.NspPatch.IsUsePreviousNspPath && previous != null)
                    {
                        if (original.Id != previous.ApplicationId)
                        {
                            ValidationMessage.Value =
                                Resources.NspHandleResult_NspIsNotSameAppIdBetweenOrigianlAndPreviousError;
                            return;
                        }
                        if (original.Digest != previous.OriginalApplication?.Digest)
                        {
                            ValidationMessage.Value =
                                Resources.NspHandleResult_NspIsNotSameAppDigestBetweenOriginalAndPreviousError;
                            return;
                        }
                    }

                    ValidationMessage.Value = null;
                })
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var originalNspChecking = false;
            var originalNspPathRp = app.NspPatch
                .ToReactivePropertyAsSynchronized(x => x.OriginalNspPath,
                    ReactivePropertyMode.RaiseLatestValueOnSubscribe)
                .SetValidateNotifyError(s => File.Exists(s) == false ? Resources.FileNotFound : null)
                .SetValidateNotifyError(async x =>
                {
                    if (originalNspChecking)
                        return null;
                    originalNspChecking = true;
                    try
                    {
                        var r = await NspBuilder.TestPatchOriginalNspError(new NspBuilder.MakePatchOption
                        {
                            OriginalNspFilePath = x
                        }).ConfigureAwait(false);
                        OriginalAppContentMeta.Value = r.Original;
                        return FromNspHandleResult(r);
                    }
                    finally
                    {
                        originalNspChecking = false;
                    }
                })
                .AddTo(CompositeDisposable);

            OriginalNspPath = new FilePathStringParamVm(
                nameof(Resources.NspPatch_OriginalNsp_Caption),
                nameof(Resources.NspPatch_OriginalNsp_Comment),
                originalNspPathRp,
                nameof(Resources.DialogFilter_Nsp),
                nameof(Resources.NspFile),
                false
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var revisedNspChecking = false;
            var revisedNspPathRp = app.NspPatch
                .ToReactivePropertyAsSynchronized(x => x.CurrentNspPath,
                    ReactivePropertyMode.RaiseLatestValueOnSubscribe)
                .SetValidateNotifyError(s =>
                {
                    if (File.Exists(s) == false)
                        return Resources.FileNotFound;
                    return null;
                })
                .SetValidateNotifyError(async x =>
                {
                    if (revisedNspChecking)
                        return null;
                    revisedNspChecking = true;
                    try
                    {
                        var r = await NspBuilder.TestPatchCurrentNspError(new NspBuilder.MakePatchOption
                        {
                            CurrentNspFilePath = x
                        }).ConfigureAwait(false);
                        RevisedAppContentMeta.Value = r.Current;
                        return FromNspHandleResult(r);
                    }
                    finally
                    {
                        revisedNspChecking = false;
                    }
                })
                .AddTo(CompositeDisposable);

            RevisedNspPath = new FilePathStringParamVm(
                nameof(Resources.NspPatch_RevisedNsp_Caption),
                nameof(Resources.NspPatch_RevisedNsp_Comment),
                revisedNspPathRp,
                nameof(Resources.DialogFilter_Nsp),
                nameof(Resources.NspFile),
                false
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var previousNspChecking = false;
            var previousNspPathRp = app.NspPatch
                .ToReactivePropertyAsSynchronized(x => x.PreviousNspPath,
                    ReactivePropertyMode.RaiseLatestValueOnSubscribe)
                .SetValidateNotifyError(s =>
                {
                    if (app.NspPatch.IsUsePreviousNspPath && File.Exists(s) == false)
                        return Resources.FileNotFound;
                    return null;
                })
                .SetValidateNotifyError(async x =>
                {
                    if (app.NspPatch.IsUsePreviousNspPath == false)
                    {
                        ValidationMessage.Value = null;
                        return null;
                    }
                    if (previousNspChecking)
                        return null;
                    previousNspChecking = true;
                    try
                    {
                        var r = await NspBuilder.TestPatchPreviousNspError(new NspBuilder.MakePatchOption
                        {
                            PreviousNspFilePath = x
                        }).ConfigureAwait(false);
                        PreviousPatchContentMeta.Value = r.Previous;
                        return FromNspHandleResult(r);
                    }
                    finally
                    {
                        previousNspChecking = false;
                    }
                })
                .AddTo(CompositeDisposable);

            PreviousNspPath = new FilePathStringParamVm(
                nameof(Resources.NspPatch_PreviousNsp_Caption),
                nameof(Resources.NspPatch_PreviousNsp_Comment),
                previousNspPathRp,
                nameof(Resources.DialogFilter_Nsp),
                nameof(Resources.NspFile),
                false
            ).AddTo(CompositeDisposable);
            PreviousNspPath.IsUse = app.NspPatch
                .ToReactivePropertyAsSynchronized(x => x.IsUsePreviousNspPath)
                .AddTo(CompositeDisposable);
            PreviousNspPath.CaptionIsUseTag = nameof(Resources.IsUse);

            Observable
                .Merge(PreviousNspPath.IsUse.ToUnit())
                .Subscribe(_ => previousNspPathRp.ForceValidate())
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var applicationDescPathRp = app.NspPatch
                .ToReactivePropertyAsSynchronized(x => x.ApplicationDescPath,
                    ReactivePropertyMode.RaiseLatestValueOnSubscribe)
                .SetValidateNotifyError(s => File.Exists(s) == false ? Resources.FileNotFound : null)
                .AddTo(CompositeDisposable);

            ApplicationDescPath = new FilePathStringParamVm(
                nameof(Resources.NspPatch_ApplicationDesc_Caption),
                nameof(Resources.NspPatch_ApplicationDesc_Comment),
                applicationDescPathRp,
                nameof(Resources.DialogFilter_Desc),
                nameof(Resources.NspPatch_ApplicationDesc_Caption),
                false
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////

            Params = new ParamVm[]
            {
                OriginalNspPath,
                RevisedNspPath,
                PreviousNspPath,
                ApplicationDescPath
            };

            MakePatchCommand = new[]
                {
                    originalNspPathRp.ObserveHasErrors,
                    revisedNspPathRp.ObserveHasErrors,
                    applicationDescPathRp.ObserveHasErrors,
                    previousNspPathRp.ObserveHasErrors,
                    ValidationMessage.ObserveHasErrors
                }
                .CombineLatestValuesAreAllFalse()
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);

            MakePatchCommand
                .Subscribe(_ =>
                {
                    if (MakePatchCheck() == false)
                        return;

                    var r = Messenger.GetResponse(new SavingFileSelectionMessage(GuiConstants.MessageKey_FileSave)
                    {
                        Filter = Resources.DialogFilter_Nsp,
                        Title = Resources.DialogTitle_OutputNspFile
                    });
                    var outputPath = r?.Response?.FirstOrDefault();
                    if (string.IsNullOrEmpty(outputPath))
                        return;

                    app.NspPatch.OutputNspPath = outputPath;
                    IsMakePatch = true;
                    Close();
                });
        }

        private bool MakePatchCheck()
        {
            var previous = PreviousPatchContentMeta.Value;
            var revised = RevisedAppContentMeta.Value;

            // Previous パッチと Current アプリのリリースバージョンの差異が 1 より大きい場合、意図した操作かを確認
            if (NewPatchVersionIncrementStatus == IncrementStatus.NotIncremented)
            {
                var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Warning,
                    Caption = Resources.Confirmation,
                    Text = string.Format(
                        Resources.DialogMessage_NewPatchVersionIsNotIncremented,
                        previous.ReleaseVersion,
                        revised.ReleaseVersion),
                    StandardButtons = DialogMessage.StandardButtonsType.OkCancel,
                });
                if (r?.Response?.DialogResult != DialogMessage.DialogResultType.Ok)
                    return false;
            }

            return true;
        }

        private void Close()
        {
            Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction"));
        }

        private string FromNspHandleResult(NspBuilder.PatchResultType r)
        {
            return r?.Result != NspHandleResult.Ok
                ? _diContainer.GetInstance<StringHelper>().GetNspHandleResultMessage(r.Result)
                : null;
        }
    }
}
