﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class BcatPageVm : PageVmBase
    {
        public BoolOnOffParamVm IsUseBcat { get; }
        public CapacityParamVm BcatDeliveryCacheStorageSize { get; }
        public StringParamVm BcatPassphrase { get; }

        public BcatPageVm(Container diContainer, ApplicationMeta appMeta)
            : base(nameof(Resources.Bcat))
        {
            var isReadOnlyRp = appMeta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly)
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var isUseBcatRp = appMeta.Application.ToReactivePropertyAsSynchronized(x => x.IsUseBcat);

            IsUseBcat = new BoolOnOffParamVm(
                nameof(Resources.IsUseBcat_Caption),
                nameof(Resources.IsUseBcat_Comment),
                isUseBcatRp
            ).AddTo(CompositeDisposable);
            IsUseBcat.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            var bcatDeliveryCacheStorageSizeRp = appMeta.Application.ToReactivePropertyAsSynchronized(
                    x => x.BcatDeliveryCacheStorageSize)
                .SetValidateNotifyError(
                    s =>
                        diContainer.GetInstance<PageValidations>()
                            .BcatDeliveryCacheStorageSize(s, appMeta.Application.IsUseBcat));
            BcatDeliveryCacheStorageSize = new CapacityParamVm(
                nameof(Resources.BcatDeliveryCacheStorageSize_Caption),
                nameof(Resources.BcatDeliveryCacheStorageSize_Comment),
                bcatDeliveryCacheStorageSizeRp,
                8,
                0x00000000,
                0xFFFFFFFF,
                null
            ).AddTo(CompositeDisposable);
            BcatDeliveryCacheStorageSize.IsReadOnly = isReadOnlyRp;
            BcatDeliveryCacheStorageSize.IsUse = isUseBcatRp;
            BcatDeliveryCacheStorageSize.EditingStyle = CapacityEditBox.EditingStyles.MegaBytes;
            BcatDeliveryCacheStorageSize.EditingStyleCandidates = new[]
            {
                CapacityEditBox.EditingStyles.Bytes,
                CapacityEditBox.EditingStyles.KiroBytes,
                CapacityEditBox.EditingStyles.MegaBytes
            };

            /////////////////////////////////////////////////////////
            var bcatPassphraseRp = appMeta.Application.ToReactivePropertyAsSynchronized(
                x => x.BcatPassphrase,
                x => x?.ToLowerInvariant(),
                x => x)
                .SetValidateNotifyError(
                    s => diContainer.GetInstance<PageValidations>().BcatPassphrase(s, appMeta.Application.IsUseBcat));
            BcatPassphrase = new StringParamVm(
                nameof(Resources.BcatPassphrase_Caption),
                nameof(Resources.BcatPassphrase_Comment),
                bcatPassphraseRp
            ).AddTo(CompositeDisposable);
            BcatPassphrase.Width = GuiConstants.LargeWidth + 150;
            BcatPassphrase.IsUse = isUseBcatRp;
            BcatPassphrase.IsReadOnly = isReadOnlyRp;
            BcatPassphrase.MaxLength = Application.BcatPassphraseLength;

            /////////////////////////////////////////////////////////
            isUseBcatRp.Subscribe(x =>
            {
                bcatDeliveryCacheStorageSizeRp.ForceValidate();
                bcatPassphraseRp.ForceValidate();
            }).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                IsUseBcat,
                BcatDeliveryCacheStorageSize,
                BcatPassphrase
            };

            /////////////////////////////////////////////////////////
            Observable
                .Merge(appMeta.Application.ObserveProperty(x => x.IsUseBcat).ToUnit())
                .Merge(appMeta.Application.ObserveProperty(x => x.ValidationBcatDeliveryCacheStorageSize).ToUnit())
                .Merge(appMeta.Application.ObserveProperty(x => x.ValidationBcatPassphrase).ToUnit())
                .Where(_ => isReadOnlyRp.Value == false)
                .Subscribe(_ =>
                {
                    HasErrors.Value =
                        appMeta.Application.ValidationBcatDeliveryCacheStorageSize !=
                        Application.BcatDeliveryCacheStorageSizeValidationType.Ok ||
                        appMeta.Application.ValidationBcatPassphrase !=
                        Application.BcatPassphraseValidationType.Ok;
                })
                .AddTo(CompositeDisposable);
        }
    }
}
