﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Reactive.Threading.Tasks;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using Nintendo.Authoring.AuthoringEditor.Properties;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class BasicPageVm : PageVmBase
    {
        public StringParamVm DisplayVersion { get; }
        public StringParamVm ReleaseVersion { get; }
        public HexParamVm ApplicationId { get; }
        public EnumParamVm LogoType { get; }
        public BoolOnOffParamVm IsDemo { get; }
        public FilePathStringParamVm LegalInformationFilePath { get; }
        public DirPathStringParamVm HtmlDocumentPath { get; }
        public FilePathStringParamVm AccessibleUrlsFilePath { get; }
        public FilePathStringParamVm FilterDescriptionFilePath { get; }

        private Container DiContainer { get; }
        private ReactiveProperty<bool> IsReadOnly { get; }

        private readonly PageValidations _validations;

        public BasicPageVm(Container diContainer, Project project)
            : base(nameof(Resources.Basic))
        {
            _validations = diContainer.GetInstance<PageValidations>();

            DiContainer = diContainer;
            IsReadOnly = project.Meta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly).AddTo(CompositeDisposable);

            var appProfile = diContainer.GetInstance<AppProfile>();
            var isNspMode = appProfile.AppMode == AppModeType.ApplicationNsp ||
                            appProfile.AppMode == AppModeType.PatchNsp;
            var app = project.Meta.Application;

            /////////////////////////////////////////////////////////
            var appIdRp =
                project.Meta.Core.ToReactivePropertyAsSynchronized(x => x.ApplicationId)
                    .SetValidateNotifyError(i => diContainer.GetInstance<PageValidations>().ApplicationId(i));

            /////////////////////////////////////////////////////////
            var displayVersionRp =
                app.ToReactivePropertyAsSynchronized(x => x.DisplayVersion)
                    .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().DisplayVersion(s));

            DisplayVersion = new StringParamVm(
                nameof(Resources.DisplayVersion_Caption),
                nameof(Resources.DisplayVersion_Comment),
                displayVersionRp
            ).AddTo(CompositeDisposable);
            DisplayVersion.IsReadOnly = IsReadOnly;
            DisplayVersion.Width = GuiConstants.SmallWidth;
            DisplayVersion.MaxLength = Core.Application.MaxDisplayVersionLength;

            /////////////////////////////////////////////////////////
            var releaseVersionRp =
                app.ToReactivePropertyAsSynchronized(x => x.ReleaseVersion);

            ReleaseVersion = new StringParamVm(
                nameof(Resources.ReleaseVersion_Caption),
                nameof(Resources.ReleaseVersion_Comment),
                releaseVersionRp
            ).AddTo(CompositeDisposable);
            ReleaseVersion.IsReadOnly = IsReadOnly;
            ReleaseVersion.Width = GuiConstants.SmallWidth;

            /////////////////////////////////////////////////////////
            var isFullRange = Constants.ProgramIdMinimum == ulong.MinValue &&
                              Constants.ProgramIdMaximum == ulong.MaxValue;

            ApplicationId = new HexParamVm(
                nameof(Resources.ApplicationId_Caption),
                isFullRange
                    ? nameof(Resources.ApplicationId_Comment_FullRange)
                    : nameof(Resources.ApplicationId_Comment),
                appIdRp,
                16,
                Constants.ProgramIdMinimum,
                Constants.ProgramIdMaximum
            ).AddTo(CompositeDisposable);
            ApplicationId.IsReadOnly.Value = isNspMode;

            /////////////////////////////////////////////////////////
            LogoType = EnumParamVm.Factory(
                nameof(Resources.LogoType_Caption),
                nameof(Resources.LogoType_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.LogoType)
            ).AddTo(CompositeDisposable);
            LogoType.IsReadOnly = IsReadOnly;

            /////////////////////////////////////////////////////////
            IsDemo = new BoolOnOffParamVm(
                nameof(Resources.Attribute_Demo_Caption),
                nameof(Resources.Attribute_Demo_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.IsDemo)
            ).AddTo(CompositeDisposable);
            IsDemo.IsReadOnly = IsReadOnly;

            /////////////////////////////////////////////////////////
            LegalInformationFilePath = PrepareExpandableFilePathParamVm(
                CreateExpandableFilePathParamVm(project,
                    nameof(Resources.LegalInformationFilePath_Caption),
                    nameof(Resources.LegalInformationFilePath_Comment),
                    app.LegalInformationFilePath,
                    nameof(Resources.DialogFilter_LegalInformationFilePath),
                    x => _validations.LegalInformationFilePath(x)
                ),
                app.ToReactivePropertyAsSynchronized(x => x.IsReplaceLegalInformationFilePath));

            /////////////////////////////////////////////////////////
            HtmlDocumentPath = PrepareExpandableFilePathParamVm(
                CreateExpandableDirPathParamVm(project,
                    nameof(Resources.HtmlDocumentDirectoryPath_Caption),
                    nameof(Resources.HtmlDocumentDirectoryPath_Comment),
                    app.HtmlDocumentPath,
                    x => _validations.HtmlDocumentPath(x)
                ),
                app.ToReactivePropertyAsSynchronized(x => x.IsReplaceHtmlDocumentPath));

            /////////////////////////////////////////////////////////
            AccessibleUrlsFilePath = PrepareExpandableFilePathParamVm(
                CreateExpandableFilePathParamVm(project,
                    nameof(Resources.AccessibleUrlsFilePath_Caption),
                    nameof(Resources.AccessibleUrlsFilePath_Comment),
                    app.AccessibleUrlsFilePath,
                    nameof(Resources.DialogFilter_Text),
                    x => _validations.AccessibleUrlsFilePath(x)
                ),
                app.ToReactivePropertyAsSynchronized(x => x.IsReplaceAccessibleUrlsFilePath));

            /////////////////////////////////////////////////////////
            if (project.AppCapability.IsSupportFilterDescriptionFileInMeta && isNspMode == false)
            {
                var filePathRp = app
                    .ToReactivePropertyAsSynchronized(x => x.FilterDescriptionFilePath,
                        ReactivePropertyMode.RaiseLatestValueOnSubscribe)
                    .SetValidateNotifyError(_ => _validations.FilterDescriptionFilePath(app));
                FilterDescriptionFilePath = PrepareExpandableFilePathParamVm(
                    new FilePathStringParamVm(
                        nameof(Resources.FilterDescriptionFilePath_Caption),
                        nameof(Resources.FilterDescriptionFilePath_Comment),
                        filePathRp,
                        nameof(Resources.DialogFilter_FilterDescriptionFile),
                        null,
                        false,
                        filePathRp.Select(x => project.ToAbsolutePath(x))
                    ).AddTo(CompositeDisposable),
                    app.ToReactivePropertyAsSynchronized(x => x.IsUseFilterDescriptionFilePath)
                );
            }

            /////////////////////////////////////////////////////////
            PrepareLegalInformationNca(project, app);
            PrepareHtmlDocumentNca(project, app);

            /////////////////////////////////////////////////////////
            var @params = new ParamVm[]
            {
                ApplicationId,
                DisplayVersion,
                ReleaseVersion,
                LogoType,
                IsDemo,
                LegalInformationFilePath,
                HtmlDocumentPath,
                AccessibleUrlsFilePath,
                FilterDescriptionFilePath
            }.Where(x => x != null).ToArray();

            /////////////////////////////////////////////////////////
            var hasErrorObservable = Observable
                .Merge(project.Meta.Core.ObserveProperty(x => x.ApplicationId).ToUnit())
                .Merge(app.ObserveProperty(x => x.ValidationDisplayVersion).ToUnit())
                //
                .Merge(LegalInformationFilePath.FilePathProperty.ToUnit())
                .Merge(LegalInformationFilePath.IsUse.ToUnit())
                .Merge(app.LegalInformationFilePath.ObserveProperty(x => x.IsExpandEnvironmentVariable).ToUnit())
                //
                .Merge(HtmlDocumentPath.FilePathProperty.ToUnit())
                .Merge(HtmlDocumentPath.IsUse.ToUnit())
                .Merge(app.HtmlDocumentPath.ObserveProperty(x => x.IsExpandEnvironmentVariable).ToUnit())
                //
                .Merge(AccessibleUrlsFilePath.FilePathProperty.ToUnit())
                .Merge(AccessibleUrlsFilePath.IsUse.ToUnit())
                .Merge(app.AccessibleUrlsFilePath.ObserveProperty(x => x.IsExpandEnvironmentVariable).ToUnit());

            if (FilterDescriptionFilePath != null)
            {
                hasErrorObservable = hasErrorObservable
                    .Merge(FilterDescriptionFilePath.FilePathProperty.ToUnit())
                    .Merge(FilterDescriptionFilePath.IsUse.ToUnit());
            }

            hasErrorObservable
                .Where(_ => IsReadOnly.Value == false)
                .Subscribe(_ =>
                    HasErrors.Value =
                        project.Meta.Core.ValidationApplicationId !=
                        Core.Core.ApplicationIdValidationType.Ok ||
                        app.ValidationDisplayVersion !=
                        Core.Application.DisplayVersionValidationType.Ok ||
                        app.ValidationLegalInformationFilePath !=
                        Core.Application.LegalInformationFilePathValidationType.Ok ||
                        app.ValidationHtmlDocumentPath !=
                        Core.Application.HtmlDocumentPathValidationType.Ok ||
                        app.ValidationAccessibleUrlsFilePath !=
                        Core.Application.AccessibleUrlsFilePathValidationType.Ok ||
                        app.ValidationFilterDescriptionFilePath !=
                        Core.Application.FilterDescriptionFileValidationType.Ok
                ).AddTo(CompositeDisposable);

            Params = @params.Where(x => x != null).ToArray();
        }

        private TParamVm PrepareExpandableFilePathParamVm<TParamVm>(
            TParamVm paramVm,
            ReactiveProperty<bool> isUseProperty) where TParamVm : ParamVm
        {
            var filePathRp = ((IFilePathPropertyParamVm) paramVm).FilePathProperty;

            paramVm.IsUse = isUseProperty.AddTo(CompositeDisposable);
            paramVm.CaptionIsUseTag = nameof(Resources.IsUse);

            Observable
                .Merge(paramVm.IsUse.ToUnit())
                .Subscribe(_ => filePathRp.ForceValidate())
                .AddTo(CompositeDisposable);

            return paramVm;
        }

        private DirPathStringParamVm CreateExpandableDirPathParamVm(
            Project project,
            string captionTag,
            string commentTag,
            ExpandablePath filePath,
            Func<Application, string> validator)
        {
            return Utilities.CreateExpandablePathVm(
                project,
                filePath,
                filePathRp => new DirPathStringParamVm(
                    captionTag,
                    commentTag,
                    filePathRp,
                    AbsolutePathSource(project, filePathRp))
                {
                    IsReadOnly = IsReadOnly
                },
                _ => validator(project.Meta.Application),
                CompositeDisposable
            );
        }

        private FilePathStringParamVm CreateExpandableFilePathParamVm(
            Project project,
            string captionTag,
            string commentTag,
            ExpandablePath filePath,
            string fileFilter,
            Func<Application, string> validator)
        {
            return Utilities.CreateExpandablePathVm(
                project,
                filePath,
                filePathRp => new FilePathStringParamVm(
                    captionTag,
                    commentTag,
                    filePathRp,
                    fileFilter,
                    null,
                    false,
                    AbsolutePathSource(project, filePathRp))
                {
                    IsReadOnly = IsReadOnly
                },
                _ => validator(project.Meta.Application),
                CompositeDisposable);
        }

        private void PrepareLegalInformationNca(Project project, Application app)
        {
            project.ExtraResourceImporter?.ReadLegalInformationAsync()
                .ToObservable()
                .ObserveOnUIDispatcher()
                .Where(ok => ok)
                .Subscribe(_ =>
                {
                    var filePath = app.OriginalLegalInformationPath;
                    if (Directory.Exists(filePath))
                        LegalInformationFilePath.CaptionIsUseTag = nameof(Resources.Replace);

                    ReplaceAdditionalVm(LegalInformationFilePath.Additional, () =>
                    {
                        if (Directory.Exists(filePath) == false)
                            return null;
                        return new LegalInformationAddtionalVm(
                            app.ToReactivePropertyAsSynchronized(x => x.OriginalLegalInformationPath));
                    });
                })
                .AddTo(CompositeDisposable);
        }

        private void PrepareHtmlDocumentNca(Project project, Application app)
        {
            project.ExtraResourceImporter?.ReadHtmlDocumentAsync().ToObservable()
                .ObserveOnUIDispatcher()
                .Where(ok => ok)
                .Subscribe(_ =>
                {
                    SetOriginalHtmlDocumentPath(app);
                    SetOriginalAccessibleUrlsFilePath(app);
                })
                .AddTo(CompositeDisposable);
        }

        private void SetOriginalAccessibleUrlsFilePath(Application app)
        {
            var filePath = app.OriginalAccessibleUrlsFilePath;
            if (File.Exists(filePath))
                AccessibleUrlsFilePath.CaptionIsUseTag = nameof(Resources.Replace);

            ReplaceAdditionalVm(AccessibleUrlsFilePath.Additional, () =>
            {
                if (File.Exists(filePath) == false)
                    return null;
                return new FileOpenerVm(
                    nameof(Resources.OpenNspContainsAccessibleUrlsFile_Caption),
                    app.ToReactivePropertyAsSynchronized(x => x.OriginalAccessibleUrlsFilePath));
            });
        }

        private void SetOriginalHtmlDocumentPath(Application app)
        {
            var filePath = app.OriginalHtmlDocumentPath;
            if (Directory.Exists(filePath))
                HtmlDocumentPath.CaptionIsUseTag = nameof(Resources.Replace);

            ReplaceAdditionalVm(HtmlDocumentPath.Additional, () =>
            {
                if (Directory.Exists(filePath) == false)
                    return null;
                return new FileOpenerVm(
                    nameof(Resources.OpenNspContainsHtmlDocument_Caption),
                    app.ToReactivePropertyAsSynchronized(x => x.OriginalHtmlDocumentPath));
            });
        }

        private static IObservable<string> AbsolutePathSource(Project project, ReactiveProperty<string> property)
        {
            return property.Select(project.ToAbsolutePath);
        }

        private static void ReplaceAdditionalVm<T>(ViewModelBase paramVm, Func<T> replaceVmFactory) where T : ViewModelBase
        {
            var paramsVm = (paramVm as CompositeParamVm)?.Params;
            if (paramsVm == null)
                return;
            var matchedParamVm = paramsVm.FirstOrDefault(x => x is T);
            if (matchedParamVm != null)
                paramsVm.RemoveOnScheduler(matchedParamVm);
            var replaceVm = replaceVmFactory();
            if (replaceVm != null)
                paramsVm.AddOnScheduler(replaceVm);
        }
    }
}
