﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel.ParamTreeBuilder
{
    public class AppNspParamTreeBuilder : ParamTreeBuilderBase
    {
        protected virtual Param.ResultValue GetProgramParamValue(Action<float> onProgress = null)
            => GetNcaIdenticalParamValue(ContentType.Program, null, onProgress);

        protected virtual Param.ResultValue GetLegalInformationParamValue(Action<float> onProgress = null)
            => GetNcaIdenticalParamValue(ContentType.LegalInformation, null, onProgress);

        // オフライン HTML とアクセス可能な URL のリストは常に厳密な nca のバイナリ比較をする

        protected virtual Param.ResultValue GetAccessibleUrlsParamValue(Action<float> onProgress = null)
             => GetStrictNcaIdenticalParamValue(ContentType.HtmlDocument, "fs0/accessible-urls", onProgress);

        protected virtual Param.ResultValue GetHtmlDocumentParamValue(Action<float> onProgress = null)
            => GetStrictNcaIdenticalParamValue(ContentType.HtmlDocument, "fs0/html-document", onProgress);

        private Param.ResultValue GetNcaIdenticalParamValue(ContentType contentType, string prefixPath = null, Action<float> onProgress = null)
        {
            if (HasPatchContentMeta)
                return GetStrictNcaIdenticalParamValue(contentType, prefixPath, onProgress);

            var content0 = Project0.ApplicationContentMeta.GetContent(contentType);
            var content1 = Project1.ApplicationContentMeta.GetContent(contentType);
            return new Param.ResultValue
            {
                IsIdentical = content0?.Hash == content1?.Hash,
                HasValue0 = content0 != null,
                HasValue1 = content1 != null
            };
        }

        private Param.ResultValue GetStrictNcaIdenticalParamValue(ContentType contentType, string prefixPath = null, Action<float> onProgress = null)
        {
            try
            {
                var result = Project0.NspFile.GetNcaIdenticalResult(Project1.NspFile, contentType.ToString(), prefixPath, onProgress, _cancelToken.Token);
                return new Param.ResultValue
                {
                    IsIdentical = result.HasFlag(NcaCompareResult.Identical),
                    HasValue0 = result.HasFlag(NcaCompareResult.SourceNcaExists) && result.HasFlag(NcaCompareResult.SourceFileExists),
                    HasValue1 = result.HasFlag(NcaCompareResult.TargetNcaExists) && result.HasFlag(NcaCompareResult.TargetFileExists)
                };
            }
            catch
            {
                // 何らかの理由でデータを読み取れなかった場合、差異なしの「読み取れません」表示とする
                return new Param.ResultValue { IsNotRead = true };
            }
        }

        protected virtual string GetNxIconFilePath(Project p, Title t) => t?.OriginalNxIconFilePath;
        protected virtual string GetIconFilePath(Project p, Title t) => t?.OriginalIconFilePath;

        private readonly CancellationTokenSource _cancelToken = new CancellationTokenSource();

        public AppNspParamTreeBuilder(Container diContainer, ObservableCollection<ComparisonPanelVm.Target> targets)
            : base(diContainer, targets)
        {
        }

        public override ObservableCollection<Param> UpdataParams(Config config)
        {
            ///////////////////////////////////////////////////////////////

            var appInfo = new Param(config, nameof(Resources.Application));
            var nspSizeInfo = new Param(config, nameof(Resources.SizeInformation));
            var basic = new Param(config, nameof(Resources.Basic));
            var language = new Param(config, nameof(Resources.Language));
            var title = new Param(config, nameof(Resources.Title));
            var limitation = new Param(config, nameof(Resources.Limitation));

            ///////////////////////////////////////////////////////////////
            var general = new Param(config, nameof(Resources.General));
            var core = new Param(config, nameof(Resources.Core));
            var account = new Param(config, nameof(Resources.Account));
            var bcat = new Param(config, nameof(Resources.Bcat));
            var userSave = new Param(config, nameof(Resources.SaveData));
            var tempDataArea = new Param(config, nameof(Resources.TemporaryDataArea));
            var deviceSave = new Param(config, nameof(Resources.DeviceSaveData));
            var fsAccess = new Param(config, nameof(Resources.FsAccessControl));
            var feature = new Param(config, nameof(Resources.Feature));
            var localComm = new Param(config, nameof(Resources.LocalCommunication));
            var playLogs = new Param(config, nameof(Resources.PlayLogs));
            var cardSpec = new Param(config, nameof(Resources.NxDevelopmentCard));

            ///////////////////////////////////////////////////////////////
            if (HasProgramInfo)
                UpdataParams_AppInfo(config, appInfo);
            if (HasNspSizeInfo)
                UpdataParams_NspSizeInfo(config, nspSizeInfo);
            UpdataParams_Basic(config, basic);
            UpdataParams_Language(config, language);
            UpdataParams_Title(config, title);
            UpdataParams_Limitation(config, limitation);

            ///////////////////////////////////////////////////////////////
            UpdataParams_General(config, general);
            if (HasProgramInfo == false)
                UpdateParams_Core(config, core);
            UpdataParams_Account(config, account);
            if (IsSupportBcat)
                UpdataParams_Bcat(config, bcat);
            UpdataParams_UserSave(config, userSave);
            UpdataParams_DeviceSave(config, deviceSave);
            UpdataParams_UserSave(config, userSave);
            if (IsSupportTempAndCacheStorage)
                UpdataParams_TempDataArea(config, tempDataArea);
            UpdateParams_FsAccessControlData(config, fsAccess);
            UpdataParams_Feature(config, feature);
            UpdateParams_LocalComm(config, localComm);
            UpdateParams_PlayLogs(config, playLogs);
            UpdataParams_CardSpec(config, cardSpec);

            ///////////////////////////////////////////////////////////////

            var rootBasic = new Param(config, nameof(Resources.Basic));
            var rootAdvanced = new Param(config, nameof(Resources.Advanced));

            if (HasProgramInfo)
                rootBasic.AddChild(appInfo);
            if (HasNspSizeInfo)
                rootBasic.AddChild(nspSizeInfo);
            rootBasic.AddChild(basic);
            rootBasic.AddChild(language);
            rootBasic.AddChild(title);
            rootBasic.AddChild(limitation);

            rootAdvanced.AddChild(general);
            if (HasProgramInfo == false)
                rootAdvanced.AddChild(core);
            rootAdvanced.AddChild(account);
            if (IsSupportBcat)
                rootAdvanced.AddChild(bcat);
            rootAdvanced.AddChild(userSave);
            if (deviceSave.Children.Count > 0)
                rootAdvanced.AddChild(deviceSave);
            if (fsAccess.Children.Count > 0)
                rootAdvanced.AddChild(fsAccess);
            rootAdvanced.AddChild(feature);
            rootAdvanced.AddChild(localComm);
            if (playLogs.Children.Count > 0)
                rootAdvanced.AddChild(playLogs);
            rootAdvanced.AddChild(cardSpec);

            ///////////////////////////////////////////////////////////////

            return new ObservableCollection<Param>
            {
                rootBasic,
                rootAdvanced
            };
        }

        private void UpdataParams_AppInfo(Config config, Param dst)
        {
            var program = new Param(config, nameof(Resources.Program),
                p => Task.Run(() => GetProgramParamValue(p)),
                TrimSecond);

            var appHashDigest = new Param(config, nameof(Resources.RomDigest_Caption),
                () => Project0.Digest,
                () => Project1.Digest);

            var sdkVersion = new Param(config, nameof(Resources.SdkVersion_Caption),
                () => ProgramInfo0.SdkVersion,
                () => ProgramInfo1.SdkVersion);

            var toolVersion = new Param(config, nameof(Resources.ToolVersion_Caption),
                () => ProgramInfo0.ToolVersion,
                () => ProgramInfo1.ToolVersion);

            var patchToolVersion = new Param(config, nameof(Resources.PatchToolVersion_Caption),
                () => ProgramInfo0.PatchToolVersion, Project0.PatchContentMeta != null,
                () => ProgramInfo1.PatchToolVersion, Project1.PatchContentMeta != null);

            var buildType = new Param(config, nameof(Resources.BuildType_Caption),
                () => ProgramInfo0.BuildType,
                () => ProgramInfo1.BuildType);

            dst.AddChild(program);
            dst.AddChild(appHashDigest);
            dst.AddChild(sdkVersion);
            dst.AddChild(toolVersion);
            if (HasPatchContentMeta)
                dst.AddChild(patchToolVersion);
            dst.AddChild(buildType);

            var middlewares = new Param(config, nameof(Resources.Middlewares_Caption));
            {
                var count = Math.Max(ProgramInfo0.Middlewares.Count, ProgramInfo1.Middlewares.Count);
                if (count == 0)
                {
                    middlewares.IsExistsValue0 = false;
                    middlewares.IsExistsValue1 = false;
                }

                for (var i = 0; i != count; ++i)
                {
                    var m0 = Element(ProgramInfo0.Middlewares, i);
                    var m1 = Element(ProgramInfo1.Middlewares, i);

                    var p = new Param(config, $"[{i}]",
                        () => Join(m0?.ModuleName, m0?.VendorName), m0 != null,
                        () => Join(m1?.ModuleName, m1?.VendorName), m1 != null);

                    middlewares.AddChild(p);
                }
            }
            dst.AddChild(middlewares);

            if (HasPatchContentMeta)
            {
                UpdateParams_PatchHistory(config, dst);
                UpdateParams_IsUsePatchDelta(config, dst);
            }
        }

        private void UpdateParams_PatchHistory(Config config, Param dst)
        {
            var histories = new Param(config, nameof(Resources.PatchHistory_Caption));
            {
                var count = Math.Max(
                    Project0.PatchContentMeta?.History.Count ?? 0,
                    Project1.PatchContentMeta?.History.Count ?? 0);
                if (count == 0)
                {
                    histories.IsExistsValue0 = false;
                    histories.IsExistsValue1 = false;
                }

                for (var i = 0; i != count; ++i)
                {
                    var history = new Param(config, $"[{i}]");

                    var m0 = Element(Project0.PatchContentMeta?.History, i);
                    var m1 = Element(Project1.PatchContentMeta?.History, i);

                    var type = new Param(config, nameof(Resources.Type),
                        () => m0?.Type.ToString(), m0 != null,
                        () => m1?.Type.ToString(), m1 != null);

                    var version = new Param(config, nameof(Resources.ReleaseVersion_Caption),
                        () => (m0?.Version >> 16)?.ToString(), m0 != null,
                        () => (m1?.Version >> 16)?.ToString(), m1 != null);

                    var digest = new Param(config, nameof(Resources.RomDigest_Caption),
                        () => m0?.Digest, m0 != null,
                        () => m1?.Digest, m1 != null);

                    history.AddChild(type);
                    history.AddChild(version);
                    history.AddChild(digest);

                    histories.AddChild(history);
                }
            }
            dst.AddChild(histories);
        }

        private void UpdateParams_IsUsePatchDelta(Config config, Param dst)
        {
            var isProduction0 = Project0.PatchContentMeta != null && Project0.PatchContentMeta.IsProduction;
            var isProduction1 = Project1.PatchContentMeta != null && Project1.PatchContentMeta.IsProduction;

            // 両方のパッチが製品化処理されたものでなければ、パッチ間差分についての情報は省略する
            if (isProduction0 == false && isProduction1 == false)
                return;

            {
                var isUseDelta = new Param(config, nameof(Resources.ComparisonPatchDeutaUse_Caption),
                    () => Bool_Use(Project0.PatchContentMeta?.IsUsePatchDelta == true), isProduction0,
                    () => Bool_Use(Project1.PatchContentMeta?.IsUsePatchDelta == true), isProduction1);
                dst.AddChild(isUseDelta);
            }

            {
                var isUseDeltaBefore3Nup = new Param(config, nameof(Resources.ComparisonPatchDeutaUseBefore3Nup_Caption),
                    () => Bool_Use(Project0.PatchContentMeta?.IsUsePatchDeltaBefore3NUP == true), isProduction0,
                    () => Bool_Use(Project1.PatchContentMeta?.IsUsePatchDeltaBefore3NUP == true), isProduction1);
                dst.AddChild(isUseDeltaBefore3Nup);
            }
        }

        private void UpdataParams_NspSizeInfo(Config config, Param dst)
        {
            var isPatch = Prop0.Type == ContentMetaType.Patch;

            var downloadSize = new Param(config,
                isPatch ? nameof(Resources.PatchDownloadSize_Caption) : nameof(Resources.DownloadSize_Caption),
                () => Prop0.Size.DownLoad.ToReadableSizeString(),
                () => Prop1.Size.DownLoad.ToReadableSizeString());

            var applicationUsedAreaOnCard = new Param(config,
                isPatch ? nameof(Resources.PatchAreaUsedOnCard_Caption) : nameof(Resources.ApplicationAreaUsedOnCard_Caption),
                () => Prop0.Size.ApplicationAreaUsedOnCard.ToReadableSizeString(),
                () => Prop1.Size.ApplicationAreaUsedOnCard.ToReadableSizeString());

            var applicationAvailableAreaOnCard = new Param(config, nameof(Resources.ApplicationAreaAvailableOnCard_Caption),
                () => Prop0.Size.ApplicationAreaAvailableOnCard.ToReadableSizeString(),
                () => Prop1.Size.ApplicationAreaAvailableOnCard.ToReadableSizeString());

            var minimumSaveDataSize = new Param(config, nameof(Resources.MinimumSaveDataSize_Caption),
                () => Prop0.Size.MinimumSaveData.ToReadableSizeString(),
                () => Prop1.Size.MinimumSaveData.ToReadableSizeString());

            var eachUserAccountSaveDataSize = new Param(config, nameof(Resources.EachUserAccountSaveDataSize_Caption),
                () => Prop0.Size.EachUserAccountSaveData.ToReadableSizeString(),
                () => Prop1.Size.EachUserAccountSaveData.ToReadableSizeString());

            dst.AddChild(downloadSize);
            dst.AddChild(applicationUsedAreaOnCard);
            dst.AddChild(applicationAvailableAreaOnCard);
            dst.AddChild(minimumSaveDataSize);
            dst.AddChild(eachUserAccountSaveDataSize);
        }

        private void UpdataParams_Basic(Config config, Param dst)
        {
            var applicationId = new Param(config, nameof(Resources.ApplicationId_Caption),
                () => Core0.ApplicationId.ToHex(),
                () => Core1.ApplicationId.ToHex());

            var displayVersion = new Param(config, nameof(Resources.DisplayVersion_Caption),
                () => App0.DisplayVersion,
                () => App1.DisplayVersion);

            var releaseVersion = new Param(config, nameof(Resources.ReleaseVersion_Caption),
                () => App0.ReleaseVersion.ToString(),
                () => App1.ReleaseVersion.ToString());

            var logoType = new Param(config, nameof(Resources.LogoType_Caption),
                () => EnumString(App0.LogoType),
                () => EnumString(App1.LogoType));

            var isDemo = new Param(config, nameof(Resources.Attribute_Demo_Caption),
                () => Bool_YesNo(App0.IsDemo),
                () => Bool_YesNo(App1.IsDemo));

            dst.AddChild(applicationId);
            dst.AddChild(displayVersion);
            dst.AddChild(releaseVersion);
            dst.AddChild(logoType);
            dst.AddChild(isDemo);

            var legalInfo = new Param(config, nameof(Resources.LegalInformationFilePath_Caption),
                p => Task.Run(() => GetLegalInformationParamValue(p)),
                TrimSecond);
            dst.AddChild(legalInfo);

            var htmlDocument = new Param(config, nameof(Resources.HtmlDocumentDirectoryPath_Caption),
                p => Task.Run(() => GetHtmlDocumentParamValue(p)),
                TrimSecond);
            dst.AddChild(htmlDocument);

            var accessibleUrls = new Param(config, nameof(Resources.AccessibleUrlsFilePath_Caption),
                p => Task.Run(() => GetAccessibleUrlsParamValue(p)),
                TrimSecond);
            dst.AddChild(accessibleUrls);
        }

        private void UpdataParams_Language(Config config, Param dst)
        {
            var america = new Param(config, nameof(Resources.America));
            var asia = new Param(config, nameof(Resources.Asia));
            var europe = new Param(config, nameof(Resources.Europe));

            foreach (var l in Enum.GetValues(typeof(LanguageType)).Cast<LanguageType>())
            {
                var lp = new Param(config, $"LanguageType_{l}_Caption",
                    () => Bool_Supported(App0.SupportedLanguage.Any(x => x == l.ToString())),
                    () => Bool_Supported(App1.SupportedLanguage.Any(x => x == l.ToString())));

                switch (LanguageHelper.LanguageToArea(l))
                {
                    case AreaType.America:
                        america.AddChild(lp);
                        break;

                    case AreaType.Asia:
                        asia.AddChild(lp);
                        break;

                    case AreaType.Europe:
                        europe.AddChild(lp);
                        break;

                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            dst.AddChild(america);
            dst.AddChild(asia);
            dst.AddChild(europe);
        }

        private void UpdataParams_Title(Config config, Param dst)
        {
            foreach (var l in Enum.GetValues(typeof(LanguageType)).Cast<LanguageType>())
            {
                var lp = new Param(config, $"LanguageType_{l}_Caption");

                {
                    var model0 = App0.Titles.FirstOrDefault(x => x.Language == l);
                    var model1 = App1.Titles.FirstOrDefault(x => x.Language == l);

                    if (model0 == null && model1 == null)
                        continue;

                    var name = new Param(config, nameof(Resources.Title_Name_Caption),
                        () => model0?.Name, model0 != null,
                        () => model1?.Name, model1 != null);

                    var publisher = new Param(config, nameof(Resources.Title_Publisher_Caption),
                        () => model0?.Publisher, model0 != null,
                        () => model1?.Publisher, model1 != null);

                    var icon = new Param(config, nameof(Resources.Title_IconFilePath_Caption),
                        new Param.ResultValue
                        {
                            IsIdentical = IsEqualIcon(model0, model1),
                            HasValue0 = File.Exists(GetIconFilePath(Project0, model0)),
                            HasValue1 = File.Exists(GetIconFilePath(Project1, model1))
                        },
                        model0 != null,
                        model1 != null,
                        TrimSecond);

                    lp.AddChild(name);
                    lp.AddChild(publisher);
                    lp.AddChild(icon);
                }

                dst.AddChild(lp);
            }
        }

        private bool IsEqualIcon(Title t0, Title t1)
        {
            var t0nx = BinaryFile(GetNxIconFilePath(Project0, t0), new byte[0]);
            var t1nx = BinaryFile(GetNxIconFilePath(Project1, t1), new byte[0]);
            if (t0nx.SequenceEqual(t1nx) == false)
                return false;

            var t0icon = BinaryFile(GetIconFilePath(Project0, t0), new byte[0]);
            var t1icon = BinaryFile(GetIconFilePath(Project1, t1), new byte[0]);
            if (t0icon.SequenceEqual(t1icon) == false)
                return false;

            return true;
        }

        private void UpdataParams_Limitation(Config config, Param dst)
        {
            var parentalControl = new Param(config, nameof(Resources.ParentalControl_Caption),
                () => EnumString(App0.ParentalControl),
                () => EnumString(App1.ParentalControl));

            var rating = new Param(config, nameof(Resources.Rating_Caption));
            {
                var count = Math.Max(App0.Ratings.Count, App1.Ratings.Count);
                if (count == 0)
                {
                    rating.IsExistsValue0 = false;
                    rating.IsExistsValue1 = false;
                }

                var app0Ratings = App0.Ratings.ToDictionary(x => x.Organization);
                var app1Ratings = App1.Ratings.ToDictionary(x => x.Organization);

                foreach (var d in Constants.AllRatingData)
                {
                    Rating r0;
                    Rating r1;
                    app0Ratings.TryGetValue(d.Key, out r0);
                    app1Ratings.TryGetValue(d.Key, out r1);

                    if (r0 == null && r1 == null)
                        continue;

                    var p = new Param(config, $"{d.Key}",
                        () => r0?.Age.ToString(), r0?.Age != null,
                        () => r1?.Age.ToString(), r1?.Age != null);

                    rating.AddChild(p);
                }
            }

            dst.AddChild(parentalControl);

            var requiredNetworkServiceLicenseOnLaunch = GetParam_RequiredNetworkServiceLicense(config);
            if (requiredNetworkServiceLicenseOnLaunch != null)
            {
                dst.AddChild(requiredNetworkServiceLicenseOnLaunch);
            }

            dst.AddChild(rating);

        }

        private Param GetParam_RequiredNetworkServiceLicense(Config config)
        {
            if (Project.AppCapability.IsSupportRequiredNetworkServiceLicenseOnLaunch == false)
                return null;

            // 非公開 nmeta 項目のため、比較元 / 比較先どちらにも値がない時は非表示
            if (!App0.RequiredNetworkServiceLicenseOnLaunch.HasValue &&
                !App1.RequiredNetworkServiceLicenseOnLaunch.HasValue)
            {
                return null;
            }

            return new Param(config,
                nameof(Resources.RequiredNetworkServiceLicenseOnLaunch_Caption),
                // App0
                () => App0.RequiredNetworkServiceLicenseOnLaunch.HasValue
                    ? EnumString(App0.RequiredNetworkServiceLicenseOnLaunch)
                    : string.Empty,
                App0.RequiredNetworkServiceLicenseOnLaunch.HasValue,
                // App1
                () => App1.RequiredNetworkServiceLicenseOnLaunch.HasValue
                    ? EnumString(App1.RequiredNetworkServiceLicenseOnLaunch)
                    : string.Empty,
                App1.RequiredNetworkServiceLicenseOnLaunch.HasValue);
        }

        private void UpdataParams_General(Config config, Param dst)
        {
            var logoHandling = new Param(config, nameof(Resources.LogoHandling_Caption),
                () => EnumString(App0.LogoHandling),
                () => EnumString(App1.LogoHandling));

            var presenceGroupId = new Param(config, nameof(Resources.PresenceGroupId_Caption),
                () => Join(Bool_Use(App0.IsUsePresenceGroupIdAppId), App0.PresenceGroupId.ToHex()),
                () => Join(Bool_Use(App1.IsUsePresenceGroupIdAppId), App1.PresenceGroupId.ToHex()));

            var appErrorCodeCategory = new Param(config, nameof(Resources.ApplicationErrorCodeCategory_Caption),
                () => Join(Bool_Use(App0.IsUseApplicationErrorCode), App0.ApplicationErrorCodeCategory),
                () => Join(Bool_Use(App1.IsUseApplicationErrorCode), App1.ApplicationErrorCodeCategory));

            dst.AddChild(logoHandling);
            dst.AddChild(presenceGroupId);
            dst.AddChild(appErrorCodeCategory);

            if (Project.AppCapability.IsSupportSeedForPseudoDeviceId)
            {
                var seedForPseudoDeviceId = new Param(config, nameof(Resources.SeedForPseudoDeviceId_Caption),
                    () => Join(Bool_Use(App0.IsUseSeedForPseudoDeviceAppId == false), App0.SeedForPseudoDeviceId.ToHex()),
                    () => Join(Bool_Use(App1.IsUseSeedForPseudoDeviceAppId == false), App1.SeedForPseudoDeviceId.ToHex()));
                dst.AddChild(seedForPseudoDeviceId);
            }

            // プレイログの設定項目は非公開なので、設定が存在する時だけ表示
            if (App0.PlayLogPolicySpecified || App1.PlayLogPolicySpecified)
            {
                var playLogPolicy = new Param(config, nameof(Resources.PlayLogPolicy_Caption),
                    () => App0.PlayLogPolicySpecified ? EnumString(App0.PlayLogPolicy) : string.Empty, App0.PlayLogPolicySpecified,
                    () => App1.PlayLogPolicySpecified ? EnumString(App1.PlayLogPolicy) : string.Empty, App1.PlayLogPolicySpecified);
                dst.AddChild(playLogPolicy);
            }

            // HDCP の設定項目は非公開なので、比較においては一方の設定が Required になっているときだけ表示
            if (App0.Hdcp == Application.HdcpType.Required ||
                App1.Hdcp == Application.HdcpType.Required)
            {
                var hdcpFlags = new Param(config, nameof(Resources.Hdcp_Caption),
                    () => App0.Hdcp.HasValue ? EnumString(App0.Hdcp) : string.Empty, App0.Hdcp.HasValue,
                    () => App1.Hdcp.HasValue ? EnumString(App1.Hdcp) : string.Empty, App1.Hdcp.HasValue);
                dst.AddChild(hdcpFlags);
            }
        }

        private void UpdateParams_Core(Config config, Param dst)
        {
            var stackSize = new Param(config, nameof(Resources.MainThreadStackSize_Caption),
                () => Core0.MainThreadStackSize.ToReadableSizeString(),
                () => Core1.MainThreadStackSize.ToReadableSizeString());
            dst.AddChild(stackSize);

            var addressSpace = new Param(config, nameof(Resources.ProcessAddressSpace_Caption),
                () => EnumString(Core0.ProcessAddressSpace),
                () => EnumString(Core1.ProcessAddressSpace));
            dst.AddChild(addressSpace);

            if (Project0.AppCapability.IsSupportSystemResourceSize == false)
                return;

            var systemResourceSize = new Param(config, nameof(Resources.SystemResourceSize_Caption),
                () => Core0.SystemResourceSize.ToReadableSizeString(),
                () => Core1.SystemResourceSize.ToReadableSizeString());
            dst.AddChild(systemResourceSize);
        }

        private void UpdataParams_Account(Config config, Param dst)
        {
            var startupUserAccount = new Param(config, nameof(Resources.StartupUserAccount_Caption),
                () => EnumString(App0.StartupUserAccount),
                () => EnumString(App1.StartupUserAccount));

            dst.AddChild(startupUserAccount);
        }

        private void UpdataParams_Bcat(Config config, Param dst)
        {
            var deliveryCacheStorageSize = new Param(config, nameof(Resources.BcatDeliveryCacheStorageSize_Caption),
                () => App0.BcatDeliveryCacheStorageSize.ToReadableSizeString(), App0.BcatDeliveryCacheStorageSize > 0,
                () => App1.BcatDeliveryCacheStorageSize.ToReadableSizeString(), App1.BcatDeliveryCacheStorageSize > 0);

            var passPhrase = new Param(config, nameof(Resources.BcatPassphrase_Caption),
                () => App0.BcatPassphrase ?? string.Empty, string.IsNullOrEmpty(App0.BcatPassphrase) == false,
                () => App1.BcatPassphrase ?? string.Empty, string.IsNullOrEmpty(App1.BcatPassphrase) == false);

            dst.AddChild(deliveryCacheStorageSize);
            dst.AddChild(passPhrase);
        }

        private void UpdataParams_UserSave(Config config, Param dst)
        {
            var summary = new Param(config, nameof(Resources.TotalSaveData_Caption),
                () => App0.TotalSaveDataSize.ToReadableSizeString(),
                () => App1.TotalSaveDataSize.ToReadableSizeString());
            dst.AddChild(summary);

            var saveDataSize = new Param(config, nameof(Resources.SaveDataSize_Caption),
                () => Join(Bool_Use(App0.IsUseSaveData), App0.SaveDataSize.ToReadableSizeString()),
                () => Join(Bool_Use(App1.IsUseSaveData), App1.SaveDataSize.ToReadableSizeString()));
            dst.AddChild(saveDataSize);

            var journalSize = new Param(config, nameof(Resources.SaveDataJournalSize_Caption),
                () => Join(Bool_Use(App0.IsSpecifiedSaveDataJournal), App0.SaveDataJournalSize.ToReadableSizeString()),
                () => Join(Bool_Use(App1.IsSpecifiedSaveDataJournal), App1.SaveDataJournalSize.ToReadableSizeString()));
            dst.AddChild(journalSize);

            var dataLossConf = new Param(config, nameof(Resources.DataLossConfirmation_Caption),
                () => EnumString(App0.DataLossConfirmation),
                () => EnumString(App1.DataLossConfirmation));
            dst.AddChild(dataLossConf);

            if (Project0.AppCapability.IsSupportSaveDataSizeExtend)
            {
                var saveDataSizeMax = new Param(config, nameof(Resources.SaveDataSizeMax_Caption),
                    () => Join(Bool_Use(App0.IsUseUserAccountSaveDataSizeMax), App0.UserAccountSaveDataSizeMax.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsUseUserAccountSaveDataSizeMax), App1.UserAccountSaveDataSizeMax.ToReadableSizeString()));
                dst.AddChild(saveDataSizeMax);

                var journalSizeMax = new Param(config, nameof(Resources.SaveDataJournalSizeMax_Caption),
                    () => Join(Bool_Use(App0.IsUseUserAccountSaveDataJournalSizeMax), App0.UserAccountSaveDataJournalSizeMax.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsUseUserAccountSaveDataJournalSizeMax), App1.UserAccountSaveDataJournalSizeMax.ToReadableSizeString()));
                dst.AddChild(journalSizeMax);
            }
        }

        private void UpdataParams_DeviceSave(Config config, Param dst)
        {
            // デバイスセーブデータは非公開項目なので、値が明示的に指定されていた場合にのみ表示する
            if (App0.IsSpecifiedDeviceSaveDataSize || App1.IsSpecifiedDeviceSaveDataSize)
            {
                var saveDataSize = new Param(config, nameof(Resources.SaveDataSize_Caption),
                    () => Join(Bool_Use(App0.IsSpecifiedDeviceSaveDataSize), App0.DeviceSaveDataSize.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsSpecifiedDeviceSaveDataSize), App1.DeviceSaveDataSize.ToReadableSizeString()));
                dst.AddChild(saveDataSize);
            }

            if (App0.IsSpecifiedDeviceSaveDataJournalSize || App1.IsSpecifiedDeviceSaveDataJournalSize)
            {
                var journalSize = new Param(config, nameof(Resources.SaveDataJournalSize_Caption),
                    () => Join(Bool_Use(App0.IsSpecifiedDeviceSaveDataJournalSize), App0.DeviceSaveDataJournalSize.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsSpecifiedDeviceSaveDataJournalSize), App1.DeviceSaveDataJournalSize.ToReadableSizeString()));
                dst.AddChild(journalSize);
            }

            if (App0.IsUseDeviceSaveDataSizeMax || App1.IsUseDeviceSaveDataSizeMax)
            {
                var saveDataSizeMax = new Param(config, nameof(Resources.SaveDataSizeMax_Caption),
                    () => Join(Bool_Use(App0.IsUseDeviceSaveDataSizeMax), App0.DeviceSaveDataSizeMax.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsUseDeviceSaveDataSizeMax), App1.DeviceSaveDataSizeMax.ToReadableSizeString()));
                dst.AddChild(saveDataSizeMax);
            }

            if (App0.IsUseDeviceSaveDataJournalSizeMax || App1.IsUseDeviceSaveDataJournalSizeMax)
            {
                var journalSizeMax = new Param(config, nameof(Resources.SaveDataJournalSizeMax_Caption),
                    () => Join(Bool_Use(App0.IsUseDeviceSaveDataJournalSizeMax), App0.DeviceSaveDataJournalSizeMax.ToReadableSizeString()),
                    () => Join(Bool_Use(App1.IsUseDeviceSaveDataJournalSizeMax), App1.DeviceSaveDataJournalSizeMax.ToReadableSizeString()));
                dst.AddChild(journalSizeMax);
            }
        }

        private void UpdataParams_TempDataArea(Config config, Param dst)
        {
            var tempStorageSize = new Param(config, nameof(Resources.TemporaryStorageSize_Caption),
                () => Join(Bool_Use(App0.IsUseTemporaryStorageSize), App0.TemporaryStorageSize.ToReadableSizeString()),
                () => Join(Bool_Use(App1.IsUseTemporaryStorageSize), App1.TemporaryStorageSize.ToReadableSizeString()));
            dst.AddChild(tempStorageSize);

            var cacheStorageTotal = new Param(config, nameof(Resources.CacheStorage_TotalSaveData_Caption),
                () => App0.TotalCacheStorageSize.ToReadableSizeString(),
                () => App1.TotalCacheStorageSize.ToReadableSizeString());
            dst.AddChild(cacheStorageTotal);

            var cacheStorageSize = new Param(config, nameof(Resources.CacheStorageSize_Caption),
                () => Join(Bool_Use(App0.IsUseCacheStorageSize), App0.CacheStorageSize.ToReadableSizeString()),
                () => Join(Bool_Use(App1.IsUseCacheStorageSize), App1.CacheStorageSize.ToReadableSizeString()));
            dst.AddChild(cacheStorageSize);

            var cacheStorageJournalSize = new Param(config, nameof(Resources.CacheStorageJournalSize_Caption),
                () => Join(Bool_Use(App0.IsUseCacheStorageJournalSize), App0.CacheStorageJournalSize.ToReadableSizeString()),
                () => Join(Bool_Use(App1.IsUseCacheStorageJournalSize), App1.CacheStorageJournalSize.ToReadableSizeString()));
            dst.AddChild(cacheStorageJournalSize);
        }

        private void UpdateParams_FsAccessControlData(Config config, Param dst)
        {
            if (HasProgramInfo == false)
            {
                var flagPresets = new Param(config, nameof(Resources.FlagPresets_Caption),
                    () => Core0.FsAccessControlData.FlagPresets,
                    () => Core1.FsAccessControlData.FlagPresets);

                dst.AddChild(flagPresets);
            }

            if (Project.AppCapability.IsSupportSaveDataOwnerIds == false)
                return;

            var saveDataOwnerIds0 = HasProgramInfo
                ? Project0.ProgramInfo.FsAccessControlData.SaveDataOwnerIds
                : Core0.FsAccessControlData.SaveDataOwnerIds;
            var saveDataOwnerIds1 = HasProgramInfo
                ? Project1.ProgramInfo.FsAccessControlData.SaveDataOwnerIds
                : Core1.FsAccessControlData.SaveDataOwnerIds;

            var ownerIds = new Param(config, nameof(Resources.SaveDataOwnerId_Caption));
            {
                var count = Math.Max(saveDataOwnerIds0.Count, saveDataOwnerIds1.Count);
                if (count == 0)
                {
                    ownerIds.IsExistsValue0 = false;
                    ownerIds.IsExistsValue1 = false;
                }

                for (var i = 0; i < count; ++i)
                {
                    var m0 = Element(saveDataOwnerIds0, i);
                    var m1 = Element(saveDataOwnerIds1, i);

                    var value0 = m0?.ApplicationId.ToHex() ?? string.Empty;
                    var value1 = m1?.ApplicationId.ToHex() ?? string.Empty;

                    if (m0?.Accessibility != null)
                        value0 += $",{m0.Accessibility}";
                    if (m1?.Accessibility != null)
                        value1 += $",{m1.Accessibility}";

                    var p = new Param(config, $"[{i}]",
                        () => value0, m0 != null,
                        () => value1, m1 != null);
                    ownerIds.AddChild(p);
                }
            }
            dst.AddChild(ownerIds);
        }

        private void UpdataParams_Feature(Config config, Param dst)
        {
            var screenshot = new Param(config, nameof(Resources.Screenshot_Caption),
                () => Bool_Allow(App0.IsAllowScreenshot),
                () => Bool_Allow(App1.IsAllowScreenshot));

            dst.AddChild(screenshot);

            if (Project.AppCapability.IsSupportVideoCapture)
            {
                var videoCapture = new Param(config, nameof(Resources.VideoCapture_Caption),
                    () => App0.VideoCapture.HasValue ? EnumString(App0.VideoCapture) : string.Empty, App0.VideoCapture.HasValue,
                    () => App1.VideoCapture.HasValue ? EnumString(App1.VideoCapture) : string.Empty, App1.VideoCapture.HasValue);

                dst.AddChild(videoCapture);
            }

            if (Project.AppCapability.IsSupportRuntimeAddOnContentInstall)
            {
                var runtimeAddOnContentInstall = new Param(config, nameof(Resources.RuntimeAddOnContentInstall_Caption),
                    () => App0.RuntimeAddOnContentInstall.HasValue
                        ? EnumString(App0.RuntimeAddOnContentInstall)
                        : string.Empty, App0.RuntimeAddOnContentInstall.HasValue,
                    () => App1.RuntimeAddOnContentInstall.HasValue
                        ? EnumString(App1.RuntimeAddOnContentInstall)
                        : string.Empty, App1.RuntimeAddOnContentInstall.HasValue);

                dst.AddChild(runtimeAddOnContentInstall);
            }

            if (Project.AppCapability.IsSupportCrashReport)
            {
                var crashReport = new Param(config, nameof(Resources.CrashReport_Caption),
                    () => App0.CrashReport.HasValue ? EnumString(App0.CrashReport) : string.Empty, App0.CrashReport.HasValue,
                    () => App1.CrashReport.HasValue ? EnumString(App1.CrashReport) : string.Empty, App1.CrashReport.HasValue);

                dst.AddChild(crashReport);
            }
        }

        private void UpdateParams_PlayLogs(Config config, Param dst)
        {
            if (Project.AppCapability.IsSupportPlayLogs == false)
                return;

            var playLogQueryCapability = new Param(config, nameof(Resources.PlayLogQueryCapability_Caption),
                () => App0.PlayLogQueryCapability.HasValue ? EnumString(App0.PlayLogQueryCapability) : string.Empty, App0.PlayLogQueryCapability.HasValue,
                () => App1.PlayLogQueryCapability.HasValue ? EnumString(App1.PlayLogQueryCapability) : string.Empty, App1.PlayLogQueryCapability.HasValue);

            dst.AddChild(playLogQueryCapability);

            var queryableAppIds = new Param(config, nameof(Resources.PlayLogQueryableApplicationId_Caption));
            {
                var count = Math.Max(
                    App0.PlayLogQueryableApplicationIds.Count,
                    App1.PlayLogQueryableApplicationIds.Count);
                if (count == 0)
                {
                    queryableAppIds.IsExistsValue0 = false;
                    queryableAppIds.IsExistsValue1 = false;
                }

                for (var i = 0; i != count; ++i)
                {
                    var m0 = Element(App0.PlayLogQueryableApplicationIds, i);
                    var m1 = Element(App1.PlayLogQueryableApplicationIds, i);

                    var p = new Param(config, $"[{i}]",
                        () => m0?.Id.ToHex(), m0 != null,
                        () => m1?.Id.ToHex(), m1 != null);

                    queryableAppIds.AddChild(p);
                }
            }

            dst.AddChild(queryableAppIds);
        }

        private void UpdateParams_LocalComm(Config config, Param dst)
        {
            var comm = new Param(config, nameof(Resources.LocalCommunicationId_Caption));
            {
                var count = Math.Max(App0.LocalCommunicationIds.Count, App1.LocalCommunicationIds.Count);
                if (count == 0)
                {
                    comm.IsExistsValue0 = false;
                    comm.IsExistsValue1 = false;
                }

                for (var i = 0; i != count; ++i)
                {
                    var m0 = Element(App0.LocalCommunicationIds, i);
                    var m1 = Element(App1.LocalCommunicationIds, i);

                    var p = new Param(config, $"[{i}]",
                        () => m0?.Id.ToHex(), m0 != null,
                        () => m1?.Id.ToHex(), m1 != null);

                    comm.AddChild(p);
                }
            }

            dst.AddChild(comm);
        }

        private void UpdataParams_CardSpec(Config config, Param dst)
        {
            var s = DiContainer.GetInstance<StringHelper>();
            var isNspMode = HasNspSizeInfo;

            var size = new Param(config, nameof(Resources.CardSpecSize_Caption),
                () => s.GetCardSizeReadableString(Meta0.CardSpec, isNspMode),
                () => s.GetCardSizeReadableString(Meta1.CardSpec, isNspMode));

            var clock = new Param(config, nameof(Resources.CardSpecClockRate_Caption),
                () => s.GetCardClockRateReadableString(Meta0.CardSpec, isNspMode),
                () => s.GetCardClockRateReadableString(Meta1.CardSpec, isNspMode));

            dst.AddChild(size);
            dst.AddChild(clock);
        }

        private bool HasNspSizeInfo => Prop0?.Size != null && Prop1?.Size != null;

        private bool HasProgramInfo
            =>
                string.IsNullOrEmpty(Content0(ContentType.Program)?.Hash) == false &&
                string.IsNullOrEmpty(Content1(ContentType.Program)?.Hash) == false;

        private bool HasPatchContentMeta
            => Project0.PatchContentMeta != null || Project1.PatchContentMeta != null;

        private bool IsSupportBcat
            => Project0.AppCapability.IsSupportBcat;

        private bool IsSupportTempAndCacheStorage
            => Project0.AppCapability.IsSupportTempAndCacheStorage;

        private bool IsSupportUserAccountSaveDataOperation
            => Project0.AppCapability.IsSupportUserAccoutnSaveDataOperation;

        public override void Dispose()
        {
            _cancelToken?.Cancel();
            _cancelToken?.Dispose();
        }
    }
}
