﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Utilities;
using Livet.Commands;
using Livet.Messaging.IO;
using Microsoft.Win32;

namespace Nintendo.Authoring.AuthoringEditor.Controls
{
    public class FilePathEditBoxSlim : Control
    {
        static FilePathEditBoxSlim()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata(typeof(FilePathEditBoxSlim)));
        }

        #region Path

        public string Path
        {
            get { return (string) GetValue(PathProperty); }
            set { SetValue(PathProperty, value); }
        }

        public static readonly DependencyProperty PathProperty =
            DependencyProperty.Register(
                "Path",
                typeof(string),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata()
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #endregion

        #region Title

        public string Title
        {
            get { return (string) GetValue(TitleProperty); }
            set { SetValue(TitleProperty, value); }
        }

        public static readonly DependencyProperty TitleProperty =
            DependencyProperty.Register(
                "Title",
                typeof(string),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #endregion

        #region Filter

        public string Filter
        {
            get { return (string) GetValue(FilterProperty); }
            set { SetValue(FilterProperty, value); }
        }

        public static readonly DependencyProperty FilterProperty =
            DependencyProperty.Register(
                "Filter",
                typeof(string),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #endregion

        #region InitialDirectory

        public string InitialDirectory
        {
            get { return (string) GetValue(InitialDirectoryProperty); }
            set { SetValue(InitialDirectoryProperty, value); }
        }

        public static readonly DependencyProperty InitialDirectoryProperty =
            DependencyProperty.Register(
                "InitialDirectory",
                typeof(string),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #endregion

        #region IsReadOnly

        public bool IsReadOnly
        {
            get { return (bool) GetValue(IsReadOnlyProperty); }
            set { SetValue(IsReadOnlyProperty, value); }
        }

        public static readonly DependencyProperty IsReadOnlyProperty =
            DependencyProperty.Register(
                "IsReadOnly",
                typeof(bool),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(bool)
                }
            );

        #endregion

        #region IsSaveDialog

        public bool IsSaveDialog
        {
            get { return (bool) GetValue(IsSaveDialogProperty); }
            set { SetValue(IsSaveDialogProperty, value); }
        }

        public static readonly DependencyProperty IsSaveDialogProperty =
            DependencyProperty.Register(
                nameof(IsSaveDialog),
                typeof(bool),
                typeof(FilePathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    DefaultValue = default(bool),
                    BindsTwoWayByDefault = true
                }
            );

        #endregion

        private ListenerCommand<FileSelectionMessage> _OpenFileCommand;

        public ListenerCommand<FileSelectionMessage> OpenFileCommand
            => _OpenFileCommand ?? (_OpenFileCommand = new ListenerCommand<FileSelectionMessage>(OpenFile, CanOpenFile))
            ;

        public void OpenFile(FileSelectionMessage m)
        {
            var path = Path;
            string initDir = null;

            try
            {
                initDir = string.IsNullOrEmpty(path)
                    ? InitialDirectory
                    : System.IO.Path.GetDirectoryName(Environment.ExpandEnvironmentVariables(path));

                // Windowsのパスにする
                initDir = initDir?.ToPathString();
            }
            catch
            {
                // ignored
            }

            string selectedFilePath;
            var result = OpenFileDialog(Title, Filter, initDir, out selectedFilePath);
            if (result == true)
                Path = selectedFilePath;
        }

        private bool? OpenFileDialog(string title, string filter, string initialDirectory, out string selectedFilePath)
        {
            bool? result;

            try
            {
                var dialog = IsSaveDialog
                    ? (FileDialog) new SaveFileDialog
                    {
                        Title = title,
                        Filter = filter,
                        InitialDirectory = initialDirectory
                    }
                    : new OpenFileDialog
                    {
                        Title = title,
                        Filter = filter,
                        InitialDirectory = initialDirectory
                    };

                result = dialog.ShowDialog();
                selectedFilePath = dialog.FileName;
            }
            catch
            {
                // 失敗したので初期ディレクトリ無しで再挑戦
                var dialog = IsSaveDialog
                    ? (FileDialog) new SaveFileDialog
                    {
                        Title = title,
                        Filter = filter,
                    }
                    : new OpenFileDialog
                    {
                        Title = title,
                        Filter = filter,
                    };

                result = dialog.ShowDialog();
                selectedFilePath = dialog.FileName;
            }

            return result;
        }

        public static bool CanOpenFile()
        {
            return true;
        }

        private ListenerCommand<object> _ClearCommand;

        public ListenerCommand<object> ClearCommand
            => _ClearCommand ?? (_ClearCommand = new ListenerCommand<object>(Clear, CanClear));

        public void Clear(object m)
        {
            if (string.IsNullOrEmpty(Path))
            {
                return;
            }
            Path = string.Empty;
        }

        public static bool CanClear()
        {
            return true;
        }

        private void UserControl_Drop(object sender, DragEventArgs e)
        {
            var files = e.Data.GetData(DataFormats.FileDrop) as string[];

            var path = files?.FirstOrDefault(File.Exists);
            if (path != null)
                Path = path;
        }

        private static void UserControl_PreviewDragOver(object sender, DragEventArgs e)
        {
            e.Effects = DragDropEffects.None;

            if (e.Data.GetDataPresent(DataFormats.FileDrop, true))
            {
                var files = e.Data.GetData(DataFormats.FileDrop) as string[];
                if (files != null)
                {
                    if (files.Any(File.Exists))
                        e.Effects = DragDropEffects.Copy;
                }
            }

            e.Handled = true;
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            var basePanel = this.FindChild<Grid>(null);

            basePanel.Drop += UserControl_Drop;
            basePanel.PreviewDragOver += UserControl_PreviewDragOver;
        }
    }
}
