﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using BezelEditor.Foundation;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public static class YamlHelper
    {
        private static Serializer _Serializer;

        private static Serializer Serializer
        {
            get
            {
                return LazyInitializer.EnsureInitialized(ref _Serializer,
                    () => new SerializerBuilder().EmitDefaults().Build());
            }
        }

        public static bool Save<T>(PathString filePath, T data)
        {
            try
            {
                if (Directory.Exists(filePath.Parent) == false)
                    Directory.CreateDirectory(filePath.Parent);

                using (var tw = new StringWriter())
                {
                    Serializer.Serialize(tw, data);
                    File.WriteAllText(filePath, tw.ToString(), Encoding.UTF8);
                }
                return true;
            }
            catch
            {
                return false;
            }
        }

        public static string SaveToString(object data)
        {
            using (var tw = new StringWriter())
            {
                Serializer.Serialize(tw, data);
                return tw.ToString();
            }
        }

        public static T Load<T>(PathString filePath, Func<T> onError)
        {
            try
            {
                return LoadFromString<T>(File.ReadAllText(filePath));
            }
            catch
            {
                return onError();
            }
        }

        public static T Load<T>(byte[] fileImage, Func<T> onError)
        {
            try
            {
                // BOMを飛ばす
                if (fileImage.Length >= 3)
                    if ((fileImage[0] == 0xEF) &&
                        (fileImage[1] == 0xBB) &&
                        (fileImage[2] == 0xBF))
                        fileImage = fileImage.Skip(3).ToArray();

                return LoadFromString<T>(Encoding.UTF8.GetString(fileImage));
            }
            catch
            {
                return onError();
            }
        }

        public static T LoadFromString<T>(string source)
        {
            var yaml = new StringReader(source);
            return new DeserializerBuilder().IgnoreUnmatchedProperties().Build().Deserialize<T>(yaml);
        }
    }
}
