﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reactive.Linq;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Foundation.Utilities;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public delegate void ProgressEventHandler(object sender, int progress);

    public class AuthoringToolJob : IDisposable
    {
        public Process Process { get; private set; }

        public bool HasExited
        {
            get
            {
                try
                {
                    return Process.HasExited;
                }
                catch
                {
                    // 関連付けられているプロセスがない or プロセスの終了コードを取得できない
                    // いずれの状況もプロセス自体は "終了している" とみなす
                    return true;
                }
            }
        }

        private bool _IsTerminateAsSuccessful;

        public int? ExitCode
        {
            get
            {
                if (_IsTerminateAsSuccessful)
                    return 0;
                if (HasExited == false)
                    return null;
                return Process?.ExitCode;
            }
        }

        #region StandardOutput

        private List<string> _StandardOutput;

        public IReadOnlyList<string> StandardOutput => _StandardOutput;

        public bool IsRedirectStandardOutput
        {
            get { return _StandardOutput != null; }
            set
            {
                Debug.Assert(Process != null);

                if (value && _StandardOutput == null)
                {
                    _StandardOutput = new List<string>();
                    Process.OutputDataReceived += ProcessOnOutputDataReceived;
                }
                else if (value == false && _StandardOutput != null)
                {
                    Process.OutputDataReceived -= ProcessOnOutputDataReceived;
                    _StandardOutput = null;
                }
            }
        }

        private void ProcessOnOutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data == null)
            {
                _waitForOutputSync.Set();
                return;
            }
            _StandardOutput?.Add(e.Data);

        }

        #endregion

        #region StandardError

        private List<string> _StandardError;

        public IReadOnlyList<string> StandardError => _StandardError;

        public bool IsRedirectStandardError
        {
            get { return _StandardError != null; }
            set
            {
                Debug.Assert(Process != null);

                if (value && _StandardError == null)
                {
                    _StandardError = new List<string>();
                    Process.ErrorDataReceived += ProcessOnErrorDataReceived;
                }
                else if (value == false && _StandardError != null)
                {
                    Process.ErrorDataReceived -= ProcessOnErrorDataReceived;
                    _StandardError = null;
                }
            }
        }

        private void ProcessOnErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data == null)
            {
                _waitForErrorSync.Set();
                return;
            }
            _StandardError?.Add(e.Data);
        }

        #endregion

        public event ProgressEventHandler ProgressEvent;

        public AuthoringToolJob(Process process)
        {
            Process = process;
        }

        public void Start()
        {
            Debug.Assert(Process != null);

            Process.OutputDataReceived += (_, e) =>
            {
                var progress = GetProgressPercent(e.Data);
                if (progress.HasValue)
                    ProgressEvent?.Invoke(this, progress.Value);
            };

            Process.Start();

            if (IsRedirectStandardError)
                _waitForErrorSync.Reset();
            else
                _waitForErrorSync.Set();

            if (IsRedirectStandardOutput)
                _waitForOutputSync.Reset();
            else
                _waitForOutputSync.Set();

            if (Process.StartInfo.RedirectStandardOutput)
                Process.BeginOutputReadLine();

            if (Process.StartInfo.RedirectStandardError)
                Process.BeginErrorReadLine();

        }

        public void WaitForExit()
        {
            Process.WaitForExit();
            _waitForOutputSync.Wait();
            _waitForErrorSync.Wait();
            if (Process.ExitCode == 0)
                ProgressEvent?.Invoke(this, 100);
        }

        private readonly ManualResetEventSlim _waitForOutputSync = new ManualResetEventSlim(false);
        private readonly ManualResetEventSlim _waitForErrorSync = new ManualResetEventSlim(false);

        public Task WaitForExitAsync(CancellationToken cancellationToken = default(CancellationToken))
        {
            var source = new TaskCompletionSource<object>();
            IDisposable d = null;
            d = Observable.FromEvent<EventHandler, EventArgs>(
                h => (sender, e) => h(e),
                h => Process.Exited += h,
                h => Process.Exited -= h
            ).Subscribe(_ =>
            {
                d?.Dispose();
                _waitForOutputSync.Wait(cancellationToken);
                _waitForErrorSync.Wait(cancellationToken);
                if (Process.ExitCode == 0)
                    ProgressEvent?.Invoke(this, 100);
                source.TrySetResult(null);
            });
            if (cancellationToken != default(CancellationToken))
                cancellationToken.Register(() => source.TrySetCanceled());
            Process.EnableRaisingEvents = true;
            return source.Task;
        }

        public Task<T> WaitForExitAsync<T>(
            Func<AuthoringToolJob, T> exitAction,
            CancellationToken cancellationToken = default(CancellationToken))
        {
            var task = new Task<T>(() => exitAction(this), cancellationToken);
            IDisposable d = null;
            d = Observable.FromEvent<EventHandler, EventArgs>(
                h => (sender, e) => h(e),
                h => Process.Exited += h,
                h => Process.Exited -= h
            ).Subscribe(_ =>
            {
                d?.Dispose();
                try
                {
                    _waitForOutputSync.Wait(cancellationToken);
                    _waitForErrorSync.Wait(cancellationToken);
                }
                catch (Exception e) when (e is OperationCanceledException)
                {
                    // タスクキャンセル系の例外は無視する
                }
                if (Process.ExitCode == 0)
                    ProgressEvent?.Invoke(this, 100);
                if (task.IsCanceled == false)
                    task.Start();
            });
            Process.EnableRaisingEvents = true;
            return task;
        }

        public void Terminate()
        {
            try
            {
                Process.Kill();
            }
            catch
            {
                // ignored
            }
        }

        public void TerminateAsSuccessful()
        {
            Terminate();
            _IsTerminateAsSuccessful = true;
        }

        public void Dispose()
        {
            try
            {
                if (Process.HasExited == false)
                {
                    Terminate();
                    WaitForExit();
                }
            }
            catch
            {
                // ignore
            }
            finally
            {
                _waitForOutputSync.Dispose();
                _waitForErrorSync.Dispose();
            }
        }

        public NspHandleResultType MakeResultType(NspHandleResult result)
        {
            return new NspHandleResultType
            {
                Result = result,
                ErrorMessages = GetOutputMessages()
            };
        }

        public NspHandleResultType MakeResultType()
        {
            return new NspHandleResultType
            {
                Result = ExitCode == 0 ? NspHandleResult.Ok : NspHandleResult.Error,
                ErrorMessages = GetOutputMessages()
            };
        }

        private string[] GetOutputMessages()
        {
            var stdout = StandardOutput.ToList();
            var stderr = StandardError.ToArray();
            stdout.AddRange(stderr);
            return stdout.ToArray();
        }

        private static readonly Regex ProgressRegex = new Regex(@"^\[Progress\].* (\d*) %$");

        private static int? GetProgressPercent(string cmdLine)
        {
            if (string.IsNullOrEmpty(cmdLine))
                return null;

            var m = ProgressRegex.Match(cmdLine);
            if (m.Success == false)
                return null;

            return int.Parse(m.Groups[1].Value);
        }
    }

    public static class AuthoringToolWrapper
    {
        public static AuthoringToolJob Run(string args, Action<int> progressChanged = null)
        {
            var job = Create(args);

            if (progressChanged != null)
                job.ProgressEvent += (_, progress) => progressChanged(progress);

            job.IsRedirectStandardError = true;
            job.IsRedirectStandardOutput = true;

            job.Start();

            return job;
        }

        public static AuthoringToolJob Create(string args)
        {
            var process = ProcessUtility.CreateProcess(AuthoringToolHelper.AuthoringToolExe, args);
            var job = new AuthoringToolJob(process);
            return job;
        }

        public static string Version()
        {
            using (var job = Run("showversion"))
            {
                job.WaitForExit();
                var version = job.StandardOutput.FirstOrDefault();
                if (string.IsNullOrEmpty(version))
                    return null;
                return version;
            }
        }

        public static string NxAddonVersion()
        {
            var version = Version();
            return version.Split(':').First().Replace('_', '.');
        }
    }
}
