﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class InvalidApplicationMetaItemException : Exception
    {
        public string Name { get; }
        public string Value { get; }

        public InvalidApplicationMetaItemException(string itemName, string itemValue)
        {
            Name = itemName;
            Value = itemValue;
        }
    }

    public partial class ApplicationMeta : DisposableModelBase
    {
        #region Core

        private Core _Core = new Core();

        public Core Core
        {
            get { return _Core; }
            set
            {
                var old = _Core;
                if (SetProperty(ref _Core, value))
                {
                    old?.Dispose();
                    UpdateHasErrorObserver();
                }
            }
        }

        #endregion

        #region Application

        private Application _Application;

        public Application Application
        {
            get { return _Application; }
            set
            {
                var old = _Application;
                if (SetProperty(ref _Application, value))
                {
                    old?.Dispose();
                    UpdateHasErrorObserver();
                }
            }
        }

        #endregion

        #region CardSpec

        private CardSpec _CardSpec = new CardSpec();

        [XmlIgnore]
        public CardSpec CardSpec
        {
            get { return _CardSpec; }
            set
            {
                var old = _CardSpec;
                if (SetProperty(ref _CardSpec, value))
                {
                    old?.Dispose();
                    UpdateHasErrorObserver();
                }
            }
        }

        [YamlIgnore]
        [XmlElement("CardSpec")]
        public CardSpec CardSpecForXml
        {
            get
            {
                if (CardSpec.IsAutomaticSettingClockRate && CardSpec.IsAutomaticSettingSize)
                    return null;
                return _CardSpec;
            }
            set { CardSpec = value; }
        }

        #endregion

        #region IsReadOnly

        private bool _IsReadOnly;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsReadOnly
        {
            get { return _IsReadOnly; }
            set { SetProperty(ref _IsReadOnly, value); }
        }

        #endregion

        #region FileFormatErrors

        private ObservableCollection<ApplicationMetaError> _FileFormatErrors = new ObservableCollection<ApplicationMetaError>();

        [YamlIgnore]
        [XmlIgnore]
        public ObservableCollection<ApplicationMetaError> FileFormatErrors
        {
            get { return _FileFormatErrors; }
            set { SetProperty(ref _FileFormatErrors, value); }
        }

        #endregion

        #region UnpublishableErrors

        private ObservableCollection<ApplicationMetaError> _UnpublishableErrors = new ObservableCollection<ApplicationMetaError>();

        [YamlIgnore]
        [XmlIgnore]
        public ObservableCollection<ApplicationMetaError> UnpublishableErrors
        {
            get { return _UnpublishableErrors; }
            set { SetProperty(ref _UnpublishableErrors, value); }
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public Container DiContainer
        {
            get { return _DiContainer; }
            set
            {
                _DiContainer = value;
                Application.DiContainer = value;

                // ReSharper disable once ExplicitCallerInfoArgument
                RaisePropertyChanged(nameof(DiContainer));
            }
        }

        private Container _DiContainer;

        public ApplicationMeta()
        {
            _Application = new Application();

            CompositeDisposable.Add(() => Core?.Dispose());
            CompositeDisposable.Add(() => Application?.Dispose());
            CompositeDisposable.Add(() => CardSpec?.Dispose());

            InitializeValidation();
        }
    }

    public partial class Core : PropertyTrackingDispoableModel<Core>
    {
        #region Name

        private string _Name = "Application";

        public string Name
        {
            get { return _Name; }
            set { SetProperty(ref _Name, value); }
        }

        #endregion

        #region Is64BitInstruction

        [YamlIgnore]
        public string Is64BitInstruction
        {
            get { return IsProcessAddressSpace64Bit ? "True" : "False"; }

            // ReSharper disable once ValueParameterNotUsed
            set { }
        }

        #endregion

        #region ApplicationId

        private ulong _ApplicationId = Constants.DefaultMetaProgramId;

        [XmlIgnore]
        public ulong ApplicationId
        {
            get { return _ApplicationId; }
            set { SetProperty(ref _ApplicationId, value); }
        }

        [YamlIgnore]
        [XmlElement("ApplicationId")]
        public string ApplicationIdHex
        {
            get { return ApplicationId.ToHex(); }
            set
            {
                ulong appId;
                if (value.TryToUlong(out appId))
                    ApplicationId = appId;
            }
        }

        #endregion

        #region MainThreadStackSize

        private ulong _MainThreadStackSize = 0x100000;

        [XmlIgnore]
        public ulong MainThreadStackSize
        {
            get { return _MainThreadStackSize; }
            set { SetProperty(ref _MainThreadStackSize, value); }
        }

        [YamlIgnore]
        [XmlElement("MainThreadStackSize")]
        public string MainThreadStackSizeHex
        {
            get { return MainThreadStackSize.ToHex(8); }
            set
            {
                ulong stackSize;
                if (value.TryToUlong(out stackSize))
                    MainThreadStackSize = stackSize;
            }
        }

        #endregion

        #region ProcessAddressSpace

        private ProcessAddressSpaceType _ProcessAddressSpace = ProcessAddressSpaceType.AddressSpace64Bit;

        [XmlIgnore]
        public ProcessAddressSpaceType ProcessAddressSpace
        {
            get { return _ProcessAddressSpace; }
            set
            {
                switch (value)
                {
                    case ProcessAddressSpaceType.AddressSpace32Bit:
                    case ProcessAddressSpaceType.AddressSpace32BitNoReserved:
                        IsProcessAddressSpace64Bit = false;
                        break;

                    case ProcessAddressSpaceType.AddressSpace64Bit:
                        IsProcessAddressSpace64Bit = true;
                        break;

                    default:
                        throw new ArgumentOutOfRangeException(nameof(value), value, null);
                }
                SetProperty(ref _ProcessAddressSpace, value);
            }
        }

        [YamlIgnore]
        [XmlElement("ProcessAddressSpace")]
        public string ProcessAddressSpaceForXml
        {
            get { return ProcessAddressSpace.ToString(); }
            set
            {
                ProcessAddressSpaceType spaceType;
                if (Enum.TryParse(value, out spaceType))
                    ProcessAddressSpace = spaceType;
            }
        }

        #endregion

        #region IsProcessAddressSpace64bit

        [YamlIgnore]
        [XmlIgnore]
        public bool IsProcessAddressSpace64Bit { get; private set; } = true;

        #endregion

        #region FsAccessControlData

        private FsAccessControlData _FsAccessControlData = new FsAccessControlData()
        {
            FlagPresets = FsAccessControlData.GetDefaultFlagPresets()
        };

        public FsAccessControlData FsAccessControlData
        {
            get { return _FsAccessControlData; }
            set { SetProperty(ref _FsAccessControlData, value); }
        }

        #endregion

        #region IsUseSystemResourceSize

        private bool _IsUseSystemResourceSize = true;

        [XmlIgnore]
        public bool IsUseSystemResourceSize
        {
            get { return _IsUseSystemResourceSize; }
            set { SetProperty(ref _IsUseSystemResourceSize, value); }
        }

        #endregion

        #region SystemResourceSize

        private uint _SystemResourceSize;

        [XmlIgnore]
        public uint SystemResourceSize
        {
            get { return _SystemResourceSize; }
            set { SetProperty(ref _SystemResourceSize, value); }
        }

        [YamlIgnore]
        [XmlElement("SystemResourceSize")]
        public string SystemResourceSizeHex
        {
            get
            {
                // * SystemResourceSize の使用が有効
                // * プロセスが 64bit アドレス空間
                // * SystemResourceSize が 0 より大きい (デフォルト値なら書き出しをスキップ)
                return IsUseSystemResourceSize && IsProcessAddressSpace64Bit && SystemResourceSize > 0
                    ? "0x" + SystemResourceSize.ToString("x8")
                    : null;
            }
            set
            {
                try
                {
                    SystemResourceSize = Convert.ToUInt32(value, 16);
                }
                catch
                {
                    // 値コンバート中のエラーは無視
                }
            }
        }

        #endregion

        public Core()
        {
            InitializeValidation();
        }

        public override IEnumerable<string> GetChangedPropertyNames()
        {
            foreach (var prop in new[]
            {
                nameof(Name),
                nameof(ApplicationId),
                nameof(MainThreadStackSize),
                nameof(ProcessAddressSpace),
                nameof(SystemResourceSize)
            })
            {
                if (IsPropertyChanged(prop))
                    yield return prop;
            }

            if (IsPropertyChanged(nameof(FsAccessControlData)))
            {
                foreach (var prop in FsAccessControlData.GetChangedPropertyNames())
                {
                    yield return $"{nameof(FsAccessControlData)}/{prop}";
                }
            }
        }

        public override void MergePropertyValues(Core value)
        {
            value.TrySetChangedProperty(x => x.Name, this);
            value.TrySetChangedProperty(x => x.ApplicationId, this);
            value.TrySetChangedProperty(x => x.MainThreadStackSize, this);
            value.TrySetChangedProperty(x => x.ProcessAddressSpace, this);
            {
                FsAccessControlData fsAccessControlData;
                if (value.TryGetChangedProperty(x => x.FsAccessControlData, out fsAccessControlData))
                    FsAccessControlData.MergePropertyValues(fsAccessControlData);
            }
            value.TrySetChangedProperty(x => x.IsUseSystemResourceSize, this);
            value.TrySetChangedProperty(x => x.SystemResourceSize, this);
        }
    }

    public class Icon : DisposableModelBase
    {
        #region Language

        private LanguageType _Language;

        [XmlIgnore]
        public LanguageType Language
        {
            get { return _Language; }
            set { SetProperty(ref _Language, value); }
        }

        [YamlIgnore]
        [XmlElement("Language")]
        public string LanguageForXml
        {
            get { return Language.ToString(); }
            set
            {
                LanguageType lang;
                if (Enum.TryParse(value, out lang))
                    Language = lang;
            }
        }

        #endregion

        #region IconPath

        private ExpandablePath _IconPath = new ExpandablePath();

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath IconPath
        {
            get { return _IconPath; }
            set
            {
                if (_IconPath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(IconPath));
            }
        }

        [YamlIgnore]
        [XmlElement(nameof(IconPath))]
        public ExpandablePath IconPathForXml
        {
            get { return IconPath.IsEmpty ? null : IconPath; }
            set { IconPath = value; }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(IconPath))]
        public string IconPathForYaml
        {
            get { return IconPath.Path; }
            set { IconPath.Path = value; }
        }

        [XmlIgnore]
        public bool IconPathIsExpandEnvironmentVariableForYaml
        {
            get { return IconPath.IsExpandEnvironmentVariable; }
            set { IconPath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region NxIconPath

        private ExpandablePath _NxIconPath = new ExpandablePath();

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath NxIconPath
        {
            get { return _NxIconPath; }
            set { SetProperty(ref _NxIconPath, new ExpandablePath(value)); }
        }

        [YamlIgnore]
        [XmlElement(nameof(NxIconPath))]
        public ExpandablePath NxIconPathForXml
        {
            get { return NxIconPath.IsEmpty ? null : NxIconPath; }
            set { NxIconPath = value; }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(NxIconPath))]
        public string NxIconPathForYaml
        {
            get { return NxIconPath.Path; }
            set { NxIconPath.Path = value; }
        }

        [XmlIgnore]
        public bool NxIconPathIsExpandEnvironmentVariableForYaml
        {
            get { return NxIconPath.IsExpandEnvironmentVariable; }
            set { NxIconPath.IsExpandEnvironmentVariable = value; }
        }

        #endregion
    }

    public partial class Application : PropertyTrackingDispoableModel<Application>
    {
        #region Titles

        private ObservableCollection<Title> _Titles = new ObservableCollection<Title>();

        [XmlElement("Title")]
        public ObservableCollection<Title> Titles
        {
            get { return _Titles; }
            set { SetProperty(ref _Titles, value); }
        }

        #endregion

        #region IconsFromMetaXml

        private Icon[] _Icons = {};

        /// <summary>
        /// .nmeta からの読み込み時にのみ使用する。。
        /// 格納している値の実体は Titles プロパティ側に存在しているので、必ず Title 側を参照すること。
        /// </summary>
        [YamlIgnore]
        [XmlElement("Icon")]
        public Icon[] IconsFromMetaXml
        {
            get { return _Icons; }
            set { SetProperty(ref _Icons, value); }
        }

        #endregion

        #region ReleaseVersion

        private ushort _ReleaseVersion;

        [XmlIgnore]
        public ushort ReleaseVersion
        {
            get { return _ReleaseVersion; }
            set { SetProperty(ref _ReleaseVersion, value); }
        }

        [YamlIgnore]
        [XmlElement("ReleaseVersion")]
        public string ReleaseVersionForXml
        {
            get { return ReleaseVersion.ToString(); }
            set
            {
                try
                {
                    ReleaseVersion = Convert.ToUInt16(value);
                }
                catch
                {
                }
            }
        }

        #endregion

        #region DisplayVersion

        private string _DisplayVersion = string.Empty;

        public string DisplayVersion
        {
            get { return _DisplayVersion; }
            set { SetProperty(ref _DisplayVersion, value); }
        }

        #endregion

        #region StartupUserAccount

        private StartupUserAccountType _StartupUserAccount = StartupUserAccountType.None;

        [XmlIgnore]
        public StartupUserAccountType StartupUserAccount
        {
            get { return _StartupUserAccount; }
            set { SetProperty(ref _StartupUserAccount, value); }
        }

        [YamlIgnore]
        [XmlElement("StartupUserAccount")]
        public string StartupUserAccountForXml
        {
            get { return StartupUserAccount.ToString(); }
            set
            {
                StartupUserAccountType startupType;
                if (Enum.TryParse(value, out startupType))
                    StartupUserAccount = startupType;
            }
        }

        #endregion

        #region ParentalControl

        private ParentalControlType _ParentalControl = ParentalControlType.None;

        [XmlIgnore]
        public ParentalControlType ParentalControl
        {
            get { return _ParentalControl; }
            set { SetProperty(ref _ParentalControl, value); }
        }

        [YamlIgnore]
        [XmlElement("ParentalControl")]
        public string ParentalControlForXml
        {
            get
            {
                var parentalControl = ParentalControl;
                switch (parentalControl)
                {
                    case ParentalControlType.FreeCommunication:
                        return parentalControl.ToString();
                    case ParentalControlType.None:
                        return null;
                    default:
                        throw new ArgumentOutOfRangeException(nameof(ParentalControl));
                }
            }
            set
            {
                ParentalControlType parentalControl;
                if (Enum.TryParse(value, out parentalControl))
                {
                    switch (parentalControl)
                    {
                        case ParentalControlType.FreeCommunication:
                            ParentalControl = parentalControl;
                            break;
                        default:
                            // FreeCommunication 以外の場合、値は反映しない
                            break;
                    }
                }
            }
        }

        [YamlIgnore]
        [XmlIgnore]
        public bool ParentalControlSpecified => ParentalControl != ParentalControlType.None;

        #endregion

        #region SupportedLanguages

        private ObservableCollection<SupportedLanguage> _SupportedLanguages =
            Enum.GetValues(typeof(LanguageType))
                .Cast<LanguageType>()
                .Select(l => new SupportedLanguage
                {
                    Language = l
                }).ToObservableCollection();

        [XmlIgnore]
        public ObservableCollection<SupportedLanguage> SupportedLanguages
        {
            get { return _SupportedLanguages; }
            set { SetProperty(ref _SupportedLanguages, value); }
        }

        [YamlIgnore]
        [XmlElement("SupportedLanguage")]
        public string[] SupportedLanguage
        {
            get
            {
                return SupportedLanguages
                    .Where(x => x.IsSupported)
                    .Select(x => x.Language.ToString())
                    .ToArray();
            }

            set
            {
                if (value == null)
                {
                    foreach (var l in SupportedLanguages)
                        l.IsSupported = false;
                }
                else
                {
                    var langHash = value.ToHashSet();

                    foreach (var l in SupportedLanguages)
                        l.IsSupported = langHash.Contains(l.Language.ToString());
                }
            }
        }

        #endregion

        #region PresenceGroupId

        private ulong _PresenceGroupId;

        [XmlIgnore]
        public ulong PresenceGroupId
        {
            get { return _PresenceGroupId; }
            set { SetProperty(ref _PresenceGroupId, value); }
        }

        [YamlIgnore]
        [XmlElement("PresenceGroupId")]
        public string PresenceGroupIdHex
        {
            get { return IsUsePresenceGroupIdAppId ? null : PresenceGroupId.ToHex(); }
            set
            {
                ulong groupId;
                if (value.TryToUlong(out groupId))
                    PresenceGroupId = groupId;
            }
        }

        #endregion

        #region IsUsePresenceGroupIdAppId

        private bool _IsUsePresenceGroupIdAppId = true;

        [XmlIgnore]
        public bool IsUsePresenceGroupIdAppId
        {
            get { return _IsUsePresenceGroupIdAppId; }
            set { SetProperty(ref _IsUsePresenceGroupIdAppId, value); }
        }

        #endregion

        #region Screenshot

        [YamlIgnore]
        [XmlIgnore]
        public ScreenshotType Screenshot
        {
            get { return IsAllowScreenshot ? ScreenshotType.Allow : ScreenshotType.Deny; }
            set { IsAllowScreenshot = value == ScreenshotType.Allow; }
        }

        [YamlIgnore]
        [XmlElement("Screenshot")]
        public string ScreenshotForXml
        {
            get { return Screenshot.ToString(); }
            set
            {
                ScreenshotType screenshot;
                if (Enum.TryParse(value, out screenshot))
                    Screenshot = screenshot;
            }
        }

        #endregion

        #region IsAllowScreenshot

        private bool _IsAllowScreenshot = true;

        [XmlIgnore]
        public bool IsAllowScreenshot
        {
            get { return _IsAllowScreenshot; }
            set { SetProperty(ref _IsAllowScreenshot, value); }
        }

        #endregion

        #region DataLossConfirmation

        private DataLossConfirmationType _DataLossConfirmation = DataLossConfirmationType.None;

        [XmlIgnore]
        public DataLossConfirmationType DataLossConfirmation
        {
            get { return _DataLossConfirmation; }
            set { SetProperty(ref _DataLossConfirmation, value); }
        }

        [YamlIgnore]
        [XmlElement("DataLossConfirmation")]
        public string DataLossConfirmationForXml
        {
            get { return DataLossConfirmation.ToString(); }
            set
            {
                DataLossConfirmationType dataLoss;
                if (Enum.TryParse(value, out dataLoss))
                    DataLossConfirmation = dataLoss;
            }
        }

        #endregion

        #region SourceXml

        private string _SourceXmlText;

        [YamlIgnore]
        [XmlIgnore]
        public string SourceXmlText
        {
            get { return _SourceXmlText; }
            set
            {
                if (SetProperty(ref _SourceXmlText, value))
                    DetectBcatElements();
            }
        }

        // Bcat 未サポート環境 (NX Addon 0.12 系, 1.x 系) においてチェックを行う
        private void DetectBcatElements()
        {
            var appXml = XDocument.Parse(SourceXmlText);
            var app = appXml.Root;
            if (app == null)
                return;

            ContainsBcatElement = ApplicationMetaBcatElement.None;

            var cacheStorageSize = app.Element("BcatDeliveryCacheStorageSize")?.Value;
            if (cacheStorageSize != null &&
                cacheStorageSize.IsUlong() && cacheStorageSize.ToUlong() > 0)
            {
                ContainsBcatElement |= ApplicationMetaBcatElement.BcatDeliveryCacheStorageSize;
            }

            var passPhrase = app.Element("BcatPassphrase")?.Value;
            if (string.IsNullOrEmpty(passPhrase) == false)
            {
                ContainsBcatElement |= ApplicationMetaBcatElement.BcatPassphrase;
            }

            var saveDataSize = app.Element("BcatSaveDataSize")?.Value;
            if (saveDataSize != null &&
                saveDataSize.IsUlong() && saveDataSize.ToUlong() > 0)
            {
                ContainsBcatElement |= ApplicationMetaBcatElement.BcatSaveDataSize;
            }
        }

        #endregion

        #region ContainsBcatElement

        private ApplicationMetaBcatElement _ContainsBcatElement;

        [XmlIgnore]
        [YamlIgnore]
        public ApplicationMetaBcatElement ContainsBcatElement
        {
            get { return _ContainsBcatElement; }
            set { SetProperty(ref _ContainsBcatElement, value); }
        }

        #endregion

        [XmlIgnore]
        [YamlIgnore]
        public long TotalSaveDataSize
        {
            get
            {
                var size = 0L;

                var saveDataSize = IsUseSaveData ? SaveDataSize : 0;

                size += saveDataSize;
                size += IsSpecifiedSaveDataJournal ? SaveDataJournalSize : saveDataSize;

                return size;
            }
        }

        [XmlIgnore]
        [YamlIgnore]
        public long TotalDeviceSaveDataSize
        {
            get
            {
                var size = 0L;

                size += DeviceSaveDataSize;
                size += DeviceSaveDataJournalSize;

                return size;
            }
        }

        [XmlIgnore]
        [YamlIgnore]
        public long TotalCacheStorageSize
        {
            get
            {
                var size = 0L;

                size += CacheStorageSize;
                size += CacheStorageJournalSize;

                return size;
            }
        }

        #region IsUseSaveData

        private bool _IsUseSaveData;

        [XmlIgnore]
        public bool IsUseSaveData
        {
            get { return _IsUseSaveData; }
            set { SetProperty(ref _IsUseSaveData, value); }
        }

        #endregion

        #region UserAccountSaveDataSize

        private long _saveDataSize;

        [XmlIgnore]
        public long SaveDataSize
        {
            get { return _saveDataSize; }
            set { SetProperty(ref _saveDataSize, value); }
        }

        [YamlIgnore]
        [XmlElement("UserAccountSaveDataSize")]
        public string SaveDataSizeHex
        {
            get { return (IsUseSaveData ? SaveDataSize : 0).ToHex(); }
            set
            {
                long saveDataSize;
                if (value.TryToLong(out saveDataSize))
                    SaveDataSize = saveDataSize;
            }
        }

        #endregion

        #region IsSpecifiedSaveDataJournal

        private bool _IsSpecifiedSaveDataJournal;

        [XmlIgnore]
        public bool IsSpecifiedSaveDataJournal
        {
            get { return _IsSpecifiedSaveDataJournal; }
            set { SetProperty(ref _IsSpecifiedSaveDataJournal, value); }
        }

        #endregion

        #region UserAccountSaveDataJournalSize

        private long _saveDataJournalSize;

        [XmlIgnore]
        public long SaveDataJournalSize
        {
            get { return _saveDataJournalSize; }
            set { SetProperty(ref _saveDataJournalSize, value); }
        }

        [YamlIgnore]
        [XmlElement("UserAccountSaveDataJournalSize")]
        public string SaveDataJournalSizeHex
        {
            get { return IsSpecifiedSaveDataJournal ? SaveDataJournalSize.ToHex() : SaveDataSizeHex; }
            set
            {
                long saveDataJournalSize;
                if (value.TryToLong(out saveDataJournalSize))
                    SaveDataJournalSize = saveDataJournalSize;
            }
        }

        #endregion

        #region ApplicationErrorCodeCategory

        private string _ApplicationErrorCodeCategory;

        [XmlIgnore]
        public string ApplicationErrorCodeCategory
        {
            get { return _ApplicationErrorCodeCategory; }
            set { SetProperty(ref _ApplicationErrorCodeCategory, value); }
        }

        [YamlIgnore]
        [XmlElement("ApplicationErrorCodeCategory")]
        public string ApplicationErrorCodeCategoryForXml
        {
            get { return IsUseApplicationErrorCode == false ? null : _ApplicationErrorCodeCategory; }
            set { SetProperty(ref _ApplicationErrorCodeCategory, value); }
        }

        #endregion

        #region IsUseApplicationErrorCode

        private bool _IsUseApplicationErrorCode;

        [XmlIgnore]
        public bool IsUseApplicationErrorCode
        {
            get { return _IsUseApplicationErrorCode; }
            set { SetProperty(ref _IsUseApplicationErrorCode, value); }
        }

        #endregion

        #region LogoType

        private LogoTypeType _LogoType = LogoTypeType.LicensedByNintendo;

        [XmlIgnore]
        public LogoTypeType LogoType
        {
            get { return _LogoType; }
            set { SetProperty(ref _LogoType, value); }
        }

        [YamlIgnore]
        [XmlElement("LogoType")]
        public string LogoTypeForXml
        {
            get { return LogoType.ToString(); }
            set
            {
                LogoTypeType logoType;
                if (Enum.TryParse(value, out logoType))
                    LogoType = logoType;
            }
        }

        #endregion

        #region LogoHandling

        private LogoHandlingType _LogoHandling = LogoHandlingType.Auto;

        [XmlIgnore]
        public LogoHandlingType LogoHandling
        {
            get { return _LogoHandling; }
            set { SetProperty(ref _LogoHandling, value); }
        }

        [YamlIgnore]
        [XmlElement("LogoHandling")]
        public string LogoHandlingForXml
        {
            get { return LogoHandling.ToString(); }
            set
            {
                LogoHandlingType logoHandling;
                if (Enum.TryParse(value, out logoHandling))
                    LogoHandling = logoHandling;
            }
        }

        #endregion

        #region LocalCommunicationIds

        private ObservableCollection<NintendoApplicationId> _LocalCommunicationIds =
            new ObservableCollection<NintendoApplicationId>();

        [XmlIgnore]
        public ObservableCollection<NintendoApplicationId> LocalCommunicationIds
        {
            get { return _LocalCommunicationIds; }
            set { SetProperty(ref _LocalCommunicationIds, value); }
        }

        [YamlIgnore]
        [XmlElement("LocalCommunicationId")]
        public string[] LocalCommunicationId
        {
            get { return LocalCommunicationIds.Select(i => i.Id.ToHex()).ToArray(); }

            set
            {
                LocalCommunicationIds.Clear();

                if (value == null)
                    return;

                foreach (var id in value)
                {
                    ulong communicationId;
                    if (id.TryToUlong(out communicationId))
                        LocalCommunicationIds.Add(new NintendoApplicationId {Id = communicationId});
                }
            }
        }

        #endregion

        public const int MaxLocalCommunicationIdCount = 8;

        #region Ratings

        private ObservableCollection<Rating> _Ratings = new ObservableCollection<Rating>(
            Constants.AllRatingOrganizations.Select(
                org => new Rating { IsUse = false, Organization = org }));

        [YamlIgnore]
        [XmlIgnore]
        public ObservableCollection<Rating> Ratings
        {
            get { return _Ratings; }
            set { SetProperty(ref _Ratings, value); }
        }

        // レーティング情報をシリアライズする場合にはレーティング機関名で昇順ソートした値を使う
        [XmlElement(nameof(Rating))]
        [YamlMember(Alias = nameof(Ratings))] // こちらだけ "s" あり
        public Rating[] RatingsForXmlOrYaml
        {
            get
            {
                return _Ratings
                    .Where(x => x.RatingData != null && x.Age.HasValue && x.IsUse)
                    .OrderBy(x => x.Organization)
                    .ToArray();
            }
            set { Ratings = value?.ToObservableCollection() ?? new ObservableCollection<Rating>(); }
        }

        #endregion

        #region IsDemo

        private bool _IsDemo;

        [XmlIgnore]
        public bool IsDemo
        {
            get { return _IsDemo; }
            set { SetProperty(ref _IsDemo, value); }
        }

        #endregion

        #region IsRetailInteractiveDisplay

        private bool _IsRetailInteractiveDisplay;

        [XmlIgnore]
        public bool IsRetailInteractiveDisplay
        {
            get { return _IsRetailInteractiveDisplay; }
            set { SetProperty(ref _IsRetailInteractiveDisplay, value); }
        }

        #endregion

        #region IsDownloadPlay

        private bool _IsDownloadPlay;

        [XmlIgnore]
        public bool IsDownloadPlay
        {
            get { return _IsDownloadPlay; }
            set { SetProperty(ref _IsDownloadPlay, value); }
        }

        #endregion

        #region Attribute

        [YamlIgnore]
        [XmlElement("Attribute")]
        public string[] AttributeForXml
        {
            get { return Attribute.Length == 0 ? null : Attribute; }
            set { Attribute = value; }
        }

        [YamlIgnore]
        [XmlIgnore]
        public string[] Attribute
        {
            get
            {
                var l = new List<string>();

                if (IsDemo)
                    l.Add(AttributeDemo);
                if (IsRetailInteractiveDisplay)
                    l.Add(AttributeRetailInteractiveDisplay);
                if (IsDownloadPlay)
                    l.Add(AttributeDownloadPlay);

                return l.ToArray();
            }

            set
            {
                if (value == null)
                {
                    IsDemo = false;
                    IsDownloadPlay = false;
                    IsRetailInteractiveDisplay = false;
                    return;
                }
                var h = value.ToHashSet();
                IsDemo = h.Contains(AttributeDemo);
                IsRetailInteractiveDisplay = h.Contains(AttributeRetailInteractiveDisplay);
                IsDownloadPlay = h.Contains(AttributeDownloadPlay);
            }
        }

        private const string AttributeDemo = "Demo";
        private const string AttributeRetailInteractiveDisplay = "RetailInteractiveDisplay";
        private const string AttributeDownloadPlay = "DownloadPlay";

        #endregion

        #region IsReplaceHtmlDocumentPath

        private bool _IsReplaceHtmlDocumentPath;

        [XmlIgnore]
        public bool IsReplaceHtmlDocumentPath
        {
            get { return _IsReplaceHtmlDocumentPath; }
            set { SetProperty(ref _IsReplaceHtmlDocumentPath, value); }
        }

        #endregion

        // オフライン HTML として nsp に格納するディレクトリのパスを指定

        #region HtmlDocumentPath

        private ExpandablePath _HtmlDocumentPath = new ExpandablePath();

        [XmlIgnore]
        [YamlIgnore]
        public ExpandablePath HtmlDocumentPath
        {
            get { return _HtmlDocumentPath; }
            set
            {
                if (_HtmlDocumentPath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(HtmlDocumentPath));
            }
        }

        [YamlIgnore]
        [XmlElement(nameof(HtmlDocumentPath))]
        public ExpandablePath HtmlDocumentPathForXml
        {
            get { return HtmlDocumentPath.IsEmpty ? null : HtmlDocumentPath; }
            set { HtmlDocumentPath = value; }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(HtmlDocumentPath))]
        public string HtmlDocumentPathForYaml
        {
            get { return HtmlDocumentPath.Path; }
            set { HtmlDocumentPath.Path = value; }
        }

        [XmlIgnore]
        public bool HtmlDocumentPathIsExpandEnvironmentVariableForYaml
        {
            get { return HtmlDocumentPath.IsExpandEnvironmentVariable; }
            set { HtmlDocumentPath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region IsReplaceAccessibleUrlsFilePath

        private bool _IsReplaceAccessibleUrlsFilePath;

        [XmlIgnore]
        public bool IsReplaceAccessibleUrlsFilePath
        {
            get { return _IsReplaceAccessibleUrlsFilePath; }
            set { SetProperty(ref _IsReplaceAccessibleUrlsFilePath, value); }
        }

        #endregion

        #region OriginalHtmlDocumentPath

        private string _OriginalHtmlDocumentPath;

        [XmlIgnore]
        [YamlIgnore]
        public string OriginalHtmlDocumentPath
        {
            get { return _OriginalHtmlDocumentPath; }
            set { SetProperty(ref _OriginalHtmlDocumentPath, value); }
        }

        #endregion

        #region AccessibleUrlsFilePath

        private ExpandablePath _AccessibleUrlsFilePath = new ExpandablePath();

        [XmlIgnore]
        [YamlIgnore]
        public ExpandablePath AccessibleUrlsFilePath
        {
            get { return _AccessibleUrlsFilePath; }
            set
            {
                if (_AccessibleUrlsFilePath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(AccessibleUrlsFilePath));
            }
        }

        [YamlIgnore]
        [XmlElement(nameof(AccessibleUrlsFilePath))]
        public ExpandablePath AccessibleUrlsFilePathForXml
        {
            get { return AccessibleUrlsFilePath.IsEmpty ? null : AccessibleUrlsFilePath; }
            set { AccessibleUrlsFilePath = value; }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(AccessibleUrlsFilePath))]
        public string AccessibleUrlsFilePathForYaml
        {
            get { return AccessibleUrlsFilePath.Path; }
            set { AccessibleUrlsFilePath.Path = value; }
        }

        [XmlIgnore]
        public bool AccessibleUrlsFilePathIsExpandEnvironmentVariableForYaml
        {
            get { return AccessibleUrlsFilePath.IsExpandEnvironmentVariable; }
            set { AccessibleUrlsFilePath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region IsReplaceLegalInformationFilePath

        private bool _IsReplaceLegalInformationFilePath;

        [XmlIgnore]
        public bool IsReplaceLegalInformationFilePath
        {
            get { return _IsReplaceLegalInformationFilePath; }
            set { SetProperty(ref _IsReplaceLegalInformationFilePath, value); }
        }

        #endregion

        #region OriginalAccessibleUrlsFilePath

        private string _OriginalAccessibleUrlsFilePath;

        [XmlIgnore]
        [YamlIgnore]
        public string OriginalAccessibleUrlsFilePath
        {
            get { return _OriginalAccessibleUrlsFilePath; }
            set { SetProperty(ref _OriginalAccessibleUrlsFilePath, value); }
        }

        #endregion

        // SLIM から発行されたリーガル情報 zip ファイルのパスを指定

        #region LegalInformationFilePath

        private ExpandablePath _LegalInformationFilePath = new ExpandablePath();

        [XmlIgnore]
        [YamlIgnore]
        public ExpandablePath LegalInformationFilePath
        {
            get { return _LegalInformationFilePath; }
            set
            {
                if (_LegalInformationFilePath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(LegalInformationFilePath));
            }
        }

        [YamlIgnore]
        [XmlElement(nameof(LegalInformationFilePath))]
        public ExpandablePath LegalInformationFilePathForXml
        {
            get { return LegalInformationFilePath.IsEmpty ? null : LegalInformationFilePath; }
            set { LegalInformationFilePath = value; }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(LegalInformationFilePath))]
        public string LegalInformationFilePathForYaml
        {
            get { return LegalInformationFilePath.Path; }
            set { LegalInformationFilePath.Path = value; }
        }

        [XmlIgnore]
        public bool LegalInformationFilePathIsExpandEnvironmentVariableForYaml
        {
            get { return LegalInformationFilePath.IsExpandEnvironmentVariable; }
            set { LegalInformationFilePath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        // .nsp からリーガル情報を展開した時のルートディレクトリのパス

        #region OriginalLegalInformationPath

        private string _OriginalLegalInformationPath;

        [XmlIgnore]
        [YamlIgnore]
        public string OriginalLegalInformationPath
        {
            get { return _OriginalLegalInformationPath; }
            set { SetProperty(ref _OriginalLegalInformationPath, value); }
        }

        #endregion

        #region SeedForPseudoDeviceId

        private ulong _SeedForPseudoDeviceId;

        [XmlIgnore]
        public ulong SeedForPseudoDeviceId
        {
            get { return _SeedForPseudoDeviceId; }
            set { SetProperty(ref _SeedForPseudoDeviceId, value); }
        }

        [YamlIgnore]
        [XmlElement("SeedForPseudoDeviceId")]
        public string SeedForPseudoDeviceIdHex
        {
            get { return IsUseSeedForPseudoDeviceAppId ? null : SeedForPseudoDeviceId.ToHex(); }
            set
            {
                ulong deviceId;
                if (value.TryToUlong(out deviceId))
                {
                    SeedForPseudoDeviceId = value.ToUlong();
                    IsUseSeedForPseudoDeviceAppId = true;
                }
            }
        }

        #endregion

        #region IsUseSeedForPseudoDeviceAppId

        private bool _IsUseSeedForPseudoDeviceAppId = true;

        [XmlIgnore]
        public bool IsUseSeedForPseudoDeviceAppId
        {
            get { return _IsUseSeedForPseudoDeviceAppId; }
            set { SetProperty(ref _IsUseSeedForPseudoDeviceAppId, value); }
        }

        #endregion

        // BCAT (Background Contents Asymmetric sync delivery and Transmission)
        #region IsUseBcat

        private bool _IsUseBcat;

        [XmlIgnore]
        public bool IsUseBcat
        {
            get { return _IsUseBcat; }
            set { SetProperty(ref _IsUseBcat, value); }
        }

        #endregion

        #region BcatDeliveryCacheStorageSize

        private long _BcatDeliveryCacheStorageSize;

        [XmlIgnore]
        public long BcatDeliveryCacheStorageSize
        {
            get { return _BcatDeliveryCacheStorageSize; }
            set { SetProperty(ref _BcatDeliveryCacheStorageSize, value); }
        }

        [XmlElement("BcatDeliveryCacheStorageSize")]
        [YamlIgnore]
        public string BcatDeliveryCacheStorageSizeHex
        {
            get { return IsUseBcat ? BcatDeliveryCacheStorageSize.ToHex() : null; }
            set
            {
                long storageSize;
                if (value.TryToLong(out storageSize))
                    BcatDeliveryCacheStorageSize = storageSize;
            }
        }

        #endregion

        #region BcatPassphrase

        private string _BcatPassphrase;

        [XmlIgnore]
        public string BcatPassphrase
        {
            get { return _BcatPassphrase; }
            set { SetProperty(ref _BcatPassphrase, value); }
        }

        [XmlElement("BcatPassphrase")]
        [YamlIgnore]
        public string BcatPassphraseForXml
        {
            get { return IsUseBcat ? BcatPassphrase : null; }
            set { BcatPassphrase = value; }
        }

        #endregion

        #region VideoCapture

        private VideoCaptureType? _VideoCapture;

        [XmlIgnore]
        public VideoCaptureType? VideoCapture
        {
            get { return _VideoCapture; }
            set { SetProperty(ref _VideoCapture, value); }
        }

        [YamlIgnore]
        [XmlElement(nameof(VideoCapture))]
        public string VideoCaptureForXml
        {
            get { return VideoCapture.HasValue ? VideoCapture.ToString() : null; }
            set
            {
                VideoCaptureType t;
                if (Enum.TryParse(value, out t))
                    VideoCapture = t;
            }
        }

        [YamlIgnore]
        [XmlIgnore]
        public bool VideoCaptureSpecified => VideoCapture.HasValue;

        #endregion

        #region RuntimeAddOnContentInstall

        private RuntimeAddOnContentInstallType? _RuntimeAddOnContentInstall;

        [XmlIgnore]
        public RuntimeAddOnContentInstallType? RuntimeAddOnContentInstall
        {
            get { return _RuntimeAddOnContentInstall; }
            set { SetProperty(ref _RuntimeAddOnContentInstall, value); }
        }

        [YamlIgnore]
        [XmlElement(nameof(RuntimeAddOnContentInstall))]
        public string RuntimeAddOnContentInstallForXml
        {
            get { return RuntimeAddOnContentInstall.HasValue ? RuntimeAddOnContentInstall.ToString() : null; }
            set
            {
                RuntimeAddOnContentInstallType t;
                if (Enum.TryParse(value, out t))
                    RuntimeAddOnContentInstall = t;
            }
        }

        #endregion

        #region CrashReport

        private CrashReportType? _CrashReport;

        [XmlIgnore]
        public CrashReportType? CrashReport
        {
            get { return _CrashReport; }
            set { SetProperty(ref _CrashReport, value); }
        }

        [YamlIgnore]
        [XmlElement(nameof(CrashReport))]
        public string CrashReportForXml
        {
            get { return CrashReport.HasValue ? CrashReport.ToString() : null; }
            set
            {
                CrashReportType c;
                if (Enum.TryParse(value, out c))
                    CrashReport = c;
            }
        }

        #endregion

        #region PlayLogQueryCapability

        private PlayLogQueryCapabilityType? _PlayLogQueryCapability;

        [XmlIgnore]
        public PlayLogQueryCapabilityType? PlayLogQueryCapability
        {
            get { return _PlayLogQueryCapability; }
            set { SetProperty(ref _PlayLogQueryCapability, value); }
        }

        [YamlIgnore]
        [XmlElement(nameof(PlayLogQueryCapability))]
        public string PlayLogQueryCapabilityForXml
        {
            get { return PlayLogQueryCapability.HasValue ? PlayLogQueryCapability.ToString() : null; }
            set
            {
                PlayLogQueryCapabilityType p;
                if (Enum.TryParse(value, out p))
                    PlayLogQueryCapability = p;
            }
        }

        #endregion

        #region PlayLogQueryableApplicationId

        public const int MaxPlayLogQueryableApplicationIdCount = 16;

        private ObservableCollection<NintendoApplicationId> _PlayLogQueryableApplicationIds =
            new ObservableCollection<NintendoApplicationId>();

        [XmlIgnore]
        public ObservableCollection<NintendoApplicationId> PlayLogQueryableApplicationIds
        {
            get { return _PlayLogQueryableApplicationIds; }
            set { SetProperty(ref _PlayLogQueryableApplicationIds, value); }
        }

        [YamlIgnore]
        [XmlElement("PlayLogQueryableApplicationId")]
        public string[] PlayLogQueryableApplicationIdForXml
        {
            get { return PlayLogQueryableApplicationIds.Select(i => i.Id.ToHex()).ToArray(); }
            set
            {
                PlayLogQueryableApplicationIds.Clear();
                if (value == null)
                    return;
                foreach (var id in value)
                {
                    ulong applicationId;
                    if (id.TryToUlong(out applicationId))
                        PlayLogQueryableApplicationIds.Add(new NintendoApplicationId {Id = applicationId});
                }
            }
        }

        #endregion

        #region 非公開プロパティ: http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=123997863

        #region Isbn

        private string _Isbn;

        public string Isbn
        {
            get { return _Isbn; }
            set { SetProperty(ref _Isbn, value); }
        }

        #endregion

        #region SaveDataOwnerId

        private string _SaveDataOwnerId;

        public string SaveDataOwnerId
        {
            get { return _SaveDataOwnerId; }
            set { SetProperty(ref _SaveDataOwnerId, value); }
        }

        #endregion

        #region DeviceSaveDataSize

        private long _DeviceSaveDataSize;

        [XmlIgnore]
        public long DeviceSaveDataSize
        {
            get { return _DeviceSaveDataSize; }
            set
            {
                _IsInvalidDeviceDataSize = false;
                SetProperty(ref _DeviceSaveDataSize, value);
            }
        }

        private bool _IsInvalidDeviceDataSize;

        [YamlIgnore]
        [XmlElement("DeviceSaveDataSize")]
        public string DeviceSaveDataSizeHex
        {
            get
            {
                return _IsInvalidDeviceDataSize == false && IsSpecifiedDeviceSaveDataSize
                    ? DeviceSaveDataSize.ToHex()
                    : null;
            }
            set
            {
                if (value?.IsUlong() == false)
                {
                    _IsInvalidDeviceDataSize = true;
                }
                else
                {
                    DeviceSaveDataSize = value?.ToLong() ?? 0L;
                }
            }
        }

        private bool _IsSpecifiedDeviceSaveDataSize;

        [XmlIgnore]
        public bool IsSpecifiedDeviceSaveDataSize
        {
            get { return _IsSpecifiedDeviceSaveDataSize; }
            set { SetProperty(ref _IsSpecifiedDeviceSaveDataSize, value); }
        }

        #endregion

        #region DeviceSaveDataJournalSize

        private long _DeviceSaveDataJournalSize;

        [XmlIgnore]
        public long DeviceSaveDataJournalSize
        {
            get { return _DeviceSaveDataJournalSize; }
            set
            {
                _IsInvalidDeviceDataJournalSize = false;
                SetProperty(ref _DeviceSaveDataJournalSize, value);
            }
        }

        private bool _IsInvalidDeviceDataJournalSize;

        [YamlIgnore]
        [XmlElement("DeviceSaveDataJournalSize")]
        public string DeviceSaveDataJournalSizeHex
        {
            get
            {
                return _IsInvalidDeviceDataJournalSize == false && IsSpecifiedDeviceSaveDataJournalSize
                    ? DeviceSaveDataJournalSize.ToHex()
                    : null;
            }
            set
            {
                if (value?.IsUlong() == false)
                {
                    _IsInvalidDeviceDataJournalSize = true;
                }
                else
                {
                    DeviceSaveDataJournalSize = value?.ToLong() ?? 0L;
                }
            }
        }

        private bool _IsSpecifiedDeviceSaveDataJournalSize;

        [XmlIgnore]
        public bool IsSpecifiedDeviceSaveDataJournalSize
        {
            get { return _IsSpecifiedDeviceSaveDataJournalSize; }
            set { SetProperty(ref _IsSpecifiedDeviceSaveDataJournalSize, value); }
        }

        #endregion

        #region PlayLogPolicy

        private PlayLogPolicyType? _PlayLogPolicy;

        public PlayLogPolicyType? PlayLogPolicy
        {
            get { return _PlayLogPolicy; }
            set { SetProperty(ref _PlayLogPolicy, value); }
        }

        [YamlIgnore]
        [XmlIgnore]
        public bool PlayLogPolicySpecified => PlayLogPolicy.HasValue;

        #endregion

        #region HDCP

        public enum HdcpType
        {
            None,
            Required
        }

        private HdcpType? _Hdcp;

        [XmlIgnore]
        public HdcpType? Hdcp
        {
            get { return _Hdcp; }
            set { SetProperty(ref _Hdcp, value); }
        }

        [YamlIgnore]
        [XmlElement("Hdcp")]
        public string HdcpForXml
        {
            get { return Hdcp.HasValue ? Hdcp.ToString() : null; }
            set
            {
                HdcpType t;
                if (Enum.TryParse(value, out t))
                {
                    Hdcp = t;
                }
            }
        }

        #endregion

        #region DeviceSaveDataSizeMax

        private long _DeviceSaveDataSizeMax;

        [XmlIgnore]
        public long DeviceSaveDataSizeMax
        {
            get { return _DeviceSaveDataSizeMax; }
            set { SetProperty(ref _DeviceSaveDataSizeMax, value); }
        }

        [YamlIgnore]
        [XmlElement("DeviceSaveDataSizeMax")]
        public string DeviceSaveDataSizeMaxHex
        {
            get { return IsUseDeviceSaveDataSizeMax ? DeviceSaveDataSizeMax.ToHex() : null; }
            set { DeviceSaveDataSizeMax = value.ToLong(); }
        }

        private bool _IsUseDeviceSaveDataSizeMax;

        [XmlIgnore]
        public bool IsUseDeviceSaveDataSizeMax
        {
            get { return _IsUseDeviceSaveDataSizeMax; }
            set { SetProperty(ref _IsUseDeviceSaveDataSizeMax, value); }
        }

        #endregion

        #region DeviceSaveDataJournalSizeMax

        private long _DeviceSaveDataJournalSizeMax;

        [XmlIgnore]
        public long DeviceSaveDataJournalSizeMax
        {
            get { return _DeviceSaveDataJournalSizeMax; }
            set { SetProperty(ref _DeviceSaveDataJournalSizeMax, value); }
        }

        [YamlIgnore]
        [XmlElement("DeviceSaveDataJournalSizeMax")]
        public string DeviceSaveDataJournalSizeMaxHex
        {
            get { return IsUseDeviceSaveDataJournalSizeMax ? DeviceSaveDataJournalSizeMax.ToHex() : null; }
            set { DeviceSaveDataJournalSizeMax = value.ToLong(); }
        }

        private bool _IsUseDeviceSaveDataJournalSizeMax;

        [XmlIgnore]
        public bool IsUseDeviceSaveDataJournalSizeMax
        {
            get { return _IsUseDeviceSaveDataJournalSizeMax; }
            set { SetProperty(ref _IsUseDeviceSaveDataJournalSizeMax, value); }
        }

        #endregion

        #region RequiredNetworkServiceLicenseOnLaunch

        private RequiredNetworkServiceLicenseOnLaunchType? _RequiredNetworkServiceLicenseOnLaunchType;

        [XmlIgnore]
        public RequiredNetworkServiceLicenseOnLaunchType? RequiredNetworkServiceLicenseOnLaunch
        {
            get { return _RequiredNetworkServiceLicenseOnLaunchType; }
            set { SetProperty(ref _RequiredNetworkServiceLicenseOnLaunchType, value); }
        }

        [YamlIgnore]
        [XmlElement(nameof(RequiredNetworkServiceLicenseOnLaunch))]
        public string RequiredNetworkServiceLicenseOnLaunchForXml
        {
            get
            {
                return RequiredNetworkServiceLicenseOnLaunch.HasValue
                    ? RequiredNetworkServiceLicenseOnLaunch.ToString()
                    : null;
            }
            set
            {
                RequiredNetworkServiceLicenseOnLaunchType t;
                if (Enum.TryParse(value, out t))
                {
                    RequiredNetworkServiceLicenseOnLaunch = t;
                }
            }
        }

        #endregion

        #endregion

        #region 非推奨プロパティ (設定されていると削除される)

        [XmlElement]
        public XmlElement[] UserAccountSaveDataOperation
        {
            get { return null; }
            // ReSharper disable once ValueParameterNotUsed
            set { }
        }

        #endregion

        #region UserAccountSaveDataSizeMax

        private long _UserAccountSaveDataSizeMax;

        [XmlIgnore]
        public long UserAccountSaveDataSizeMax
        {
            get { return _UserAccountSaveDataSizeMax; }
            set { SetProperty(ref _UserAccountSaveDataSizeMax, value); }
        }

        [YamlIgnore]
        [XmlElement("UserAccountSaveDataSizeMax")]
        public string UserAccountSaveDataSizeMaxHex
        {
            get { return IsUseUserAccountSaveDataSizeMax ? UserAccountSaveDataSizeMax.ToHex() : null; }
            set { UserAccountSaveDataSizeMax = value.ToLong(); }
        }

        private bool _IsUseUserAccountSaveDataSizeMax;

        [XmlIgnore]
        public bool IsUseUserAccountSaveDataSizeMax
        {
            get { return _IsUseUserAccountSaveDataSizeMax; }
            set { SetProperty(ref _IsUseUserAccountSaveDataSizeMax, value); }
        }

        #endregion

        #region UserAccountSaveDataJournalSizeMax

        private long _UserAccountSaveDataJournalSizeMax;

        [XmlIgnore]
        public long UserAccountSaveDataJournalSizeMax
        {
            get { return _UserAccountSaveDataJournalSizeMax; }
            set { SetProperty(ref _UserAccountSaveDataJournalSizeMax, value); }
        }

        [YamlIgnore]
        [XmlElement("UserAccountSaveDataJournalSizeMax")]
        public string UserAccountSaveDataJournalSizeMaxHex
        {
            get { return IsUseUserAccountSaveDataJournalSizeMax ? UserAccountSaveDataJournalSizeMax.ToHex() : null; }
            set { UserAccountSaveDataJournalSizeMax = value.ToLong(); }
        }

        private bool _IsUseUserAccountSaveDataJournalSizeMax;

        [XmlIgnore]
        public bool IsUseUserAccountSaveDataJournalSizeMax
        {
            get { return _IsUseUserAccountSaveDataJournalSizeMax; }
            set { SetProperty(ref _IsUseUserAccountSaveDataJournalSizeMax, value); }
        }

        #endregion

        #region TemporaryStorageSize

        private long _TemporaryStorageSize;

        [XmlIgnore]
        public long TemporaryStorageSize
        {
            get { return _TemporaryStorageSize; }
            set { SetProperty(ref _TemporaryStorageSize, value); }
        }

        [YamlIgnore]
        [XmlElement("TemporaryStorageSize")]
        public string TemporaryStorageSizeHex
        {
            get { return IsUseTemporaryStorageSize ? TemporaryStorageSize.ToHex() : null; }
            set { TemporaryStorageSize = value.ToLong(); }
        }

        private bool _IsUseTemporaryStorageSize;

        [XmlIgnore]
        public bool IsUseTemporaryStorageSize
        {
            get { return _IsUseTemporaryStorageSize; }
            set { SetProperty(ref _IsUseTemporaryStorageSize, value); }
        }

        #endregion

        #region CacheStorageSize

        private long _CacheStorageSize;

        [XmlIgnore]
        public long CacheStorageSize
        {
            get { return _CacheStorageSize; }
            set { SetProperty(ref _CacheStorageSize, value); }
        }

        [YamlIgnore]
        [XmlElement("CacheStorageSize")]
        public string CacheStorageSizeHex
        {
            get { return IsUseCacheStorageSize ? CacheStorageSize.ToHex() : null; }
            set { CacheStorageSize = value.ToLong(); }
        }

        private bool _IsUseCacheStorageSize;

        [XmlIgnore]
        public bool IsUseCacheStorageSize
        {
            get { return _IsUseCacheStorageSize; }
            set { SetProperty(ref _IsUseCacheStorageSize, value); }
        }

        #endregion

        #region CacheStorageJournalSize

        private long _CacheStorageJournalSize;

        [XmlIgnore]
        public long CacheStorageJournalSize
        {
            get { return _CacheStorageJournalSize; }
            set { SetProperty(ref _CacheStorageJournalSize, value); }
        }

        [YamlIgnore]
        [XmlElement("CacheStorageJournalSize")]
        public string CacheStorageJournalSizeHex
        {
            get { return IsUseCacheStorageJournalSize ? CacheStorageJournalSize.ToHex() : null; }
            set { CacheStorageJournalSize = value.ToLong(); }
        }

        private bool _IsUseCacheStorageJournalSize;

        [XmlIgnore]
        public bool IsUseCacheStorageJournalSize
        {
            get { return _IsUseCacheStorageJournalSize; }
            set { SetProperty(ref _IsUseCacheStorageJournalSize, value); }
        }

        #endregion

        #region FilterDescriptionFilePath

        private string _FilterDescriptionFilePath;

        [XmlIgnore]
        public string FilterDescriptionFilePath
        {
            get { return _FilterDescriptionFilePath; }
            set { SetProperty(ref _FilterDescriptionFilePath, value); }
        }

        [YamlIgnore]
        [XmlElement("FilterDescriptionFilePath")]
        public string FilterDescriptionFilePathForXml
        {
            get { return IsUseFilterDescriptionFilePath ? FilterDescriptionFilePath : null; }
            set { FilterDescriptionFilePath = value; }
        }

        private bool _IsUseFilterDescriptionFilePath;

        [XmlIgnore]
        public bool IsUseFilterDescriptionFilePath
        {
            get { return _IsUseFilterDescriptionFilePath; }
            set { SetProperty(ref _IsUseFilterDescriptionFilePath, value); }
        }

        #endregion

        // 未知の要素は保持しておく
        [YamlIgnore]
        [XmlAnyElement]
        public XmlElement[] UnknownXmlElements { get; set; }

        // 非推奨プロパティとして無視する
        [YamlIgnore]
        [XmlElement]
        public string TouchScreenUsage
        {
            get { return null; }
            // ReSharper disable once ValueParameterNotUsed
            set {}
        }

        [YamlIgnore]
        [XmlIgnore]
        public Container DiContainer
        {
            get { return _DiContainer; }
            set
            {
                _DiContainer = value;

                Titles.ObserveAddChanged()
                    .Subscribe(t => t.DiContainer = value)
                    .AddTo(CompositeDisposable);

                foreach (var t in Titles)
                    t.DiContainer = value;

                if (_DiContainer != null)
                    UpdateTitlesObservers();

                // ReSharper disable once ExplicitCallerInfoArgument
                RaisePropertyChanged(nameof(DiContainer));
            }
        }

        private Container _DiContainer;

        public Application()
        {
            InitializeValidation();
        }

        public override IEnumerable<string> GetChangedPropertyNames()
        {
            if (!SupportedLanguages.Where(x => x.IsSupported).IsEmpty())
                yield return nameof(SupportedLanguage); // nmeta 上の項目名には "s" がつかない
            if (!Titles.IsEmpty())
                yield return nameof(Title); // nmeta 上の項目名には "s" がつかない
            if (!Ratings.IsEmpty())
                yield return nameof(Rating); // nmeta 上の項目名には "s" がつかない
            if (!LocalCommunicationIds.IsEmpty())
                yield return nameof(LocalCommunicationId); // nmeta 上の項目名には "s" がつかない

            foreach (var prop in new[]
            {
                nameof(ReleaseVersion),
                nameof(DisplayVersion),
                nameof(StartupUserAccount),
                nameof(ParentalControl),
                nameof(PresenceGroupId),
                nameof(Screenshot),
                nameof(DataLossConfirmation),
                // TODO: セーブデータ系項目の項目名が nmeta と異なるので SIGLO-66469 で直す
                nameof(SaveDataSize),
                nameof(SaveDataJournalSize),
                nameof(ApplicationErrorCodeCategory),
                nameof(LogoType),
                nameof(LogoHandling),
                nameof(SeedForPseudoDeviceId),
                nameof(VideoCapture),
                nameof(RuntimeAddOnContentInstall),
                nameof(HtmlDocumentPath),
                nameof(AccessibleUrlsFilePath),
                nameof(LegalInformationFilePath),
                nameof(BcatDeliveryCacheStorageSize),
                nameof(BcatPassphrase),
                nameof(Attribute),
                // ここから非公開プロパティ
                nameof(Isbn),
                nameof(SaveDataOwnerId),
                nameof(DeviceSaveDataSize),
                nameof(DeviceSaveDataJournalSize),
                nameof(PlayLogPolicy),
                nameof(Hdcp)
            })
            {
                if (IsPropertyChanged(prop))
                    yield return prop;
            }
        }

        public override void MergePropertyValues(Application value)
        {
            MergeGeneralProperties(value);
            MergeSaveDataProperties(value);
            MergeFeatureProperties(value);
            MergeExternalPathProperties(value);
            MergeBcatProperties(value);
            MergeAttributeProperties(value);
            MergePrivateProperties(value);
        }

        private void MergeGeneralProperties(Application value)
        {
            SupportedLanguages.Merge(value.SupportedLanguages,
                (x, y) => y.IsSupported && x.Language == y.Language,
                (src, newValue) => src.IsSupported = newValue.IsSupported);

            if (!value.Titles.IsEmpty())
            {
                Titles.Replace(value.Titles,
                    (x, y) => x.Language == y.Language,
                    (src, newValue) =>
                    {
                        src.Name = newValue.Name;
                        src.Publisher = newValue.Publisher;

                        src.IconFilePath = newValue.IconFilePath;
                        if (!string.IsNullOrEmpty(src.IconFilePath))
                            src.IsReplaceIcon = true;

                        src.NxIconFilePath = newValue.NxIconFilePath;
                        if (!string.IsNullOrEmpty(src.NxIconFilePath))
                            src.IsReplaceNxIcon = true;
                    });
            }

            if (!value.Ratings.IsEmpty())
            {
                // レーティング情報は FixupRating によって全レーティングの情報が反映された状態
                foreach (var oldValue in Ratings)
                {
                    // インポートによって既存の値をすべてリセットし、インポートした nmeta に存在するレーティング情報のみ有効化
                    oldValue.IsUse = false;
                    oldValue.Age = null;

                    var newValue = value.Ratings
                        .FirstOrDefault(x => x.Organization == oldValue.Organization && x.IsUse);
                    if (newValue != null)
                    {
                        oldValue.IsUse = true;
                        oldValue.Age = newValue.Age;
                    }

                    oldValue.RaisePropertyChanged(nameof(oldValue.Age));
                }
            }

            value.TrySetChangedProperty(x => x.ReleaseVersion, this);
            value.TrySetChangedProperty(x => x.DisplayVersion, this);

            value.TrySetChangedProperty(x => x.StartupUserAccount, this);
            value.TrySetChangedProperty(x => x.ParentalControl, this);

            value.TrySetChangedProperty(x => x.PresenceGroupId, this);
            value.TrySetChangedProperty(x => x.IsUsePresenceGroupIdAppId, this);

            value.TrySetChangedProperty(x => x.Screenshot, this);
            value.TrySetChangedProperty(x => x.IsAllowScreenshot, this);

            value.TrySetChangedProperty(x => x.DataLossConfirmation, this);
        }

        private void MergeSaveDataProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.IsUseSaveData, this);
            value.TrySetChangedProperty(x => x.SaveDataSize, this);
            value.TrySetChangedProperty(x => x.IsSpecifiedSaveDataJournal, this);
            value.TrySetChangedProperty(x => x.SaveDataJournalSize, this);
        }

        private void MergeFeatureProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.ApplicationErrorCodeCategory, this);
            value.TrySetChangedProperty(x => x.IsUseApplicationErrorCode, this);
            value.TrySetChangedProperty(x => x.LogoType, this);
            value.TrySetChangedProperty(x => x.LogoHandling, this);

            if (!value.LocalCommunicationIds.IsEmpty())
            {
                LocalCommunicationIds.Replace(value.LocalCommunicationIds,
                    (x, y) => x.Id == y.Id,
                    (src, newValue) => src.Id = newValue.Id);
            }

            value.TrySetChangedProperty(x => x.SeedForPseudoDeviceId, this);
            value.TrySetChangedProperty(x => x.VideoCapture, this);
            value.TrySetChangedProperty(x => x.RuntimeAddOnContentInstall, this);

            value.TrySetChangedProperty(x => x.PlayLogQueryCapability, this);
            if (!value.PlayLogQueryableApplicationIds.IsEmpty())
            {
                PlayLogQueryableApplicationIds.Replace(value.PlayLogQueryableApplicationIds,
                    (x, y) => x.Id == y.Id,
                    (src, newValue) => src.Id = newValue.Id);
            }
        }

        private void MergeExternalPathProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.IsReplaceHtmlDocumentPath, this);
            value.TrySetChangedProperty(x => x.HtmlDocumentPath, this);

            value.TrySetChangedProperty(x => x.IsReplaceAccessibleUrlsFilePath, this);
            value.TrySetChangedProperty(x => x.AccessibleUrlsFilePath, this);

            value.TrySetChangedProperty(x => x.IsReplaceLegalInformationFilePath, this);
            value.TrySetChangedProperty(x => x.LegalInformationFilePath, this);
        }

        private void MergeBcatProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.IsUseBcat, this);
            value.TrySetChangedProperty(x => x.BcatDeliveryCacheStorageSize, this);
            value.TrySetChangedProperty(x => x.BcatPassphrase, this);
        }

        private void MergeAttributeProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.IsDemo, this);
            value.TrySetChangedProperty(x => x.IsRetailInteractiveDisplay, this);
            value.TrySetChangedProperty(x => x.IsDownloadPlay, this);
        }

        private void MergePrivateProperties(Application value)
        {
            value.TrySetChangedProperty(x => x.Isbn, this);
            value.TrySetChangedProperty(x => x.SaveDataOwnerId, this);
            value.TrySetChangedProperty(x => x.DeviceSaveDataSize, this);
            value.TrySetChangedProperty(x => x.DeviceSaveDataJournalSize, this);
            value.TrySetChangedProperty(x => x.PlayLogPolicy, this);
            value.TrySetChangedProperty(x => x.Hdcp, this);
        }
    }

    public partial class SaveDataOperation : DisposableModelBase
    {
        #region Copy

        private CopyType _Copy;

        [XmlIgnore]
        public CopyType Copy
        {
            get { return _Copy; }
            set { SetProperty(ref _Copy, value); }
        }

        [YamlIgnore]
        [XmlElement("Copy")]
        public string CopyForXml
        {
            get { return Copy == CopyType.Unspecified ? null : Copy.ToString(); }
            set
            {
                CopyType c;
                if (CopyType.TryParse(value, false, out c) && c != CopyType.Unspecified)
                    Copy = c;
            }
        }

        #endregion

        #region Rollback

        private RollbackType _Rollback;

        [XmlIgnore]
        public RollbackType Rollback
        {
            get { return _Rollback; }
            set { SetProperty(ref _Rollback, value); }
        }

        [YamlIgnore]
        [XmlElement("Rollback")]
        public string RollbackForXml
        {
            get
            {
                switch (Rollback)
                {
                    case RollbackType.Unspecified:
                        return null;
                    case RollbackType.Allow:
                    case RollbackType.RestrictForOnceADay:
                        return "Allow";
                    case RollbackType.Deny:
                        return "Deny";
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
            set
            {
                RollbackType t;
                if (RollbackType.TryParse(value, false, out t)
                    && t != RollbackType.Unspecified
                    && t != RollbackType.RestrictForOnceADay)
                {
                    Rollback = t;
                }
            }
        }

        #endregion

        #region RollbackTerm

        private int _RollbackTerm;

        [XmlIgnore]
        public int RollbackTerm
        {
            get { return _RollbackTerm; }
            set { SetProperty(ref _RollbackTerm, value); }
        }

        [YamlIgnore]
        [XmlElement("RollbackTerm")]
        public string RollbackTermForXml
        {
            get
            {
                switch (Rollback)
                {
                    case RollbackType.Unspecified:
                    case RollbackType.Deny:
                        return null;
                    case RollbackType.Allow:
                    case RollbackType.RestrictForOnceADay:
                        return RollbackTerm.ToString();
                    default:
                        throw new ArgumentOutOfRangeException();

                }
            }

            set
            {
                int hour;
                if (int.TryParse(value, out hour))
                    RollbackTerm = hour;
            }
        }

        #endregion

        public SaveDataOperation()
        {
            InitializeValidation();
        }
    }

    public enum CopyType
    {
        Unspecified,
        Allow,
        Deny
    }

    public enum RollbackType
    {
        Unspecified,
        Deny,
        Allow,
        RestrictForOnceADay
    }

    public partial class CardSpec : PropertyTrackingDispoableModel<CardSpec>
    {
        #region IsAutomaticSettingSize

        private bool _IsAutomaticSettingSize = true;

        [XmlIgnore]
        public bool IsAutomaticSettingSize
        {
            get { return _IsAutomaticSettingSize; }
            set { SetProperty(ref _IsAutomaticSettingSize, value); }
        }

        #endregion

        #region Size

        private int _Size = 2;

        public int Size
        {
            get { return _Size; }
            set { SetProperty(ref _Size, value); }
        }

        [YamlIgnore]
        [XmlIgnore]
        public bool SizeSpecified => IsAutomaticSettingSize == false;

        #endregion

        #region IsAutomaticSettingClockRate

        private bool _IsAutomaticSettingClockRate = true;

        [XmlIgnore]
        public bool IsAutomaticSettingClockRate
        {
            get { return _IsAutomaticSettingClockRate; }
            set { SetProperty(ref _IsAutomaticSettingClockRate, value); }
        }

        #endregion

        #region ClockRate

        private int _ClockRate = 25;

        public int ClockRate
        {
            get { return _ClockRate; }
            set { SetProperty(ref _ClockRate, value); }
        }

        [YamlIgnore]
        [XmlIgnore]
        public bool ClockRateSpecified => IsAutomaticSettingClockRate == false;

        #endregion

        public CardSpec()
        {
            InitializeValidation();
        }

        public override IEnumerable<string> GetChangedPropertyNames()
        {
            foreach (var prop in new[]
            {
                nameof(Size),
                nameof(ClockRate)
            })
            {
                if (IsPropertyChanged(prop))
                    yield return prop;
            }
        }

        public override void MergePropertyValues(CardSpec value)
        {
            if (value.TrySetChangedProperty(x => x.Size, this))
                IsAutomaticSettingSize = false;
            if (value.TrySetChangedProperty(x => x.ClockRate, this))
                IsAutomaticSettingClockRate = false;
        }
    }

    public partial class Title : ModelBase
    {
        [YamlIgnore]
        [XmlIgnore]
        public Container DiContainer { get; set; }

        #region Language

        private LanguageType _Language = LanguageType.Japanese;

        [XmlIgnore]
        public LanguageType Language
        {
            get { return _Language; }
            set { SetProperty(ref _Language, value); }
        }

        [YamlIgnore]
        [XmlElement("Language")]
        public string LanguageForXml
        {
            get { return Language.ToString(); }
            set
            {
                LanguageType lang;
                if (Enum.TryParse(value, out lang))
                    Language = lang;
            }
        }

        #endregion

        #region Name

        private string _Name = string.Empty;

        public string Name
        {
            get { return _Name; }
            set { SetProperty(ref _Name, value); }
        }

        #endregion

        #region Publisher

        private string _Publisher = string.Empty;

        public string Publisher
        {
            get { return _Publisher; }
            set { SetProperty(ref _Publisher, value); }
        }

        #endregion

        #region IsReplaceIcon

        private bool _IsReplaceIcon = true;

        [XmlIgnore]
        public bool IsReplaceIcon
        {
            get { return _IsReplaceIcon; }
            set { SetProperty(ref _IsReplaceIcon, value); }
        }

        #endregion

        #region IconFilePath

        private ExpandablePath _IconFilePath = string.Empty;

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath IconFilePath
        {
            get { return _IconFilePath; }
            set
            {
                if (_IconFilePath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(IconFilePath));
            }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(IconFilePath))]
        public string IconFilePathForYaml
        {
            get { return IconFilePath.Path; }
            set { IconFilePath.Path = value; }
        }

        [XmlIgnore]
        public bool IconFilePathIsExpandEnvironmentVariableForYaml
        {
            get { return IconFilePath.IsExpandEnvironmentVariable; }
            set { IconFilePath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region IsReplaceNxIcon

        private bool _IsReplaceNxIcon;

        [XmlIgnore]
        public bool IsReplaceNxIcon
        {
            get { return _IsReplaceNxIcon; }
            set { SetProperty(ref _IsReplaceNxIcon, value); }
        }

        #endregion

        #region NxIconFilePath

        private ExpandablePath _nxIconFilePath = string.Empty;

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath NxIconFilePath
        {
            get { return _nxIconFilePath; }
            set { SetProperty(ref _nxIconFilePath, new ExpandablePath(value)); }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(NxIconFilePath))]
        public string NxIconFilePathForYaml
        {
            get { return NxIconFilePath.Path; }
            set { NxIconFilePath.Path = value; }
        }

        [XmlIgnore]
        public bool NxIconFilePathIsExpandEnvironmentVariableForYaml
        {
            get { return NxIconFilePath.IsExpandEnvironmentVariable; }
            set { NxIconFilePath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region OriginalNxIconFilePath

        private ExpandablePath _OriginalNxIconFilePath = string.Empty;

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath OriginalNxIconFilePath
        {
            get { return _OriginalNxIconFilePath; }
            set { SetProperty(ref _OriginalNxIconFilePath, new ExpandablePath(value)); }
        }

        #endregion

        #region OriginalIconFilePath

        private ExpandablePath _OriginalIconFilePath = string.Empty;

        [YamlIgnore]
        [XmlIgnore]
        public ExpandablePath OriginalIconFilePath
        {
            get { return _OriginalIconFilePath; }
            set { SetProperty(ref _OriginalIconFilePath, new ExpandablePath(value)); }
        }

        #endregion

        // nsp から展開したアイコンファイルのパス (.jpg)
        #region NspRawIconFilePath

        private string _NspRawIconFilePath;

        [YamlIgnore]
        [XmlIgnore]
        public string NspRawIconFilePath
        {
            get { return _NspRawIconFilePath; }
            set { SetProperty(ref _NspRawIconFilePath, value); }
        }

        #endregion

        // nsp から展開した NX 用アイコンのファイルのパス (.dat)
        #region NspNxIconFilePath

        private string _NspNxIconFilePath;

        [YamlIgnore]
        [XmlIgnore]
        public string NspNxIconFilePath
        {
            get { return _NspNxIconFilePath; }
            set { SetProperty(ref _NspNxIconFilePath, value); }
        }

        #endregion

        // 強制的にタイトルのバリデーションをトリガする
        public void ForceValidation()
        {
            IsForceValidation = true;
        }

        private bool _IsForceValidation;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsForceValidation
        {
            get { return _IsForceValidation; }
            set
            {
                _IsForceValidation = value;
                // ReSharper disable once ExplicitCallerInfoArgument
                RaisePropertyChanged(nameof(_IsForceValidation));
            }
        }
    }

    [DebuggerDisplay("IsSupported = {IsSupported}")]
    public class SupportedLanguage : ModelBase
    {
        #region Language

        private LanguageType _Language;

        public LanguageType Language
        {
            get { return _Language; }
            set { SetProperty(ref _Language, value); }
        }

        #endregion

        #region IsSupported

        private bool _IsSupported;

        public bool IsSupported
        {
            get { return _IsSupported; }
            set { SetProperty(ref _IsSupported, value); }
        }

        #endregion

        #region HasTitle

        private bool _HasTitle;

        public bool HasTitle
        {
            get { return _HasTitle; }
            set { SetProperty(ref _HasTitle, value); }
        }

        #endregion

        // 対応言語が "未使用の中韓台言語" かどうか
        public bool IsNotSupportedChineseOrKoreanLanguage
        {
            get
            {
                if (IsSupported)
                    return false;
                return Language == LanguageType.Korean ||
                       Language == LanguageType.SimplifiedChinese ||
                       Language == LanguageType.TraditionalChinese;
            }
        }
    }

    public class Rating : ModelBase
    {
        #region Organization

        private string _Organization = string.Empty;

        public string Organization
        {
            get { return _Organization; }
            set { SetProperty(ref _Organization, value); }
        }

        #endregion

        #region Age

        private int? _Age;

        public int? Age
        {
            get { return _Age; }
            set { SetProperty(ref _Age, value);  }
        }

        #endregion

        #region RatingData

        [XmlIgnore]
        [YamlIgnore]
        public RatingData RatingData
        {
            get
            {
                var org = Organization;
                if (string.IsNullOrEmpty(org))
                    return null;
                RatingData ratingData;
                if (Constants.AllRatingData.TryGetValue(org, out ratingData))
                    return ratingData;
                return null;
            }
        }

        #endregion

        #region IsUse

        // デシリアライズした段階でレーティングは有効
        // (データ上ではインポートしてきたものはすべて有効で、UI 上でのみ無効にできる)
        private bool _isUse = true;

        [XmlIgnore]
        [YamlIgnore]
        public bool IsUse
        {
            get { return _isUse; }
            set { SetProperty(ref _isUse, value); }
        }

        #endregion
    }

    public class NintendoApplicationId : ModelBase
    {
        #region Id

        private ulong _Id;

        public ulong Id
        {
            get { return _Id; }
            set { SetProperty(ref _Id, value); }
        }

        #endregion
    }

    public enum ProcessAddressSpaceType
    {
        AddressSpace32Bit,
        AddressSpace32BitNoReserved,
        AddressSpace64Bit
    }

    public enum LanguageType
    {
        [AreaAttirbute(AreaType.America)] AmericanEnglish,
        [AreaAttirbute(AreaType.America)] CanadianFrench,
        [AreaAttirbute(AreaType.America)] LatinAmericanSpanish,
        [AreaAttirbute(AreaType.Asia)] Japanese,
        [AreaAttirbute(AreaType.Asia)] SimplifiedChinese,
        [AreaAttirbute(AreaType.Asia)] TraditionalChinese,
        [AreaAttirbute(AreaType.Asia)] Korean,
        [AreaAttirbute(AreaType.Europe)] BritishEnglish,
        [AreaAttirbute(AreaType.Europe)] French,
        [AreaAttirbute(AreaType.Europe)] German,
        [AreaAttirbute(AreaType.Europe)] Spanish,
        [AreaAttirbute(AreaType.Europe)] Italian,
        [AreaAttirbute(AreaType.Europe)] Dutch,
        [AreaAttirbute(AreaType.Europe)] Portuguese,
        [AreaAttirbute(AreaType.Europe)] Russian
    }

    public static class LanguageHelper
    {
        public static AreaType LanguageToArea(LanguageType lang)
        {
            var attr = lang.GetType().GetField(lang.ToString()).GetCustomAttributes(typeof(AreaAttirbute), false);
            var area = attr[0] as AreaAttirbute;

            Debug.Assert(area != null);

            return area.Area;
        }

        public static LanguageType[] AreaToLanguages(AreaType area)
        {
            switch (area)
            {
                case AreaType.America:
                    return America;

                case AreaType.Asia:
                    return Asia;

                case AreaType.Europe:
                    return Europe;

                default:
                    throw new ArgumentOutOfRangeException(nameof(area), area, null);
            }
        }

        static LanguageHelper()
        {
            var america = new List<LanguageType>();
            var asia = new List<LanguageType>();
            var europe = new List<LanguageType>();

            foreach (LanguageType lang in Enum.GetValues(typeof(LanguageType)))
            {
                switch (LanguageToArea(lang))
                {
                    case AreaType.America:
                        america.Add(lang);
                        break;

                    case AreaType.Asia:
                        asia.Add(lang);
                        break;

                    case AreaType.Europe:
                        europe.Add(lang);
                        break;

                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            America = america.ToArray();
            Asia = asia.ToArray();
            Europe = europe.ToArray();
        }

        private static readonly LanguageType[] America;
        private static readonly LanguageType[] Asia;
        private static readonly LanguageType[] Europe;
    }

    public enum StartupUserAccountType
    {
        Required,
        RequiredWithNetworkServiceAccountAvailable,
        None
    }

    public enum ParentalControlType
    {
        None,
        FreeCommunication,
    }

    public enum ScreenshotType
    {
        Allow,
        Deny
    }

    public enum DataLossConfirmationType
    {
        None,
        Required
    }

    public enum LogoTypeType
    {
        LicensedByNintendo,
        Nintendo
    }

    public enum LogoHandlingType
    {
        Auto,
        Manual
    }

    public enum VideoCaptureType
    {
        // >= NX Addon 4.2.0 環境では無効なパラメーターとして例外扱い
        Deny,
        Allow,
        Manual,
        // >= NX Addon 4.2.0
        Disable,
        Enable
    }

    public enum PlayLogPolicyType
    {
        All,
        LogOnly,
        None
    }

    public enum PowerConsumptionType
    {
        Normal,
        High
    }

    public enum AreaType
    {
        America,
        Asia,
        Europe
    }

    public class AreaAttirbute : Attribute
    {
        public AreaType Area { get; }

        public AreaAttirbute(AreaType area)
        {
            Area = area;
        }
    }

    public enum RuntimeAddOnContentInstallType
    {
        AllowAppend,
        Deny
    }

    public enum CrashReportType
    {
        Deny,
        Allow
    }

    public enum PlayLogQueryCapabilityType
    {
        None,
        WhiteList,
        All
    }

    public enum RequiredNetworkServiceLicenseOnLaunchType
    {
        None,
        Common
    }
}
