﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NAudio.Wave;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Threading.Tasks;

namespace SystemAudioMonitor
{
    public class WaveRecorder : IDisposable, IWaveProcessor
    {
        private static readonly UInt32 WaveFileSizeMax = Int32.MaxValue;
        private WaveFormat _format;
        private WaveFilePathGenerator _generator;
        private WaveFileWriter _writer;
        private List<Task> _disposeTaskList;

        public WaveRecorder(WaveformFormat format, string dirPath)
        {
            Debug.Assert(dirPath != null);
            _format = new WaveFormat(format.SampleRate, format.BitDepth, format.ChannelCount);
            if(!Directory.Exists(dirPath))
            {
                Directory.CreateDirectory(dirPath);
            }
            _generator = new WaveFilePathGenerator(dirPath);
            _writer = new WaveFileWriter(_generator.Generate(), _format);
            _disposeTaskList = new List<Task>();
        }

        public void AddSamples(byte[] buffer, int offset, int count)
        {
            if(((UInt32)(_writer.Length) + (UInt32)(count) > WaveFileSizeMax))
            {
                var disposeWriter = _writer;
                var waveFileWriterDisposeTask = Task.Run(
                    () =>
                    {
                        disposeWriter.Dispose();
                    });
                _disposeTaskList.Add(waveFileWriterDisposeTask);

                _writer = new WaveFileWriter(_generator.Generate(), _format);
            }

            _writer?.Write(buffer, 0, count);
        }

        public void Dispose()
        {
            foreach(var disposeTask in _disposeTaskList)
            {
                disposeTask.Wait();
            }
            _writer?.Dispose();
        }

        public void OnCaptureStopped(Exception e)
        {
            Dispose();
        }
    }
}
