﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Shapes;

namespace SystemAudioMonitor
{
    public partial class LoudnessMeter : UserControl
    {
        public enum BarStyle
        {
            Simple,
            Block,
        };

        public static readonly DependencyProperty BarStyleProperty = DependencyProperty.Register(
            nameof(Style),
            typeof(BarStyle),
            typeof(LoudnessMeter),
            new PropertyMetadata(BarStyle.Simple));

        public new BarStyle Style
        {
            get { return (BarStyle)GetValue(BarStyleProperty); }
            set { SetValue(BarStyleProperty, value); }
        }

        public static readonly DependencyProperty LabelProperty = DependencyProperty.Register(
            nameof(Label),
            typeof(string),
            typeof(LoudnessMeter),
            new PropertyMetadata(string.Empty));

        public string Label
        {
            get { return (string)GetValue(LabelProperty); }
            set { SetValue(LabelProperty, value); }
        }

        public static readonly DependencyProperty ValueProperty = DependencyProperty.Register(
            nameof(Value),
            typeof(double),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                0.0,
                (sender, e) => (sender as LoudnessMeter).OnValuePropertyChanged(sender, e)));

        public double Value
        {
            get { return (double)GetValue(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }

        public static readonly DependencyProperty MaxValueProperty = DependencyProperty.Register(
            nameof(MaxValue),
            typeof(double),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                6.0,
                (sender, e) => (sender as LoudnessMeter).OnMaxValuePropertyChanged(sender, e)));

        public double MaxValue
        {
            get { return (double)GetValue(MaxValueProperty); }
            set { SetValue(MaxValueProperty, value); }
        }

        public static readonly DependencyProperty MinValueProperty = DependencyProperty.Register(
            nameof(MinValue),
            typeof(double),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                -96.0,
                (sender, e) => (sender as LoudnessMeter).OnMinValuePropertyChanged(sender, e)));

        public double MinValue
        {
            get { return (double)GetValue(MinValueProperty); }
            set { SetValue(MinValueProperty, value); }
        }

        public static readonly DependencyProperty CautionValueProperty = DependencyProperty.Register(
            nameof(CautionValue),
            typeof(double),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                -24.0,
                (sender, e) => (sender as LoudnessMeter).OnCautionValuePropertyChanged(sender, e)));

        public double CautionValue
        {
            get { return (double)GetValue(CautionValueProperty); }
            set { SetValue(CautionValueProperty, value); }
        }

        public static readonly DependencyProperty WarningValueProperty = DependencyProperty.Register(
            nameof(WarningValue),
            typeof(double),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                0.0,
                (sender, e) => (sender as LoudnessMeter).OnWarningValuePropertyChanged(sender, e)));

        public double WarningValue
        {
            get { return (double)GetValue(WarningValueProperty); }
            set { SetValue(WarningValueProperty, value); }
        }

        public static readonly DependencyProperty NormalBrushProperty = DependencyProperty.Register(
            nameof(NormalBrush),
            typeof(Brush),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                Brushes.LightGreen,
                (sender, e) => (sender as LoudnessMeter).OnNormalBrushPropertyChanged(sender, e)));

        public Brush NormalBrush
        {
            get { return (Brush)GetValue(NormalBrushProperty); }
            set { SetValue(NormalBrushProperty, value); }
        }

        public static readonly DependencyProperty CautionBrushProperty = DependencyProperty.Register(
            nameof(CautionBrush),
            typeof(Brush),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                Brushes.Yellow,
                (sender, e) => (sender as LoudnessMeter).OnCautionBrushPropertyChanged(sender, e)));

        public Brush CautionBrush
        {
            get { return (Brush)GetValue(CautionBrushProperty); }
            set { SetValue(CautionBrushProperty, value); }
        }

        public static readonly DependencyProperty WarningBrushProperty = DependencyProperty.Register(
            nameof(WarningBrush),
            typeof(Brush),
            typeof(LoudnessMeter),
            new PropertyMetadata(
                Brushes.Red,
                (sender, e) => (sender as LoudnessMeter).OnWarningBrushPropertyChanged(sender, e)));

        public Brush WarningBrush
        {
            get { return (Brush)GetValue(WarningBrushProperty); }
            set { SetValue(WarningBrushProperty, value); }
        }

        private void OnValuePropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnMaxValuePropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnMinValuePropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnCautionValuePropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnWarningValuePropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnNormalBrushPropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnCautionBrushPropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        private void OnWarningBrushPropertyChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            Update();
        }

        protected override void OnRender(DrawingContext dc)
        {
            Update();
        }

        private void Update()
        {
            var barWidth = canvas.ActualWidth;
            var barHeight = canvas.ActualHeight;

            var value = Math.Max(Math.Min(Value, MaxValue), MinValue);

            var lowerBounds = new double[] { MinValue, CautionValue, WarningValue, MaxValue };
            var colors = new Brush[] { NormalBrush, CautionBrush, WarningBrush };

            canvas.Children.Clear();
            switch (Style)
            {
                case BarStyle.Simple:
                    for (var i = 0; i < colors.Length; ++i)
                    {
                        if (value >= lowerBounds[i])
                        {
                            var rect = new Rectangle();
                            Canvas.SetTop(rect, 0);
                            Canvas.SetLeft(rect, ((double)barWidth / (MaxValue - MinValue) * Math.Floor(lowerBounds[i] - MinValue)));
                            var v = Math.Min(value, lowerBounds[i + 1] + 1);
                            rect.Width = (int)((double)barWidth / (MaxValue - MinValue) * Math.Ceiling(v - (lowerBounds[i])));
                            rect.Height = barHeight;
                            rect.Fill = colors[i];
                            canvas.Children.Add(rect);
                        }
                    }
                    break;
                case BarStyle.Block:
                    var c = (MaxValue - MinValue) / 2;
                    var cellWidth = (double)barWidth / c;
                    for (var i = 0; i < c; ++i)
                    {
                        if ((double)barWidth * (value - MinValue) / (MaxValue - MinValue) > cellWidth * i)
                        {
                            var rect = new Rectangle();
                            Canvas.SetTop(rect, 0);
                            Canvas.SetLeft(rect, barWidth * i / c);
                            rect.Width = Math.Floor(cellWidth * 0.8);
                            rect.Height = barHeight;
                            rect.Fill = NormalBrush;
                            for (var j = 0; j < colors.Length; ++j)
                            {
                                if ((double)(MaxValue - MinValue) / c * i >= (lowerBounds[j] - MinValue))
                                {
                                    rect.Fill = colors[j];
                                }
                            }
                            canvas.Children.Add(rect);
                        }
                    }
                    break;
                default:
                    break;
            }
        }

        public LoudnessMeter()
        {
            InitializeComponent();
        }
    }
}
