﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include "Wavfile.h"


u32 * str_riff = (u32 *)"RIFF";
u32 * str_wave = (u32 *)"WAVE";
u32 * str_fmt  = (u32 *)"fmt ";
u32 * str_data = (u32 *)"data";
u32 * str_smpl = (u32 *)"smpl";


/*---------------------------------------------------------------------------*
  Name:         wavCreateHeader

  Description:  create a Wave file header.

  Arguments:    wc            pointer of a WAVECHUNK structure.
                channels      num of channels.
                samples       num of samples.
                bitPerSample  bit width per a sample.
                sampleRate    sampling frequency.
                loopEnd       end samlpe number of a loop.

  Returns:      none.
 *---------------------------------------------------------------------------*/

void wavCreateHeader
(
    WAVECHUNK *wc,
    int       channels,
    int       samples,
    int       bitsPerSample,
    int       sampleRate,
    int       loopEnd
)
{
    u32 datasize, riffchunksize;
    u16 blocksize;

    *((u32*)(wc->chunkId))      = *str_riff;
    *((u32*)(wc->formType))     = *str_wave;
    *((u32*)(wc->fmt.chunkId))  = *str_fmt;
    *((u32*)(wc->data.chunkId)) = *str_data;

    blocksize     = channels * ( bitsPerSample / 8 );
    datasize      = blocksize * samples;
    riffchunksize = datasize + 36;

    if (loopEnd)
    {
        riffchunksize += sizeof(SMPLCHUNK) + sizeof(SMPLLOOP);
    }

    wc->chunkSize = riffchunksize;

    wc->fmt.chunkSize      = 16;
    wc->fmt.waveFormatType = 1;
    wc->fmt.channel        = channels;
    wc->fmt.samplesPerSec  = sampleRate;
    wc->fmt.bytesPerSec    = sampleRate * blocksize;
    wc->fmt.blockSize      = blocksize;
    wc->fmt.bitsPerSample  = bitsPerSample;

    wc->data.chunkSize = datasize;

    return;
}


/*---------------------------------------------------------------------------*
  Name:         wavWriteHeader

  Description:  output a Wave file header.

  Arguments:    wc            pointer of a WAVECHUNK structure.
                outfile       pointer of a FILE object.

  Returns:      none.
 *---------------------------------------------------------------------------*/

void wavWriteHeader
(
    WAVECHUNK *wc,
    FILE      *outfile
)
{
    fwrite(wc, 1, sizeof(WAVECHUNK), outfile);

    return;
}


/*---------------------------------------------------------------------------*
  Name:         wavCreateSmplChunk

  Description:  create a SMPL chunk.

  Arguments:    wc            pointer of a WAVECHUNK structure.
                sc            pointer of a SMPLCHUNK structure.
                sl            pointer of a SMPLLOOP structure.
                loopStart     position of loop start
                loopEnd       position of loop end

  Returns:      none.
 *---------------------------------------------------------------------------*/

void wavCreateSmplChunk
(
    WAVECHUNK *wc,
    SMPLCHUNK *sc,
    SMPLLOOP  *sl,
    u32       loopStart,
    u32       loopEnd
)
{
    *((u32*)(sc->chunkId)) = *str_smpl;

    sc->chunkSize         = sizeof(SMPLCHUNK) + sizeof(SMPLLOOP) - 8;
    sc->manufacturer      = 0; // (ignore)
    sc->product           = 0; // (ignore)
    sc->samplePeriod      = 0; // (ignore)
    sc->midiUnityNote     = 0; // (ignore)
    sc->midiPitchFraction = 0; // (ignore)
    sc->smpteFormat       = 0; // (ignore)
    sc->smpteOffset       = 0; // (ignore)
    sc->sampleLoops       = 1; // only 1 loop
    sc->samplerData       = 0; // (ignore)

    sl->id                = 0; // (ignore)
    sl->type              = 0; // forward loop
    sl->start             = loopStart;   // loop start
    sl->end               = loopEnd - 1; // loop end
    sl->fraction          = 0; // (ignore)
    sl->playCount         = 0; // infinte loop
}


/*---------------------------------------------------------------------------*
  Name:         wavWriteSmplChunk

  Description:  output a SMPL chunk.

  Arguments:    sc            pointer of a SMPLCHUNK structure.
                sl            pointer of a SMPLLOOP structure.
                outfile       pointer of a FILE object.

  Returns:      none.
 *---------------------------------------------------------------------------*/

void wavWriteSmplChunk(SMPLCHUNK *sc, SMPLLOOP *sl, FILE *outfile)
{
    fwrite(sc, 1, sizeof(SMPLCHUNK), outfile);
    fwrite(sl, 1, sizeof(SMPLLOOP),  outfile);
}


/*---------------------------------------------------------------------------*
  Name:         wavReadHeader

  Description:  get & parse a Wave file header.

  Arguments:    wc            pointer of a WAVECHUNK structure.
                li            pointer of a LOOPINFO structure.
                infile        pointer of a FILE object.

  Returns:      TRUE                     OK.
                FALSE                    invalid file.
 *---------------------------------------------------------------------------*/

int wavReadHeader
(
    WAVECHUNK   *wc,
    LOOPINFO    *li,
    FILE        *infile
)
{
    u32       d, riffchunksize, size;
    int       state;
    int       remaining;
    int       top = 0;
    SMPLCHUNK smpl = { 0 };
    SMPLLOOP  loop;

    state = 0;

    if (0 != fseek(infile, 0, SEEK_SET))
    {
        printf("Read File seek error\n");
        return FALSE;
    }

    // check chunk ID = "RIFF"
    if(!fread(&d, 1, sizeof(u32), infile))
    {
        printf("File read error\n");
        return FALSE;
    }


    if (d != *str_riff)
    {
        printf("Not an RIFF file");
        return FALSE;
    }

    *(u32*)(wc->chunkId) = d;

    // get chunk size
    if (!fread(&riffchunksize, 1, sizeof(u32), infile))
    {
        printf("File read error\n");
        return FALSE;
    }

    wc->chunkSize = riffchunksize;

    // check RIFF type = "WAVE"
    if (!fread(&d, 1, sizeof(u32), infile))
    {
        printf("File read error\n");
        return FALSE;
    }

    if (d != *str_wave)
    {
        printf("Not a WAVE file content type");
        return FALSE;
    }

    *(u32*)(wc->formType) = d;

    riffchunksize -= 4;

    // check chunks
    while (riffchunksize >= 8)
    {
        BOOL isEssential = TRUE;

        // get chunk ID
        if (!fread(&d, 1, sizeof(u32), infile))
        {
            printf("File read error\n");
            return FALSE;
        }
        // get chunk size
        if (!fread(&size, 1, sizeof(u32), infile))
        {
            printf("File read error\n");
            return FALSE;
        }

        if (size & 1)
        {
            fprintf(stderr, "ID = %8.8x, Size = %d\n", d, size);
        }

        // check chunk ID
        if (d == *str_fmt)
        {
            // fmt chunk

            *(u32*)(wc->fmt.chunkId) = d;
            wc->fmt.chunkSize = size;

            state++;

            if (!fread(&wc->fmt.waveFormatType, 1, sizeof(FMTCHUNK) - 8, infile))
            {
                printf("File read error\n");
                return FALSE;
            }

            remaining = size - (sizeof(FMTCHUNK) - 8);

            if (remaining > 0)
            {
                fseek(infile, remaining, SEEK_CUR);
            }
        }

        else if (d == *str_data)
        {
            // data chunk

            *(u32*)(wc->data.chunkId) = d;
            wc->data.chunkSize = size;

            state++;
            top = ftell(infile);

            fseek(infile, size, SEEK_CUR);
        }

        else if (d == *str_smpl)
        {
            // smpl chunk

            if (!fread(&smpl.manufacturer, 1, sizeof(SMPLCHUNK) - 8, infile))
            {
                printf("File read error\n");
                return FALSE;
            }

            remaining = size - (sizeof(SMPLCHUNK) - 8);

            if (smpl.sampleLoops)
            {
                if(!fread(&loop, 1, sizeof(SMPLLOOP), infile))
                {
                    printf("File read error\n");
                    return FALSE;
                }

                remaining -= sizeof(SMPLLOOP);
            }

            if (remaining > 0)
            {
                fseek(infile, remaining, SEEK_CUR);
            }
        }

        else
        {
            // other chunks
            fseek(infile, size, SEEK_CUR);
            isEssential = FALSE;
        }

        riffchunksize -= (2 * sizeof(u32)) + size;

        //RIFF data is always WORD aligned, so even if the chunk contains an odd number of bytes,
        //an even number of bytes must be skipped to access the next element.

        // bug 5800: except for bad format such as output sometimes by ProTools or SoundForge
        // so only skip if this was not the data chunk (the last chunk)
        // bug 5830: if it was the last chunk, make it OK if the chunk size agrees with the file size
        // otherwise throw an error
        // if (size &1) {
        if (size &1)
        {
            if (state != 2) {
                riffchunksize--;
                fseek(infile, 1, SEEK_CUR);
            }
            else
            if (riffchunksize != 0)     // size does not agree
            {
                fprintf(stderr, "illegal DATA chunk size!\n");
                if (isEssential == TRUE)
                {
                    return FALSE;
                }
                else
                {
                    riffchunksize--;
                    fseek(infile, 1, SEEK_CUR);
                }
            }
        }
    }

    if (riffchunksize < 0)
    {
        printf("DATA chunk size bigger than file!\n");
        return FALSE;
    }

    if (state != 2)
    {
        printf("No DATA chunk!\n");
        return FALSE;
    }

    if (0 == wc->data.chunkSize)
    {
        printf("Empty DATA chunk!\n");
        return FALSE;
    }

    // fill LOOPINFO
    li->start = 0;
    li->end   = 0;
    if (smpl.sampleLoops && !loop.type)
    {
        li->start = loop.start;
        li->end   = loop.end + 1; // adjust to AIFF
    }

    if (top == 0)  // data chunk not found
    {
        return FALSE;
    }

    // return to the top of samples
    fseek(infile, top, SEEK_SET);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         wavGetSamples

  Description:  return num of samples.

  Arguments:    wc            pointer of a WAVECHUNK structure.

  Returns:      num of samples.
 *---------------------------------------------------------------------------*/

u32 wavGetSamples(WAVECHUNK *wc)
{
    u32 a, bps;

    bps = wc->fmt.channel * wc->fmt.bitsPerSample / 8;
    a = wc->data.chunkSize / bps;

    return a;
}


/*---------------------------------------------------------------------------*
  Name:         wavGetChannels

  Description:  return num of channels.

  Arguments:    wc            pointer of a WAVECHUNK structure.

  Returns:      num of channels.
 *---------------------------------------------------------------------------*/

u32 wavGetChannels(WAVECHUNK *wc)
{
    return (u32)wc->fmt.channel;
}


/*---------------------------------------------------------------------------*
  Name:         wavGetBitsPerSample

  Description:  return bit width per a sample.

  Arguments:    wc            pointer of a WAVECHUNK structure.

  Returns:      bit width per a sample.
 *---------------------------------------------------------------------------*/

u32 wavGetBitsPerSample(WAVECHUNK *wc)
{
    return (u32)wc->fmt.bitsPerSample;
}


/*---------------------------------------------------------------------------*
  Name:         wavGetSampleRate

  Description:  return num of samples per 1 sec.

  Arguments:    wc            pointer of a WAVECHUNK structure.

  Returns:      num of samples per sec.
 *---------------------------------------------------------------------------*/

u32 wavGetSampleRate(WAVECHUNK *wc)
{
    return (u32)wc->fmt.samplesPerSec;
}
