﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "coredump_FormatTypes.h"

#define COREDUMP_MACRO_VALUE(a) a

#define USE_ZLIB COREDUMP_MACRO_VALUE(0)
#define USE_LZ4 ((USE_ZLIB) == 0)

#include <zlib.h>
#include <lz4.h>

#ifdef NN_NINTENDO_SDK
#include <nn/fs/fs_File.h>
#endif

//==============================================================================
namespace coredump {
//==============================================================================

//========================================================================================================

class coredump_writer
{
public:

                        coredump_writer             ( coredump_file_header*     pInfo,
                                                      char*                     pFileName );

                        ~coredump_writer            ( );

                        coredump_writer();
        bool            Init                        ( coredump_file_header*     pInfo,
                                                      char*                     pFileName );

        //==============================================================================
        // Data-adding API

        result          AddModule                   ( char*                     pModuleName,
                                                      char*                     pModuleId,
                                                      u64                       LoadAddress,
                                                      u64                       Size );

        result          AddThread                   ( coredump_thread_info*     pThreadInfo,
                                                      u64                       TPIDR );

        result          AddRegisterDefinitions      ( void*                     pData,
                                                      u64                       SizeOfData  );

        result          AddTTY                      ( char*                     pData,
                                                      u64                       SizeOfData  );

        result          AddStackFrames              ( u64                       ThreadId,
                                                      u64                       NumberOfFrames,
                                                      u64*                      pFrames );

        //==============================================================================
        // Memory-adding API.

typedef void*           (*AllocateFunction)         (void*, unsigned int, unsigned int );
typedef void            (*DeallocateFunction)       (void*, void*);

        result          CreateMemorySection         ( u64                       Address,
                                                      MemoryType                Type,
                                                      u64                       Size,
                                                      AllocateFunction          Alloc = NULL,
                                                      DeallocateFunction        Dealloc = NULL );

        result          AddMemoryData               ( void*                     pData,
                                                      u64                       Size,
                                                      void*                     pCompressedDataBuffer,
                                                      u64                       CompressionBufferSize );

        result          CloseMemorySection          ( u64                       Address );


        u64             CompressionBufferSizeRequired( u64                      Size );

        result          CreateUncompressedMemorySection
                                                    ( u64                       Address,
                                                      MemoryType                Type,
                                                      u64                       Size );

        result          AddMemoryData               ( void*                     pData,
                                                      u64                       Size );

        result          CloseUncompressedMemorySection
                                                    ( u64                       Address );

        //==============================================================================
        // Screenshot adding API.
        result          CreateScreenshot            ( ScreenshotType            FileType,
                                                      u64                       Size );

        result          AddScreenshotData           ( void*                     pData,
                                                      u64                       Size );

        result          CloseScreenshot             ( );


        //==============================================================================
        // Video adding API.
        result          CreateVideo                 ( VideoType                 FileType,
                                                      u64                       Size );

        result          AddVideoData                ( void*                     pData,
                                                      u64                       Size );

        result          CloseVideo                  ( );

        //============================================================================================
        // Program ID adding

        result          AddApplicationId            ( u64                       Id );

        //============================================================================================

        void            Close                       ( );

        //============================================================================================
        // This is meant to be used as a post-process step, done on the PC.
static  result          AddBMPImage                 ( char*                     pImageFileName,
                                                      char*                     pCoreDumpFileName );

static  result          AddMP4Video                 ( char*                     pVideoFileName,
                                                      char*                     pCoreDumpFileName );

        //============================================================================================

protected:
        void            Init                        ( );

        result          AddData                     ( const char*               pTag,
                                                      void*                     pData,
                                                      u64                       SizeOfData );

static  result          AddFileData                 ( char*                     pFileName,
                                                      const char*               pTag,
                                                      char*                     pCoreDumpFileName );

private:
        void            Open                        ( char*                     pFileName );

        result          Write                       ( void*                     pData,
                                                      u64                       SizeOfData );

        result          Write                       ( s64                       Offset,
                                                      void*                     pData,
                                                      u64                       SizeOfData );

        result          CompressMemoryData          ( void*                     pSourceData,
                                                      u64                       SizeOfSourceData,
                                                      void*                     pCompressionBuffer,
                                                      u64                       SizeOfCompressionBuffer,
                                                      u64*                      pSizeOfCompressedData );

        result          Flush                       ();

#ifdef WIN32
    FILE*               m_pOutput;
#else
    nn::fs::FileHandle  m_Output;
    nn::fs::FileHandle* m_pOutput;
#endif
    u64                 m_OpenMemorySection;
    s64                 m_FileOffset;

#ifndef WIN32
    enum
    {
        SIZE_OF_FILE_BUFFER = ((64 * 1024) * 2)
    };

    char                m_FileBuffer[SIZE_OF_FILE_BUFFER];
    s64                 m_FileBufferOffset;
    s64                 m_FileWriteOffset;
#endif

#if USE_ZLIB
    z_stream            m_MemoryCompressor;
#else
    LZ4_stream_t        m_CompressionStream;
#endif
    s64                 m_OpenMemorySectionFileOffset;

    // Media variables
    u64                 m_OpenImageSectionSize;
    u64                 m_OpenVideoSectionSize;

};

//==============================================================================
} // namespace coredump
//==============================================================================
