﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs.h>
#include <nn/svc/svc_Types.h>
#include <nn/os.h>
#include <nn/osdbg.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <cstdlib>
#include <nn/ovln/ovln_Sender.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/format/ovln_SnapShotDumpMessage.h>

#include "SnapShotDumper.h"
#include "SnapShotDumper_System.h"
#include "SnapShotDumper_Output.h"

#include "coredump\coredump_Format.h"
#include "..\..\Libraries\tmagent\dbg\dbg_RegisterDefs.h"

//============================================================================================

namespace SnapShotDumper
{
enum
{
    ARM64_REGISTER_SP           = 31,
    ARM64_REGISTER_LR           = 30,
    ARM64_REGISTER_PC           = 32, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_REGISTER_CONTROL      = 33, // TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM64_FPU_REGISTER_START    = ((ARM64_REGISTER_CONTROL) + 1),

    ARM32_REGISTER_SP           = 13,
    ARM32_REGISTER_LR           = 14,
    ARM32_REGISTER_PC           = 15,
    ARM32_REGISTER_CONTROL      = 16,// TEMP HACK: Beyond bounds of cpuRegisters[]
    ARM32_FPU_REGISTER_START    = ((ARM32_REGISTER_CONTROL) + 1),

    CF_REG_IP                   = -3, // Call Frame aliases for registers
    CF_REG_FA                   = -2,
    CF_REG_RA                   = -1,

};

enum
{
    DEFAULT_SCREENSHOT_SIZE     = (1280 * 720 * 4)
};

//=======================================================================================

#define PROFILE COREDUMP_MACRO_VALUE(0)

// Statically-declared coredump writer.  Done this way to reduce memory fragmentation
static coredump::coredump_writer s_Writer;
static unsigned char s_MemoryBuffer[MEMORY_SECTION_BUFFER_SIZE];
static unsigned char s_MemoryCompressionBuffer[LZ4_COMPRESSBOUND( sizeof( s_MemoryBuffer ) )];

//============================================================================================

Output::Output( nn::Bit64 ProcessId, nn::svc::Handle ProcessHandle, bool IsQuickDump )
:   m_ProcessHandle( ProcessHandle )
{
    Init();
    m_Data.m_ProcessId = ProcessId;
    m_Data.m_QuickDump = IsQuickDump;
}

//============================================================================================

Output::~Output()
{
}

//============================================================================================

void Output::Init()
{
    memset( &m_DebugInfoCreateProcess, 0, sizeof(m_DebugInfoCreateProcess) );
    m_AmountToWrite = 0.0f;
    m_AmountWritten = 0.0f;
    m_CurrentPercentComplete = 0;
    m_OverlayAvailable = false;
}

//============================================================================================

nn::svc::Handle Output::GetHandle()
{
    return m_ProcessHandle;
}

//============================================================================================

nn::Result Output::SetProcessType( nn::svc::DebugEventInfo* pEventInfo )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "SetProcessType" );
    m_Data.m_Is64Bit = (pEventInfo->info.createProcess.flag & nn::svc::CreateProcessParameterFlag_64Bit) != 0;
    m_Data.m_Is64BitAddressSpace = (pEventInfo->info.createProcess.flag & nn::svc::CreateProcessParameterFlag_AddressSpace64Bit ) != 0;
    memcpy( &m_DebugInfoCreateProcess, &pEventInfo->info.createProcess, sizeof(m_DebugInfoCreateProcess) );

    return nn::ResultSuccess();
}

//============================================================================================

nn::Result Output::SetException( nn::svc::DebugEventInfo* pEventInfo )
{
    m_Data.SetException( pEventInfo->info.exception.exceptionCode, pEventInfo->info.exception.exceptionAddress );
    if( pEventInfo->threadId != 0 )
    {
        m_Data.m_CurrentThreadId = pEventInfo->threadId;
    }
    SNAP_SHOT_DUMPER_TRACE( "Output", "SetException:  Code = %d, thread %d", m_Data.m_ExceptionId, pEventInfo->threadId );
    return nn::ResultSuccess();
}

//============================================================================================

nn::Result Output::CreateThread( nn::svc::DebugEventInfo* pEventInfo )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "CreateThread Id = %lld", pEventInfo->threadId );
    nn::osdbg::ThreadInfo threadInfo;
    nn::Result result = nn::osdbg::InitializeThreadInfo( &threadInfo, m_ProcessHandle, &m_DebugInfoCreateProcess, &pEventInfo->info.createThread );
    if( result.IsSuccess() )
    {
        char ThreadName[32];
        nn::Result res = nn::dbg::ReadDebugProcessMemory( (uintptr_t)ThreadName, m_ProcessHandle, threadInfo._namePointer, sizeof(ThreadName) );
        if (res.IsSuccess())
        {
            // Create this guy.
            m_Data.AddThread( pEventInfo->threadId, ThreadName, threadInfo._stackSize, threadInfo._stack );
        }
    }
    else
    {
        NN_SDK_LOG("SnapShotDumper::Output: CreateThread: nn::osdbg::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n", pEventInfo->threadId, result.GetInnerValueForDebug() );
    }

    return nn::ResultSuccess();
}

//============================================================================================

nn::Result Output::ExitThread( nn::svc::DebugEventInfo* pEventInfo )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "ExitThread Id = %lld", pEventInfo->threadId );
    return nn::ResultSuccess();
}

//============================================================================================

bool Output::Open( char* pOutputName, coredump::coredump_writer** pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "Open %s", pOutputName );

    //Create and write our header
    coredump::coredump_file_header Header;
    Header.m_ProcessId = m_Data.m_ProcessId;
    strcpy( Header.m_ProcessName, m_DebugInfoCreateProcess.programName );
    Header.m_ExceptionNumber = m_Data.m_ExceptionId;
    memcpy( Header.m_Args, m_Data.m_Args, sizeof(m_Data.m_Args) );

    strcpy( Header.m_Architecture, "NX" );
    strcpy( Header.m_OSVersion,    "1.0.0" );
    *pWriter = &s_Writer;
    return s_Writer.Init( &Header, pOutputName );
}

//============================================================================================

void Output::AddTTY( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddTTY" );
    //FIXME:  Get TTY/Console
    //char* pTTY = "Can't get TTY at the moment.\n"; //SnapShotDumper::System::GetProcessTTY( (s32)ProcessId );
    if( m_Data.m_pTTY != NULL )
    {
        pWriter->AddTTY( m_Data.m_pTTY, strlen(m_Data.m_pTTY) );
        OnWrote( strlen( m_Data.m_pTTY ) );
    }
}

//============================================================================================

void Output::AddRegisterDefinitions( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddRegisterDefinitions" );
    u32 RegisterDefinitionSize = sizeof( s_MemoryBuffer );
    tma::dbg::GetRegisterDefinitions( m_Data.m_Is64Bit, m_Data.m_Is64BitAddressSpace, RegisterDefinitionSize, s_MemoryBuffer );
    pWriter->AddRegisterDefinitions( s_MemoryBuffer, RegisterDefinitionSize );
    OnWrote( RegisterDefinitionSize );
}

//============================================================================================

bool Output::FindModuleName( uint64_t ModuleAddress, char* pName )
{
    bool Ret = false;
    nn::svc::PageInfo PageInfo;
    nn::svc::MemoryInfo MemInfo;

    //Get EX size to determine the start of the RO section. First entry should be a Pascal string with the .nss path
    nn::Result result = nn::dbg::QueryDebugProcessMemory( &MemInfo, &PageInfo, m_ProcessHandle, ModuleAddress );
    if( result.IsSuccess() )
    {
        u64 Buffer = 0;
        u64 Address = MemInfo.baseAddress + MemInfo.size;
        result = nn::dbg::ReadDebugProcessMemory( (uintptr_t)&Buffer, m_ProcessHandle, (uintptr_t)Address, sizeof(u64) );
        if( result.IsSuccess() )
        {
            u64 Length = Buffer >> 32;
            Length |= (Buffer << 32);

            if( Length > 0 && Length < sizeof( coredump::coredump_module_info::m_ModuleName ) )
            {
                nn::dbg::ReadDebugProcessMemory( (uintptr_t)pName, m_ProcessHandle, (uintptr_t)(Address + sizeof(u64)), Length );
                pName[Length] = '\0';
                Ret = true;
                SNAP_SHOT_DUMPER_TRACE( "Output", "Found module name %s", pName );
            }
        }
    }
    return Ret;
}

//============================================================================================

enum
{
    MAX_NUMBER_OF_MODULES_QUERY = nn::dbg::MaxNsoModules + nn::dbg::MaxNroModules
};

nn::dbg::ModuleInfo s_ModuleBuffer[MAX_NUMBER_OF_MODULES_QUERY];

//============================================================================================

nn::dbg::ModuleInfo* Output::GetModules( s32* pNumberOfModules )
{
    nn::os::ProcessId Id;
    Id.value = m_Data.m_ProcessId;
    *pNumberOfModules = 0;
    nn::Result Res = nn::dbg::GetProcessModuleInfo( pNumberOfModules, s_ModuleBuffer, MAX_NUMBER_OF_MODULES_QUERY, Id );
    if( Res.IsSuccess() )
    {
        return s_ModuleBuffer;
    }
    return NULL;
}

//============================================================================================

void Output::AddModules( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddModules from process %lld", m_Data.m_ProcessId );
    s32 NumberOfModules = 0;
    nn::dbg::ModuleInfo* pModuleBuffer = GetModules( &NumberOfModules );
    if( pModuleBuffer != NULL )
    {
        SNAP_SHOT_DUMPER_TRACE( "Output", "AddModules found %d", NumberOfModules );

        for(int ModuleIndex = 0; ModuleIndex < NumberOfModules; ModuleIndex += 1 )
        {
            nn::dbg::ModuleInfo* pDefined = &pModuleBuffer[ModuleIndex];

            //=================================================================================================
            // The build Id is only 20 characters, double-check that we have don't have any trailing garbage.
            //=================================================================================================
            NN_SDK_ASSERT( (pDefined->moduleId[20] == 0) );

            char ModuleName[1024];
            memset( ModuleName, 0, sizeof(ModuleName) );
            if( FindModuleName( pDefined->address, ModuleName ) == false )
            {
                //=================================================================================================
                // Until we get a proper file name and path from the OS, we need to tell whoever's looking at
                // this file that all we have is a 20-byte identifier.  We'll do that by using this "ID="
                // string, then putting the identifier after that.
                //=================================================================================================
                strcpy( ModuleName, "ID=");
                char* pWriteId = ModuleName + 3;
                memcpy( pWriteId, pDefined->moduleId, sizeof(pDefined->moduleId) );
            }

            SNAP_SHOT_DUMPER_TRACE( "Output", "AddModules adding %s", ModuleName );
            pWriter->AddModule( ModuleName, (char*)pDefined->moduleId, pDefined->address, pDefined->size );
            OnWrote( sizeof( coredump::coredump_compiled_data_module ) );
        }
    }
}

//========================================================================================================

nn::svc::ThreadContext s_Context;

enum
{
    NUMBER_OF_CONTEXT_GP_REGISTERS = sizeof( s_Context.r) / sizeof( s_Context.r[0]),
    NUMBER_OF_CONTEXT_FP_REGISTERS = 64, //sizeof(sContext.v) / sizeof(sContext.v[0]),
};

//========================================================================================================

nn::svc::ThreadContext* Output::GetContext( u64 ThreadId )
{
    coredump::coredump_thread_info Info;
    nn::Result result = nn::dbg::GetDebugThreadContext( &s_Context, m_ProcessHandle, ThreadId, ALL_REGISTERS );
    if( result.IsFailure() )
    {
        NN_LOG( "Output::AddThread nn::dbg::GetDebugThreadContext failed.\n" );
        return NULL;
    }

    //====================================================================================================
    // If we had an exception, sometimes the PC is set to the exception handler address.  We'll
    // fix that by setting the PC to the exception address here, but only for the exception thread.
    //====================================================================================================
    // Doesn't work - The exception address passed in is 0xF at this point in time.
    //if( m_Data.m_CurrentThreadId == ThreadId )
    //{
    //    s_Context.pc = m_Data.m_ExceptionAddr;
    //}

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)

    if( s_Context.sp == 0 ) // !pProcess->Is64Bit() ) // Fixup the context info
    {
        s_Context.sp = s_Context.r[ARM32_REGISTER_SP]; // Copy registers to named locations
        s_Context.lr = s_Context.r[ARM32_REGISTER_LR];
        s_Context.r[ARM32_REGISTER_PC] = s_Context.pc; // Copy registers to numeric locations
        s_Context.r[ARM32_REGISTER_CONTROL] = s_Context.pstate;
    }
#endif
    return &s_Context;
}

//========================================================================================================

coredump::u64 FPRegisters[NUMBER_OF_CONTEXT_FP_REGISTERS];
coredump::u64 GPRegisters[NUMBER_OF_CONTEXT_GP_REGISTERS + 4];

//========================================================================================================

void Output::AddThread( coredump::coredump_compiled_data_thread* pThread, coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddThread %lld", pThread->m_Id );
    coredump::coredump_thread_info Info;
    nn::svc::ThreadContext* pContext = GetContext( pThread->m_Id );
    if( pContext == NULL )
    {
        NN_LOG( "Output::AddThread nn::dbg::GetDebugThreadContext failed.\n" );
        return;
    }

    Info.m_ThreadId         = pThread->m_Id;
    Info.m_CurrentThread    = ( m_Data.m_CurrentThreadId == pThread->m_Id );
    Info.m_Priority         = SnapShotDumper::System::GetThreadPriority( m_ProcessHandle, pThread->m_Id );
    Info.m_Core             = SnapShotDumper::System::GetThreadCore( m_ProcessHandle, pThread->m_Id );
    Info.m_IP               = pContext->pc;
    Info.m_SP               = pContext->sp;
    Info.m_Status           = SnapShotDumper::System::GetThreadStatus( m_ProcessHandle, pThread->m_Id );
    strcpy(Info.m_Name, pThread->m_Name);

    //Save our registers
    Info.m_NumberOfGPRegisters = NUMBER_OF_CONTEXT_GP_REGISTERS + 4;
    int RegIndex = 0;
    for( RegIndex = 0; RegIndex < NUMBER_OF_CONTEXT_GP_REGISTERS; RegIndex += 1)
    {
        GPRegisters[ RegIndex ] = pContext->r[ RegIndex ];
    }

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    GPRegisters[ RegIndex + 0 ]   = pContext->fp;
    GPRegisters[ RegIndex + 1 ]   = pContext->lr;
    GPRegisters[ RegIndex + 2 ]   = pContext->sp;
    GPRegisters[ RegIndex + 3 ]   = pContext->pc;
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    GPRegisters[ RegIndex + 0 ]   = pContext->lr;
    GPRegisters[ RegIndex + 1 ]   = pContext->sp;
    GPRegisters[ RegIndex + 2 ]   = pContext->pc;
#endif

    Info.m_pGPRegisters = GPRegisters;

    Info.m_NumberOfGPControlRegisters = 1;
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    coredump::u64 GPControlRegisters[1] = { pContext->pstate };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    coredump::u64 GPControlRegisters[1] = { pContext->cpsr };
#endif
    Info.m_pGPControlRegisters = GPControlRegisters;

    Info.m_NumberOfFPRegisters = NUMBER_OF_CONTEXT_FP_REGISTERS;

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    memcpy( (void*)FPRegisters, (void*)&pContext->v, sizeof(FPRegisters) );
#endif
    Info.m_pFPRegisters = FPRegisters;

    Info.m_NumberOfFPControlRegisters = 2;

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    coredump::u64 FPControlRegisters[2] = { pContext->fpcr, pContext->fpsr };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    coredump::u64 FPControlRegisters[2] = { pContext->fpscr, pContext->fpexc };
#endif
    Info.m_pFPControlRegisters = FPControlRegisters;
    pWriter->AddThread( &Info, pContext->tpidr );

    OnWrote( sizeof( coredump::coredump_compiled_data_thread ) );
}

//=======================================================================================

void Output::AddThreads( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "Adding %d threads", m_Data.m_NumberOfThreads );

    for(int ThreadIndex = 0; ThreadIndex < m_Data.m_NumberOfThreads; ThreadIndex += 1 )
    {
        AddThread( &m_Data.m_pThreads[ThreadIndex], pWriter );
    }
}

//=======================================================================================

void* Alloc( void* Opaque, unsigned int ItemCount, unsigned int SizePerItem )
{
    return System::Allocate( SizePerItem * ItemCount );
}

//=======================================================================================

void Free( void* Opaque,  void* Ptr )
{
    System::Deallocate( Ptr, 0 );
}

//=======================================================================================

void Output::AddMemory( coredump::MemoryType Type, coredump::u64 SizeOfData, coredump::u64 Address, coredump::coredump_writer* pWriter )
{
    //Create the memory section.
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddMemoryBlock:  Creating memory section of type %d, size = %lld", (int)Type, SizeOfData );
    if( pWriter->CreateMemorySection( Address, Type, SizeOfData, Alloc, Free ) == coredump::RESULT_COREDUMP_OK )
    {
        //Start our copy loop.
        u64 ReadFromAddress = Address;
        u64 AmountToCopy = SizeOfData;

        while( AmountToCopy > 0 )
        {
            //====================
            // Read the memory....
            //====================
            u64 SizeToCopy = ( AmountToCopy > sizeof( s_MemoryBuffer ) ) ? sizeof( s_MemoryBuffer ) : AmountToCopy;
            nn::Result Result = nn::dbg::ReadDebugProcessMemory( (uintptr_t)s_MemoryBuffer, m_ProcessHandle, ReadFromAddress, SizeToCopy );
            if( Result.IsFailure() )
            {
                memset( s_MemoryBuffer, 0xFF, SizeToCopy );
                SNAP_SHOT_DUMPER_TRACE( "Output", "AddMemoryBlock:  nn::dbg::ReadDebugProcessMemory FAILED with 0x%x", Result.GetInnerValueForDebug() );
            }

            //=========================
            // Write it to our file.
            //=========================
            coredump::result res = pWriter->AddMemoryData( s_MemoryBuffer, SizeToCopy, s_MemoryCompressionBuffer, sizeof( s_MemoryCompressionBuffer ) );
            OnWrote( SizeToCopy );

            SNAP_SHOT_DUMPER_TRACE( "Output", "AddMemoryBlock:  Adding memory (0x%llx, Size %lld), result = %d", ReadFromAddress, SizeToCopy, res ); (void)res;
            AmountToCopy    -= SizeToCopy;
            ReadFromAddress += SizeToCopy;
        }

        SNAP_SHOT_DUMPER_TRACE( "Output", "AddMemoryBlock:  Closing memory section" );
        pWriter->CloseMemorySection( Address );
    }
}

//=======================================================================================

void Output::AddMemoryBlock( nn::svc::MemoryInfo* pBlockInfo, coredump::coredump_writer* pWriter )
{
    //Assume we can read this.
    coredump::MemoryType Type = coredump::MemoryType_Read;

    //Figure out our memory type.
    switch( pBlockInfo->permission )
    {
        default:
            Type = coredump::MemoryType_All;
            break;
        case nn::svc::MemoryPermission_Read:
            Type = coredump::MemoryType_Read;
            break;
        case nn::svc::MemoryPermission_Write:
            Type = coredump::MemoryType_Write;
            break;
        case nn::svc::MemoryPermission_Execute:
            Type = coredump::MemoryType_Execute;
            break;
        case nn::svc::MemoryPermission_ReadWrite:
            Type = coredump::MemoryType_ReadWrite;
            break;
        case nn::svc::MemoryPermission_ReadExecute:
            Type = coredump::MemoryType_ReadExecute;
            break;
    }

    AddMemory( Type, (coredump::u64)pBlockInfo->size, (coredump::u64)pBlockInfo->baseAddress, pWriter );
}

//=======================================================================================

void Output::AddMemory( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddMemory" );

    // Find the memory blocks with these permissions
    int SaveMemoryPermissionType = ( nn::svc::MemoryPermission_Read | nn::svc::MemoryPermission_Write | nn::svc::MemoryPermission_Execute );

    if( m_Data.m_QuickDump )
    {
        //============================================================
        // If it's a mini dump, we just want the Execute memory blocks
        // at this point - We'll add specific areas of memory later.
        //============================================================
        SaveMemoryPermissionType = nn::svc::MemoryPermission_Execute;
    }

    // Go through and add the relevant memory blocks
    for (uintptr_t v = 0;;)
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        nn::Result result = nn::dbg::QueryDebugProcessMemory( &blockInfo, &pageInfo, m_ProcessHandle, v );
        if( result.IsFailure() )
        {
            break;
        }
        if( blockInfo.state == nn::svc::MemoryState_Inaccessible )
        {
            break;
        }

        if( blockInfo.baseAddress + blockInfo.size < v )
        {
            break;
        }

        if( ((blockInfo.permission & SaveMemoryPermissionType ) != 0) && (blockInfo.state != nn::svc::MemoryState_Io) )
        {
            AddMemoryBlock( &blockInfo, pWriter );
        }

        v = blockInfo.baseAddress + blockInfo.size;
    }

    //=====================================================================
    // If a quick dump, we'll need to cherry-pick our memory -
    //=====================================================================
    if( m_Data.m_QuickDump )
    {
        // We'll want the stacks for each of our threads.
        for(int ThreadIndex = 0; ThreadIndex < m_Data.m_NumberOfThreads; ThreadIndex += 1 )
        {
            coredump::coredump_compiled_data_thread* pThread = &m_Data.m_pThreads[ThreadIndex];
            AddMemory( coredump::MemoryType_ReadWrite, pThread->m_StackSize, pThread->m_StackAddress, pWriter );
        }
    }
}

//=======================================================================================

bool Output::StartScreenshot( size_t ScreenshotDataSize, int ScreenshotWidth, int ScreenshotHeight, coredump::coredump_writer* pWriter )
{
#pragma pack(push, 2)

    // Screenshot BMP types
    struct BMPFileHeader
    {
        coredump::u16   bfType;
        coredump::u32   bfSize;
        coredump::u16   bfReserved1;
        coredump::u16   bfReserved2;
        coredump::u32   bfOffBits;
    };

    struct BMPInfoHeader
    {
        coredump::u32   biSize;
        coredump::s32   biWidth;
        coredump::s32   biHeight;
        coredump::u16   biPlanes;
        coredump::u16   biBitCount;
        coredump::u32   biCompression;
        coredump::u32   biSizeImage;
        coredump::s32   biXPelsPerMeter;
        coredump::s32   biYPelsPerMeter;
        coredump::u32   biClrUsed;
        coredump::u32   biClrImportant;
    };

    enum
    {
        BI_RGB = 0L
    };

#pragma pack (pop)

    coredump::u64 TotalSize = ScreenshotDataSize + sizeof( BMPFileHeader ) + sizeof( BMPInfoHeader );
    if( pWriter->CreateScreenshot( coredump::ScreenshotType_BMP, TotalSize ) == coredump::RESULT_COREDUMP_OK )
    {
        BMPFileHeader bmfh;
        memset( &bmfh, 0, sizeof( BMPFileHeader ) );
        bmfh.bfType = 0x4d42; // 0x4d42 = 'BM'
        bmfh.bfReserved1 = 0;
        bmfh.bfReserved2 = 0;
        bmfh.bfSize = ScreenshotDataSize;
        bmfh.bfOffBits = 0x36;
        SNAP_SHOT_DUMPER_TRACE( "Output", "StartScreenshot adding %d bytes for %d x %d screenshot (Total %lld)", ScreenshotDataSize, ScreenshotWidth, ScreenshotHeight, TotalSize );

        if( pWriter->AddScreenshotData( &bmfh, sizeof( bmfh ) ) == coredump::RESULT_COREDUMP_OK )
        {
            // Prepare info header...
            BMPInfoHeader info;
            memset( &info, 0, sizeof( BMPInfoHeader ) );
            info.biSize = sizeof( BMPInfoHeader );
            info.biWidth = ScreenshotWidth;
            info.biHeight = -ScreenshotHeight;
            info.biPlanes = 1;
            info.biBitCount = 32;
            info.biCompression = BI_RGB;
            info.biSizeImage = ScreenshotDataSize;
            info.biXPelsPerMeter = 0x0ec4;
            info.biYPelsPerMeter = 0x0ec4;
            info.biClrUsed = 0;
            info.biClrImportant = 0;
            if( pWriter->AddScreenshotData( &info, sizeof( info ) ) == coredump::RESULT_COREDUMP_OK )
            {
                return true;
            }
        }

        // Made it here, then there was a problem.  Close the screenshot we started.
        pWriter->CloseScreenshot();
    }

    return false;
}

//=======================================================================================

void Output::FixColors( u8* pData, size_t ReadSize )
{
    if( ReadSize > 0 )
    {
        size_t NumberOfEntries = ReadSize / (sizeof( u32 ));
        u32* pEntries = (u32*)pData;
        int Index = 0;
        for( ; Index < NumberOfEntries; Index += 1 )
        {
            u8* pValues = (u8*)(&pEntries[Index]);
            u8 Temp = pValues[0];
            pValues[0] = pValues[2];
            pValues[2] = Temp;
            pEntries[Index] = *((u32*)pValues);
        }
    }
}

//=======================================================================================

void Output::AddScreenshot( coredump::coredump_writer* pWriter )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot" );
    nn::Result result = nn::capsrv::InitializeScreenShotControl();
    if( result.IsFailure() )
    {
        SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot nn::capsrv::InitializeScreenShotControl() FAILED, returned %x", result.GetInnerValueForDebug() );
        return;
    }
    size_t ScreenshotDataSize;
    int ScreenshotWidth;
    int ScreenshotHeight;
    result = nn::capsrv::OpenRawScreenShotReadStreamForDevelop( &ScreenshotDataSize, &ScreenshotWidth, &ScreenshotHeight, nn::vi::LayerStack::LayerStack_ApplicationForDebug, nn::TimeSpan::FromSeconds( 10 ) );
    if( result.IsFailure() )
    {
        SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot nn::capsrv::OpenRawScreenShotReadStreamForDevelop() FAILED, returned %x", result.GetInnerValueForDebug() );
        nn::capsrv::FinalizeScreenShotControl();
        return;
    }

    // Start the screenshot
    if( StartScreenshot( ScreenshotDataSize, ScreenshotWidth, ScreenshotHeight, pWriter ) )
    {
        // We'll need a buffer
        size_t MemoryBufferSize = ( ScreenshotDataSize > sizeof( s_MemoryBuffer ) ) ? sizeof( s_MemoryBuffer ) : ScreenshotDataSize;
        size_t readSize;
        size_t totalReadSize = 0;
        while( totalReadSize < ScreenshotDataSize )
        {
            result = nn::capsrv::ReadRawScreenShotReadStreamForDevelop( &readSize, s_MemoryBuffer, MemoryBufferSize, totalReadSize );

            if( result.IsFailure() )
            {
                SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot nn::capsrv::ReadRawScreenShotReadStreamForDevelop() FAILED, returned %x", result.GetInnerValueForDebug() );
                break;
            }

            FixColors( s_MemoryBuffer, readSize );

            SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot:  Adding %d bytes (%d of %d)", readSize, totalReadSize, ScreenshotDataSize );
            if( pWriter->AddScreenshotData( s_MemoryBuffer, readSize ) != coredump::RESULT_COREDUMP_OK )
            {
                break;
            }

            OnWrote( readSize );
            totalReadSize += readSize;
        }

        // Close up
        pWriter->CloseScreenshot();
    }
    else
    {
        // Problem writing screenshot, so just say we wrote it.
        OnWrote( DEFAULT_SCREENSHOT_SIZE );
    }

    nn::capsrv::CloseRawScreenShotReadStreamForDevelop();
    nn::capsrv::FinalizeScreenShotControl();

    SNAP_SHOT_DUMPER_TRACE( "Output", "AddScreenshot complete" );
}

//=======================================================================================

void Output::AddApplicationID( coredump::coredump_writer* pWriter )
{
    pWriter->AddApplicationId( m_DebugInfoCreateProcess.programId );
    SNAP_SHOT_DUMPER_TRACE( "Output", "AddApplicationID added Id %llx", m_DebugInfoCreateProcess.programId );
}

//=======================================================================================

void Output::CalculateWriteAmount()
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "CalculateWriteAmount");

    //===========================================================================================
    // NOTICE that these sizes for threads and modules don't have to be EXACTLY how much we are
    // going to write, but a relative representation of how much data we need to go through.
    //===========================================================================================
    m_AmountToWrite = (float)(m_Data.m_NumberOfThreads * sizeof( coredump::coredump_compiled_data_thread ));
    m_AmountToWrite += (float)(m_Data.m_NumberOfModules * sizeof( coredump::coredump_compiled_data_module ));

    //================================================================
    // Calculate our memory write requirements with more precision
    //================================================================
    int SaveMemoryPermissionType = (nn::svc::MemoryPermission_Read | nn::svc::MemoryPermission_Write | nn::svc::MemoryPermission_Execute);
    if( m_Data.m_QuickDump == true )
    {
        //============================================================
        // If it's a mini dump, we just want the Execute memory blocks
        // at this point - We'll add specific areas of memory later.
        //============================================================
        SaveMemoryPermissionType = nn::svc::MemoryPermission_Execute;
    }

    for( uintptr_t v = 0;;)
    {
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo   pageInfo;
        nn::Result result = nn::dbg::QueryDebugProcessMemory( &blockInfo, &pageInfo, m_ProcessHandle, v );
        if( result.IsFailure() )
        {
            break;
        }
        if( blockInfo.state == nn::svc::MemoryState_Inaccessible )
        {
            break;
        }

        if( blockInfo.baseAddress + blockInfo.size < v )
        {
            break;
        }

        if( ((blockInfo.permission & SaveMemoryPermissionType) != 0) && (blockInfo.state != nn::svc::MemoryState_Io) )
        {
            m_AmountToWrite += (float)blockInfo.size;
        }

        v = blockInfo.baseAddress + blockInfo.size;
    }

    //=====================================================================
    // If a quick dump, we'll need to cherry-pick our memory -
    //=====================================================================
    if( m_Data.m_QuickDump == true )
    {
        // We'll want the stacks for each of our threads.
        for( int ThreadIndex = 0; ThreadIndex < m_Data.m_NumberOfThreads; ThreadIndex += 1 )
        {
            coredump::coredump_compiled_data_thread* pThread = &m_Data.m_pThreads[ThreadIndex];
            if( pThread->m_StackSize > 0 )
            {
                m_AmountToWrite += (float)pThread->m_StackSize;
            }
        }
    }

    // Put in a dummy amount for the screenshot.
    m_AmountToWrite += (float)DEFAULT_SCREENSHOT_SIZE;

    // Init our starting amount
    m_AmountWritten = 0;
}

//=======================================================================================

void Output::SendMessage( uint32_t percent )
{
    if( m_OverlayAvailable )
    {
        nn::ovln::Message message = {};
        message.tag = nn::ovln::format::DumpProgressTag;
        nn::ovln::format::DumpProgress data = { percent };
        message.dataSize = sizeof( data );
        std::memcpy( &message.data, &data, sizeof( data ) );
        if( nn::ovln::Send( &m_PercentageSender, message ) == false )
        {
            NN_SDK_LOG( "Failed to send message to Overlay\n" );
        }
    }
}

//=======================================================================================

void Output::OnWrote( size_t AmountWritten )
{
    m_AmountWritten += AmountWritten;
    float Percent = m_AmountWritten / m_AmountToWrite;
    uint32_t Ret = (uint32_t)(Percent * 100.0f);

    if( Ret > m_CurrentPercentComplete )
    {
        m_CurrentPercentComplete = Ret;

        // Notify anyone who's interested.
        if( m_OverlayAvailable )
        {
            SendMessage( m_CurrentPercentComplete );
        }
        else
        {
            NN_SDK_LOG( "[SSD] %d complete...\n", m_CurrentPercentComplete );
        }
    }
}

//=======================================================================================

void Output::InitializeOverlay()
{
    nn::Result result = nn::ovln::InitializeSenderLibraryForOverlay();
    if( result.IsFailure() )
    {
        NN_SDK_LOG( "Overlay initialization failed. %08X\n", result.GetInnerValueForDebug() );
        return;
    }
    result = nn::ovln::InitializeSenderForValue( &m_PercentageSender );
    if( result.IsFailure() )
    {
        NN_SDK_LOG( "Overlay initialization failed. %08X\n", result.GetInnerValueForDebug() );
        return;
    }

    m_OverlayAvailable = true;
}

//=======================================================================================

void Output::FinalizeOverlay()
{
    if( !m_OverlayAvailable )
    {
        return;
    }

    const int Timeout = 10;
    for( int i = 0; i < Timeout; i++ )
    {
        if( GetUnreceivedMessageCount( &m_PercentageSender ) == 0 )
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
    }

    nn::ovln::FinalizeSender( &m_PercentageSender );
    nn::ovln::FinalizeSenderLibraryForOverlay();
    m_OverlayAvailable = false;
}

//=======================================================================================

nn::Result Output::Write( char* pOutputName )
{
    SNAP_SHOT_DUMPER_TRACE( "Output", "Write %s", pOutputName );
    if( m_ProcessHandle.IsValid() == true )
    {
#if PROFILE
        nn::os::Tick startTick = nn::os::GetSystemTick();
#endif
        CalculateWriteAmount();

        //===========================================================================
        // Get our required pieces
        //===========================================================================
        coredump::coredump_writer* pWriter = NULL;
        if( Open( pOutputName, &pWriter ))
        {
            InitializeOverlay();
            AddTTY( pWriter );
            AddRegisterDefinitions( pWriter );
            AddModules( pWriter );
            AddThreads( pWriter );
            AddMemory( pWriter );
            AddScreenshot( pWriter );
            AddApplicationID( pWriter );
            SendMessage( 100 );
            FinalizeOverlay();

            // Now we simply close this writer, do NOT delete it:  It is statically
            // defined in this file to cut down on memory fragmentation.
            pWriter->Close();

        }
#if PROFILE
        nn::os::Tick endTick = nn::os::GetSystemTick();
        nn::TimeSpan durationSpan = (endTick - startTick).ToTimeSpan();
        uint64_t milliSeconds = durationSpan.GetMilliSeconds();
        if( milliSeconds > 0 )
        {
            float duration = (float)(milliSeconds) / 1000.0f;
            NN_SDK_LOG("Core dump took %2.4f\n", duration );
        }
#endif
    }

    return nn::ResultSuccess();
}

//============================================================================================

//============================================================================================
}// namespace SnapShotDumper
//============================================================================================

