﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cerrno>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/account_Types.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Uuid.h>

#include "SettingsManager_Utility.h"

::std::vector<::std::string> Split(
    const ::std::string& value, const ::std::string& separator) NN_NOEXCEPT
{
    auto vector = ::std::vector<::std::string>();

    if (value.length() == 0)
    {
        vector.push_back(value);
    }
    else if (separator.length() == 0)
    {
        for (char c : value)
        {
            vector.push_back(::std::string(1, c));
        }
    }
    else
    {
        size_t head = 0;

        while (NN_STATIC_CONDITION(true))
        {
            size_t tail = value.find(separator, head);

            if (tail == ::std::string::npos)
            {
                vector.push_back(value.substr(head));

                break;
            }

            vector.push_back(value.substr(head, tail - head));

            head = tail + separator.length();
        }
    }

    return vector;
}

bool GetMissingKeys(
    ::std::vector<::std::string>* pOutValue,
    const ::std::vector<::std::string>& haystack,
    const ::std::vector<::std::string>& needles) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    for (const ::std::string& key : needles)
    {
        if (::std::find(
                haystack.begin(), haystack.end(), key) == haystack.end())
        {
            pOutValue->push_back(key);
        }
    }

    return true;
}

bool EncodeInteger(
    ::std::string* pOutValue, int64_t value, int base) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    char buffer[32] = {};

    switch (base)
    {
    case  0:
    case 10:
        ::nn::util::SNPrintf(buffer, sizeof(buffer), "%d", value);
        *pOutValue = buffer;
        break;

    case 16:
        ::nn::util::SNPrintf(buffer, sizeof(buffer), "%x", value);
        *pOutValue = buffer;
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

bool DecodeInteger(
    int64_t* pOutValue, const ::std::string& string, int base) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    char* end = nullptr;

    errno = 0;

    int64_t value =
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2012)
        _strtoi64(string.c_str(), &end, base);
#else
        strtoll(string.c_str(), &end, base);
#endif

    if (errno == ERANGE)
    {
        return false;
    }

    if (end == string.c_str() || end != string.c_str() + string.length())
    {
        return false;
    }

    *pOutValue = value;

    return true;
}

bool EncodeUuid(
    ::std::string* pOutValue, const ::nn::util::Uuid& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    char buffer[37] = {};

    *pOutValue = value.ToString(buffer, sizeof(buffer));

    return true;
}

bool DecodeUuid(
    ::nn::util::Uuid* pOutValue, const ::std::string& string) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nn::util::Uuid uuid = {};

    // フォーマット 00000000-0000-0000-0000-000000000000 のオクテット開始位置
    const size_t indices[] = {
        0, 2, 4, 6, 9, 11, 14, 16, 19, 21, 24, 26, 28, 30, 32, 34
    };

    const size_t indexCount = NN_ARRAY_SIZE(indices);

    NN_STATIC_ASSERT(indexCount == sizeof(uuid.data));

    for (size_t i = 0; i < indexCount; ++i)
    {
        if (indices[i] + 2 <= string.length())
        {
            auto value = int64_t();

            if (DecodeInteger(&value, string.substr(indices[i], 2), 16))
            {
                uuid.data[i] = static_cast<::nn::Bit8>(value);

                continue;
            }
        }

        return false;
    }

    *pOutValue = uuid;

    return true;
}

::nn::util::Uuid ConvertToUuid(const ::nn::account::Uid& uid) NN_NOEXCEPT
{
    typedef nn::util::BitPack32::Field< 0, 32, uint32_t> TimeLow;

    typedef nn::util::BitPack32::Field< 0, 16, uint16_t> TimeMid;
    typedef nn::util::BitPack32::Field<16, 16, uint16_t> TimeHav;

    typedef nn::util::BitPack32::Field< 0,  8, uint8_t>  ClkHar;
    typedef nn::util::BitPack32::Field< 8,  8, uint8_t>  ClkLow;
    typedef nn::util::BitPack32::Field<16, 16, uint16_t> NodeLo;

    typedef nn::util::BitPack32::Field< 0, 32, uint32_t> NodeHi;

    const nn::util::BitPack32 data[] = {
        { static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull) },
        { static_cast<uint32_t>(uid._data[0] >> 32) },
        { static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull) },
        { static_cast<uint32_t>(uid._data[1] >> 32) },
    };

    ::nn::util::Uuid uuid = {};

    auto buffer16 = uint16_t();

    auto buffer32 = uint32_t();

    ::nn::util::StoreBigEndian(&buffer32, data[0].Get<TimeLow>());

    ::std::memcpy(&uuid.data[0], &buffer32, sizeof(buffer32));

    ::nn::util::StoreBigEndian(&buffer16, data[1].Get<TimeMid>());

    ::std::memcpy(&uuid.data[4], &buffer16, sizeof(buffer16));

    ::nn::util::StoreBigEndian(&buffer16, data[1].Get<TimeHav>());

    ::std::memcpy(&uuid.data[6], &buffer16, sizeof(buffer16));

    uuid.data[8] = data[2].Get<ClkHar>();

    uuid.data[9] = data[2].Get<ClkLow>();

    ::nn::util::StoreBigEndian(&buffer32, data[3].Get<NodeHi>());

    ::std::memcpy(&uuid.data[10], &buffer32, sizeof(buffer32));

    ::nn::util::StoreBigEndian(&buffer16, data[2].Get<NodeLo>());

    ::std::memcpy(&uuid.data[14], &buffer16, sizeof(buffer16));

    return uuid;
}

::nn::account::Uid ConvertToUid(const ::nn::util::Uuid& uuid) NN_NOEXCEPT
{
    const ::nn::account::Uid uid = {
        {
            static_cast<uint64_t>(uuid.data[ 0]) << 24 |
            static_cast<uint64_t>(uuid.data[ 1]) << 16 |
            static_cast<uint64_t>(uuid.data[ 2]) <<  8 |
            static_cast<uint64_t>(uuid.data[ 3]) <<  0 |
            static_cast<uint64_t>(uuid.data[ 4]) << 40 |
            static_cast<uint64_t>(uuid.data[ 5]) << 32 |
            static_cast<uint64_t>(uuid.data[ 6]) << 56 |
            static_cast<uint64_t>(uuid.data[ 7]) << 48 ,
            static_cast<uint64_t>(uuid.data[ 8]) <<  0 |
            static_cast<uint64_t>(uuid.data[ 9]) <<  8 |
            static_cast<uint64_t>(uuid.data[10]) << 56 |
            static_cast<uint64_t>(uuid.data[11]) << 48 |
            static_cast<uint64_t>(uuid.data[12]) << 40 |
            static_cast<uint64_t>(uuid.data[13]) << 32 |
            static_cast<uint64_t>(uuid.data[14]) << 24 |
            static_cast<uint64_t>(uuid.data[15]) << 16
        }
    };

    return uid;
}
