﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_Language.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 言語設定のキー
const char* const SettingNameLanguageSettings = "language_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< 言語コードのキー
const char* const KeyLanguageCode = "language_code";

//!< 言語コードをエクスポートします。
bool ExportLanguageCode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyLanguageCode);

        const int count = ::nn::settings::GetAvailableLanguageCodeCount();

        ::std::unique_ptr<::nn::settings::LanguageCode[]> languageCodes(
            new ::nn::settings::LanguageCode[count]());

        auto currentValueNode = Node::CreateStringNode();

        ::nn::settings::GetLanguageCode(languageCodes.get());

        COMMAND_DO(currentValueNode.SetValue(languageCodes[0].string));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(
            defaultValueNode.SetValue(
                ::nn::settings::LanguageCode::Make(
                    ::nn::settings::Language_AmericanEnglish).string));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        ::nn::settings::GetAvailableLanguageCodes(languageCodes.get(), count);

        for (int i = 0; i < count; ++i)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(
                choiceValueNode.SetValue(languageCodes[i].string));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyLanguageCode, ::std::move(node)));

    return true;
}

//!< 言語コードをインポートします。
bool ImportLanguageCode(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyLanguageCode));

    NameScope nameScope(KeyLanguageCode);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string ietf;
    COMMAND_DO(pCurrentValueNode->GetValue(&ietf));

    const int count = ::nn::settings::GetAvailableLanguageCodeCount();

    ::std::unique_ptr<::nn::settings::LanguageCode[]> languageCodes(
        new ::nn::settings::LanguageCode[count]());

    ::nn::settings::GetAvailableLanguageCodes(languageCodes.get(), count);

    for (int i = 0; i < count; ++i)
    {
        if (ietf == languageCodes[i].string)
        {
            ::nn::settings::system::SetLanguageCode(languageCodes[i]);

            return true;
        }
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), ietf);

    return false;
}

} // namespace

bool IsSettingNameLanguageSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameLanguageSettings);
}

bool ExportLanguageSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameLanguageSettings);

        COMMAND_DO(ExportLanguageCode(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameLanguageSettings, ::std::move(node)));

    return true;
}

bool ImportLanguageSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameLanguageSettings);

    COMMAND_DO(ImportLanguageCode(node));

    return true;
}
