﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

//!< エラーコードのプラットフォーム識別子です。
const int ErrorCodePlatformId = 2;

//!< エラーコードを表す列挙体です。
enum class ErrorCode
{
    None,                               //!< エラーは発生しませんでした
    CommandSpecificationMissing = 100,  //!< コマンドが指定されませんでした
    CommandNotFound,                    //!< コマンドが見つかりませんでした
    CommandNotSupported,                //!< サポートされないコマンドでした
    CommandResetModeInvalid,            //!< 不正なリセット方式が指定されました
    FileSpecificationMissing = 200,     //!< ファイルが指定されませんでした
    FileOpenFailure,                    //!< ファイルのオープンに失敗しました
    FileSizeChangeFailure,              //!< ファイルサイズの変更に失敗しました
    FileSizeAcquisitionFailure,         //!< ファイルサイズの取得に失敗しました
    FileReadingFailure,                 //!< ファイルの読み込みに失敗しました
    FileWritingFailure,                 //!< ファイルの書き込みに失敗しました
    FileFlushingFailure,                //!< ファイルのフラッシュに失敗しました
    NodeKeyMissing = 300,               //!< ノードのキーが見つかりませんでした
    NodeKeyInvalid,                     //!< 不正なノードのキーが指定されました
    NodeTypeUnsupported = 400,          //!< サポートされないノードの型でした
    NodeTypeMismatch,                   //!< ノードの型が一致しませんでした
    NodeTypeShortArray,                 //!< ノードの型が短すぎる配列でした
    NodeValueInvalid = 500,             //!< ノードの値が不正でした
    NodeValueTooLong,                   //!< ノードの値が長過ぎました
    NodeValueOverLimitation,            //!< ノードの値が有効範囲を超えました
    MemoryAllocationFailure = 1000,     //!< メモリアロケーションに失敗しました
    DataSizeInvalid,                    //!< データのサイズが不正でした
    JsonParsingFailure,                 //!< JSON のパースに失敗しました
};

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(ErrorCode code) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d\n", code);
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(ErrorCode code, size_t arg1) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%u\n", code, arg1);
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, size_t arg1, const ::std::string& arg2) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%u:%s\n", code, arg1, arg2.c_str());
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, const ::std::string& arg1) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%s\n", code, arg1.c_str());
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, const ::std::string& arg1, const ::std::string& arg2
    ) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%s:%s\n", code, arg1.c_str(), arg2.c_str());
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, const ::std::string& arg1, const ::std::string& arg2,
    const ::std::string& arg3) NN_NOEXCEPT
{
    NN_LOG(
        "ErrorCode:%04d:%s:%s:%s\n",
        code, arg1.c_str(), arg2.c_str(), arg3.c_str());
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, const ::std::string& arg1, size_t arg2, size_t arg3
    ) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%s:%u:%u\n", code, arg1.c_str(), arg2, arg3);
}

//!< エラーコードをログとして出力します。
inline void PrintErrorCode(
    ErrorCode code, const ::nn::Result& arg1, const ::std::string& arg2
    ) NN_NOEXCEPT
{
    NN_LOG("ErrorCode:%04d:%d%03d-%04d:%s\n",
        code,
        ErrorCodePlatformId, arg1.GetModule(), arg1.GetDescription(),
        arg2.c_str());
}
