﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_DataDeletion.h>

#include "SettingsManager_DataDeletion.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< おすすめ削除設定のキー
const char* const SettingNameDataDeletionSettings = "data_deletion_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 自動削除が有効か否かを表す値のキー
const char* const KeyAutomaticDeletionFlag = "automatic_deletion_flag";

//!< おすすめ削除を利用した回数のキー
const char* const KeyUseCount = "use_count";

//!< 自動削除が有効か否かを表す値をエクスポートします。
bool ExportAutomaticDeletionFlag(
    Node* pDictNode,
    const ::nn::settings::system::DataDeletionSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyAutomaticDeletionFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::DataDeletionFlag::Automatic>()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyAutomaticDeletionFlag, ::std::move(node)));

    return true;
}

//!< 自動削除が有効か否かを表す値をインポートします。
bool ImportAutomaticDeletionFlag(
    ::nn::settings::system::DataDeletionSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyAutomaticDeletionFlag));

    NameScope nameScope(KeyAutomaticDeletionFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto flag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&flag));

    pOutValue->flags.Set<
        ::nn::settings::system::DataDeletionFlag::Automatic>(flag);

    return true;
}

//!< おすすめ削除を利用した回数をエクスポートします。
bool ExportUseCount(
    Node* pDictNode,
    const ::nn::settings::system::DataDeletionSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUseCount);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(currentValueNode.SetValue(settings.useCount));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(static_cast<int32_t>(0)));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyUseCount, ::std::move(node)));

    return true;
}

//!< おすすめ削除を利用した回数をインポートします。
bool ImportUseCount(
    ::nn::settings::system::DataDeletionSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyUseCount));

    NameScope nameScope(KeyUseCount);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    COMMAND_DO(pCurrentValueNode->GetValue(&pOutValue->useCount));

    return true;
}

} // namespace

bool IsSettingNameDataDeletionSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameDataDeletionSettings);
}

bool ExportDataDeletionSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    ::nn::settings::system::DataDeletionSettings settings = {};

    ::nn::settings::system::GetDataDeletionSettings(&settings);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameDataDeletionSettings);

        COMMAND_DO(ExportAutomaticDeletionFlag(&node, settings));

        COMMAND_DO(ExportUseCount(&node, settings));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameDataDeletionSettings, ::std::move(node)));

    return true;
}

bool ImportDataDeletionSettings(const Node& node) NN_NOEXCEPT
{
    ::nn::settings::system::DataDeletionSettings settings = {};

    NameScope nameScope(SettingNameDataDeletionSettings);

    COMMAND_DO(ImportAutomaticDeletionFlag(&settings, node));

    COMMAND_DO(ImportUseCount(&settings, node));

    ::nn::settings::system::SetDataDeletionSettings(settings);

    return true;
}
