﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>

#include "SettingsManager_Account.h"
#include "SettingsManager_Audio.h"
#include "SettingsManager_Backlight.h"
#include "SettingsManager_Capture.h"
#include "SettingsManager_Clock.h"
#include "SettingsManager_Command.h"
#include "SettingsManager_Debug.h"
#include "SettingsManager_DataDeletion.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_Eula.h"
#include "SettingsManager_FirmwareDebugSettings.h"
#include "SettingsManager_Hid.h"
#include "SettingsManager_Language.h"
#include "SettingsManager_Ldn.h"
#include "SettingsManager_Network.h"
#include "SettingsManager_NetworkService.h"
#include "SettingsManager_News.h"
#include "SettingsManager_Nfc.h"
#include "SettingsManager_OnlineStorage.h"
#include "SettingsManager_Pcv.h"
#include "SettingsManager_PushNotification.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Sleep.h"
#include "SettingsManager_SystemApplication.h"
#include "SettingsManager_TemporaryDatabase.h"
#include "SettingsManager_TimeZone.h"
#include "SettingsManager_Tv.h"
#include "SettingsManager_Usb.h"
#include "SettingsManager_Utility.h"
#include "SettingsManager_Web.h"
#include "SettingsManager_WirelessLan.h"

namespace {

//!< インポータを表す構造体です。
struct Importer final
{
    //!< インポート可能か否かを表す値を返します。
    bool (*supports)(const ::std::string&) NN_NOEXCEPT;

    //!< インポートを実行します。
    bool (*import)(const Node&) NN_NOEXCEPT;
};

//!< インポータを返します。
const Importer* GetImporter(const ::std::string& name) NN_NOEXCEPT;

//!< 利用可能なファームウェアデバッグ設定の名前を返します。
const ::std::vector<::std::string>& GetAvailableNames() NN_NOEXCEPT;

} // namespace

bool ExportSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    COMMAND_DO(ExportAccountSettings(pNode));
    COMMAND_DO(ExportAudioSettings(pNode));
    COMMAND_DO(ExportBacklightSettings(pNode));
    COMMAND_DO(ExportCaptureSettings(pNode));
    COMMAND_DO(ExportClockSettings(pNode));
    COMMAND_DO(ExportDataDeletionSettings(pNode));
    COMMAND_DO(ExportDebugSettings(pNode));
    COMMAND_DO(ExportEulaVersions(pNode));
    COMMAND_DO(ExportHidSettings(pNode));
    COMMAND_DO(ExportLanguageSettings(pNode));
    COMMAND_DO(ExportLdnSettings(pNode));
    COMMAND_DO(ExportNetworkSettings(pNode));
    COMMAND_DO(ExportNetworkServiceSettings(pNode));
    COMMAND_DO(ExportNfcSettings(pNode));
    COMMAND_DO(ExportNewsSettings(pNode));
    COMMAND_DO(ExportNotificationSettings(pNode));
    COMMAND_DO(ExportAccountNotificationSettings(pNode));
    COMMAND_DO(ExportAccountOnlineStorageSettings(pNode));
    COMMAND_DO(ExportPcvSettings(pNode));
    COMMAND_DO(ExportPushNotificationSettings(pNode));
    COMMAND_DO(ExportSleepSettings(pNode));
    COMMAND_DO(ExportSystemApplicationSettings(pNode));
    COMMAND_DO(ExportTimeZoneSettings(pNode));
    COMMAND_DO(ExportTvSettings(pNode));
    COMMAND_DO(ExportUsbSettings(pNode));
    COMMAND_DO(ExportWebSettings(pNode));
    COMMAND_DO(ExportWirelessLanSettings(pNode));

    COMMAND_DO(ExportFirmwareDebugSettings(pNode, GetAvailableNames()));

    return true;
}

bool ImportSettings(const Node& node) NN_NOEXCEPT
{
    ::std::vector<::std::string> settingNames;
    COMMAND_DO(node.GetKeys(&settingNames));

    auto isFirmwareDebugSettingsImported = false;

    for (const ::std::string& settingName : settingNames)
    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(node.GetMember(&pNode, settingName));

        const Importer* const pImporter = GetImporter(settingName);

        if (pImporter != nullptr)
        {
            COMMAND_DO(pImporter->import(*pNode));
        }
        else if (Contains(GetAvailableNames(), settingName))
        {
            COMMAND_DO(ImportFirmwareDebugSettings(settingName, *pNode));

            isFirmwareDebugSettingsImported = true;
        }
        else
        {
            PrintErrorCode(ErrorCode::NodeKeyInvalid, settingName);

            return false;
        }
    }

    if (isFirmwareDebugSettingsImported)
    {
        COMMAND_DO(FlushFirmwareDebugSettings());
    }

    return true;
}


bool ResetSettings(const ::std::string& mode) NN_NOEXCEPT
{
    if (mode == "default")
    {
        ::nn::settings::fwdbg::ResetSettings(
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

        return true;
    }
    else
    {
        // リセットモードが見つからなかった場合は失敗
        PrintErrorCode(ErrorCode::CommandResetModeInvalid, mode);

        return false;
    }
}

bool DumpSettings(
    ::std::unique_ptr<::std::vector<int64_t>>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const size_t bufferSize = 6 * SettingsSectionSize;

    COMMAND_DO(AllocateBuffer(pOutValue, bufferSize / sizeof(int64_t)));

    auto buffer = reinterpret_cast<::nn::Bit8*>((*pOutValue)->data());

    auto pDatabase = reinterpret_cast<TemporaryDatabase*>(buffer);
    COMMAND_DO(DumpDebugSettings(pDatabase));

    ::nn::settings::fwdbg::ReadSettings(
        &buffer[SettingsSectionSize * 1], SettingsSectionSize,
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    ::nn::settings::fwdbg::ReadSettings(
        &buffer[SettingsSectionSize * 2], SettingsSectionSize,
        ::nn::settings::fwdbg::SettingsTarget_SystemSavePrivate);

    int32_t size = sizeof(int32_t);
    ::std::memcpy(&buffer[SettingsSectionSize * 3], &size, sizeof(size));
    ::std::memcpy(&buffer[SettingsSectionSize * 4], &size, sizeof(size));
    ::std::memcpy(&buffer[SettingsSectionSize * 5], &size, sizeof(size));

    COMMAND_DO(
        DumpFirmwareDebugSettings(
            &buffer[SettingsSectionSize * 3], SettingsSectionSize,
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug,
            GetAvailableNames()));

    COMMAND_DO(
        DumpFirmwareDebugSettings(
            &buffer[SettingsSectionSize * 4], SettingsSectionSize,
            ::nn::settings::fwdbg::SettingsTarget_SystemDataFirmwareDebug,
            GetAvailableNames()));

    COMMAND_DO(
        DumpFirmwareDebugSettings(
            &buffer[SettingsSectionSize * 5], SettingsSectionSize,
            ::nn::settings::fwdbg::
                SettingsTarget_SystemDataPlatformConfigration,
            GetAvailableNames()));

    return true;
}

namespace {

const Importer* GetImporter(const ::std::string& name) NN_NOEXCEPT
{
    static const Importer Importers[] =
    {
        { IsSettingNameAccountSettings, ImportAccountSettings },
        { IsSettingNameAudioSettings, ImportAudioSettings },
        { IsSettingNameBacklightSettings, ImportBacklightSettings },
        { IsSettingNameCaptureSettings, ImportCaptureSettings },
        { IsSettingNameClockSettings, ImportClockSettings },
        { IsSettingNameDataDeletionSettings, ImportDataDeletionSettings },
        { IsSettingNameDebugSettings, ImportDebugSettings },
        { IsSettingNameEulaVersions, ImportEulaVersions },
        { IsSettingNameHidSettings, ImportHidSettings },
        { IsSettingNameLanguageSettings, ImportLanguageSettings },
        { IsSettingNameLdnSettings, ImportLdnSettings },
        { IsSettingNameNetworkSettings, ImportNetworkSettings },
        { IsSettingNameNetworkServiceSettings, ImportNetworkServiceSettings },
        { IsSettingNameNfcSettings, ImportNfcSettings },
        { IsSettingNameNewsSettings, ImportNewsSettings },
        { IsSettingNameNotificationSettings, ImportNotificationSettings },
        { IsSettingNamePcvSettings, ImportPcvSettings },
        { IsSettingNamePushNotificationSettings,
          ImportPushNotificationSettings },
        { IsSettingNameAccountNotificationSettings,
          ImportAccountNotificationSettings },
        { IsSettingNameAccountOnlineStorageSettings,
          ImportAccountOnlineStorageSettings },
        { IsSettingNameSleepSettings, ImportSleepSettings },
        { IsSettingNameSystemApplicationSettings,
          ImportSystemApplicationSettings },
        { IsSettingNameTimeZoneSettings, ImportTimeZoneSettings },
        { IsSettingNameTvSettings, ImportTvSettings },
        { IsSettingNameUsbSettings, ImportUsbSettings },
        { IsSettingNameWebSettings, ImportWebSettings },
        { IsSettingNameWirelessLanSettings, ImportWirelessLanSettings },
    };

    for (const Importer& importer : Importers)
    {
        if (importer.supports(name))
        {
            return &importer;
        }
    }

    return nullptr;
}

const ::std::vector<::std::string>& GetAvailableNames() NN_NOEXCEPT
{
    static const ::std::vector<::std::string>& s_Names{
        "snap_shot_dump",
    };

    return s_Names;
}

} // namespace
