﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Audio.h>

#include "SettingsManager_Audio.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace nn { namespace settings { namespace system {

bool GetHeadphoneVolumeUpdateFlag() NN_NOEXCEPT;

void SetHeadphoneVolumeUpdateFlag(bool value) NN_NOEXCEPT;

}}} // namespace nn::settings::system

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameAudioSettings = "audio_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< スピーカがミュート状態か否かを表すフラグのキー
const char* const KeySpeakerMuteFlag = "speaker_mute_flag";

//!< スピーカの音量値のキー
const char* const KeySpeakerVolume = "speaker_volume";

//!< ヘッドフォンがミュート状態か否かを表すフラグのキー
const char* const KeyHeadphoneMuteFlag = "headphone_mute_flag";

//!< ヘッドフォンが出力レベルアップ状態か否かを表すフラグのキー
const char* const KeyHeadphoneHighPowerFlag = "headphone_high_power_flag";

//!< ヘッドフォンの音量値のキー
const char* const KeyHeadphoneVolume = "headphone_volume";

//!< USB オーディオがミュート状態か否かを表すフラグのキー
const char* const KeyUsbMuteFlag = "usb_mute_flag";

//!< USB オーディオの音量値のキー
const char* const KeyUsbVolume = "usb_volume";

//!< HDMI のオーディオ出力設定のキー
const char* const KeyHdmiOutputMode = "hdmi_output_mode";

//!< スピーカのオーディオ出力設定のキー
const char* const KeySpeakerOutputMode = "speaker_output_mode";

//!< ヘッドフォンのオーディオ出力設定のキー
const char* const KeyHeadphoneOutputMode = "headphone_output_mode";

//!< USB オーディオのオーディオ出力設定のキー
const char* const KeyUsbOutputMode = "usb_output_mode";

//!< 1 ch を表す本体オーディオ出力設定
const char* const AudioOutputMode1ch = "1ch";

//!< 2 ch を表す本体オーディオ出力設定
const char* const AudioOutputMode2ch = "2ch";

//!< 5.1 ch を表す本体オーディオ出力設定
const char* const AudioOutputMode51ch = "5.1ch";

//!< 5.1 ch (強制) を表す本体オーディオ出力設定
const char* const AudioOutputMode51chForced = "5.1chForced";

//!< 7.1 ch を表す本体オーディオ出力設定
const char* const AudioOutputMode71ch = "7.1ch";

//!< ヘッドフォン抜け時の強制ミュート設定のキー
const char* const KeyForceMuteFlag = "force_mute_flag";

//!< ヘッドフォン音量の警告表示回数のキー
const char* const KeyHeadphoneVolumeWarningCount =
    "headphone_volume_warning_count";

//!< ヘッドフォン音量の 3.0 NUP 時のレンジ更新が適応済みか否かを表すフラグのキー
const char* const HeadphoneVolumeUpdateFlag = "headphone_volume_update_flag";

//!< 本体音量設定がミュート状態か否かを表すフラグをエクスポートします。
bool ExportAudioVolumeMuteFlag(
    Node* pDictNode,
    const ::nn::settings::system::AudioVolume& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto currentValueNode = Node::CreateBooleanNode();

    COMMAND_DO(
        currentValueNode.SetValue(
            settings.flags.Test<
                ::nn::settings::system::AudioVolumeFlag::Mute>()));

    COMMAND_DO(
        pDictNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));

    auto defaultValueNode = Node::CreateBooleanNode();

    COMMAND_DO(defaultValueNode.SetValue(false));

    COMMAND_DO(
        pDictNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));

    return true;
}

//!< 本体音量設定が出力レベルアップ状態か否かを表すフラグをエクスポートします。
bool ExportAudioVolumeHighPowerFlag(
    Node* pDictNode,
    const ::nn::settings::system::AudioVolume& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto currentValueNode = Node::CreateBooleanNode();

    COMMAND_DO(
        currentValueNode.SetValue(
            settings.flags.Test<
                ::nn::settings::system::AudioVolumeFlag::HighPower>()));

    COMMAND_DO(
        pDictNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));

    auto defaultValueNode = Node::CreateBooleanNode();

    COMMAND_DO(defaultValueNode.SetValue(true));

    COMMAND_DO(
        pDictNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));

    return true;
}

//!< 本体音量設定の音量値をエクスポートします。
bool ExportAudioVolume(
    Node* pDictNode,
    const ::nn::settings::system::AudioVolume& settings,
    int32_t defaultVolume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto currentValueNode = Node::CreateIntegerNode();

    COMMAND_DO(currentValueNode.SetValue(settings.volume));

    COMMAND_DO(
        pDictNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));

    auto defaultValueNode = Node::CreateIntegerNode();

    COMMAND_DO(defaultValueNode.SetValue(defaultVolume));

    COMMAND_DO(
        pDictNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));

    return true;
}

//!< スピーカの音量設定をエクスポートします。
bool ExportSpeakerAudioVolume(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::GetAudioVolume(
        &settings, ::nn::settings::system::AudioVolumeTarget_Speaker);

    auto muteFlagNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySpeakerMuteFlag);

        COMMAND_DO(ExportAudioVolumeMuteFlag(&muteFlagNode, settings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySpeakerMuteFlag, ::std::move(muteFlagNode)));

    auto volumeNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySpeakerVolume);

        COMMAND_DO(ExportAudioVolume(&volumeNode, settings, 10));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySpeakerVolume, ::std::move(volumeNode)));

    return true;
}

//!< ヘッドフォンの音量設定をエクスポートします。
bool ExportHeadphoneAudioVolume(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::GetAudioVolume(
        &settings, ::nn::settings::system::AudioVolumeTarget_Headphone);

    auto muteFlagNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadphoneMuteFlag);

        COMMAND_DO(ExportAudioVolumeMuteFlag(&muteFlagNode, settings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyHeadphoneMuteFlag, ::std::move(muteFlagNode)));

    auto highPowerFlagNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadphoneHighPowerFlag);

        COMMAND_DO(
            ExportAudioVolumeHighPowerFlag(&highPowerFlagNode, settings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyHeadphoneHighPowerFlag, ::std::move(highPowerFlagNode)));

    auto volumeNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadphoneVolume);

        COMMAND_DO(ExportAudioVolume(&volumeNode, settings, 7));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyHeadphoneVolume, ::std::move(volumeNode)));

    return true;
}

//!< USB オーディオの音量設定をエクスポートします。
bool ExportUsbAudioVolume(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::settings::system::AudioVolume settings = {};

    ::nn::settings::system::GetAudioVolume(
        &settings, ::nn::settings::system::AudioVolumeTarget_Usb);

    auto muteFlagNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUsbMuteFlag);

        COMMAND_DO(ExportAudioVolumeMuteFlag(&muteFlagNode, settings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyUsbMuteFlag, ::std::move(muteFlagNode)));

    auto volumeNode = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUsbVolume);

        COMMAND_DO(ExportAudioVolume(&volumeNode, settings, 8));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyUsbVolume, ::std::move(volumeNode)));

    return true;
}

//!< スピーカの音量設定をインポートします。
bool ImportSpeakerAudioVolume(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const ::std::vector<::std::string> needles = {
        KeySpeakerMuteFlag,
        KeySpeakerVolume,
    };

    ::std::vector<::std::string> missingKeys;

    COMMAND_DO(GetMissingKeys(&missingKeys, keys, needles));

    if (missingKeys.size() == needles.size())
    {
        return true;
    }
    else if (missingKeys.size() > 0)
    {
        PrintErrorCode(
            ErrorCode::NodeKeyMissing, NameScope(missingKeys[0]).Get());

        return false;
    }

    ::nn::settings::system::AudioVolume settings = {};

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeySpeakerMuteFlag));

        NameScope nameScope(KeySpeakerMuteFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto muteFlag = bool();

        COMMAND_DO(pCurrentValueNode->GetValue(&muteFlag));

        settings.flags.Set<
            ::nn::settings::system::AudioVolumeFlag::Mute>(muteFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeySpeakerVolume));

        NameScope nameScope(KeySpeakerVolume);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto volume = int();

        COMMAND_DO(pCurrentValueNode->GetValue(&volume));

        settings.volume = static_cast<int8_t>(volume);
    }

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Speaker);

    return true;
}

//!< ヘッドフォンの音量設定をインポートします。
bool ImportHeadphoneAudioVolume(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const ::std::vector<::std::string> needles = {
        KeyHeadphoneMuteFlag,
        KeyHeadphoneHighPowerFlag,
        KeyHeadphoneVolume,
    };

    ::std::vector<::std::string> missingKeys;

    COMMAND_DO(GetMissingKeys(&missingKeys, keys, needles));

    if (missingKeys.size() == needles.size())
    {
        return true;
    }
    else if (missingKeys.size() > 0)
    {
        PrintErrorCode(
            ErrorCode::NodeKeyMissing, NameScope(missingKeys[0]).Get());

        return false;
    }

    ::nn::settings::system::AudioVolume settings = {};

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyHeadphoneMuteFlag));

        NameScope nameScope(KeyHeadphoneMuteFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto muteFlag = bool();

        COMMAND_DO(pCurrentValueNode->GetValue(&muteFlag));

        settings.flags.Set<
            ::nn::settings::system::AudioVolumeFlag::Mute>(muteFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyHeadphoneHighPowerFlag));

        NameScope nameScope(KeyHeadphoneHighPowerFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto highPowerFlag = bool();

        COMMAND_DO(pCurrentValueNode->GetValue(&highPowerFlag));

        settings.flags.Set<
            ::nn::settings::system::AudioVolumeFlag::HighPower>(highPowerFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyHeadphoneVolume));

        NameScope nameScope(KeyHeadphoneVolume);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto volume = int();

        COMMAND_DO(pCurrentValueNode->GetValue(&volume));

        settings.volume = static_cast<int8_t>(volume);
    }

    ::nn::settings::system::SetAudioVolume(
        settings,
        ::nn::settings::system::AudioVolumeTarget_Headphone);

    return true;
}

//!< USB オーディオの音量設定をインポートします。
bool ImportUsbAudioVolume(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const ::std::vector<::std::string> needles = {
        KeyUsbMuteFlag,
        KeyUsbVolume,
    };

    ::std::vector<::std::string> missingKeys;

    COMMAND_DO(GetMissingKeys(&missingKeys, keys, needles));

    if (missingKeys.size() == needles.size())
    {
        return true;
    }
    else if (missingKeys.size() > 0)
    {
        PrintErrorCode(
            ErrorCode::NodeKeyMissing, NameScope(missingKeys[0]).Get());

        return false;
    }

    ::nn::settings::system::AudioVolume settings = {};

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyUsbMuteFlag));

        NameScope nameScope(KeyUsbMuteFlag);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto muteFlag = bool();

        COMMAND_DO(pCurrentValueNode->GetValue(&muteFlag));

        settings.flags.Set<
            ::nn::settings::system::AudioVolumeFlag::Mute>(muteFlag);
    }

    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, KeyUsbVolume));

        NameScope nameScope(KeyUsbVolume);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

        auto volume = int();

        COMMAND_DO(pCurrentValueNode->GetValue(&volume));

        settings.volume = static_cast<int8_t>(volume);
    }

    ::nn::settings::system::SetAudioVolume(
        settings, ::nn::settings::system::AudioVolumeTarget_Usb);

    return true;
}

//!< 本体オーディオ出力設定をエクスポートします。
bool ExportAudioOutputMode(
    Node* pDictNode,
    ::nn::settings::system::AudioOutputModeTarget target) NN_NOEXCEPT
{
    auto currentValueNode = Node::CreateStringNode();

    switch (::nn::settings::system::GetAudioOutputMode(target))
    {
    case ::nn::settings::system::AudioOutputMode_1ch:
        COMMAND_DO(currentValueNode.SetValue(AudioOutputMode1ch));
        break;

    case ::nn::settings::system::AudioOutputMode_2ch:
        COMMAND_DO(currentValueNode.SetValue(AudioOutputMode2ch));
        break;

    case ::nn::settings::system::AudioOutputMode_5_1ch:
        COMMAND_DO(currentValueNode.SetValue(AudioOutputMode51ch));
        break;

    case ::nn::settings::system::AudioOutputMode_5_1ch_Forced:
        COMMAND_DO(currentValueNode.SetValue(AudioOutputMode51chForced));
        break;

    case ::nn::settings::system::AudioOutputMode_7_1ch:
        COMMAND_DO(currentValueNode.SetValue(AudioOutputMode71ch));
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));

    auto defaultValueNode = Node::CreateStringNode();

    COMMAND_DO(defaultValueNode.SetValue(AudioOutputMode2ch));

    COMMAND_DO(
        pDictNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));

    auto choicesValueNode = Node::CreateArrayNode();

    const char* const choices[] =
    {
        AudioOutputMode1ch,
        AudioOutputMode2ch,
        AudioOutputMode51ch,
        AudioOutputMode51chForced,
        AudioOutputMode71ch,
    };

    for (const char* const choice : choices)
    {
        auto choiceValueNode = Node::CreateStringNode();

        COMMAND_DO(choiceValueNode.SetValue(choice));

        COMMAND_DO(
            choicesValueNode.AppendElement(::std::move(choiceValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyChoices, ::std::move(choicesValueNode)));

    return true;
}

//!< HDMI のオーディオ出力設定をエクスポートします。
bool ExportHdmiOutputMode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHdmiOutputMode);

        COMMAND_DO(
            ExportAudioOutputMode(
                &node,
                ::nn::settings::system::AudioOutputModeTarget_Hdmi));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyHdmiOutputMode, ::std::move(node)));

    return true;
}

//!< スピーカのオーディオ出力設定をエクスポートします。
bool ExportSpeakerOutputMode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySpeakerOutputMode);

        COMMAND_DO(
            ExportAudioOutputMode(
                &node,
                ::nn::settings::system::AudioOutputModeTarget_Speaker));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySpeakerOutputMode, ::std::move(node)));

    return true;
}

//!< ヘッドフォンのオーディオ出力設定をエクスポートします。
bool ExportHeadphoneOutputMode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadphoneOutputMode);

        COMMAND_DO(
            ExportAudioOutputMode(
                &node,
                ::nn::settings::system::AudioOutputModeTarget_Headphone));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyHeadphoneOutputMode, ::std::move(node)));

    return true;
}

//!< USB オーディオのオーディオ出力設定をエクスポートします。
bool ExportUsbOutputMode(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUsbOutputMode);

        COMMAND_DO(
            ExportAudioOutputMode(
                &node, ::nn::settings::system::AudioOutputModeTarget_Usb));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyUsbOutputMode, ::std::move(node)));

    return true;
}

//!< 本体オーディオ出力設定をインポートします。
bool ImportAudioOutputMode(
    const Node& dictNode,
    ::nn::settings::system::AudioOutputModeTarget target) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(dictNode.GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string outputMode;
    COMMAND_DO(pCurrentValueNode->GetValue(&outputMode));

    if (outputMode == AudioOutputMode1ch)
    {
        ::nn::settings::system::SetAudioOutputMode(
            ::nn::settings::system::AudioOutputMode_1ch, target);

        return true;
    }

    if (outputMode == AudioOutputMode2ch)
    {
        ::nn::settings::system::SetAudioOutputMode(
            ::nn::settings::system::AudioOutputMode_2ch, target);

        return true;
    }

    if (outputMode == AudioOutputMode51ch)
    {
        ::nn::settings::system::SetAudioOutputMode(
            ::nn::settings::system::AudioOutputMode_5_1ch, target);

        return true;
    }

    if (outputMode == AudioOutputMode51chForced)
    {
        ::nn::settings::system::SetAudioOutputMode(
            ::nn::settings::system::AudioOutputMode_5_1ch_Forced, target);

        return true;
    }

    if (outputMode == AudioOutputMode71ch)
    {
        ::nn::settings::system::SetAudioOutputMode(
            ::nn::settings::system::AudioOutputMode_7_1ch, target);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, NameScope().Get(), outputMode);

    return false;
}

//!< HDMI のオーディオ出力設定をインポートします。
bool ImportHdmiOutputMode(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyHdmiOutputMode;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    COMMAND_DO(
        ImportAudioOutputMode(
            *pNode, ::nn::settings::system::AudioOutputModeTarget_Hdmi));

    return true;
}

//!< スピーカのオーディオ出力設定をインポートします。
bool ImportSpeakerOutputMode(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeySpeakerOutputMode;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    COMMAND_DO(
        ImportAudioOutputMode(
            *pNode, ::nn::settings::system::AudioOutputModeTarget_Speaker));

    return true;
}

//!< ヘッドフォンのオーディオ出力設定をインポートします。
bool ImportHeadphoneOutputMode(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyHeadphoneOutputMode;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    COMMAND_DO(
        ImportAudioOutputMode(
            *pNode, ::nn::settings::system::AudioOutputModeTarget_Headphone));

    return true;
}

//!< USB オーディオのオーディオ出力設定をインポートします。
bool ImportUsbOutputMode(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyUsbOutputMode;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    COMMAND_DO(
        ImportAudioOutputMode(
            *pNode, ::nn::settings::system::AudioOutputModeTarget_Usb));

    return true;
}

//!< ヘッドフォン抜け時の強制ミュート設定をエクスポートします。
bool ExportForceMuteFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyForceMuteFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::IsForceMuteOnHeadphoneRemoved()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyForceMuteFlag, ::std::move(node)));

    return true;
}

//!< ヘッドフォン抜け時の強制ミュート設定をインポートします。
bool ImportForceMuteFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyForceMuteFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto forceMuteFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&forceMuteFlag));

    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(forceMuteFlag);

    return true;
}

//!< ヘッドフォン音量の警告表示回数をエクスポートします。
int32_t ExportHeadphoneVolumeWarningCount(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyHeadphoneVolumeWarningCount);

        auto currentValueNode = Node::CreateIntegerNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::
                    GetHeadphoneVolumeWarningDisplayedEventCount()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateIntegerNode();

        COMMAND_DO(defaultValueNode.SetValue(0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyHeadphoneVolumeWarningCount, ::std::move(node)));

    return true;
}

//!< ヘッドフォン音量の警告表示回数をインポートします。
int32_t ImportHeadphoneVolumeWarningCount(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = KeyHeadphoneVolumeWarningCount;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto count = int32_t();
    COMMAND_DO(pCurrentValueNode->GetValue(&count));

    ::nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(count);

    return true;
}

//!< ヘッドフォン音量の 3.0 NUP 時のレンジ更新が適応済みか否かを表すフラグをエクスポートします。
bool ExportHeadphoneVolumeUpdateFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(HeadphoneVolumeUpdateFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::GetHeadphoneVolumeUpdateFlag()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(HeadphoneVolumeUpdateFlag, ::std::move(node)));

    return true;
}

//!< ヘッドフォン音量の 3.0 NUP 時のレンジ更新が適応済みか否かを表すフラグをインポートします。
bool ImportHeadphoneVolumeUpdateFlag(
    const Node& dictNode, const ::std::vector<::std::string>& keys) NN_NOEXCEPT
{
    const char* const key = HeadphoneVolumeUpdateFlag;

    if (::std::find(keys.begin(), keys.end(), key) == keys.end())
    {
        return true;
    }

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, key));

    NameScope nameScope(key);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto updateFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&updateFlag));

    ::nn::settings::system::SetHeadphoneVolumeUpdateFlag(updateFlag);

    return true;
}

} // namespace

bool IsSettingNameAudioSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameAudioSettings);
}

bool ExportAudioSettings(Node* pNode) NN_NOEXCEPT
{
    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameAudioSettings);

        COMMAND_DO(ExportSpeakerAudioVolume(&node));

        COMMAND_DO(ExportHeadphoneAudioVolume(&node));

        COMMAND_DO(ExportUsbAudioVolume(&node));

        COMMAND_DO(ExportHdmiOutputMode(&node));

        COMMAND_DO(ExportSpeakerOutputMode(&node));

        COMMAND_DO(ExportHeadphoneOutputMode(&node));

        COMMAND_DO(ExportUsbOutputMode(&node));

        COMMAND_DO(ExportForceMuteFlag(&node));

        COMMAND_DO(ExportHeadphoneVolumeWarningCount(&node));

        COMMAND_DO(ExportHeadphoneVolumeUpdateFlag(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameAudioSettings, ::std::move(node)));

    return true;
}

bool ImportAudioSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameAudioSettings);

    ::std::vector<::std::string> keys;

    COMMAND_DO(node.GetKeys(&keys));

    COMMAND_DO(ImportSpeakerAudioVolume(node, keys));

    COMMAND_DO(ImportHeadphoneAudioVolume(node, keys));

    COMMAND_DO(ImportUsbAudioVolume(node, keys));

    COMMAND_DO(ImportHdmiOutputMode(node, keys));

    COMMAND_DO(ImportSpeakerOutputMode(node, keys));

    COMMAND_DO(ImportHeadphoneOutputMode(node, keys));

    COMMAND_DO(ImportUsbOutputMode(node, keys));

    COMMAND_DO(ImportForceMuteFlag(node, keys));

    COMMAND_DO(ImportHeadphoneVolumeWarningCount(node, keys));

    COMMAND_DO(ImportHeadphoneVolumeUpdateFlag(node, keys));

    return true;
}
