﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cctype>
#include <sstream>
#include <string>
#include <vector>

#include <nn/crypto.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/fatal/fatal_ApiPrivate.h>
#include <nn/fatalsrv/fatalsrv_CpuContext.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/erpt/erpt_TypesPrivate.h>
#include <nn/erpt/server/erpt_Keys.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/hid/system/hid_PlayReport.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_CalendarAdditionalInfo.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "../DevMenuCommand_Common.h"
#include "../DevMenuCommand_Log.h"
#include "../DevMenuCommand_ProgramIdNameMap.h"
#include "DevMenuCommand_ErrorDump.h"

#if defined(_USE_MATH_DEFINES)
// DeMenu が _USE_MATH_DEFINES を定義するため、nn/msgpack.h の include によって DevMenuCommand in DevMenu ビルド時に重複定義になってしまうことの W/A
#undef _USE_MATH_DEFINES
#endif

#include <nn/msgpack.h>
#ifdef NN_BUILD_CONFIG_OS_WIN
    #include <nn/nn_Windows.h> // msgpack.h が Windows.h を include していることに対するワークアラウンド
#endif

using namespace nn;

namespace errors {

namespace
{
    class Writer
    {
    private:
        util::optional<fs::FileHandle> m_FileHandle;
        int64_t m_FileOffset;
        size_t m_FileWriteCachedBufferCount;
        char m_FileWriteCacheBuffer[erpt::MaxArrayBufferLength + 1];
    public:
        void BeginDumpToFile(fs::FileHandle handle, int64_t offset)
        {
            m_FileHandle = handle;
            m_FileOffset = offset;
            m_FileWriteCachedBufferCount = 0;
            m_FileWriteCacheBuffer[0] = '\0';
        }
        Result EndDumpToFile()
        {
            if( m_FileHandle && m_FileWriteCachedBufferCount > 0 )
            {
                NN_RESULT_DO(fs::WriteFile(*m_FileHandle, m_FileOffset, m_FileWriteCacheBuffer, m_FileWriteCachedBufferCount, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
            }
            m_FileHandle = nullptr;
            m_FileOffset = 0;
            m_FileWriteCachedBufferCount = 0;
            m_FileWriteCacheBuffer[0] = '\0';
            NN_RESULT_SUCCESS;
        }
        Result Dump(const char* format, std::va_list vaList)
        {
            if( m_FileHandle )
            {
                static char temp[erpt::MaxArrayBufferLength + 1];
                int l = util::VSNPrintf(temp, sizeof(temp), format, vaList);
                NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(temp)));
                NN_UNUSED(l);
                if( m_FileWriteCachedBufferCount + util::Strnlen(temp, static_cast<int>(sizeof(temp))) >= sizeof(m_FileWriteCacheBuffer) - 1)
                {
                    NN_RESULT_DO(fs::WriteFile(*m_FileHandle, m_FileOffset, m_FileWriteCacheBuffer, m_FileWriteCachedBufferCount, fs::WriteOption::MakeValue(0)));
                    m_FileOffset += m_FileWriteCachedBufferCount;
                    m_FileWriteCachedBufferCount = 0;
                    m_FileWriteCacheBuffer[0] = '\0';
                }
                std::strncat(m_FileWriteCacheBuffer + m_FileWriteCachedBufferCount, temp, l);
                m_FileWriteCachedBufferCount += util::Strnlen(temp, static_cast<int>(sizeof(temp)));
                NN_RESULT_SUCCESS;
            }
            else
            {
                DEVMENUCOMMAND_VLOG(format, vaList);
                NN_RESULT_SUCCESS;
            }
        }
        Result Dump(const char* format, ...)
        {
            va_list vaList;
            va_start(vaList, format);
            NN_UTIL_SCOPE_EXIT{ va_end(vaList); };
            return Dump(format, vaList);
        }

    } g_Writer;

    bool IsReportMetaDataZeroCleared(const erpt::ReportMetaData& reportMetaData)
    {
        bool allZero = true;
        for( size_t i = 0; i < sizeof(erpt::ReportMetaData); i++ )
        {
            if( reportMetaData.userData[i] != 0 )
            {
                allZero = false;
                break;
            }
        }
        return allZero;
    }

    typedef void(*DumperFunction)(const char* format, ...);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    const char* Reg64Names[] =
    {
        NN_FATALSRV_AARCH64_VALUES(NN_FATALSRV_GENERATE_STRING)
    };
    const char* Reg32Names[] =
    {
        NN_FATALSRV_AARCH32_VALUES(NN_FATALSRV_GENERATE_STRING)
    };
#else
    const char* Reg64Names[] =
    {
        ""
    };
    const char* Reg32Names[] =
    {
        ""
    };
#endif

    template<typename RegisterType>
    class RegisterDumper
    {
    public:
        RegisterDumper(const char** names, const char* format) :
            m_RegistersSet(false), m_FlagsSet(false), m_pNames(&names[0]), m_pFormat(format)
        {}
        void SetRegisters(const void* p, size_t size)
        {
            m_Registers.Init(p, size);
            m_RegistersSet = true;
        }
        void SetFlags(const void* p, size_t size)
        {
            m_Flags.Init(p, size);
            m_FlagsSet = true;
        }
        void Dump(const char* title)
        {
            if (!(m_RegistersSet && m_FlagsSet))
            {
                return;
            }

            RegisterType flagsRaw;
            m_Flags.GetUint(&flagsRaw);
            util::BitFlagSet<sizeof(RegisterType) * 8> flags;
            flags._storage[0] = flagsRaw;
            if (flags.IsAllOff())
            {
                return;
            }

            g_Writer.Dump("%s:\n", title);

            uint32_t count;
            m_Registers.GetArrayCount(&count);

            for (uint32_t i = 0; i < count; ++i)
            {
                if (flags.Test(i))
                {
                    RegisterType value;
                    m_Registers.At(i).GetUint(&value);
                    g_Writer.Dump(m_pFormat, m_pNames[i], value);
                }
            }
        }
    private:
        msgpack::MpWalker m_Registers;
        msgpack::MpWalker m_Flags;
        bool m_RegistersSet;
        bool m_FlagsSet;
        const char** m_pNames;
        const char* m_pFormat;
    };

    template <typename AddrType>
    class StacktraceDumper
    {
    public:
        StacktraceDumper()
        {
        }
        static const char* GetBaseAddrKey()
        {
            static char key[sizeof("ProgramMappedAddr00")];
            util::SNPrintf(key, sizeof(key), "ProgramMappedAddr%d", AddrBitWidth);
            return key;
        }
        static const char* GetStacktraceKey()
        {
            static char key[sizeof("StackBacktrace00")];
            util::SNPrintf(key, sizeof(key), "StackBacktrace%d", AddrBitWidth);
            return key;
        }
        void SetBaseAddr(const msgpack::MpWalker& value)
        {
            m_BaseAddr.emplace();
            value.GetUint(&m_BaseAddr.value());
        }
        void SetStackBacktrace(const msgpack::MpWalker& value)
        {
            m_StackBacktrace.emplace();
            m_StackBacktrace->Init(value.GetPtr(), value.GetSize());
        }
        void Dump() const
        {
            const int AddrHexFormatWidth = AddrBitWidth / 8 * 2;

            if( m_BaseAddr )
            {
                g_Writer.Dump("%s: 0x%0*llx\n", GetBaseAddrKey(), AddrHexFormatWidth, static_cast<Bit64>(*m_BaseAddr));
            }
            if( m_StackBacktrace )
            {
                uint32_t count;
                m_StackBacktrace->GetArrayCount(&count);

                g_Writer.Dump("%s: (Array : Count = %u)\n", GetStacktraceKey(), count);

                for( uint32_t i = 0; i < count; ++i )
                {
                    AddrType addr = 0;
                    m_StackBacktrace->At(i).GetUint(&addr);
                    if( m_BaseAddr )
                    {
                        g_Writer.Dump("    [%2d]: 0x%0*llx (%9llx)\n", i, AddrHexFormatWidth, static_cast<Bit64>(addr), static_cast<Bit64>(addr - *m_BaseAddr));
                    }
                    else
                    {
                        g_Writer.Dump("    [%2d]: 0x%0*llx\n", i, AddrHexFormatWidth, static_cast<Bit64>(addr));
                    }
                }
            }
        }
    private:
        static const int AddrBitWidth = sizeof(AddrType) * 8;
        util::optional<AddrType> m_BaseAddr;
        util::optional<msgpack::MpWalker> m_StackBacktrace;
    };

    void TimestampDumper(const char* format, ...)
    {
        std::va_list vaList;
        int64_t timestamp;
        va_start(vaList, format);
        timestamp = va_arg(vaList, int64_t);
        va_end(vaList);

        if (timestamp == 0) // ネットワーク時計が無効な場合に 0 が設定されている。
        {
            g_Writer.Dump("(Invalid)\n");
            return;
        }

        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        nn::time::PosixTime posixTime;
        posixTime.value = timestamp;
        auto result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
        if (result.IsSuccess())
        {
            g_Writer.Dump("%04d-%02d-%02d %02d:%02d:%02d (%s)\n",
                calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
                calendarAdditionalInfo.timeZone.standardTimeName);
        }
        else
        {
            g_Writer.Dump("result: %08x\n", result.GetInnerValueForDebug());
            g_Writer.Dump("%lld (posix time)\n", timestamp);
        }
    }

    void ResultBacktraceDumper(const char* format, ...)
    {
        std::va_list vaList;
        uint32_t resultInnerValue;
        va_start(vaList, format);
        resultInnerValue = va_arg(vaList, uint32_t);
        va_end(vaList);

        auto result = nn::result::detail::ConstructResult(resultInnerValue);

        g_Writer.Dump("%08x (Module = %d, Description = %d)\n", resultInnerValue, result.GetModule(), result.GetDescription());
    }

    void StorageSpaceDumper(const char* format, ...)
    {
        std::va_list vaList;
        uint64_t space;
        va_start(vaList, format);
        space = va_arg(vaList, uint64_t);
        va_end(vaList);

        auto spaceInKiB = space / 1024.0;
        if( spaceInKiB < 1024 )
        {
            g_Writer.Dump("%.2f KiB (%llu = 0x%08x)\n", spaceInKiB, space, space);
            return;
        }
        auto spaceInMiB = spaceInKiB / 1024.0;
        if( spaceInMiB < 1024 )
        {
            g_Writer.Dump("%.2f MiB (%llu = 0x%08x)\n", spaceInMiB, space, space);
            return;
        }
        g_Writer.Dump("%.2f GiB (%llu = 0x%016llx)\n", spaceInMiB / 1024.0, space, space);
    }

    void AbortTypeDumper(const char* format, ...)
    {
        std::va_list vaList;
        uint32_t v;
        va_start(vaList, format);
        v = va_arg(vaList, uint32_t);
        va_end(vaList);

        g_Writer.Dump("%s(%08x)\n", nn::fatal::GetExceptionName(v), v);
    }

    const char* GetProgramName(Bit64 id)
    {
        NN_FUNCTION_LOCAL_STATIC(ProgramIdNameMap, map);
        NN_FUNCTION_LOCAL_STATIC(bool, s_IsMapInitialized, = map.Initialize().IsSuccess());
        if( s_IsMapInitialized )
        {
            return map.GetName(id);
        }
        return "Unknown";
    }

    void ProgramIdDumper(const char* format, ...)
    {
        std::va_list vaList;
        uint64_t v;
        va_start(vaList, format);
        v = va_arg(vaList, uint64_t);
        va_end(vaList);

        g_Writer.Dump("0x%016llx (%s)\n", v, GetProgramName(v));
    }

    void ProgramIdStrDumper(const char* format, ...)
    {
        std::va_list vaList;
        char* v;
        va_start(vaList, format);
        v = va_arg(vaList, char*);
        va_end(vaList);

        auto id = strtoul(v, NULL, 16);

        g_Writer.Dump("0x%s (%s)\n", v, GetProgramName(id));
    }

    void UInt32HexDumper(const char* format, ...)
    {
        std::va_list vaList;
        uint32_t v;
        va_start(vaList, format);
        v = va_arg(vaList, uint32_t);
        va_end(vaList);

        g_Writer.Dump("%08x\n", v);
    }

    void ErrorCodeDumper(const char* format, ...)
    {
        std::va_list vaList;
        const char* errorCodeString;
        va_start(vaList, format);
        errorCodeString = va_arg(vaList, const char*);
        va_end(vaList);

        if (strlen(errorCodeString) == 9) // "2XXX-YYYY".
        {
            uint32_t category = std::strtoul(errorCodeString, NULL, 10);
            uint32_t number = std::strtoul(errorCodeString + 5, NULL, 10);

            // NX の category は 2XXX というフォーマットで、XXX が Result の Module に対応。
            // category がそれ以上の値の場合は Result からの生成ではないエラーコードとみなす。
            if( category <= 2000 + result::detail::ResultTraits::ModuleEnd - 1 )
            {
                nn::err::ErrorCode errorCode = { category, number };
                auto result = nn::err::detail::ConvertErrorCodeToResult(errorCode);
                g_Writer.Dump("%s (result=%08x)\n", errorCodeString, result.GetInnerValueForDebug());
            }
            else
            {
                g_Writer.Dump("%s\n", errorCodeString);
            }

        }
        else // ApplicationErrorCode.
        {
            g_Writer.Dump("%s\n", errorCodeString);
        }
    }

    void TickValueDumper(const char* format, ...)
    {
        std::va_list vaList;
        int64_t v;
        va_start(vaList, format);
        v = va_arg(vaList, int64_t);
        va_end(vaList);

        g_Writer.Dump("%lld (equivalent to %lld seconds)\n", v, os::Tick(v).ToTimeSpan().GetSeconds());
    }

    const char* GetErrorReportSharePermissionString(settings::system::ErrorReportSharePermission permission)
    {
        switch( permission )
        {
        case settings::system::ErrorReportSharePermission_NotConfirmed: return "NotConfirmed";
        case settings::system::ErrorReportSharePermission_Granted:      return "Granted";
        case settings::system::ErrorReportSharePermission_Denied:       return "Denied";
        default:                                                        return "(unknown)";
        }
    }

    void ErrorReportSharePermissionDumper(const char* format, ...)
    {
        std::va_list vaList;
        int v;
        va_start(vaList, format);
        v = va_arg(vaList, int);
        va_end(vaList);

        auto permission = static_cast<nn::settings::system::ErrorReportSharePermission>(v);
        g_Writer.Dump("%s\n", GetErrorReportSharePermissionString(permission));
    }

    const char* GetControllerDeviceTypeString(hid::system::PlayReportDeviceType type)
    {
        switch( type )
        {
        case hid::system::PlayReportDeviceType_JoyConLeft:          return "Joy-Con (L)";
        case hid::system::PlayReportDeviceType_JoyConRight:         return "Joy-Con (R)";
        case hid::system::PlayReportDeviceType_SwitchProController: return "Switch Pro Controller";
        case hid::system::PlayReportDeviceType_UsbController:       return "USB Controller";
        case hid::system::PlayReportDeviceType_Unknown:             return "(unknown)";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void ControllerDeviceTypeDumper(const char* format, ...)
    {
        std::va_list vaList;
        int v;
        va_start(vaList, format);
        v = va_arg(vaList, int);
        va_end(vaList);

        g_Writer.Dump("%d : %s\n", v, GetControllerDeviceTypeString(static_cast<hid::system::PlayReportDeviceType>(v)));
    }

    const char* GetControllerInterfaceTypeString(hid::system::InterfaceType type)
    {
        switch( type )
        {
        case hid::system::InterfaceType_Bluetooth:  return "Bluetooth";
        case hid::system::InterfaceType_Rail:       return "Rail";
        case hid::system::InterfaceType_Usb:        return "USB";
        case hid::system::InterfaceType_Unknown:    return "(unknown)";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void ControllerIntrafceDumper(const char* format, ...)
    {
        std::va_list vaList;
        int v;
        va_start(vaList, format);
        v = va_arg(vaList, int);
        va_end(vaList);

        g_Writer.Dump("%d : %s\n", v, GetControllerInterfaceTypeString(static_cast<hid::system::InterfaceType>(v)));
    }

    const char* GetControllerStyleString(hid::system::PlayReportPlayStyle style)
    {
        switch( style )
        {
        case hid::system::PlayReportPlayStyle_SwitchProController:      return "Switch Pro Controller";
        case hid::system::PlayReportPlayStyle_Handheld:                 return "Handheld";
        case hid::system::PlayReportPlayStyle_JoyConDual:               return "Joy-Con Dual";
        case hid::system::PlayReportPlayStyle_JoyConLeftHorizontal:     return "Joy-Con (L) Horizontal";
        case hid::system::PlayReportPlayStyle_JoyConLeftVertical:       return "Joy-Con (L) Vertical";
        case hid::system::PlayReportPlayStyle_JoyConRightHorizontal:    return "Joy-Con (R) Horizontal";
        case hid::system::PlayReportPlayStyle_JoyConRightVertical:      return "Joy-Con (R) Vertical";
        case hid::system::PlayReportPlayStyle_Unknown:                  return "(unknown)";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void ControllerStyleDumper(const char* format, ...)
    {
        std::va_list vaList;
        int v;
        va_start(vaList, format);
        v = va_arg(vaList, int);
        va_end(vaList);

        g_Writer.Dump("%d : %s\n", v, GetControllerStyleString(static_cast<hid::system::PlayReportPlayStyle>(v)));
    }

    const char* GetEnumString(ns::StartupUserAccount value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::StartupUserAccount::None:                                       return "None";
        case ns::StartupUserAccount::Required:                                   return "Required";
        case ns::StartupUserAccount::RequiredWithNetworkServiceAccountAvailable: return "RequiredWithNetworkServiceAccountAvailable";
        default:                                                                 return "(unknown)";
        }
    }

    const char* GetEnumString(ns::Screenshot value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::Screenshot::Allow: return "Allow";
        case ns::Screenshot::Deny:  return "Deny";
        default:                    return "(unknown)";
        }
    }

    const char* GetEnumString(ns::VideoCapture value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::VideoCapture::Disable: return "Disable";
        case ns::VideoCapture::Manual:  return "Manual";
        case ns::VideoCapture::Enable:  return "Enable";
        default:                        return "(unknown)";
        }
    }

    const char* GetEnumString(ns::LogoType value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::LogoType::LicensedByNintendo: return "LicensedByNintendo";
        case ns::LogoType::ObsoletedLogoType:  return "ObsoletedLogoType";
        case ns::LogoType::Nintendo:           return "Nintendo";
        default:                               return "(unknown)";
        }
    }

    const char* GetEnumString(ns::DataLossConfirmation value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::DataLossConfirmation::None:     return "None";
        case ns::DataLossConfirmation::Required: return "Required";
        default:                                 return "(unknown)";
        }
    }

    const char* GetEnumString(ns::PlayLogPolicy value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::PlayLogPolicy::All:     return "All";
        case ns::PlayLogPolicy::LogOnly: return "LogOnly";
        case ns::PlayLogPolicy::None:    return "None";
        default:                         return "(unknown)";
        }
    }
    const char* GetEnumString(ns::LogoHandling value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::LogoHandling::Auto:   return "Auto";
        case ns::LogoHandling::Manual: return "Manual";
        default:                       return "(unknown)";
        }
    }
    const char* GetEnumString(ns::AddOnContentRegistrationType value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::AddOnContentRegistrationType::AllOnLaunch: return "AllOnLaunch";
        case ns::AddOnContentRegistrationType::OnDemand:    return "OnDemand";
        default:                                            return "(unknown)";
        }
    }
    const char* GetEnumString(ns::Hdcp value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::Hdcp::None:     return "None";
        case ns::Hdcp::Required: return "Required";
        default:                 return "(unknown)";
        }
    }

    const char* GetEnumString(ns::CrashReport value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::CrashReport::Deny:  return "Deny";
        case ns::CrashReport::Allow: return "Allow";
        default:                     return "(unknown)";
        }
    }

    const char* GetEnumString(ns::RuntimeAddOnContentInstall value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::RuntimeAddOnContentInstall::Deny:        return "Deny";
        case ns::RuntimeAddOnContentInstall::AllowAppend: return "AllowAppend";
        default:                                          return "(unknown)";
        }
    }

    const char* GetEnumString(ns::PlayLogQueryCapability value) NN_NOEXCEPT
    {
        switch( value )
        {
        case ns::PlayLogQueryCapability::None:      return "None";
        case ns::PlayLogQueryCapability::WhiteList: return "WhiteList";
        case ns::PlayLogQueryCapability::All:       return "All";
        default:                                    return "(unknown)";
        }
    }

    template <typename T>
    void EnumValueDumper(const char* format, ...)
    {
        std::va_list vaList;
        int v;
        va_start(vaList, format);
        v = va_arg(vaList, int);
        va_end(vaList);

        g_Writer.Dump("%s (%d)\n", GetEnumString(static_cast<T>(v)), v);
    }

    void DefaultDumper(const char* format, ...)
    {
        std::va_list vaList;
        va_start(vaList, format);
        g_Writer.Dump(format, vaList);
        va_end(vaList);
    }

    struct DumpLogic
    {
        std::string name;
        DumperFunction dump;
        bool printName;
    };

    DumpLogic g_DumpLogicForDumpAll[] = {
        { "ProgramId", ProgramIdStrDumper, true },
        { "ErrorCode", ErrorCodeDumper, true },
        { "OccurrenceTimestamp", TimestampDumper, true },
        { "OccurrenceTimestampNet", TimestampDumper, true },
        { "NANDFreeSpace", StorageSpaceDumper, true },
        { "NANDTotalSize", StorageSpaceDumper, true },
        { "SDCardFreeSpace", StorageSpaceDumper, true },
        { "SdCardTotalSize", StorageSpaceDumper, true },
        { "SdCardUserAreaSize", StorageSpaceDumper, true },
        { "SdCardProtectedAreaSize", StorageSpaceDumper, true },
        { "ResultBacktrace", ResultBacktraceDumper, true },
        { "ProgramMappedAddr32", UInt32HexDumper, true },
        { "AbortType", AbortTypeDumper, true },
        { "RunningAppletList", ProgramIdDumper, true },
        { "FocusedAppletHistory", ProgramIdDumper, true },
        { "OccurrenceTick", TickValueDumper, true },
        { "ErrorReportSharePermission", ErrorReportSharePermissionDumper, true },
        { "ControllerTypeList", ControllerDeviceTypeDumper, true },
        { "ControllerInterfaceList", ControllerIntrafceDumper, true },
        { "ControllerStyleList", ControllerStyleDumper, true },
        { "NifmErrorCode", ErrorCodeDumper, true },
        { "AcpStartupUserAccount", EnumValueDumper<ns::StartupUserAccount>, true },
        { "AcpRegistrationType", EnumValueDumper<ns::AddOnContentRegistrationType>, true },
        { "AcpScreenShot", EnumValueDumper<ns::Screenshot>, true },
        { "AcpVideoCapture", EnumValueDumper<ns::VideoCapture>, true },
        { "AcpDataLossConfirmation", EnumValueDumper<ns::DataLossConfirmation>, true },
        { "AcpPlayLogPolicy", EnumValueDumper<ns::PlayLogPolicy>, true },
        { "AcpUserAccountSaveDataSize", StorageSpaceDumper, true },
        { "AcpUserAccountSaveDataJournalSize", StorageSpaceDumper, true },
        { "AcpDeviceSaveDataSize", StorageSpaceDumper, true },
        { "AcpDeviceSaveDataJournalSize", StorageSpaceDumper, true },
        { "AcpBcatDeliveryCacheStorageSize", StorageSpaceDumper, true },
        { "AcpLogoType", EnumValueDumper<ns::LogoType>, true },
        { "AcpLogoHandling", EnumValueDumper<ns::LogoHandling>, true },
        { "AcpRuntimeAddOnContentInstall", EnumValueDumper<ns::RuntimeAddOnContentInstall>, true },
        { "AcpCrashReport", EnumValueDumper<ns::CrashReport>, true },
        { "AcpHdcp", EnumValueDumper<ns::Hdcp>, true },
        { "AcpUserAccountSaveDataSizeMax", StorageSpaceDumper, true },
        { "AcpUserAccountSaveDataJournalSizeMax", StorageSpaceDumper, true },
        { "AcpDeviceSaveDataSizeMax", StorageSpaceDumper, true },
        { "AcpDeviceSaveDataJournalSizeMax", StorageSpaceDumper, true },
        { "AcpTemporaryStorageSize", StorageSpaceDumper, true },
        { "AcpCacheStorageSize", StorageSpaceDumper, true },
        { "AcpCacheStorageJournalSize", StorageSpaceDumper, true },
        { "AcpCacheStorageDataAndJournalSizeMax", StorageSpaceDumper, true },
        { "AcpPlayLogQueryCapability", EnumValueDumper<ns::PlayLogQueryCapability>, true },
    };
    const int DumpLogicForDumpAllCount = sizeof(g_DumpLogicForDumpAll) / sizeof(DumpLogic);

    // INFO: https://github.com/msgpack/msgpack/blob/master/spec.md#formats-bin
    void DumpMessagePackValue(nn::msgpack::MpWalker& mpWalker, DumperFunction dumper)
    {
        auto pRaw = reinterpret_cast<const Bit8*>(mpWalker.GetPtr());
        auto valueType = pRaw[0];

        if ((valueType & 0xe0) == erpt::ValueTypeTag::Str ||
            valueType == erpt::ValueTypeTag::Str256 ||
            valueType == erpt::ValueTypeTag::Str16384)
        {
            const char* pStr;
            uint32_t size;
            mpWalker.GetString(&pStr, &size);
            std::unique_ptr<char[]> strBuffer(new char[size + 1]);
            std::memcpy(strBuffer.get(), pStr, size);
            strBuffer[size] = '\0';
            dumper("%s\n", strBuffer.get());
        }
        else if (valueType == erpt::ValueTypeTag::U8 || valueType == erpt::ValueTypeTag::U16 || valueType == erpt::ValueTypeTag::U32)
        {
            uint32_t v;
            mpWalker.GetUint(&v);
            dumper("%u\n", v);
        }
        else if (valueType == erpt::ValueTypeTag::I8 || valueType == erpt::ValueTypeTag::I16 || valueType == erpt::ValueTypeTag::I32)
        {
            int32_t  v;
            mpWalker.GetInt(&v);
            dumper("%d\n", v);
        }
        else if (valueType == erpt::ValueTypeTag::I64)
        {
            int64_t v;
            mpWalker.GetInt(&v);
            dumper("%lld\n", v);
        }
        else if (valueType == erpt::ValueTypeTag::U64)
        {
            uint64_t v;
            mpWalker.GetUint(&v);
            dumper("0x%016llx\n", v);
        }
        else if (valueType == erpt::ValueTypeTag::False)
        {
            dumper("False\n");
        }
        else if (valueType == erpt::ValueTypeTag::True)
        {
            dumper("True\n");
        }
        else if ((valueType & 0xf0) == erpt::ValueTypeTag::Array16)
        {
            // TORIAEZU : MpWalker では fixarray を GetArrayCount, At で解釈できないようなので Skip を使って多少無理やりに表示。
            uint32_t count (valueType & (0x0f));
            g_Writer.Dump("(Array : Count = %d)\n", count);
            for( uint32_t i = 0; i < count; i++ )
            {
                g_Writer.Dump("    [%2d] ", i);
                nn::msgpack::MpWalker arrayItemWalker(&pRaw[1], mpWalker.GetSize());
                for( uint32_t j = 0; j < i; j++ )
                {
                    arrayItemWalker = arrayItemWalker.Skip();
                }
                DumpMessagePackValue(arrayItemWalker, dumper);
            }
        }
        else if (valueType == erpt::ValueTypeTag::Array16384)
        {
            uint32_t count;
            mpWalker.GetArrayCount(&count);
            g_Writer.Dump("(Array : Count = %d)\n", count);
            for( uint32_t i = 0; i < count; i++ )
            {
                g_Writer.Dump("    [%2d] ", i);
                auto arrayItemWalker = mpWalker.At(i);
                DumpMessagePackValue(arrayItemWalker, dumper);
            }
        }
        else if (valueType == erpt::ValueTypeTag::Bin || valueType == erpt::ValueTypeTag::Bin16384 )
        {
            const void* binArray;
            uint32_t length;
            mpWalker.GetBinary(&binArray, &length);
            g_Writer.Dump("(BinaryArray : Count = %u)\n", length);
            if( dumper == DefaultDumper )
            {
                for( uint32_t i = 0; i < length; i += 16 )
                {
                    g_Writer.Dump("0x%04x | ", i);
                    for( uint32_t j = i; j < i + 16 && j < length; ++j )
                    {
                        g_Writer.Dump("%02x ", reinterpret_cast<const char*>(binArray)[j]);
                    }
                    g_Writer.Dump("\n");
                }
            }
            else
            {
                for( uint32_t i = 0; i < length; i++ )
                {
                    g_Writer.Dump("    [%2d] ", i);
                    dumper("", reinterpret_cast<const char*>(binArray)[i]);
                }
            }
        }
        else
        {
            g_Writer.Dump("* Unknown format * : valueType = 0x%02x\n", valueType);
            auto p = reinterpret_cast<const Bit8*>(mpWalker.GetPtr());
            for (size_t i = 0; i < mpWalker.GetSize(); i += 16)
            {
                for (size_t j = i; j < i + 16 && j < mpWalker.GetSize(); ++j)
                {
                    g_Writer.Dump("%02x ", p[j]);
                }
                g_Writer.Dump("\n");
            }
        }
    } // NOLINT(impl/function_size)

    bool DumpValue(const DumpLogic* logicTable, int logicCount, const char* keyBuffer, size_t keySize, msgpack::MpWalker& value)
    {
        for (int i = 0; i < logicCount; ++i)
        {
            auto& logic = logicTable[i];
            if (nn::util::Strncmp(keyBuffer, logic.name.c_str(), static_cast<int>(keySize)) == 0)
            {
                if (logic.printName)
                {
                    g_Writer.Dump("%s: ", keyBuffer);
                }
                DumpMessagePackValue(value, logic.dump);
                return true;
            }
        }
        return false;
    }

    bool IsEncryptedField(const char* key, size_t keyLength)
    {
        for( size_t fieldId = 0; fieldId < NN_ARRAY_SIZE(nn::erpt::srv::FieldString); fieldId++ )
        {
            if( util::Strncmp(key, nn::erpt::srv::FieldString[fieldId], static_cast<int>(keyLength)) == 0 )
            {
                return (nn::erpt::srv::FieldToFlagMap[fieldId] == nn::erpt::srv::FieldFlag::FieldFlag_Encrypt);
            }
        }
        return false;
    }

    // 開発機用のフィールド暗号鍵の秘密鍵成分
    const uint8_t g_ErrorReportFieldRsaPrivateExponentForDev[] =
    {
        0x00, 0xa7, 0xf6, 0xd8, 0x7e, 0x84, 0xa9, 0xec, 0xe1, 0x4e, 0xc8, 0x77, 0x3e, 0xf0, 0xdc,
        0x91, 0x06, 0xe7, 0x03, 0x4f, 0x89, 0xeb, 0xef, 0x19, 0xa6, 0x4e, 0xb5, 0x03, 0x1d, 0x35,
        0x45, 0x8f, 0xc2, 0xb6, 0x0a, 0x98, 0x02, 0xee, 0x27, 0x65, 0x05, 0xf0, 0x85, 0x73, 0x3e,
        0x45, 0xa7, 0x6e, 0xc6, 0xb7, 0x3b, 0x8e, 0x3a, 0xc8, 0x5f, 0x08, 0x3b, 0x8f, 0x09, 0x13,
        0x65, 0x4f, 0xd4, 0xe2, 0x89, 0xfc, 0x2a, 0xe7, 0x5b, 0xa5, 0x24, 0x0b, 0xd9, 0x02, 0xce,
        0x00, 0x64, 0xe3, 0x35, 0x97, 0x89, 0x1a, 0x46, 0xf5, 0xa2, 0xfc, 0x2a, 0x82, 0xc7, 0xa5,
        0x87, 0xbf, 0x4c, 0x90, 0x87, 0xdd, 0x29, 0x30, 0xa9, 0x88, 0xbc, 0xcc, 0x80, 0x19, 0x66,
        0xa6, 0xe9, 0x68, 0x95, 0xf0, 0x52, 0x76, 0x8c, 0xb7, 0xe0, 0x64, 0xa1, 0x08, 0xfb, 0xf5,
        0x66, 0x2d, 0x5e, 0xcb, 0x8b, 0x19, 0x6e, 0xad, 0x31, 0xd8, 0x0f, 0xf9, 0xfe, 0x13, 0xe2,
        0xfe, 0x37, 0xce, 0x99, 0x58, 0x61, 0xfb, 0x13, 0x27, 0x5f, 0xf9, 0xc3, 0x47, 0x20, 0x5a,
        0x8f, 0xd5, 0x21, 0x92, 0xab, 0x39, 0x29, 0xc5, 0xbd, 0xb5, 0x21, 0xff, 0x38, 0xda, 0x55,
        0x86, 0x25, 0x40, 0x8e, 0x81, 0xb1, 0x5b, 0x33, 0x89, 0x63, 0x1f, 0xf3, 0x0d, 0xfd, 0x31,
        0x9b, 0x7c, 0x57, 0x55, 0x2c, 0x5e, 0xde, 0xda, 0xa5, 0x08, 0x0f, 0x8b, 0xda, 0x91, 0x68,
        0x80, 0x2c, 0xc0, 0x95, 0x51, 0x79, 0x86, 0x89, 0x82, 0x3d, 0x3f, 0x69, 0xa0, 0xeb, 0x9d,
        0xab, 0x65, 0xb0, 0x8c, 0xaa, 0xe3, 0x2b, 0x6d, 0x28, 0xc3, 0xf9, 0xd2, 0xdb, 0x90, 0x28,
        0xf7, 0x66, 0xa8, 0x6c, 0x69, 0x6b, 0xed, 0x2a, 0xf7, 0xfe, 0xf2, 0x71, 0xf9, 0xb7, 0x28,
        0x62, 0x04, 0x85, 0x19, 0x45, 0xfc, 0x3f, 0xff, 0xee, 0xbd, 0x32, 0x97, 0xa7, 0x67, 0x14,
        0x6e, 0xa1
    };

    // EncryptedFieldHeader = erpt::srv::Cipher::Header (Programs/Eris/Sources/Libraries/erpt/server/erptsrv_Cipher.h)
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4200)
#endif
    struct EncryptedFieldHeader
    {
        uint32_t magic;
        uint32_t fieldType;
        uint32_t elementCount;
        uint32_t reserved;
        uint8_t  data[0];
    };
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

    bool IsZeroCleared(const void* buffer, size_t bufferLength)
    {
        for( uint32_t i = 0; i < bufferLength; i++ )
        {
            if( reinterpret_cast<const Bit8*>(buffer)[i] != 0u )
            {
                return false;
            }
        }
        return true;
    }

    bool GetCipherKey(Bit8 (&outAesKey)[16], Bit8 (&outAesIv)[16], uint8_t* pReportData, size_t reportDataSize)
    {
        nn::msgpack::MpWalker mpWalker;
        mpWalker.Init(pReportData, reportDataSize);
        const void* bin = nullptr;
        uint32_t size;
        mpWalker.Find("CipherKey").GetBinary(&bin, &size);
        if( bin == nullptr || size != 256u )
        {
            return false;
        }

        if( IsZeroCleared(bin, size) )
        {
            return false;
        }

        crypto::RsaOaepDecryptor<256, nn::crypto::Sha256Generator> rsa;
        rsa.Initialize(erpt::srv::PublicKeyModulusForDev, sizeof(erpt::srv::PublicKeyModulusForDev), g_ErrorReportFieldRsaPrivateExponentForDev, sizeof(g_ErrorReportFieldRsaPrivateExponentForDev));
        Bit8 plain[128];
        auto plainSize = rsa.Decrypt(plain, sizeof(plain), bin, size);

        // Decrypted CipherKey is 48 bytes : [16byte key | 16byte IV | 16byte numbers (They are sequential (0, 1, 2, 3...) if the report is created in Non-Release build FW. If not, They are random.)]
        if( plainSize != 48 )
        {
            g_Writer.Dump("Failed to decrypt CipherKey.\n");
            return false;
        }
#if 0
        for( size_t i = 32; i < 48; i++ )
        {
            if( plain[i] != i - 32 )
            {
                g_Writer.Dump("Decrypted CipherKey has unexpected value.\n");
                return false;
            }
        }
#endif
        std::memcpy(outAesKey, plain, 16);
        std::memcpy(outAesIv, plain + 16, 16);
        return true;
    }

    const char* GetFiledTypeString(erpt::FieldType fieldType)
    {
        switch( fieldType )
        {
        case erpt::FieldType_NumericU64: return "NumericU64";
        case erpt::FieldType_NumericU32: return "NumericU32";
        case erpt::FieldType_NumericI64: return "NumericI64";
        case erpt::FieldType_NumericI32: return "NumericI32";
        case erpt::FieldType_String:     return "String";
        case erpt::FieldType_U8Array:    return "U8Array";
        case erpt::FieldType_U32Array:   return "U32Array";
        case erpt::FieldType_U64Array:   return "U64Array";
        case erpt::FieldType_I32Array:   return "I32Array";
        case erpt::FieldType_I64Array:   return "I64Array";
        case erpt::FieldType_Bool:       return "Bool";
        case erpt::FieldType_NumericU16: return "NumericU16";
        case erpt::FieldType_NumericU8:  return "NumericU8";
        case erpt::FieldType_NumericI16: return "NumericI16";
        case erpt::FieldType_NumericI8:  return "NumericI8";
        default:
            return "(unknown)";
        }
    }

    void DumpEncryptedField(nn::msgpack::MpWalker& value, const Bit8 (&aesKey)[16], const Bit8 (&aesIv)[16])
    {
        const void* bin;
        uint32_t len;
        value.GetBinary(&bin, &len);
        if( bin != nullptr && len >= sizeof(EncryptedFieldHeader) )
        {
            auto pEncryptedFieldHeader = reinterpret_cast<const EncryptedFieldHeader*>(bin);
            NN_SDK_ASSERT_EQUAL(pEncryptedFieldHeader->magic, static_cast<uint32_t>('C' | 'R' << 8 | 'P' << 16 | 'T' << 24));
            g_Writer.Dump("    Encrypted Field : FieldType = %s, ElementCount = %u\n",
                GetFiledTypeString(static_cast<erpt::FieldType>(pEncryptedFieldHeader->fieldType)), pEncryptedFieldHeader->elementCount);
            auto encryptedLength = len - sizeof(EncryptedFieldHeader);

            std::unique_ptr<Bit8[]> plain(new Bit8[encryptedLength]);
            crypto::Aes128CtrDecryptor decryptor;
            auto plainSize = crypto::DecryptAes128Ctr(plain.get(), encryptedLength, aesKey, sizeof(aesKey), aesIv, sizeof(aesIv), pEncryptedFieldHeader->data, encryptedLength);
            NN_UNUSED(plainSize);
            switch( pEncryptedFieldHeader->fieldType )
            {
            case erpt::FieldType::FieldType_U8Array:
                for( uint32_t i = 0; i < pEncryptedFieldHeader->elementCount; i++ )
                {
                    g_Writer.Dump("    [%2u] %u\n", i, reinterpret_cast<uint8_t*>(plain.get())[i]);
                }
                break;
            case erpt::FieldType::FieldType_U32Array:
                for( uint32_t i = 0; i < pEncryptedFieldHeader->elementCount; i++ )
                {
                    g_Writer.Dump("    [%2u] %u\n", i, reinterpret_cast<uint32_t*>(plain.get())[i]);
                }
                break;
            case erpt::FieldType::FieldType_U64Array:
                for( uint32_t i = 0; i < pEncryptedFieldHeader->elementCount; i++ )
                {
                    g_Writer.Dump("    [%2u] %llu\n", i, reinterpret_cast<uint64_t*>(plain.get())[i]);
                }
                break;
            case erpt::FieldType::FieldType_I32Array:
                for( uint32_t i = 0; i < pEncryptedFieldHeader->elementCount; i++ )
                {
                    g_Writer.Dump("    [%2u] %d\n", i, reinterpret_cast<int32_t*>(plain.get())[i]);
                }
                break;
            case erpt::FieldType::FieldType_I64Array:
                for( uint32_t i = 0; i < pEncryptedFieldHeader->elementCount; i++ )
                {
                    g_Writer.Dump("    [%2u] %lld\n", i, reinterpret_cast<int64_t*>(plain.get())[i]);
                }
                break;
            case erpt::FieldType::FieldType_String:
                g_Writer.Dump("    %s\n", plain.get());
                break;
            default:
                g_Writer.Dump("Unexpected field type for encrypted data.\n");
                break;
            }
        }
    }
} // ~anonimous namespace

void BeginDumpOutputToFile(nn::fs::FileHandle fileHandle, int64_t fileOffset)
{
    g_Writer.BeginDumpToFile(fileHandle, fileOffset);
}

void EndDumpOutputToFile()
{
    g_Writer.EndDumpToFile();
}

Result DumpReportCore(nn::erpt::ReportId& reportId, const nn::erpt::ReportMetaData& reportMetaData, uint8_t* pReportData, size_t reportDataSize,
    util::optional<std::vector<std::string>> categories, bool useCategoriesToExclude,
    nn::util::optional<std::vector<std::string>> fields, bool useFieldsToExclude)
{
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    Bit8 aesKey[16];
    Bit8 aesIv[16];
    auto hasCipherKey = GetCipherKey(aesKey, aesIv, pReportData, reportDataSize);
#endif

    g_Writer.Dump("----------------------------------------------------------\n");

    char idStr[64];
    reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
    g_Writer.Dump("ReportId: %s\n", idStr);
    g_Writer.Dump("ReportSize: %zu bytes\n", reportDataSize);
    if( !IsReportMetaDataZeroCleared(reportMetaData) )
    {
        g_Writer.Dump("ReportMetaData: ");
        for( size_t i = 0; i < sizeof(erpt::ReportMetaData); i++ )
        {
            g_Writer.Dump("%02x", reportMetaData.userData[i]);
        }
        g_Writer.Dump("\n");
    }

    RegisterDumper<Bit64> registerDumper64(Reg64Names, "  %s: %016llx\n");
    RegisterDumper<Bit32> registerDumper32(Reg32Names, "  %s: %08llx\n");

    StacktraceDumper<Bit64> stacktraceDumper64;
    StacktraceDumper<Bit32> stacktraceDumper32;

    nn::msgpack::MpWalker mpWalker;
    mpWalker.Init(pReportData, reportDataSize);
    uint32_t count;
    mpWalker.GetMapCount(&count);
    for( uint32_t i = 0; i < count; ++i )
    {
        const char* key;
        size_t keySize;
        mpWalker.At(i, &key, &keySize);
        std::unique_ptr<char[]> keyBuffer(new char[keySize + 1]);
        keyBuffer[keySize] = '\0';
        std::memcpy(keyBuffer.get(), key, keySize);

        if( fields )
        {
            bool shouldDump = useFieldsToExclude;
            for( const auto& f : *fields )
            {
                if( f.length() == keySize && nn::util::Strncmp(f.c_str(), keyBuffer.get(), static_cast<int>(keySize)) == 0 )
                {
                    shouldDump = !useFieldsToExclude;
                    break;
                }
            }
            if( !shouldDump )
            {
                continue;
            }
        }

        if( categories )
        {
            // 出力しようとしているフィールド（key）が指定したカテゴリに含まれているものかを調べ、含まれていれば出力する。
            util::optional<int> fieldIndex(nullptr);
            for( size_t j = 0; j < sizeof(erpt::srv::FieldString) / sizeof(erpt::srv::FieldString[0]); j++ )
            {
                if( nn::util::Strncmp(keyBuffer.get(), erpt::srv::FieldString[j], static_cast<int>(keySize)) == 0 )
                {
                    fieldIndex = static_cast<int>(j);
                    break;
                }
            }
            if( !fieldIndex )
            {
                // 新しい FW で保存されたエラーレポートを処理する場合、fieldIndex が見つからない場合がある。
                continue;
            }

            auto categoryIndex = erpt::srv::FieldToCategoryMap[*fieldIndex];
            auto categoryStr = erpt::srv::CategoryString[categoryIndex];
            bool shouldDump = useCategoriesToExclude;
            for( const auto& c : *categories )
            {
                if( c.length() == std::strlen(categoryStr) && nn::util::Strncmp(c.c_str(), categoryStr, static_cast<int>(std::strlen(categoryStr))) == 0 )
                {
                    shouldDump = !useCategoriesToExclude;
                    break;
                }
            }
            if( !shouldDump )
            {
                continue;
            }
        }

        auto value = mpWalker[keyBuffer.get()];

        if( nn::util::Strncmp(keyBuffer.get(), "GeneralRegisterAarch32", static_cast<int>(keySize)) == 0 )
        {
            registerDumper32.SetRegisters(value.GetPtr(), value.GetSize());
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), "RegisterSetFlag32", static_cast<int>(keySize)) == 0 )
        {
            registerDumper32.SetFlags(value.GetPtr(), value.GetSize());
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), "GeneralRegisterAarch64", static_cast<int>(keySize)) == 0 )
        {
            registerDumper64.SetRegisters(value.GetPtr(), value.GetSize());
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), "RegisterSetFlag64", static_cast<int>(keySize)) == 0 )
        {
            registerDumper64.SetFlags(value.GetPtr(), value.GetSize());
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), stacktraceDumper32.GetBaseAddrKey(), static_cast<int>(keySize)) == 0 )
        {
            stacktraceDumper32.SetBaseAddr(value);
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), stacktraceDumper64.GetBaseAddrKey(), static_cast<int>(keySize)) == 0 )
        {
            stacktraceDumper64.SetBaseAddr(value);
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), stacktraceDumper32.GetStacktraceKey(), static_cast<int>(keySize)) == 0 )
        {
            stacktraceDumper32.SetStackBacktrace(value);
            continue;
        }
        else if( nn::util::Strncmp(keyBuffer.get(), stacktraceDumper64.GetStacktraceKey(), static_cast<int>(keySize)) == 0 )
        {
            stacktraceDumper64.SetStackBacktrace(value);
            continue;
        }

        bool dumped = DumpValue(g_DumpLogicForDumpAll, DumpLogicForDumpAllCount, keyBuffer.get(), keySize, value);
        if( !dumped )
        {
            g_Writer.Dump("%s: ", keyBuffer.get());
            DumpMessagePackValue(value, DefaultDumper);
        }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        if( hasCipherKey && IsEncryptedField(key, keySize) )
        {
            DumpEncryptedField(value, aesKey, aesIv);
        }
#endif
    }

    registerDumper32.Dump("Arm32 Registers");
    registerDumper64.Dump("Arm64 Registers");

    stacktraceDumper32.Dump();
    stacktraceDumper64.Dump();

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result DumpReport(nn::erpt::ReportId& reportId, const nn::erpt::ReportMetaData& reportMetaData,
    util::optional<std::vector<std::string>> categories, bool useCategoriesToExclude,
    nn::util::optional<std::vector<std::string>> fields, bool userFieldsToExclude)
{
    nn::erpt::Report report;
    NN_RESULT_DO(report.Open(reportId));

    int64_t reportSize;
    NN_RESULT_DO(report.GetSize(&reportSize));

    std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(reportSize)]);
    NN_RESULT_THROW_UNLESS(reportData != nullptr, nn::erpt::ResultOutOfMemory());

    uint32_t readCount;
    NN_RESULT_DO(report.Read(&readCount, reportData.get(), static_cast<uint32_t>(reportSize)));

    NN_RESULT_DO(DumpReportCore(reportId, reportMetaData, reportData.get(), readCount, categories, useCategoriesToExclude, fields, userFieldsToExclude));

    NN_RESULT_SUCCESS;
}

}
