﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecuteCleanAction.h"

#include <vector>
#include <string>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>

#include "DevMenuCommandAlbum_StorageUtility.h"

namespace album {

namespace {

    bool IsAlbumFilename(const char* name, size_t length) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(length >= 16 + 1 + 32 + 4);

        // 前半の日付部分の簡易チェック
        for (int i=0; i<15; ++i)
        {
            if (!std::isdigit( name[i] ))
            {
                return false;
            }
        }

        // 途中のハイフン
        if (name[16] != '-')
        {
            return false;
        }

        // 後半の難読化した ApplicationId 部分の簡易チェック
        for (int i=0; i<32; ++i)
        {
            if (!std::isxdigit( name[17 + i] ))
            {
                return false;
            }
        }

        // 拡張子
        if (!(std::strcmp(name + length - 4, ".MP4") == 0 ||
              std::strcmp(name + length - 4, ".mp4") == 0 ||
              std::strcmp(name + length - 4, ".JPG") == 0 ||
              std::strcmp(name + length - 4, ".jpg") == 0 ))
        {
            return false;
        }
        return true;
    }

    bool IsExtraAlbumFile(const char* name) NN_NOEXCEPT
    {
        // YYYYMMDDHHMMSSNN-<ENCRYPTEDAPPLICATIONIDANDFLAGS>X.suf
        auto length = std::strlen(name);
        if (length != 16 + 1 + 32 + 1 + 4)
        {
            return false;
        }
        if (!IsAlbumFilename(name, length))
        {
            return false;
        }
        if (name[length - 5] != 'X')
        {
            return false;
        }
        return true;
    }

    bool IsRegularAlbumFile(const char* name) NN_NOEXCEPT
    {
        // YYYYMMDDHHMMSSNN-<ENCRYPTEDAPPLICATIONIDANDFLAGS>.suf
        auto length = std::strlen(name);
        if (length != 16 + 1 + 32 + 4)
        {
            return false;
        }
        if (!IsAlbumFilename(name, length))
        {
            return false;
        }
        return true;
    }

    nn::Result ListFilesRecursivelyImpl(int64_t& totalCount, const std::string& path, const ProgramOption& opts)
    {
        nn::fs::DirectoryHandle hDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hDir, path.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hDir);
        };

        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> srcEntryList;
        srcEntryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, srcEntryList.data(), hDir, static_cast<int64_t>(srcEntryList.size())));

        int64_t fileCount = 0;
        for (auto& e : srcEntryList)
        {
            std::string entryPath = path + "/" + e.name;

            // ディレクトリの場合
            if (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(ListFilesRecursivelyImpl(totalCount, entryPath, opts));
                continue;
            }

            // ファイルの場合
            if (opts.IsExtraAlbumFileOnly() && !IsExtraAlbumFile(e.name))
            {
                continue;
            }
            if (opts.IsRegularAlbumFileOnly() && !IsRegularAlbumFile(e.name))
            {
                continue;
            }
            NN_LOG(" %s:/Nintendo/Album/%s\n", opts.GetStorageName(), entryPath.c_str() + 7);
            ++fileCount;
        }

        totalCount += fileCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result ListAllFilesRecursively(const char* mountName, const ProgramOption& opts) NN_NOEXCEPT
    {
        std::string rootPath = std::string(mountName) + ":/";

        int64_t totalCount = 0;
        NN_RESULT_DO( ListFilesRecursivelyImpl(totalCount, rootPath, opts) );

        NN_LOG("Found %d files\n", static_cast<int>(totalCount));
        NN_RESULT_SUCCESS;
    }

}   // namespace

bool ExecuteListAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("List files: storage=\x22%s\x22\n", opts.GetStorageName());

    if (opts.IsEmptyCheckRequired())
    {
        NN_LOG("Failed. '--check-empty' option is not available for list command.\n");
        return false;
    }
    if (!MountTargetStorage(*opts.GetStorage()))
    {
        NN_LOG("List files ... cannot mount target storage.\n");
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        UnmountTargetStorage();
    };

    if (!ListAllFilesRecursively("TARG", opts).IsSuccess())
    {
        NN_LOG("List files ... failed to complete\n");
        return false;
    }
    NN_LOG("List files ... completed.\n");
    return true;
}

}   // namespace album

