﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>

#include <nn/nifm.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/ntc/shim/ntc_shim.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_EphemeralNetworkSystemClock.h>
#include <nn/time/time_EphemeralNetworkSystemClockPrivilegeApi.h>
#include <nn/time/time_EphemeralNetworkSystemClockTestApi.h>
#include <nn/time/time_StandardUserSystemClockPrivilegeApi.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_MakeArgv.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_TimeCommand.h"
#include "DevMenuCommand_TimeZone.h"

using namespace nn;

namespace
{
    // ヘルプ用メッセージ
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " time set-auto-network-clock-ensure-policy <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " time get-auto-network-clock-ensure-policy\n"
        "       " DEVMENUCOMMAND_NAME " time invalidate-network-clock\n"
        "       " DEVMENUCOMMAND_NAME " time reset-steady-clock\n"
        "       " DEVMENUCOMMAND_NAME " time add-steady-clock-test-offset <minutes>\n"
        "       " DEVMENUCOMMAND_NAME " time get-steady-clock-test-offset\n"
        "       " DEVMENUCOMMAND_NAME " time set-user-clock --year [2000..2100] --month [1..12] --day [1..31] --hour [0..23] --minute [0..59]\n"
        "       " DEVMENUCOMMAND_NAME " time set-network-clock --year [2000..2100] --month [1..12] --day [1..31] --hour [0..23] --minute [0..59]\n"
        "       " DEVMENUCOMMAND_NAME " time list-location-name\n"
        "       " DEVMENUCOMMAND_NAME " time get-location-name\n"
        "       " DEVMENUCOMMAND_NAME " time set-location-name <location name>\n"

        "       " DEVMENUCOMMAND_NAME " time set-user-clock-auto-correction <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " time get-user-clock-auto-correction\n"
        "       " DEVMENUCOMMAND_NAME " time correct-network-clock\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " time info\n"
#endif
        ;

    // ユースケースとしてあり得ないくらい大きなオフセットを設定されても困るので、
    // 十分そうな適度な範囲でオフセットを限定しておく
    const nn::TimeSpan SteadyClockTestOffsetMax = nn::TimeSpan::FromDays(365 * 30); // 30年

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    template <typename T>
    void SetFwdbgValue(const char* moduleName, const char* key, const T& value)
    {
        settings::fwdbg::SetSettingsItemValue(moduleName, key, &value, sizeof(T));
    }

    template <typename T>
    void GetFwdbgValue(T* pOut, const char* moduleName, const char* key) NN_NOEXCEPT
    {
        auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(
            pOut,
            sizeof(T),
            moduleName,
            key);
        NN_ABORT_UNLESS_EQUAL(outBytes, sizeof(T));
    }

    // 自動補正処理の有効性取得
    Result GetAutoNetworkClockEnsurePolicy( bool* outValue, const Option& option )
    {
        NN_UNUSED(option);

        bool value;
        GetFwdbgValue(&value, "ntc", "is_autonomic_correction_enabled");
        if(value)
        {
            DEVMENUCOMMAND_LOG("enabled\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("disabled\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // 自動補正処理の有効性設定
    Result SetAutoNetworkClockEnsurePolicy( bool* outValue, const Option& option )
    {
        auto modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            DEVMENUCOMMAND_LOG("You must specify a policy. Set one of [enable|disable].\n");
            *outValue = false;
        }
        else if(std::string(modeString) == "enable")
        {
            SetFwdbgValue("ntc", "is_autonomic_correction_enabled", true);
            *outValue = true;
        }
        else if(std::string(modeString) == "disable")
        {
            SetFwdbgValue("ntc", "is_autonomic_correction_enabled", false);
            *outValue = true;
        }
        else
        {
            DEVMENUCOMMAND_LOG("%s is not valid policy. Set one of [enable|disable].\n", modeString);
            *outValue = false;
        }

        if(*outValue)
        {
            DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");
        }

        NN_RESULT_SUCCESS;
    }

    Result SetAutoNetworkClockEnsureIntervalSeconds( bool* outValue, const Option& option )
    {
        auto target = option.GetTarget();
        if( std::strlen(target) == 0 )
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " time set-auto-network-clock-ensure-interval <seconds>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if( std::string(target) == "0" )
        {
            SetFwdbgValue("ntc", "autonomic_correction_interval_seconds", static_cast<uint32_t>(0));
            DEVMENUCOMMAND_LOG("0 [seconds]\n");
            DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        char* endptr;
        auto seconds = STR_TO_ULL(target, &endptr, 10); // decimal

        if( seconds == 0 || *endptr != '\0' )
        {
            DEVMENUCOMMAND_LOG("%s is not valid.\n", target);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if( !nn::util::IsIntValueRepresentable<uint32_t>(seconds) )
        {
            DEVMENUCOMMAND_LOG("%s is too large.\n", target);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        const auto setValue = static_cast<uint32_t>(seconds);
        SetFwdbgValue("ntc", "autonomic_correction_interval_seconds", setValue);
        DEVMENUCOMMAND_LOG("%d seconds\n", setValue);
        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetAutoNetworkClockEnsureIntervalSeconds( bool* outValue, const Option& option )
    {
        NN_UNUSED(option);

        uint32_t value;
        GetFwdbgValue(&value, "ntc", "autonomic_correction_interval_seconds");

        DEVMENUCOMMAND_LOG("%d seconds\n", value);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // ネットワーク時計無効化
    Result InvalidateNetworkClock( bool* outValue, const Option& option )
    {
        NN_UNUSED(option);

        auto context = nn::settings::system::SystemClockContext();
        context.timeStamp.sourceId = nn::util::InvalidUuid;

        ::nn::settings::system::SetNetworkSystemClockContext(context);
        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // StandardSteadyClock リセット(RTC電池切れのシミュレート)
    Result ResetSteadyClock( bool* outValue, const Option& option )
    {
        NN_UNUSED(option);

        nn::settings::system::SetExternalSteadyClockSourceId(nn::util::InvalidUuid);

        // テストオフセットも初期化
        SetFwdbgValue("time", "standard_steady_clock_test_offset_minutes", static_cast<uint32_t>(0UL));

        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // StandardSteadyClock TestOffset 追加
    Result AddSteadyClockTestOffsetMinutes(bool* outValue, const Option& option)
    {
        auto target = option.GetTarget();
        if( std::strlen(target) == 0 )
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " time add-steady-clock-test-offset <minutes>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if( std::string(target) == "0" )
        {
            DEVMENUCOMMAND_LOG("0 is not valid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* endptr;
        auto minutes = STR_TO_ULL(target, &endptr, 10); // decimal
        if( minutes == 0 || *endptr != '\0' || !nn::util::IsIntValueRepresentable<uint32_t>(minutes) )
        {
            DEVMENUCOMMAND_LOG("%s is not valid.\n", target);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        uint32_t currentValue;
        GetFwdbgValue(&currentValue, "time", "standard_steady_clock_test_offset_minutes");

        const auto addValue = static_cast<uint32_t>(minutes);
        if( (static_cast<int64_t>(addValue) + static_cast<int64_t>(currentValue)) > SteadyClockTestOffsetMax.GetMinutes() )
        {
            DEVMENUCOMMAND_LOG("%s is too large.\n", target);
            DEVMENUCOMMAND_LOG("Steady clock test offset supports until up to %lld years (%lld minutes).\n",
                SteadyClockTestOffsetMax.GetDays() / 365LL, SteadyClockTestOffsetMax.GetMinutes());
            DEVMENUCOMMAND_LOG("Current value is %lu minutes.\n", currentValue);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        const uint32_t setValue = addValue + currentValue;
        SetFwdbgValue("time", "standard_steady_clock_test_offset_minutes", setValue);
        DEVMENUCOMMAND_LOG("Added %lu minutes. Total offset is %lu minutes.\n", addValue, setValue);
        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow new settings to take effect.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // StandardSteadyClock TestOffset 取得
    Result GetSteadyClockTestOffsetMinutes( bool* outValue, const Option& option )
    {
        NN_UNUSED(option);

        uint32_t value;
        GetFwdbgValue(&value, "time", "standard_steady_clock_test_offset_minutes");

        DEVMENUCOMMAND_LOG("%lu minutes\n", value);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool RetrieveDateTime(nn::time::CalendarTime* pOut, const Option& option)
    {
        auto GetParam = [&](const char* sub, int min, int max) -> int
        {
            const auto str = option.GetValue(sub);
            if( !str || std::strlen(str) == 0 )
            {
                DEVMENUCOMMAND_LOG("%s option is required.\n", sub);
                return -1;
            }

            char* endptr;
            const auto value = STR_TO_ULL(str, &endptr, 10); // decimal

            if( *endptr != '\0' || value < min || value > max)
            {
                DEVMENUCOMMAND_LOG("%s: %s is not valid.\n", sub, str);
                return -1;
            }

            return static_cast<int>(value);
        };

        auto year = GetParam("--year", 2000, 2100);
        auto month = GetParam("--month", 1, 12);
        auto day = GetParam("--day", 1, 31);
        auto hour = GetParam("--hour", 0, 23);
        auto minute = GetParam("--minute", 0, 59);
        if( year < 0 || month < 0 || day < 0 || hour < 0 || minute < 0 )
        {
            return false;
        }

        if( !nn::time::IsValidDate(year, month, day) )
        {
            DEVMENUCOMMAND_LOG("year:%d month:%d day:%d is not valid date.\n", year, month, day);
            return false;
        }

        pOut->year = static_cast<int16_t>(year);
        pOut->month = static_cast<int8_t>(month);
        pOut->day = static_cast<int8_t>(day);
        pOut->hour = static_cast<int8_t>(hour);
        pOut->minute = static_cast<int8_t>(minute);
        pOut->second = 0;
        return true;
    }

    Result SetUserClock( bool* outValue, const Option& option )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        nn::time::CalendarTime c;
        if( !RetrieveDateTime(&c, option) )
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        {
            int outCount;
            nn::time::PosixTime outPosix;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToPosixTime(&outCount, &outPosix, 1, c));

            if( outCount == 0 )
            {
                DEVMENUCOMMAND_LOG("year:%d month:%d day:%d hour:%d minute:%d is not valid date-time.\n",
                    c.year, c.month, c.day, c.hour, c.minute);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::SetStandardLocalSystemClockCurrentTime(outPosix));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // ユーザー時計の自動補正ON/OFFの設定
    Result SetUserClockAutoCorrection( bool* outValue, const Option& option )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        auto modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            DEVMENUCOMMAND_LOG("You must specify a policy. Set one of [enable|disable].\n");
            *outValue = false;
        }
        else if(std::string(modeString) == "enable")
        {
            nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(true);
            *outValue = true;
        }
        else if(std::string(modeString) == "disable")
        {
            nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled(false);
            *outValue = true;
        }
        else
        {
            DEVMENUCOMMAND_LOG("%s is not valid. Set one of [enable|disable].\n", modeString);
            *outValue = false;
        }


        NN_RESULT_SUCCESS;
    }

    // ユーザー時計の自動補正ON/OFFの取得
    Result GetUserClockAutoCorrection ( bool* outValue, const Option&)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        if(nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled())
        {
            DEVMENUCOMMAND_LOG("enabled\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("disabled\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetNetworkClock( bool* outValue, const Option& option )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForSystem());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        nn::time::CalendarTime c;
        if( !RetrieveDateTime(&c, option) )
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        {
            int outCount;
            nn::time::PosixTime outPosix;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToPosixTime(&outCount, &outPosix, 1, c));

            if( outCount == 0 )
            {
                DEVMENUCOMMAND_LOG("year:%d month:%d day:%d hour:%d minute:%d is not valid date-time.\n",
                    c.year, c.month, c.day, c.hour, c.minute);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::SetStandardNetworkSystemClockCurrentTime(outPosix));

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::SetEphemeralNetworkSystemClockCurrentTime(outPosix));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListLocationName( bool* outValue, const Option& )
    {
        auto pTimeZoneList = devmenuUtil::GetTimeZoneList();
        int count = devmenuUtil::GetCountOfTimeZone();
        for (int i = 0 ; i < count ; ++i)
        {
            DEVMENUCOMMAND_LOG("%s\n", pTimeZoneList[i].locationName._value);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetLocationName( bool* outValue, const Option& )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        nn::time::LocationName locationName;
        nn::time::GetDeviceLocationName(&locationName);
        DEVMENUCOMMAND_LOG("%s\n", locationName._value);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetLocationName( bool* outValue, const Option& option )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        auto target = option.GetTarget();

        if ( std::string(target) == "" )
        {
            DEVMENUCOMMAND_LOG("You must specify a policy. Set one of available location name.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto pTimeZoneList = devmenuUtil::GetTimeZoneList();
        int count = devmenuUtil::GetCountOfTimeZone();
        for (int i = 0 ; i < count ; ++i)
        {
            if(std::string(target) == std::string(pTimeZoneList[i].locationName._value))
            {
                nn::time::SetDeviceLocationName(pTimeZoneList[i].locationName);

                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        DEVMENUCOMMAND_LOG("%s is invalid location name.\n", target);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    void PrintSteadyClockTimePoint(const nn::time::SteadyClockTimePoint& timePoint, const char* header)
    {
        char buffer[nn::util::Uuid::StringSize];

        DEVMENUCOMMAND_LOG("    %s.value : %lld\n", header, timePoint.value);
        DEVMENUCOMMAND_LOG("    %s.sourceId : %s\n", header, timePoint.sourceId.ToString(buffer, sizeof(buffer)));
    }

    void PrintPosixTime(const nn::time::PosixTime posixTime, const char* header)
    {
        nn::time::CalendarTime c;
        nn::time::CalendarAdditionalInfo info;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&c, &info, posixTime));

        DEVMENUCOMMAND_LOG("    %s : %lld (%04d/%02d/%02d %02d:%02d:%02d[%s], DST : %s, UTC Offset : %d)\n",
            header,
            posixTime.value,
            c.year, c.month, c.day, c.hour, c.minute, c.second,
            info.timeZone.standardTimeName,
            info.timeZone.isDaylightSavingTime ? "true" : "false",
            info.timeZone.utcOffsetSeconds);
    }

    Result Info( bool* outValue, const Option&)
    {

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        {
            DEVMENUCOMMAND_LOG("nn::time::StandardUserSystemClock\n");

            nn::time::PosixTime userCurrent;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&userCurrent));
            PrintPosixTime(userCurrent, "Current Time");

            nn::time::SystemClockContext userContext;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetSystemClockContext(&userContext));
            DEVMENUCOMMAND_LOG("    SystemClockContext.offset : %lld\n", userContext.offset);
            PrintSteadyClockTimePoint(userContext.timeStamp, "SystemClockContext.timeStamp");

            DEVMENUCOMMAND_LOG("    Auto Correction : %s\n", nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled() ? "enabled" : "disabled");
        }

        {
            DEVMENUCOMMAND_LOG("nn::time::StandardNetworkSystemClock\n");

            nn::time::PosixTime networkCurrent;
            auto networkCurrentResult = nn::time::StandardNetworkSystemClock::GetCurrentTime(&networkCurrent);
            if(networkCurrentResult.IsSuccess())
            {
                PrintPosixTime(networkCurrent, "Current Time");
            }
            else
            {
                DEVMENUCOMMAND_LOG("    PosixTime : Not corrected\n");
            }
            nn::time::SystemClockContext networkContext;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardNetworkSystemClock::GetSystemClockContext(&networkContext));
            DEVMENUCOMMAND_LOG("    SystemClockContext.offset : %lld\n", networkContext.offset);
            PrintSteadyClockTimePoint(networkContext.timeStamp, "SystemClockContext.timeStamp");

            bool enabled;
            GetFwdbgValue(&enabled, "ntc", "is_autonomic_correction_enabled");
            DEVMENUCOMMAND_LOG("    Auto Clock Ensure : %s\n", enabled ? "enabled" : "disabled");
        }

        {
            DEVMENUCOMMAND_LOG("nn::time::StandardSteadyClock\n");

            nn::time::SteadyClockTimePoint currentTp;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint (&currentTp));

            PrintSteadyClockTimePoint(currentTp, "SteadyClockTimePoint");
            DEVMENUCOMMAND_LOG("    Current RTC : %lld\n", nn::time::GetStandardSteadyClockRtcValue());

            DEVMENUCOMMAND_LOG("    RTC Reset Detected : %s\n", nn::time::IsRtcResetDetected() ? "true" : "false");

            DEVMENUCOMMAND_LOG("    Setup Result Value : 0x%08x\n", nn::time::GetStandardSteadyClockSetupResultValue());

            uint32_t testOffset;
            GetFwdbgValue(&testOffset, "time", "standard_steady_clock_test_offset_minutes");
            DEVMENUCOMMAND_LOG("    Test offset : %lld\n", testOffset);

            auto internalOffset = nn::time::GetStandardSteadyClockInternalOffset();
            DEVMENUCOMMAND_LOG("    Internal offset : %lld\n", internalOffset.GetSeconds());
        }

        {
            DEVMENUCOMMAND_LOG("nn::time::EphemeralNetworkSystemClock\n");

            nn::time::PosixTime posixTime;
            auto result = nn::time::EphemeralNetworkSystemClock::GetCurrentTime(&posixTime);
            if(result.IsSuccess())
            {
                PrintPosixTime(posixTime, "Current Time");
            }
            else
            {
                DEVMENUCOMMAND_LOG("    PosixTime : Not corrected\n");
            }

            nn::time::SystemClockContext context;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::GetEphemeralNetworkSystemClockContext(&context));
            DEVMENUCOMMAND_LOG("    SystemClockContext.offset : %lld\n", context.offset);
            PrintSteadyClockTimePoint(context.timeStamp, "SystemClockContext.timeStamp");
        }

        {
            DEVMENUCOMMAND_LOG("Other\n");

            nn::time::LocationName locationName;
            nn::time::GetDeviceLocationName(&locationName);
            DEVMENUCOMMAND_LOG("    Location name : %s\n", locationName._value);

            auto tick = nn::os::GetSystemTick();
            DEVMENUCOMMAND_LOG("    Current tick : %lld ( %lld seconds )\n", tick.GetInt64Value(), tick.ToTimeSpan().GetSeconds());
        }

        {
            DEVMENUCOMMAND_LOG("Settings\n");

            char buffer[nn::util::Uuid::StringSize];

            nn::time::SourceId externalSourceId;
            nn::settings::system::GetExternalSteadyClockSourceId(&externalSourceId);
            DEVMENUCOMMAND_LOG("    ExternalSteadyClockSourceId %s\n", externalSourceId.ToString(buffer, sizeof(buffer)));

            DEVMENUCOMMAND_LOG("    ExternalSteadyClockInternalOffset %lld\n", nn::settings::system::GetExternalSteadyClockInternalOffset());

            nn::time::SystemClockContext userContext;
            nn::settings::system::GetUserSystemClockContext(&userContext);
            DEVMENUCOMMAND_LOG("    UserSystemClockContext.offset : %lld\n", userContext.offset);
            PrintSteadyClockTimePoint(userContext.timeStamp, "UserSystemClockContext.timeStamp");

            nn::time::SystemClockContext networkContext;
            nn::settings::system::GetNetworkSystemClockContext(&networkContext);
            DEVMENUCOMMAND_LOG("    NetworkSystemClockContext.offset : %lld\n", networkContext.offset);
            PrintSteadyClockTimePoint(networkContext.timeStamp, "NetworkSystemClockContext.timeStamp");

            DEVMENUCOMMAND_LOG("    IsUserSystemClockAutomaticCorrectionEnabled : %s\n", nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled() ? "enabled" : "disabled");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result CorrectNetworkClock( bool* outValue, const Option& )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::ntc::shim::CorrectionNetworkClockAsyncTask task(
            nn::os::EventClearMode_AutoClear,
            nn::ntc::EnsureNetworkClockAvailabilityMode_ForcibleDownload // 必ずサーバーアクセスして時刻ダウンロード
            );

        auto result = task.StartTask();
        if(result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("Failed to start task : 0x%08x.\n", result.GetInnerValueForDebug());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        task.GetFinishNotificationEvent().Wait();

        result = task.GetResult();
        if(result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("Failed to execute task : 0x%08x.\n", result.GetInnerValueForDebug());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

} // namespace

//----------------------------------------------------------------
namespace{
    // サブコマンド情報
    const SubCommand g_SubCommands[] =
    {
        { "set-auto-network-clock-ensure-policy", SetAutoNetworkClockEnsurePolicy },
        { "get-auto-network-clock-ensure-policy", GetAutoNetworkClockEnsurePolicy },

        { "invalidate-network-clock", InvalidateNetworkClock },
        { "reset-steady-clock", ResetSteadyClock },

        { "add-steady-clock-test-offset", AddSteadyClockTestOffsetMinutes },
        { "get-steady-clock-test-offset", GetSteadyClockTestOffsetMinutes },

        { "set-user-clock", SetUserClock },
        { "set-network-clock", SetNetworkClock },

        { "list-location-name", ListLocationName },
        { "get-location-name", GetLocationName },
        { "set-location-name", SetLocationName },

        { "set-user-clock-auto-correction", SetUserClockAutoCorrection },
        { "get-user-clock-auto-correction", GetUserClockAutoCorrection },

        { "correct-network-clock", CorrectNetworkClock },

        // 現時点で公開する必要もないので以下は --help ではいったん出力しない
        { "set-auto-network-clock-ensure-interval", SetAutoNetworkClockEnsureIntervalSeconds },
        { "get-auto-network-clock-ensure-interval", GetAutoNetworkClockEnsureIntervalSeconds },
        { "info", Info },
    };
}

Result TimeCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto secondArg = option.GetSubCommand();
    if (std::string(secondArg) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // サブコマンド処理
    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    // 不正なサブコマンドを指定
    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu time command. See '" DEVMENUCOMMAND_NAME " time --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
