﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <string>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_Tv.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_SettingsCommand.h"

#define NN_DEVMENUDOMMAND_SETTINGS_SET_TV_RESOLUTION_USAGE          "settings set-tv-resolution <auto|1080p|720p|480p>"
#define NN_DEVMENUDOMMAND_SETTINGS_GET_TV_RESOLUTION_USAGE          "settings get-tv-resolution"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_RGB_RANGE_USAGE              "settings set-rgb-range <auto|full|limited>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_RGB_RANGE_USAGE              "settings get-rgb-range"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_BURNIN_REDUCITION_USAGE "settings set-burnin-reduction <enabled|disabled>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_BURNIN_REDUCITION_USAGE "settings get-burnin-reduction"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_CEC_USAGE                    "settings set-cec <enabled|disabled>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_CEC_USAGE                    "settings get-cec"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_TV_UNDERSCAN_USAGE           "settings set-tv-underscan <0..20>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_TV_UNDERSCAN_USAGE           "settings get-tv-underscan"

#define NN_DEVMENUCOMMAND_SETTINGS_SET_REGION_USAGE                 "settings set-region <Japan|Usa|Europe|Australia>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_REGION_USAGE                 "settings get-region"

#define NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_USAGE             "settings set-auto-sleep [--handheld <1m|3m|5m|10m|30m|never>] [--console <1h|2h|3h|6h|12h|never>] [--mediaplay <enabled|disabled>]"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_USAGE             "settings get-auto-sleep --<handheld|console|mediaplay|all>"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_HANDHELD_USAGE    "settings set-auto-sleep --handheld <1m|3m|5m|10m|30m|never>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_HANDHELD_USAGE    "settings get-auto-sleep --handheld"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_CONSOLE_USAGE     "settings set-auto-sleep --console <1h|2h|3h|6h|12h|never>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_CONSOLE_USAGE     "settings get-auto-sleep --console"
#define NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_MEDIA_PLAY_USAGE  "settings set-auto-sleep --mediaplay <enabled|disabled>"
#define NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_MEDIA_PLAY_USAGE  "settings get-auto-sleep --mediaplay"

namespace {

    void OutputBooleanValueString( bool isEnabled, const char* caption = "" ) NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG( "%s%s\n", caption, isEnabled ? "enabled" : "disabled" );
    }

    /*
     * TV 設定
     */
    bool SetTvResolution( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUDOMMAND_SETTINGS_SET_TV_RESOLUTION_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        if ( "auto" == target )
        {
            tvSettings.tvResolution = nn::settings::system::TvResolution_Auto;
        }
        else if ( "1080p" == target )
        {
            tvSettings.tvResolution = nn::settings::system::TvResolution_1080p;
        }
        else if ( "720p" == target )
        {
            tvSettings.tvResolution = nn::settings::system::TvResolution_720p;
        }
        else if ( "480p" == target )
        {
            tvSettings.tvResolution = nn::settings::system::TvResolution_480p;
        }
        else
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUDOMMAND_SETTINGS_SET_TV_RESOLUTION_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetTvSettings( tvSettings );
        return true;
    }

    bool GetTvResolution( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        switch ( tvSettings.tvResolution )
        {
        case nn::settings::system::TvResolution_Auto:
            DEVMENUCOMMAND_LOG( "Auto\n" );
            break;
        case nn::settings::system::TvResolution_1080p:
            DEVMENUCOMMAND_LOG( "1080p\n" );
            break;
        case nn::settings::system::TvResolution_720p:
            DEVMENUCOMMAND_LOG( "720p\n" );
            break;
        case nn::settings::system::TvResolution_480p:
            DEVMENUCOMMAND_LOG( "480p\n" );
            break;
        default:
            DEVMENUCOMMAND_LOG( "Unknown\n" );
            break;
        }

        return true;
    }

    bool SetRgbRange( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_RGB_RANGE_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        if ( "auto" == target )
        {
            tvSettings.rgbRange = nn::settings::system::RgbRange_Auto;
        }
        else if ( "full" == target )
        {
            tvSettings.rgbRange = nn::settings::system::RgbRange_Full;
        }
        else if ( "limited" == target )
        {
            tvSettings.rgbRange = nn::settings::system::RgbRange_Limited;
        }
        else
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_RGB_RANGE_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetTvSettings( tvSettings );
        return true;
    }

    bool GetRgbRange( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        switch ( tvSettings.rgbRange )
        {
        case nn::settings::system::RgbRange_Auto:
            DEVMENUCOMMAND_LOG( "Auto\n" );
            break;
        case nn::settings::system::RgbRange_Full:
            DEVMENUCOMMAND_LOG( "Full\n" );
            break;
        case nn::settings::system::RgbRange_Limited:
            DEVMENUCOMMAND_LOG( "Limited\n" );
            break;
        default:
            DEVMENUCOMMAND_LOG( "Unknown\n" );
            break;
        }

        return true;
    }

    bool SetBurnInReductionEnabled( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_BURNIN_REDUCITION_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        if ( "enabled" == target )
        {
            tvSettings.flags.Set< nn::settings::system::TvFlag::PreventsScreenBurnIn >();
        }
        else if ( "disabled" == target )
        {
            tvSettings.flags.Reset< nn::settings::system::TvFlag::PreventsScreenBurnIn >();
        }
        else
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_BURNIN_REDUCITION_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetTvSettings( tvSettings );
        return true;
    }

    bool GetBurnInReductionEnabled( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );
        OutputBooleanValueString( tvSettings.flags.Test< nn::settings::system::TvFlag::PreventsScreenBurnIn >() );

        return true;
    }

    bool SetCecEnabled( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_CEC_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        if ( "enabled" == target )
        {
            tvSettings.flags.Set< nn::settings::system::TvFlag::AllowsCec >();
        }
        else if ( "disabled" == target )
        {
            tvSettings.flags.Reset< nn::settings::system::TvFlag::AllowsCec >();
        }
        else
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_CEC_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetTvSettings( tvSettings );
        return true;
    }

    bool GetCecEnabled( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );
        OutputBooleanValueString( tvSettings.flags.Test< nn::settings::system::TvFlag::AllowsCec >() );

        return true;
    }

    bool SetTvUnderscan( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_TV_UNDERSCAN_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        // 数値判定
        auto isSuccess = false;
        if ( std::all_of( target.cbegin(), target.cend(), isdigit ) )
        {
            auto tvUnderscan = std::atoi( target.c_str() );
            if ( 0 <= tvUnderscan && tvUnderscan <= 20 )
            {
                tvSettings.tvUnderscan = tvUnderscan;
                isSuccess = true;
            }
        }

        if ( !isSuccess )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_TV_UNDERSCAN_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetTvSettings( tvSettings );
        return true;
    }

    bool GetTvUnderscan( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::TvSettings tvSettings;
        nn::settings::system::GetTvSettings( &tvSettings );

        DEVMENUCOMMAND_LOG( "%d\n", tvSettings.tvUnderscan );

        return true;
    }

    /*
    * Region 設定
    */
    bool SetRegion( const Option& option ) NN_NOEXCEPT
    {
        if ( !option.HasTarget() )
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_REGION_USAGE "\n" );
            return false;
        }

        const std::string target = option.GetTarget();

        nn::settings::system::RegionCode regionCode;

        if ( "Japan" == target )
        {
            regionCode = nn::settings::system::RegionCode_Japan;
        }
        else if ( "Usa" == target )
        {
            regionCode = nn::settings::system::RegionCode_Usa;
        }
        else if ( "Europe" == target )
        {
            regionCode = nn::settings::system::RegionCode_Europe;
        }
        else if ( "Australia" == target )
        {
            regionCode = nn::settings::system::RegionCode_Australia;
        }
        else
        {
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_REGION_USAGE "\n" );
            DEVMENUCOMMAND_LOG( " - Unknown target is specified: '%s'\n", target.c_str() );
            return false;
        }

        nn::settings::system::SetRegionCode( regionCode );
        return true;
    }

    bool GetRegion( const Option& option ) NN_NOEXCEPT
    {
        NN_UNUSED( option );

        nn::settings::system::RegionCode regionCode;
        nn::settings::system::GetRegionCode( &regionCode );

        switch ( regionCode )
        {
        case nn::settings::system::RegionCode_Japan:
            DEVMENUCOMMAND_LOG( "Japan\n" );
            break;
        case nn::settings::system::RegionCode_Usa:
            DEVMENUCOMMAND_LOG( "Usa\n" );
            break;
        case nn::settings::system::RegionCode_Europe:
            DEVMENUCOMMAND_LOG( "Europe\n" );
            break;
        case nn::settings::system::RegionCode_Australia:
            DEVMENUCOMMAND_LOG( "Australia\n" );
            break;
        case nn::settings::system::RegionCode_China:
            DEVMENUCOMMAND_LOG( "China\n" );
            break;
        case nn::settings::system::RegionCode_Korea:
            DEVMENUCOMMAND_LOG( "Korea\n" );
            break;
        case nn::settings::system::RegionCode_Taiwan:
            DEVMENUCOMMAND_LOG( "Taiwan\n" );
            break;
        default:
            DEVMENUCOMMAND_LOG( "Unknown\n" );
            break;
        }

        return true;
    }

    /*
     * スリープモード設定
     */
    const char* SleepModeTypeNames[] = { "--handheld", "--console", "--mediaplay", "--all" };

    template < typename T >
    struct AutoSleepKeyValue {
        const char* key;
        T value;
    };

    const AutoSleepKeyValue< nn::settings::system::HandheldSleepPlan > HandheldKeyValueList[] = {
        { "1m", nn::settings::system::HandheldSleepPlan_1Min },
        { "3m", nn::settings::system::HandheldSleepPlan_3Min },
        { "5m", nn::settings::system::HandheldSleepPlan_5Min },
        { "10m", nn::settings::system::HandheldSleepPlan_10Min },
        { "30m", nn::settings::system::HandheldSleepPlan_30Min },
        { "never", nn::settings::system::HandheldSleepPlan_Never },
    };

    const AutoSleepKeyValue< nn::settings::system::ConsoleSleepPlan > ConsoleKeyValueList[] = {
        { "1h", nn::settings::system::ConsoleSleepPlan_1Hour },
        { "2h", nn::settings::system::ConsoleSleepPlan_2Hour },
        { "3h", nn::settings::system::ConsoleSleepPlan_3Hour },
        { "6h", nn::settings::system::ConsoleSleepPlan_6Hour },
        { "12h", nn::settings::system::ConsoleSleepPlan_12Hour },
        { "never", nn::settings::system::ConsoleSleepPlan_Never },
    };

    bool SetAutoSleep( const Option& option ) NN_NOEXCEPT
    {
        struct OptionHandler {
            const char* key;
            std::function< bool( nn::settings::system::SleepSettings* pOutSleepSettings, const std::string& targetValue ) > handler;
        } handlerList[] = {
            {
                "--handheld",
                [&]( nn::settings::system::SleepSettings* pOutSleepSettings, const std::string& targetValue ) -> bool
                {
                    for ( const auto& iter : HandheldKeyValueList )
                    {
                        if ( iter.key == targetValue )
                        {
                            pOutSleepSettings->handheldSleepPlan = iter.value;
                            return true;
                        }
                    }

                    DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_HANDHELD_USAGE "\n" );
                    DEVMENUCOMMAND_LOG( " - Unknown target value is specified: '%s'\n", targetValue.c_str() );
                    return false;
                },
            },
            {
                "--console",
                [&]( nn::settings::system::SleepSettings* pOutSleepSettings, const std::string& targetValue ) -> bool
                {
                    for ( const auto& iter : ConsoleKeyValueList )
                    {
                        if ( iter.key == targetValue )
                        {
                            pOutSleepSettings->consoleSleepPlan = iter.value;
                            return true;
                        }
                    }

                    DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_CONSOLE_USAGE "\n" );
                    DEVMENUCOMMAND_LOG( " - Unknown target value is specified: '%s'\n", targetValue.c_str() );
                    return false;
                },
            },
            {
                "--mediaplay",
                [&]( nn::settings::system::SleepSettings* pOutSleepSettings, const std::string& targetValue ) -> bool
                {
                    if ( "enabled" == targetValue )
                    {
                        pOutSleepSettings->flags.Set< nn::settings::system::SleepFlag::SleepsWhilePlayingMedia >();
                    }
                    else if ( "disabled" == targetValue )
                    {
                        pOutSleepSettings->flags.Reset< nn::settings::system::SleepFlag::SleepsWhilePlayingMedia >();
                    }
                    else
                    {
                        DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_MEDIA_PLAY_USAGE "\n" );
                        DEVMENUCOMMAND_LOG( " - Unknown target value is specified: '%s'\n", targetValue.c_str() );
                        return false;
                    }

                    return true;
                },
            },
        };

        bool isAnyValidOpitionsDetected = false;
        bool isAnyOptionHandlersFailed = false;
        nn::settings::system::SleepSettings sleepSettings;
        nn::settings::system::GetSleepSettings( &sleepSettings );

        for ( const auto& name : SleepModeTypeNames )
        {
            if ( !option.HasKey( name ) )
            {
                continue;
            }

            const auto targetValue = option.GetValue( name );

            for ( const auto& iter : handlerList )
            {
                if ( 0 == nn::util::Strncmp( name, iter.key, sizeof( name ) ) )
                {
                    isAnyValidOpitionsDetected = true;
                    isAnyOptionHandlersFailed |= ( false == iter.handler( &sleepSettings, targetValue ) );
                }
            }
        }

        if ( true == isAnyValidOpitionsDetected )
        {
            if ( false == isAnyOptionHandlersFailed )
            {
                nn::settings::system::SetSleepSettings( sleepSettings );
                return true;
            }
        }
        else
        {
            DEVMENUCOMMAND_LOG( "Cannot detect any valid options. Keeps current setting.\n" );
            DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_USAGE  "\n" );
            return true;
        }

        return false;
    }

    bool GetAutoSleep( const Option& option ) NN_NOEXCEPT
    {
        auto outputHandheldSleepModeSetting = []( const nn::settings::system::SleepSettings& sleepSettings, const char* caption ) -> bool
        {
            for ( const auto& iter : HandheldKeyValueList )
            {
                if ( iter.value == sleepSettings.handheldSleepPlan )
                {
                    DEVMENUCOMMAND_LOG( "%s%s\n", caption, iter.key );
                    return true;
                }
            }

            DEVMENUCOMMAND_LOG( "handheld: Unexpected value is being set.\n" );
            return false;
        };

        auto outputConsoleSleepModeSetting = []( const nn::settings::system::SleepSettings& sleepSettings, const char* caption ) -> bool
        {
            for ( const auto& iter : ConsoleKeyValueList )
            {
                if ( iter.value == sleepSettings.consoleSleepPlan )
                {
                    DEVMENUCOMMAND_LOG( "%s%s\n", caption, iter.key );
                    return true;
                }
            }

            DEVMENUCOMMAND_LOG( "console: Unexpected value is being set.\n" );
            return false;
        };

        auto outputMediaPlaySleepModeSetting = []( const nn::settings::system::SleepSettings& sleepSettings, const char* caption ) -> bool
        {
            OutputBooleanValueString( sleepSettings.flags.Test< nn::settings::system::SleepFlag::SleepsWhilePlayingMedia >(), caption );
            return true;
        };

        struct OptionHandler {
            const char* key;
            std::function< bool( const nn::settings::system::SleepSettings& sleepSettings ) > handler;
        } handlerList[] = {
            {
                "--handheld",
                [&]( const nn::settings::system::SleepSettings& sleepSettings ) -> bool
                {
                    return outputHandheldSleepModeSetting( sleepSettings, "" );
                },
            },
            {
                "--console",
                [&]( const nn::settings::system::SleepSettings& sleepSettings ) -> bool
                {
                    return outputConsoleSleepModeSetting( sleepSettings, "" );
                },
            },
            {
                "--mediaplay",
                [&]( const nn::settings::system::SleepSettings& sleepSettings ) -> bool
                {
                    return outputMediaPlaySleepModeSetting( sleepSettings, "" );
                },
            },
            {
                "--all",
                [&]( const nn::settings::system::SleepSettings& sleepSettings ) -> bool
                {
                    return outputHandheldSleepModeSetting( sleepSettings, "handheld: " ) &&
                        outputConsoleSleepModeSetting( sleepSettings, "console: " ) &&
                        outputMediaPlaySleepModeSetting( sleepSettings, "mediaplay: " );
                },
            },
        };

        nn::settings::system::SleepSettings sleepSettings;
        nn::settings::system::GetSleepSettings( &sleepSettings );

        for ( const auto& name : SleepModeTypeNames )
        {
            if ( !option.HasKey( name ) )
            {
                continue;
            }

            for ( const auto& iter : handlerList )
            {
                if ( 0 == nn::util::Strncmp( name, iter.key, sizeof( name ) ) )
                {
                    return iter.handler( sleepSettings );
                }
            }
        }

        DEVMENUCOMMAND_LOG( "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_USAGE "\n" );
        return false;
    }

} // ~namespace <anonymous>

//------------------------------------------------------------------------------------------------

nn::Result SettingsCommand( bool* outValue, const Option& option ) NN_NOEXCEPT
{
    static const struct
    {
        std::string name;
        bool ( *function )( const Option& );
    } SubCommands[] = {
        { "set-tv-resolution", SetTvResolution },
        { "get-tv-resolution", GetTvResolution },
        { "set-rgb-range", SetRgbRange },
        { "get-rgb-range", GetRgbRange },
        { "set-burnin-reduction", SetBurnInReductionEnabled },
        { "get-burnin-reduction", GetBurnInReductionEnabled },
        { "set-cec", SetCecEnabled },
        { "get-cec", GetCecEnabled },
        { "set-tv-underscan", SetTvUnderscan },
        { "get-tv-underscan", GetTvUnderscan },
        { "set-region", SetRegion },
        { "get-region", GetRegion },
        { "set-auto-sleep", SetAutoSleep },
        { "get-auto-sleep", GetAutoSleep },
    };

    static const char HelpMessage[] =
        // TV settings
        "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENUDOMMAND_SETTINGS_SET_TV_RESOLUTION_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUDOMMAND_SETTINGS_GET_TV_RESOLUTION_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_RGB_RANGE_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_RGB_RANGE_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_BURNIN_REDUCITION_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_BURNIN_REDUCITION_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_CEC_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_CEC_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_TV_UNDERSCAN_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_TV_UNDERSCAN_USAGE "\n"
        // Region settings
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_REGION_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_REGION_USAGE "\n"
        // Sleep
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_SET_AUTO_SLEEP_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENUCOMMAND_SETTINGS_GET_AUTO_SLEEP_USAGE "\n"
        "";

    if ( !option.HasSubCommand() )
    {
        DEVMENUCOMMAND_LOG( HelpMessage );
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if ( std::string( option.GetSubCommand() ) == "--help" )
    {
        DEVMENUCOMMAND_LOG( HelpMessage );
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for ( const auto& subCommand : SubCommands )
    {
        if ( subCommand.name == option.GetSubCommand() )
        {
            *outValue = subCommand.function( option );
            NN_RESULT_SUCCESS;
        }
    }

    DEVMENUCOMMAND_LOG( "'%s' is not a DevMenuCommand settings command. See '" DEVMENUCOMMAND_NAME " settings --help'.\n", option.GetSubCommand() );
    *outValue = false;
    NN_RESULT_SUCCESS;
}
