﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <functional>
#include <sstream>
#include <string>
#include <vector>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os/os_Result.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_NotifyEventApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/pdm/pdm_QueryLastPlayTimeApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_PlayDataCommand.h"

using namespace nn;
using namespace nn::pdm;

//------------------------------------------------------------------------------------------------

#define NN_DEVMENU_COMMAND_QUERY_PLAYSTATISTICS         "playdata query-playstatistics <application-id> [--account <index>]"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
#define NN_DEVMENU_COMMAND_QUERY_PLAYEVENT_USAGE        "playdata query-playevent [--offset <offset>] [--count <count>]"
#define NN_DEVMENU_COMMAND_QUERY_ACCOUNTPLAYEVENT_USAGE "playdata query-playevent-account [--account <index>] [--offset <offset>] [--count <count>]"
#define NN_DEVMENU_COMMAND_QUERY_LAST_PLAY_TIME_USAGE   "playdata query-lastplaytime [application-id]"
#define NN_DEVMENU_COMMAND_ENABLE_PLAYLOG               "playdata enable-playlog"
#define NN_DEVMENU_COMMAND_DISABLE_PLAYLOG              "playdata disable-playlog"
#define NN_DEVMENU_COMMAND_ADD_PLAYSTATISTICS           "playdata add-playstatistics --id <application-id> --starttime <YYYYMMDDhhmmss> --playtime <minutes> [--account <index>]"
#define NN_DEVMENU_COMMAND_ADD_PLAYSTATISTICS_FROM_CSV  "playdata add-playstatistics-from-csv <csv-file>"
#define NN_DEVMENU_COMMAND_SET_LASTPLAYTIME             "playdata set-lastplaytime --id <application-id> [--user-clock-time <YYYYMMDDhhmmss>] [--network-clock-time <YYYYMMDDhhmmss>] [--elapsed <minutes>]"
#define NN_DEVMENU_COMMAND_CLEAR_PLAYEVENT              "playdata clear-playevent"
#define NN_DEVMENU_COMMAND_ENABLE_FORCED_MIGRATION      "playdata enable-forced-migration"
#define NN_DEVMENU_COMMAND_DISABLE_FORCED_MIGRATION     "playdata disable-forced-migration"
#endif

namespace {

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_QUERY_PLAYSTATISTICS "\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_QUERY_PLAYEVENT_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_QUERY_ACCOUNTPLAYEVENT_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_QUERY_LAST_PLAY_TIME_USAGE "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ENABLE_PLAYLOG "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_DISABLE_PLAYLOG "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ADD_PLAYSTATISTICS "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ADD_PLAYSTATISTICS_FROM_CSV "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_SET_LASTPLAYTIME "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_CLEAR_PLAYEVENT "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_ENABLE_FORCED_MIGRATION "\n"
        "       " DEVMENUCOMMAND_NAME " " NN_DEVMENU_COMMAND_DISABLE_FORCED_MIGRATION "\n"
#endif
        ""; // 終端

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    util::optional<account::Uid> GetUserAccountId(int index)
    {
        account::InitializeForAdministrator();
        int count;
        account::Uid users[account::UserCountMax];
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
        if( index >= count )
        {
            DEVMENUCOMMAND_LOG("The number of registered accounts is %d. Index %d is out of range.\n", count, index);
            return nullptr;
        }
        return users[index];
    }

    util::optional<account::Uid> GetUserAccountId(const Option& option)
    {
        auto indexStr = option.GetValue("--account");
        if( indexStr == nullptr || strlen(indexStr) == 0 )
        {
            DEVMENUCOMMAND_LOG("Please provide index value for --account option.\n");
            return nullptr;
        }
        char* endPtr;
        int index = static_cast<int>(STR_TO_ULL(indexStr, &endPtr, 10));
        if( *endPtr != '\0' )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid for --account option. Please provide a number.\n", indexStr);
            return nullptr;
        }

        return GetUserAccountId(index);
    }

    void GetDateTimeString(char* outBuffer, size_t outBufferSize, time::PosixTime posixTime)
    {
        if( posixTime < time::InputPosixTimeMin || posixTime > time::InputPosixTimeMax )
        {
            util::SNPrintf(outBuffer, outBufferSize, "(Invalid)");
            return;
        }

        time::CalendarTime calendarTime;
        time::CalendarAdditionalInfo calendarAdditionalInfo;
        auto result = time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
        if( result.IsSuccess() )
        {
            util::SNPrintf(outBuffer, outBufferSize, "%04d-%02d-%02d %02d:%02d:%02d (%s)",
                calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
                calendarAdditionalInfo.timeZone.standardTimeName);
        }
        else
        {
            util::SNPrintf(outBuffer, outBufferSize, "(GetDateTimeString failed : 0x%08x)", result.GetInnerValueForDebug());
        }
    }

    time::PosixTime GetPosixTimeFromElapsedMinutes(uint32_t elapsedMinSincePosixTimeMin)
    {
        return time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(static_cast<int64_t>(elapsedMinSincePosixTimeMin));
    }

    void DumpEventTimeData(const pdm::EventTimeData& eventTimeData)
    {
        const auto TimeStringSize = 64;

        char userTimeString[TimeStringSize];
        char networkTimeString[TimeStringSize];
        GetDateTimeString(userTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(eventTimeData.userClockTime));
        if( eventTimeData.networkClockTime == 0 )
        {
            util::Strlcpy(networkTimeString, "(Invalid)", sizeof(networkTimeString));
        }
        else
        {
            GetDateTimeString(networkTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(eventTimeData.networkClockTime));
        }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        DEVMENUCOMMAND_LOG("Index         : %d\n", eventTimeData.eventIndex);
#endif
        DEVMENUCOMMAND_LOG("UserClockTime : %s\n", userTimeString);
        DEVMENUCOMMAND_LOG("NetClockTime  : %s\n", networkTimeString);
    }

    void DumpPlayStatistics(const pdm::PlayStatistics& playStatistics)
    {
        auto separator = "------------------------------\n";
        DEVMENUCOMMAND_LOG(separator);
        DEVMENUCOMMAND_LOG("ApplicationId : 0x%016llx\n", playStatistics.applicationId.value);
        DEVMENUCOMMAND_LOG("Play Count    : %u\n", playStatistics.totalPlayCount);
        DEVMENUCOMMAND_LOG("Play Time     : %u minutes\n", playStatistics.totalPlayTime);
        DEVMENUCOMMAND_LOG("[First Play]\n");
        DumpEventTimeData(playStatistics.firstEventTime);
        DEVMENUCOMMAND_LOG("[Latest Play]\n");
        DumpEventTimeData(playStatistics.latestEventTime);
        DEVMENUCOMMAND_LOG(separator);
    }

    Result QueryStatisticsCommand(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        if( idString == nullptr || strlen(idString) == 0 )
        {
            DEVMENUCOMMAND_LOG("usage: %s\n", NN_DEVMENU_COMMAND_QUERY_PLAYSTATISTICS);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ncm::ApplicationId applicationId = { STR_TO_ULL(idString, NULL, 16) };

        if( applicationId == ncm::ApplicationId::GetInvalidId() )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid ApplicationId.\n", idString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if( option.HasKey("--account") )
        {
            auto uid = GetUserAccountId(option);
            if( !uid )
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            auto playStatistics = pdm::QueryPlayStatistics(applicationId, *uid);
            if( playStatistics )
            {
                DumpPlayStatistics(*playStatistics);
            }
            else
            {
                DEVMENUCOMMAND_LOG("0x%016llx for the specified account is not found.\n", applicationId);
            }
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            auto playStatistics = pdm::QueryPlayStatistics(applicationId);
            if( playStatistics )
            {
                DumpPlayStatistics(*playStatistics);
            }
            else
            {
                DEVMENUCOMMAND_LOG("0x%016llx is not found.\n", applicationId);
            }
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
    }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

    bool IsLibraryApplet(applet::AppletId id)
    {
        switch( id )
        {
        case applet::AppletId::AppletId_None:
        case applet::AppletId::AppletId_Application:
        case applet::AppletId::AppletId_OverlayApplet:
        case applet::AppletId::AppletId_SystemAppletMenu:
        case applet::AppletId::AppletId_SystemApplication:
            return false;
        default:
            return true;
        }
    }

    void ShowPleaseRebootLog()
    {
        DEVMENUCOMMAND_LOG("*** Please reboot the target to allow the change to take effect.\n");
    }

    uint32_t Hi(uint64_t value) NN_NOEXCEPT
    {
        return static_cast<uint32_t>((value >> 32) & 0xffffffff);
    }

    uint32_t Low(uint64_t value) NN_NOEXCEPT
    {
        return static_cast<uint32_t>((value >> 0) & 0xffffffff);
    }

    uint64_t ToUint64(uint32_t hi, uint32_t low) NN_NOEXCEPT
    {
        return (static_cast<uint64_t>(hi) << 32) | low;
    }

    ncm::ProgramId GetProgramId(const PlayEvent& playEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(PlayEventCategory::Applet, playEvent.eventCategory);
        nn::ncm::ProgramId id{ ToUint64(playEvent.appletEventData.programIdHi, playEvent.appletEventData.programIdLow) };
        return id;
    }

    nn::account::Uid GetUid(const PlayEvent& playEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
        nn::account::Uid uid;
        uid._data[0] = ToUint64(playEvent.userAccountEventData.userId0Hi, playEvent.userAccountEventData.userId0Low);
        uid._data[1] = ToUint64(playEvent.userAccountEventData.userId1Hi, playEvent.userAccountEventData.userId1Low);
        return uid;
    }

    nn::account::NetworkServiceAccountId GetNetworkServiceAccountId(const PlayEvent& playEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(PlayEventCategory::UserAccount, playEvent.eventCategory);
        NN_SDK_ASSERT_EQUAL(UserAccountEventType::NetworkServiceAccountAvailable, playEvent.userAccountEventData.eventType);
        nn::account::NetworkServiceAccountId nid;
        nid.id = ToUint64(playEvent.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdHi,
            playEvent.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdLow);
        return nid;
    }

    PlayEvent MakePlayEventCommon(PlayEventCategory category, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
    {
        PlayEvent playEvent;
        memset(&playEvent, 0, sizeof(PlayEvent));
        playEvent.eventCategory = category;
        playEvent.userTime = userClockTime;
        playEvent.networkTime = networkClockTime;
        playEvent.steadyTime = steadyClockTime;
        return playEvent;
    }

    PlayEvent MakeApplicationPlayEvent(AppletEventType eventType, ncm::ApplicationId applicationId, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
    {
        PlayEvent pe = MakePlayEventCommon(PlayEventCategory::Applet, userClockTime, networkClockTime, steadyClockTime);
        pe.appletEventData.eventType = eventType;
        pe.appletEventData.programIdHi = Hi(applicationId.value);
        pe.appletEventData.programIdLow = Low(applicationId.value);
        pe.appletEventData.version = 0;
        pe.appletEventData.appletId = applet::AppletId_Application;
        pe.appletEventData.storageId = ncm::StorageId::Card;
        pe.appletEventData.playLogPolicy = ns::PlayLogPolicy::All;
        return pe;
    }

    PlayEvent MakeUserAccountEvent(UserAccountEventType eventType, const nn::account::Uid& user, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(eventType != UserAccountEventType::NetworkServiceAccountAvailable);
        PlayEvent pe = MakePlayEventCommon(PlayEventCategory::UserAccount, userClockTime, networkClockTime, steadyClockTime);
        pe.userAccountEventData.eventType = eventType;
        pe.userAccountEventData.userId0Hi = Hi(user._data[0]);
        pe.userAccountEventData.userId0Low = Low(user._data[0]);
        pe.userAccountEventData.userId1Hi = Hi(user._data[1]);
        pe.userAccountEventData.userId1Low = Low(user._data[1]);
        return pe;
    }

    bool ParseDateTimeStrYYYYMMDDhhmmss(time::PosixTime* pOutValue, const char* datetimeStr)
    {
        if( strlen(datetimeStr) != strlen("YYYYMMDDhhmmss") )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is not valid format. Must be \"YYYYMMDDhhmmss\"\n", datetimeStr);
            return false;
        }

        char yearStr[5];
        util::Strlcpy(yearStr, datetimeStr, 5);
        char monthStr[3];
        util::Strlcpy(monthStr, datetimeStr + 4, 3);
        char dayStr[3];
        util::Strlcpy(dayStr, datetimeStr + 6, 3);
        char hourStr[3];
        util::Strlcpy(hourStr, datetimeStr + 8, 3);
        char minuteStr[3];
        util::Strlcpy(minuteStr, datetimeStr + 10, 3);
        char secondStr[3];
        util::Strlcpy(secondStr, datetimeStr + 12, 3);

        auto year = STR_TO_ULL(yearStr, NULL, 10);
        auto month = STR_TO_ULL(monthStr, NULL, 10);
        auto day = STR_TO_ULL(dayStr, NULL, 10);
        auto hour = STR_TO_ULL(hourStr, NULL, 10);
        auto minute = STR_TO_ULL(minuteStr, NULL, 10);
        auto second = STR_TO_ULL(secondStr, NULL, 10);

        time::CalendarTime calendarTime;
        calendarTime.year = static_cast<int16_t>(year);
        calendarTime.month = static_cast<int8_t>(month);
        calendarTime.day = static_cast<int8_t>(day);
        calendarTime.hour = static_cast<int8_t>(hour);
        calendarTime.minute = static_cast<int8_t>(minute);
        calendarTime.second = static_cast<int8_t>(second);

        int outCount;
        auto result = time::ToPosixTime(&outCount, pOutValue, 1, calendarTime);
        if( result.IsFailure() )
        {
            DEVMENUCOMMAND_LOG("Failed to convert \"%s\" to PosixTime : %08x.\n", datetimeStr, result.GetInnerValueForDebug());
            return false;
        }
        return true;
    }

    const char* GetPlayEventCategoryString(pdm::PlayEventCategory category) NN_NOEXCEPT
    {
        switch( category )
        {
        case pdm::PlayEventCategory::Applet:                return "Applet";
        case pdm::PlayEventCategory::UserAccount:           return "Account";
        case pdm::PlayEventCategory::PowerStateChange:      return "Power";
        case pdm::PlayEventCategory::OperationModeChange:   return "OperationModeChange";
        case pdm::PlayEventCategory::SteadyClockReset:      return "SteadyClockReset";
        default:                                            return "(unknown)";
        }
    }

    const char* GetAppletEventTypeString(pdm::AppletEventType appletEventType) NN_NOEXCEPT
    {
        switch( appletEventType )
        {
        case pdm::AppletEventType::Launch:      return "Launch";
        case pdm::AppletEventType::Exit:        return "Exit";
        case pdm::AppletEventType::InFocus:     return "InFocus";
        case pdm::AppletEventType::OutOfFocus:  return "OutOfFocus";
        case pdm::AppletEventType::Background:  return "Background";
        case pdm::AppletEventType::AbnormalExit:return "AbnExit";
        case pdm::AppletEventType::Terminate:   return "Terminate";
        default:                                return "(unknown)";
        }
    }

    const char* GetUserAccountEventTypeString(pdm::UserAccountEventType accountEventType) NN_NOEXCEPT
    {
        switch( accountEventType )
        {
        case pdm::UserAccountEventType::Open:    return "Open";
        case pdm::UserAccountEventType::Close:   return "Close";
        case pdm::UserAccountEventType::NetworkServiceAccountAvailable:   return "NsaAvaiable";
        case pdm::UserAccountEventType::NetworkServiceAccountUnavailable: return "NsaUnavailable";
        default:                                                          return "(unknown)";
        }
    }

    const char* GetAppletIdString(uint8_t appletId) NN_NOEXCEPT
    {
        switch( appletId )
        {
        case applet::AppletId_None:                         return "None";
        case applet::AppletId_Application:                  return "A";
        case applet::AppletId_OverlayApplet:                return "OA";
        case applet::AppletId_SystemAppletMenu:             return "SA";
        case applet::AppletId_SystemApplication:            return "SysApp";
        case applet::AppletId_LibraryAppletAuth:            return "Auth";
        case applet::AppletId_LibraryAppletCabinet:         return "Cabinet";
        case applet::AppletId_LibraryAppletController:      return "Controller";
        case applet::AppletId_LibraryAppletDataErase:       return "DataErase";
        case applet::AppletId_LibraryAppletError:           return "Error";
        case applet::AppletId_LibraryAppletNetConnect:      return "NetConnect";
        case applet::AppletId_LibraryAppletPlayerSelect:    return "PlayerSelect";
        case applet::AppletId_LibraryAppletSwkbd:           return "Swkbd";
        case applet::AppletId_LibraryAppletMiiEdit:         return "MiiEdit";
        case applet::AppletId_LibraryAppletWeb:             return "Web";
        case applet::AppletId_LibraryAppletShop:            return "Shop";
        case applet::AppletId_LibraryAppletPhotoViewer:     return "PhotoViewer";
        case applet::AppletId_LibraryAppletSet:             return "Set";
        case applet::AppletId_LibraryAppletOfflineWeb:      return "OfflineWeb";
        case applet::AppletId_LibraryAppletLoginShare:      return "LoginShare";
        case applet::AppletId_LibraryAppletWifiWebAuth:     return "WifiWebAuth";
        case applet::AppletId_LibraryAppletMyPage:          return "MyPage";
        case applet::AppletId_LibraryAppletGift:            return "Gift";
        case applet::AppletId_LibraryAppletUserMigration:   return "UserMigration";
        case applet::AppletId_LibraryAppletEncounter:       return "Encounter";
        case applet::AppletId_LibraryAppletStory:           return "Story";
        default:                                            return "(unknown)";
        }
    }

    const char* GetLogPolicyString(ns::PlayLogPolicy policy) NN_NOEXCEPT
    {
        switch( policy )
        {
        case ns::PlayLogPolicy::All:    return "All";
        case ns::PlayLogPolicy::LogOnly:return "LogOnly";
        case ns::PlayLogPolicy::None:   return "None";
        default:                        return "(unknown)";
        }
    }

    const char* GetOperationModeString(pdm::OperationMode mode) NN_NOEXCEPT
    {
        switch( mode )
        {
        case pdm::OperationMode::Handheld:  return "HandHeld";
        case pdm::OperationMode::Console:   return "Console";
        default:                            return "(unknown)";
        }
    }

    const char* GetPowerStateString(pdm::PowerStateChangeEventType powerState) NN_NOEXCEPT
    {
        switch( powerState )
        {
        case pdm::PowerStateChangeEventType::On:                        return "On";
        case pdm::PowerStateChangeEventType::Off:                       return "Off";
        case pdm::PowerStateChangeEventType::Awake:                     return "Awake";
        case pdm::PowerStateChangeEventType::Sleep:                     return "Sleep";
        case pdm::PowerStateChangeEventType::BackgroundServicesAwake:   return "BackgroundServicesAwake";
        case pdm::PowerStateChangeEventType::Terminate:                 return "Terminate";
        default:                                                        return "(unknown)";
        }
    }

    void GetPlayEventCommonDataString(char* buffer, size_t bufferSize, const pdm::PlayEvent& playEvent) NN_NOEXCEPT
    {
        const auto TimeStringSize = 64;
        char userTimeString[TimeStringSize];
        char networkTimeString[TimeStringSize];
        GetDateTimeString(userTimeString, TimeStringSize, playEvent.userTime);
        if( playEvent.networkTime.value == 0 )
        {
            util::Strlcpy(networkTimeString, "(Invalid)", sizeof(networkTimeString));
        }
        else
        {
            GetDateTimeString(networkTimeString, TimeStringSize, playEvent.networkTime);
        }
        util::SNPrintf(buffer, bufferSize, "%s, %s, %s, %lld",
            GetPlayEventCategoryString(playEvent.eventCategory), userTimeString, networkTimeString, playEvent.steadyTime);
    }

    void GetAppletPlayEventString(char* buffer, size_t bufferSize, const pdm::PlayEvent& playEvent) NN_NOEXCEPT
    {
        auto data = playEvent.appletEventData;
        util::SNPrintf(buffer, bufferSize, "%s, 0x%016llx, %u, %s, %s, %s",
            GetAppletEventTypeString(data.eventType), GetProgramId(playEvent).value, data.version,
            GetAppletIdString(data.appletId), devmenuUtil::GetStorageIdString(data.storageId), GetLogPolicyString(data.playLogPolicy));
    }

    void GetUserAccountEventString(char* buffer, size_t bufferSize, const pdm::PlayEvent& playEvent) NN_NOEXCEPT
    {
        auto data = playEvent.userAccountEventData;
        auto uid = GetUid(playEvent);
        const auto IdStringSize = 64;
        char uidString[IdStringSize];
        util::SNPrintf(uidString, IdStringSize, "%016llx-%016llx", uid._data[0], uid._data[1]);
        if( data.eventType == pdm::UserAccountEventType::NetworkServiceAccountAvailable )
        {
            char nsaIdString[IdStringSize];
            util::SNPrintf(nsaIdString, IdStringSize, "%016llx", GetNetworkServiceAccountId(playEvent).id);
            util::SNPrintf(buffer, bufferSize, "%s, %s, %s",
                GetUserAccountEventTypeString(data.eventType), uidString, nsaIdString);
        }
        else
        {
            util::SNPrintf(buffer, bufferSize, "%s, %s",
                GetUserAccountEventTypeString(data.eventType), uidString);
        }
    }

    void DumpPlayEvent(const pdm::PlayEvent& playEvent) NN_NOEXCEPT
    {
        char commonString[128]{};
        char dataString[128]{};
        GetPlayEventCommonDataString(commonString, 128, playEvent);
        switch( playEvent.eventCategory )
        {
        case pdm::PlayEventCategory::Applet:
            GetAppletPlayEventString(dataString, sizeof(dataString), playEvent);
            break;
        case pdm::PlayEventCategory::UserAccount:
            GetUserAccountEventString(dataString, sizeof(dataString), playEvent);
            break;
        case pdm::PlayEventCategory::OperationModeChange:
            util::Strlcpy(dataString, GetOperationModeString(playEvent.operationModeEventData.operationMode), sizeof(dataString));
            break;
        case pdm::PlayEventCategory::PowerStateChange:
            util::Strlcpy(dataString, GetPowerStateString(playEvent.powerStateChangeEventData.eventType), sizeof(dataString));
            break;
        case pdm::PlayEventCategory::SteadyClockReset:
            // データなし。
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        DEVMENUCOMMAND_LOG("%s, %s\n", commonString, dataString);
    }

    void DumpLastPlayTime(const pdm::LastPlayTime& lastPlayTime)
    {
        const auto TimeStringSize = 64;

        char userTimeString[TimeStringSize];
        char networkTimeString[TimeStringSize];
        GetDateTimeString(userTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(lastPlayTime.userClockTime));
        if( lastPlayTime.networkClockTime == 0 )
        {
            util::Strlcpy(networkTimeString, "(Invalid)", sizeof(networkTimeString));
        }
        else
        {
            GetDateTimeString(networkTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(lastPlayTime.networkClockTime));
        }

        char steadyTimeString[TimeStringSize];
        if( lastPlayTime.isElapsedMinutesAvailable )
        {
            nn::util::SNPrintf(steadyTimeString, TimeStringSize, "%u minutes ago", lastPlayTime.elapsedMinutesSinceLastPlay);
        }
        else
        {
            const char* unavailableMessage = "(unavailable)";
            nn::util::Strlcpy(steadyTimeString, unavailableMessage, TimeStringSize);
        }

        DEVMENUCOMMAND_LOG("0x%016llx, %s, %s, %s.\n", lastPlayTime.applicationId.value, userTimeString, networkTimeString, steadyTimeString);
    }

    void Trim(std::string &s)
    {
        s.erase(s.begin(), std::find_if(s.begin(), s.end(), std::not1(std::ptr_fun<int, int>(std::isspace))));
        s.erase(std::find_if(s.rbegin(), s.rend(), std::not1(std::ptr_fun<int, int>(std::isspace))).base(), s.end());
    }

    std::vector<std::string> Split(const std::string& input, char delimiter)
    {
        std::istringstream stream(input);

        std::string field;
        std::vector<std::string> result;
        while( std::getline(stream, field, delimiter) )
        {
            Trim(field);
            result.push_back(field);
        }
        return result;
    }

    Result QueryPlayEventCommand(bool* outValue, const Option& option)
    {
        int storedPlayEventStartIndex;
        int storedPlayEventLastIndex;
        auto storedPlayEventCount = pdm::GetAvailablePlayEventRange(&storedPlayEventStartIndex, &storedPlayEventLastIndex);
        DEVMENUCOMMAND_LOG("The number of stored PlayEvent : %d (%d - %d)\n", storedPlayEventCount, storedPlayEventStartIndex, storedPlayEventLastIndex);

        if( storedPlayEventCount == 0 )
        {
            DEVMENUCOMMAND_LOG("No PlayEvent is stored.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        int offset = 0;
        if( option.HasKey("--offset") )
        {
            offset = static_cast<int>(STR_TO_ULL(option.GetValue("--offset"), NULL, 10));
        }
        if( offset >= storedPlayEventCount )
        {
            DEVMENUCOMMAND_LOG("--offset value must be less than the number of stored PlayEvent.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int count = storedPlayEventCount;
        if( option.HasKey("--count") )
        {
            count = static_cast<int>(STR_TO_ULL(option.GetValue("--count"), NULL, 10));
            if( count == 0 )
            {
                DEVMENUCOMMAND_LOG("--count value must be greater than 0.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        if( offset + count > storedPlayEventCount )
        {
            count = storedPlayEventCount - offset;
        }

        DEVMENUCOMMAND_LOG("Start to read %d events from offset %d.\n", count, offset);

        const int PlayEventBufferSize = 2048;
        std::unique_ptr<pdm::PlayEvent[]> playEventBuffer(new pdm::PlayEvent[PlayEventBufferSize]);

        auto readCountSum = 0;
        while( NN_STATIC_CONDITION(true) )
        {
            auto readCount = (readCountSum + PlayEventBufferSize <= count) ? PlayEventBufferSize : count - readCountSum;
            auto actualReadCount = pdm::QueryPlayEvent(playEventBuffer.get(), readCount, offset + readCountSum);
            NN_SDK_ASSERT(actualReadCount == readCount);
            for( int i = 0; i < actualReadCount; i++ )
            {
                DEVMENUCOMMAND_LOG("[%d] ", readCountSum + i + offset);
                DumpPlayEvent(playEventBuffer[i]);
            }
            readCountSum += actualReadCount;
            if( readCountSum == count )
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }
    }

    //!----------------------------------------------------------------------------------
    //! @name AccountPlayEvent
    //@{

    class AccountPlayEventDumper
    {
    public:
        const char* DumpCommon(const pdm::AccountPlayEvent& entry) NN_NOEXCEPT
        {
            char user[32];
            char network[32];
            GetDateTimeString(user, NN_ARRAY_SIZE(user), entry.time.user);
            GetDateTimeString(network, NN_ARRAY_SIZE(network), entry.time.network);
            util::SNPrintf(m_Common, NN_ARRAY_SIZE(m_Common), "%8s, %7s, %26s, %26s",
                GetCategoryString(entry.category),
                GetPowerChangeString(entry.powerChange),
                user, network
            );
            return m_Common;
        }

        const char* DumpContext(const pdm::AccountPlayEvent& entry) NN_NOEXCEPT
        {
            switch (entry.category)
            {
            case pdm::AccountPlayEvent::Category::InOpen:
                return DumpContext(entry.context.inOpen);
            case pdm::AccountPlayEvent::Category::InFocus:
                return DumpContext(entry.context.inFocus);
            case pdm::AccountPlayEvent::Category::NsaAvailable:
                return DumpContext(entry.context.nsaAvailable);
            case pdm::AccountPlayEvent::Category::NsaUnavailable:
                return DumpContext(entry.context.nsaUnavailable);
            default:
                break;
            }
            return "Nothing";
        }

        const char* DumpContext(const pdm::AccountPlayEvent::Context::InOpen& entry) NN_NOEXCEPT
        {
            char exInfoStr[16];
            if( IsLibraryApplet(static_cast<applet::AppletId>(entry.applet.appletId)) )
            {
                util::SNPrintf(exInfoStr, NN_ARRAY_SIZE(exInfoStr), "(%u, %u)", entry.applet.laInfo.version, entry.applet.laInfo.libraryAppletMode);
            }
            else
            {
                util::SNPrintf(exInfoStr, NN_ARRAY_SIZE(exInfoStr), "0x%08lx", entry.applet.version);
            }
            util::SNPrintf(m_Context, NN_ARRAY_SIZE(m_Context), "%6lu sec, %12s, %9s, [ 0x%016llx, %16s, %15s, %13s ]",
                entry.duration,
                GetRecordCauseString(entry.applet.cause),
                GetOperationModeString(entry.applet.operationMode),
                entry.applet.programId.ToValue64(),
                exInfoStr,
                GetAppletIdString(entry.applet.appletId),
                devmenuUtil::GetStorageIdString(entry.applet.storageId)
            );
            return m_Context;
        }

        const char* DumpContext(const pdm::AccountPlayEvent::Context::NetworkServiceAccountAvailable& entry) NN_NOEXCEPT
        {
            util::SNPrintf(m_Context, NN_ARRAY_SIZE(m_Context), "0x%016llx", entry.nsaId.ToValue64());
            return m_Context;
        }

        const char* DumpContext(const pdm::AccountPlayEvent::Context::NetworkServiceAccountUnavailable& entry) NN_NOEXCEPT
        {
            NN_UNUSED(entry);
            return "Nothing";
        }

        static const char* GetRecordCauseString(const pdm::AccountPlayEvent::RecordCause cause) NN_NOEXCEPT
        {
            const char* s_Types[] = {"None", "PowerOff", "FocusOut", "AccountClose"};
            const auto typeIndex = static_cast<int>(cause);
            return s_Types[typeIndex];
        }

        static const char* GetCategoryString(const pdm::AccountPlayEvent::Category type) NN_NOEXCEPT
        {
            const char* s_Types[] = {"InOpen", "InFocus", "NsaOn", "NsaOff"};
            const auto typeIndex = static_cast<int>(type);
            return s_Types[typeIndex];
        }

        static const char* GetPowerChangeString(const pdm::AccountPlayEvent::PowerChange type) NN_NOEXCEPT
        {
            const char* s_Types[] = {"None", "PowerOn"};
            const auto typeIndex = static_cast<int>(type);
            return s_Types[typeIndex];
        }

    private:
        char    m_Common[96];
        char    m_Context[160];
    };

    void DumpEvent(int nowOffset, const pdm::AccountPlayEvent& entry) NN_NOEXCEPT
    {
        AccountPlayEventDumper dumper;
        DEVMENUCOMMAND_LOG("[%6d] %s: %s\n", nowOffset, dumper.DumpCommon(entry), dumper.DumpContext(entry));
    }

    bool DumpAccountPlayEvents(const account::Uid& uid, const int offset, const int count) NN_NOEXCEPT
    {
        int lastIndex;
        int startIndex;
        const auto storedCount = pdm::GetAvailableAccountPlayEventRange(&startIndex, &lastIndex, uid);
        if (0 >= storedCount)
        {
            DEVMENUCOMMAND_LOG("Could not find the available AccountPlayEvent on UID[0x%016llx-0x%016llx].\n", uid._data[0], uid._data[1]);
            return true;
        }

        // 指定 offset 値のチェック。
        if (offset >= storedCount)
        {
            DEVMENUCOMMAND_LOG("--offset value must be less than the number of stored AccountPlayEvent on UID[0x%016llx-0x%016llx].\n", uid._data[0], uid._data[1]);
            return false;
        }

        // 読み込み総数調整。
        auto totalCount = (count > 0) ? count : storedCount;
        if ((offset + totalCount) > storedCount)
        {
            totalCount = totalCount - offset;
        }
        DEVMENUCOMMAND_LOG("====<< UID[0x%016llx-0x%016llx]: Start to read %d events from offset %d >>=====\n", uid._data[0], uid._data[1], totalCount, offset);
        DEVMENUCOMMAND_LOG("[ index] Category,   Power, %26s, %26s: Detail of context\n", "UserClock(Posix)", "NetworkClock(Posix)");

        const int BufferSize = 2048;
        const std::unique_ptr<pdm::AccountPlayEvent[]> buffer(new pdm::AccountPlayEvent[BufferSize]);

        int readedCount = 0;
        while (readedCount < totalCount)
        {
            const auto nowOffset = offset + readedCount;
            const auto expectReadCount = ((readedCount + BufferSize) <= totalCount) ? BufferSize : (totalCount - readedCount);
            const auto actualReadCount = pdm::QueryAccountPlayEvent(buffer.get(), expectReadCount, nowOffset, uid);
            NN_SDK_ASSERT(actualReadCount == expectReadCount);
            for (int i = 0; i < actualReadCount; ++i)
            {
                DumpEvent(nowOffset + i, buffer[i]);
            }
            readedCount += actualReadCount;
        }
        return true;
    }

    Result QueryAccountPlayEventCommand(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        const int offset = (option.HasKey("--offset")) ? static_cast<int>(STR_TO_ULL(option.GetValue("--offset"), NULL, 10)) : 0;

        int count = 0;
        if (option.HasKey("--count") && 0 >= (count = static_cast<int>(STR_TO_ULL(option.GetValue("--count"), NULL, 10))))
        {
            DEVMENUCOMMAND_LOG("--count value must be greater than 0.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (option.HasKey("--account"))
        {
            const auto uid = GetUserAccountId(option);
            if (!uid)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            *outValue = DumpAccountPlayEvents(*uid, offset, count);
            NN_RESULT_SUCCESS;
        }

        // --account 指定なければ、全ユーザ。
        int nUsers;
        *outValue = false;
        account::InitializeForAdministrator();
        account::Uid users[account::UserCountMax];
        NN_RESULT_DO(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
        for (int i = 0;  i < nUsers; ++i)
        {
            DumpAccountPlayEvents(users[i], 0, count);
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    void ApplyForcedMigrationSettings(const bool request) NN_NOEXCEPT
    {
        bool enabled = request;
        settings::fwdbg::SetSettingsItemValue("pdm", "force_migrate_account_database", &enabled, sizeof(enabled));
        DEVMENUCOMMAND_LOG("*** Successful requested for %s the forced-migration.\n", (request ? "enable" : "disable"));
        ShowPleaseRebootLog();
    }

    Result EnableForcedMigration(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);
        ApplyForcedMigrationSettings(true);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableForcedMigration(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);
        ApplyForcedMigrationSettings(false);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    //}@
    //!----------------------------------------------------------------------------------

    Result QueryLastPlayTimeCommand(bool* outValue, const Option& option)
    {
        auto idString = option.GetTarget();
        if( strlen(idString) > 0 )
        {
            nn::ApplicationId applicationId = { STR_TO_ULL(idString, NULL, 16) };
            if( applicationId == nn::ApplicationId::GetInvalidId() )
            {
                DEVMENUCOMMAND_LOG("\"%s\" is invalid ApplicationId.\n", idString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            pdm::LastPlayTime lastPlayTime;

            auto count = pdm::QueryLastPlayTime(&lastPlayTime, &applicationId, 1);
            if( count == 1 )
            {
                DumpLastPlayTime(lastPlayTime);
            }
            else
            {
                DEVMENUCOMMAND_LOG("0x%016llx is not found.\n", applicationId.value);
            }
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            const size_t ListBufferSize = 2014;
            std::unique_ptr<ns::ApplicationRecord[]> recordBuffer(new ns::ApplicationRecord[ListBufferSize]);
            auto count = ns::ListApplicationRecord(recordBuffer.get(), ListBufferSize, 0);
            if( count > 0 )
            {
                std::unique_ptr<nn::ApplicationId[]> applicationIdBuffer(new nn::ApplicationId[count]);
                std::unique_ptr<pdm::LastPlayTime[]> lastPlayTimeBuffer(new pdm::LastPlayTime[count]);

                for( int i = 0; i < count; i++ )
                {
                    applicationIdBuffer[i].value = recordBuffer[i].id.value;
                }

                pdm::QueryLastPlayTime(lastPlayTimeBuffer.get(), applicationIdBuffer.get(), count);

                for( int i = 0; i < count; i++ )
                {
                    if( lastPlayTimeBuffer[i].applicationId == nn::ApplicationId::GetInvalidId() )
                    {
                        DEVMENUCOMMAND_LOG("0x%016llx is not found.\n", applicationIdBuffer[i]);
                    }
                    else
                    {
                        DumpLastPlayTime(lastPlayTimeBuffer[i]);
                    }
                }
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
            else
            {
                DEVMENUCOMMAND_LOG("No application found.\n");
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }
    }

    Result EnablePlaylogCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = true;
        settings::fwdbg::SetSettingsItemValue("pdm", "save_playlog", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    Result DisablePlaylogCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool enabled = false;
        settings::fwdbg::SetSettingsItemValue("pdm", "save_playlog", &enabled, sizeof(enabled));
        *outValue = true;
        ShowPleaseRebootLog();
        NN_RESULT_SUCCESS;
    }

    void AddPlayStatisticsCore(ncm::ApplicationId applicationId, time::PosixTime startTime, int playTime, util::optional<account::Uid> uid)
    {
        /**
        終了時刻 = 開始時刻 + プレイ時間として以下のようなイベント列を入れ込むことで統計情報（プレイ時間）を追加する。

        1. [開始時刻] アプリの起動
        2. [開始時刻] アプリのフォーカス取得
        3. [開始時刻] アカウントのオープン（アカウント指定があれば）
        4. [終了時刻] アカウントのクローズ（アカウント指定があれば）
        5. [終了時刻] アプリのフォーカス喪失
        6. [終了時刻] アプリの終了
        */

        const int PlayEventCountMax = 6;
        int playEventCount = 0;
        pdm::PlayEvent playEvent[PlayEventCountMax];

        playEvent[playEventCount] = MakeApplicationPlayEvent(AppletEventType::Launch, applicationId, startTime, startTime, 0);
        playEventCount++;
        playEvent[playEventCount] = MakeApplicationPlayEvent(AppletEventType::InFocus, applicationId, startTime, startTime, 0);
        playEventCount++;

        auto endTime = startTime + nn::TimeSpan::FromMinutes(static_cast<int64_t>(playTime));

        if( uid )
        {
            playEvent[playEventCount] = MakeUserAccountEvent(pdm::UserAccountEventType::Open, *uid, startTime, startTime, 0);
            playEventCount++;
            playEvent[playEventCount] = MakeUserAccountEvent(pdm::UserAccountEventType::Close, *uid, endTime, endTime, playTime * 60);
            playEventCount++;
        }

        playEvent[playEventCount] = MakeApplicationPlayEvent(AppletEventType::Background, applicationId, endTime, endTime, playTime * 60);
        playEventCount++;
        playEvent[playEventCount] = MakeApplicationPlayEvent(AppletEventType::Exit, applicationId, endTime, endTime, playTime * 60);
        playEventCount++;

        pdm::NotifyEventForDebug(playEvent, playEventCount);
    }

    Result AddPlayStatisticsCommand(bool* outValue, const Option& option)
    {
        if( !option.HasKey("--id") )
        {
            DEVMENUCOMMAND_LOG("--id option is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto idString = option.GetValue("--id");
        ncm::ApplicationId applicationId = { STR_TO_ULL(idString, NULL, 16) };
        if( applicationId == ncm::ApplicationId::GetInvalidId() )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid ApplicationId.\n", idString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if( !option.HasKey("--starttime") )
        {
            DEVMENUCOMMAND_LOG("--starttime option is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto startTimeStr = option.GetValue("--starttime");
        time::PosixTime startTime;
        if( !ParseDateTimeStrYYYYMMDDhhmmss(&startTime, startTimeStr) )
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if( !option.HasKey("--playtime") )
        {
            DEVMENUCOMMAND_LOG("--playtime option is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto playTimeStr = option.GetValue("--playtime");
        auto playTime = STR_TO_ULL(playTimeStr, NULL, 10);

        util::optional<account::Uid> uid;
        if( option.HasKey("--account") )
        {
            uid = GetUserAccountId(option);
            if( !uid )
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        pdm::InitializeForNotification();
        NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForNotification(); };

        AddPlayStatisticsCore(applicationId, startTime, static_cast<int>(playTime), uid);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result AddPlayStatisticsFromCsvCommand(bool* outValue, const Option& option)
    {
        *outValue = false;

        auto filePath = option.GetTarget();
        if( !devmenuUtil::IsAbsolutePath(filePath) )
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", filePath);
            NN_RESULT_SUCCESS;
        }

        pdm::InitializeForNotification();
        NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForNotification(); };

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

        std::unique_ptr<char[]> buffer(new char[static_cast<int>(fileSize) + 1]);
        NN_RESULT_THROW_UNLESS(buffer != nullptr, os::ResultOutOfMemory());
        NN_RESULT_DO(fs::ReadFile(file, 0, buffer.get(), static_cast<size_t>(fileSize)));
        buffer[static_cast<int>(fileSize)] = '\0';

        std::istringstream fileStream(buffer.get());
        std::string line;
        int lineNumber = 0;
        while( std::getline(fileStream, line) )
        {
            lineNumber++;
            DEVMENUCOMMAND_LOG("Line %d : %s\n", lineNumber, line.c_str());
            // ApplicationId, StartTime, PlayTime(, account)
            auto cols = Split(line, ',');
            if( cols.size() < 3 )
            {
                DEVMENUCOMMAND_LOG("Invalid format. Skip.\n");
                continue;
            }

            char* endPtr;
            ncm::ApplicationId applicationId = { STR_TO_ULL(cols[0].c_str(), &endPtr, 16) };
            if( *endPtr != '\0' || applicationId == ncm::ApplicationId::GetInvalidId() )
            {
                DEVMENUCOMMAND_LOG("Invalid ApplicationId. Skip.\n");
                continue;
            }

            time::PosixTime startTime;
            if( !ParseDateTimeStrYYYYMMDDhhmmss(&startTime, cols[1].c_str()) )
            {
                DEVMENUCOMMAND_LOG("Invalid StartTime. Skip.\n");
                continue;
            }

            auto playTime = static_cast<int>(STR_TO_ULL(cols[2].c_str(), NULL, 10));

            util::optional<account::Uid> uid;
            if( cols.size() >= 4 )
            {
                int index = static_cast<int>(STR_TO_ULL(cols[3].c_str(), NULL, 10));
                uid = GetUserAccountId(index);
            }

            AddPlayStatisticsCore(applicationId, startTime, playTime, uid);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetLastPlayTime(bool* outValue, const Option& option)
    {
        /*
           QueryLastPlayTime は pdm に記録されたイベントを一番最近のものから遡って最初に見つかった対象のアプリケーションのイベントの時刻情報を返すので、
           適切なイベントを付け加えることで返ってくる値を設定する。
        */

        pdm::InitializeForNotification();
        NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForNotification(); };
        pdm::InitializeForQuery();
        NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForQuery(); };

        if( !option.HasKey("--id") )
        {
            DEVMENUCOMMAND_LOG("--id option is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto idString = option.GetValue("--id");
        ncm::ApplicationId applicationId = { STR_TO_ULL(idString, NULL, 16) };
        if( applicationId == ncm::ApplicationId::GetInvalidId() )
        {
            DEVMENUCOMMAND_LOG("\"%s\" is invalid ApplicationId.\n", idString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // ダミーのイベントを一個加えて、即座に Query。pdm が記録している単調増加時計ベースの経過秒の現在値を取得する。
        pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, ncm::ProgramId{ applicationId.value }, 0,
            applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
        int startIndex;
        int endIndex;
        NN_ABORT_UNLESS_GREATER(pdm::GetAvailablePlayEventRange(&startIndex, &endIndex), 0);
        pdm::PlayEvent dummyPlayEvent;
        NN_ABORT_UNLESS_EQUAL(pdm::QueryPlayEvent(&dummyPlayEvent, 1, endIndex), 1);

        int64_t elapsed = option.HasKey("--elapsed") ?
            static_cast<int64_t>(STR_TO_ULL(option.GetValue("--elapsed"), NULL, 10) * 60) : 0;

        // 「最後に遊んでからの経過時間」は「現在の単調増加時計の値 - イベント記録時の単調増加時計の値」なので、現在の単調増加時計の値以上に設定することはできない。
        if( elapsed > dummyPlayEvent.steadyTime )
        {
            DEVMENUCOMMAND_LOG("The value specified for --elapsed exceeds the current maximum value (%d = The elapsed minutes since the first event recorded after the steady clock's reset).\n", (dummyPlayEvent.steadyTime / 60));
            DEVMENUCOMMAND_LOG("To increase the maximum value, add an offset to the steady clock value using DevMenu or DevMenuCommand.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        time::PosixTime userClockTime;
        if( option.HasKey("--user-clock-time") )
        {
            auto clockTimeStr = option.GetValue("--user-clock-time");
            if( !ParseDateTimeStrYYYYMMDDhhmmss(&userClockTime, clockTimeStr) )
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&userClockTime));
            userClockTime -= nn::TimeSpan::FromSeconds(elapsed);
        }

        time::PosixTime networkClockTime;
        if( option.HasKey("--network-clock-time") )
        {
            auto clockTimeStr = option.GetValue("--network-clock-time");
            if( !ParseDateTimeStrYYYYMMDDhhmmss(&networkClockTime, clockTimeStr) )
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            if( time::StandardNetworkSystemClock::GetCurrentTime(&networkClockTime).IsSuccess() )
            {
                networkClockTime -= nn::TimeSpan::FromSeconds(elapsed);
            }
            else
            {
                networkClockTime.value = 0;
            }
        }

        PlayEvent playEvent = MakeApplicationPlayEvent(pdm::AppletEventType::Launch, applicationId,
            userClockTime, networkClockTime, dummyPlayEvent.steadyTime - elapsed);
        pdm::NotifyEventForDebug(&playEvent, 1);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearPlayEventCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        pdm::InitializeForNotification();
        NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForNotification(); };
        pdm::NotifyClearAllEvent();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const SubCommand g_SubCommands[] =
    {
        { "query-playstatistics", QueryStatisticsCommand },
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        { "query-playevent", QueryPlayEventCommand },
        { "query-playevent-account", QueryAccountPlayEventCommand },
        { "query-lastplaytime", QueryLastPlayTimeCommand },
        { "enable-playlog", EnablePlaylogCommand },
        { "disable-playlog", DisablePlaylogCommand },
        { "add-playstatistics", AddPlayStatisticsCommand },
        { "add-playstatistics-from-csv", AddPlayStatisticsFromCsvCommand },
        { "set-lastplaytime", SetLastPlayTime },
        { "clear-playevent", ClearPlayEventCommand },
        { "enable-forced-migration", EnableForcedMigration },
        { "disable-forced-migration", DisableForcedMigration },
#endif
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result PlayDataCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
    pdm::InitializeForQuery();
    NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForQuery(); };

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            auto r = subCommand.function(outValue, option);
            if (!r.IsSuccess())
            {
                DEVMENUCOMMAND_LOG("!!! Operation failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
            }
            return r;
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu playdata command. See '" DEVMENUCOMMAND_NAME " playdata --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
