﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <vector>
#include <algorithm>
#include <array>
#include <iostream>
#include <cstdlib>
#include <cstdio>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/os/os_Argument.h>
#include <nn/nn_Assert.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>

// アロケータ
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h> // 流用

#include <nn/fssystem/fs_PartitionFileSystemMeta.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Types.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/dd.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>

#include <nn/idle/idle_SystemApi.h>

#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include "../../../../../Externals/pugixml/src/pugixml.hpp"
#ifndef NN_BUILD_CONFIG_OS_WIN
#include "../../../../Iris/Include/nn/hid/system/hid_InputDetection.h"
#endif

#include "DevMenuCommand_GameCardWriterImpl.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Result.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nn::fssystem;
using namespace nn::fs::detail;

using nn::util::align_up;
using nn::util::is_aligned;

typedef Sha256PartitionFileSystemMeta::FileEntryForConstruct FileEntryForConstruct;
typedef Sha256PartitionFileSystemMeta::PartitionEntry PartitionEntry;

namespace
{
    const char* const ContentMeta  = "ContentMeta";
    const char* const Content      = "Content";
    const char* const Meta         = "Meta";
    const char* const Application  = "Application";
    const char* const AddOnContent = "AddOnContent";
    const char* const Patch        = "Patch";
    const char* const Type         = "Type";
    const char* const Id           = "Id";
    const char* const Control      = "Control";
    const char* const SystemUpdate = "SystemUpdate";
    const char* const Version      = "Version";
    const char* const ProgramInfo  = "ProgramInfo";
    const char* const ToolVersion  = "ToolVersion";
    const char* const CardSpec     = "CardSpec";
    const char* const Size         = "Size";
    const char* const ClockRate    = "ClockRate";
    const char* const MultiApplicationCard = "MultiApplicationCard";
    const char* const Digest       = "Digest";
    const int ContentMetaXmlElementLengthMax = 32;

    const size_t ContentFileMaxLength = 64;
    const size_t ToolVersionMaxLength = 32;

    // Normal
    const size_t NormalAreaIdMaxNum = 16;
    char g_NormalContentsFileList[NormalAreaIdMaxNum][ContentFileMaxLength];
    int64_t g_NormalContentsFileNum = 0; // g_NormalContentsFileList に入れる nca 数

    // スレッド用
    nn::os::ThreadType g_WriteThread;
    static const int    ThreadPriority = 9;
    static const size_t ThreadStackSize = 128 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_WriteThreadStack[ ThreadStackSize ];
    char g_ThreadName[] = "WriteThread";
    bool g_IsWriting = false;
    nn::os::EventType g_WriteStartEvent;
    nn::os::EventType g_WriteFinishEvent;

#ifdef NN_BUILD_CONFIG_OS_WIN
    // FlushToGameCard 用ストレージ
    std::shared_ptr<IStorage> g_XciStorage;
#endif

    const int64_t KeyAreaSize                = 0x8 * 0x200;
    const int64_t CardHeaderSize             = 512;
    const int64_t FsPartitionHeaderOffset    = 0x78 * 0x200; // card header + reserved + cert
    const int64_t ExpectedRootHeaderSize     = 0x200;
    const int64_t XciHeaderSize              = KeyAreaSize + FsPartitionHeaderOffset; // 64KB
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    const size_t  WorkBufferSize             = 1 * 1024 * 1024; // nn::dd::DeviceAddressSpaceMemoryRegionAlignment の倍数
#else
    const size_t  WorkBufferSize             = 8 * 1024 * 1024; // nn::dd::DeviceAddressSpaceMemoryRegionAlignment の倍数
#endif
    const size_t  WorkBufferSizeFillFileData = 256 * 1024;
    char* g_pWriteBuffer;
    nn::Result g_WriteResult = nn::ResultSuccess();

    enum GameCardPartition
    {
        GameCardPartition_Update,
        GameCardPartition_Normal,
        GameCardPartition_Secure,
    };

    struct PartitionContents
    {
        size_t  headerSize;
        int64_t bodySize;
        int64_t hardwareOffset;
        std::unique_ptr<char[]> headerBuffer;
    };

    struct WritePageInfo
    {
        size_t offset;
        char* writeBuffer;
    };

    struct WriteProgress
    {
        int64_t writtenSize;
        int64_t totalSize;
        void Initialize()
        {
            writtenSize = 0;
            totalSize = 0;
        }
        void SetTotalSize(int64_t size)
        {
            totalSize   = size;
            writtenSize = 0;
        }
        nn::Result AddWrittenSize(int64_t size)
        {
            if(totalSize == 0 || writtenSize + size > totalSize)
            {
                DEVMENUCOMMAND_LOG("AddWrittenSize Unexpected Error\n");
                return ResultOutOfRange();
            }
            writtenSize += size;
            PrintProgress();
            NN_RESULT_SUCCESS;
        }
        void PrintProgress()
        {
            DEVMENUCOMMAND_LOG("%lld / %lld (%.1llf%)\n", writtenSize, totalSize, (static_cast<double>(writtenSize) / totalSize) * 100.0);
        }
    };

    WriteProgress g_WriteProgress;

    struct ProgramInfoXml
    {
        char toolVersion[ToolVersionMaxLength];
    };

    struct CardInfo
    {
        int64_t normalAreaSize;
        int64_t totalSize;
        nn::fs::GameCardSize cardSize;
        nn::fs::GameCardClockRate cardClockRate;
        int32_t cupVersion;
        int64_t cupId;
        char uppHashForLotCheck[8];

        CardInfo()
        {
            normalAreaSize = 0;
            totalSize = 0;
            memset(&cardSize , 0x00, sizeof(nn::fs::GameCardSize));
            memset(&cardClockRate , 0x00, sizeof(nn::fs::GameCardClockRate));
            cupVersion = 0;
            cupId = 0;
            memset(&uppHashForLotCheck , 0x00, sizeof(uppHashForLotCheck));
        }

        Result SetTotalSize(int64_t writeSize)
        {
            const int64_t AvailableSizeBase = (1024 - 72) * 1024 * 1024;// 1GB に対して 72MB の予備領域が必要
            if(writeSize > AvailableSizeBase * GetMaxGameCardWritebleSizeGb())
            {
                DEVMENUCOMMAND_LOG("error : Write Size > Game Card Hardware Size.\n");
                DEVMENUCOMMAND_LOG("card hardware Size : %d -> writable Size = %lld, but write Size is %lld\n", GetMaxGameCardWritebleSizeGb(), AvailableSizeBase * GetMaxGameCardWritebleSizeGb(), writeSize);
                DEVMENUCOMMAND_LOG("Please Decrease write file size or Use Large Card.\n");
                return ResultOutOfRange();
            }

            int64_t writableSize = AvailableSizeBase * static_cast<uint8_t>(cardSize);
            if(writeSize > writableSize)
            {
                DEVMENUCOMMAND_LOG("error : Write Size > Writable Size.\n");
                DEVMENUCOMMAND_LOG("cardSize : %d -> writable Size = %lld, but write Size is %lld\n", static_cast<uint8_t>(cardSize), writableSize, writeSize);
                DEVMENUCOMMAND_LOG("Please Decrease write file size or Increase given Card Size\n");
                return ResultOutOfRange();
            }
            totalSize = writeSize;
            NN_RESULT_SUCCESS;
        }

        private:
        uint8_t GetMaxGameCardWritebleSizeGb()
        {
            nn::gc::detail::DevCardParameter param;
            memset(&param, 0x00, sizeof(nn::gc::detail::DevCardParameter));
            NN_RESULT_DO(nn::fs::ReadParamDirectly(reinterpret_cast<char*>(&param), sizeof(nn::gc::detail::DevCardParameter)));
            switch(param.nandSize)
            {
            case nn::gc::detail::DevCardNandSize_16GB:
                {
                    return 16;
                }
                break;
            case nn::gc::detail::DevCardNandSize_32GB:
                {
                    return 32;
                }
                break;
            case nn::gc::detail::DevCardNandSize_64GB:
                {
                    // 今は存在しないが今後万が一出現したときのため
                    return 64;
                }
                break;
            default:
                NN_LOG("[Error] Unknown GameCard\n");
                return 0;
            }
        }
    };

    void* AllocateTest(size_t size) NN_NOEXCEPT
    {
        return std::malloc(size);
    }

    void DeallocateTest(void* addr) NN_NOEXCEPT
    {
        std::free(addr);
    }

    // @brief GetSize のみ可能なストレージ
    class PlaceHolderStorage : public IStorage
    {
    public:

        explicit PlaceHolderStorage(int64_t size)
        : m_Size(size)
        {
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_ABORT("Invalid access to PlaceHolderStorage.");
        }
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_ABORT("Invalid access to PlaceHolderStorage.");
        }
        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = m_Size;
            NN_RESULT_SUCCESS;
        }
    private:
        int64_t m_Size;
    };

    // @brief DataAlign のサイズに切り上げて書き込むストレージ
    // TORIAEZU: リードモディファイしない
    template <int DataAlign>
    class SizeAlignmentWriteOnlyStorage : public IStorage
    {
    public:
        explicit SizeAlignmentWriteOnlyStorage(std::shared_ptr<IStorage> pBaseStorage)
        : m_pBaseStorage(pBaseStorage)
        {
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            NN_ABORT("Read access is not supported.");
        }
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES( is_aligned(offset, DataAlign) );
            if( is_aligned(size, DataAlign) )
            {
                return m_pBaseStorage->Write(offset, buffer, size);
            }
            else
            {
                size_t roundedSize = nn::util::align_down(size, DataAlign);

                NN_RESULT_DO(m_pBaseStorage->Write(offset, buffer, roundedSize));

                char tmpBuffer[DataAlign] = {0};
                memcpy(tmpBuffer, static_cast<const char*>(buffer) + roundedSize, size - roundedSize);
                NN_RESULT_DO(m_pBaseStorage->Write(offset + roundedSize, tmpBuffer, DataAlign));

                NN_RESULT_SUCCESS;
            }
        }
        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->GetSize(outValue);
        }
        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->Flush();
        }
        virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->SetSize(size);
        }
        virtual Result OperateRange(
            void* pOutBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* pInBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseStorage->OperateRange(
                pOutBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                pInBuffer,
                inBufferSize);
        }

    private:
        std::shared_ptr<IStorage> m_pBaseStorage;
    };


    // @brief ストレージを連結したストレージ。ストレージをまたぐアクセスに未対応
    class CatStorage : public IStorage
    {
    public:
        explicit CatStorage(std::vector<IStorage*> pStorages)
        : m_pStorages(pStorages)
        {
        }

        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            int64_t totalOffset = 0;
            for (auto pStorage : m_pStorages)
            {
                int64_t storageSize;
                NN_RESULT_DO(pStorage->GetSize(&storageSize));
                NN_SDK_REQUIRES(!(offset < totalOffset + storageSize && totalOffset + storageSize < static_cast<int64_t>(offset + size) ));

                if( totalOffset <= offset && static_cast<int64_t>(offset + size) <= totalOffset + storageSize )
                {
                    return pStorage->Read(offset - totalOffset, buffer, size);
                }
                else
                {
                    totalOffset += storageSize;
                }

            }
            return ResultOutOfRange();
        }
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            int64_t totalOffset = 0;
            for (auto pStorage : m_pStorages)
            {
                int64_t storageSize;
                NN_RESULT_DO(pStorage->GetSize(&storageSize));
                NN_SDK_REQUIRES(!(offset < totalOffset + storageSize && totalOffset + storageSize < static_cast<int64_t>(offset + size) ));

                if( totalOffset <= offset && static_cast<int64_t>(offset + size) <= totalOffset + storageSize )
                {
                    return pStorage->Write(offset - totalOffset, buffer, size);
                }
                else
                {
                    totalOffset += storageSize;
                }

            }
            return ResultOutOfRange();
        }

        virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            for (auto pStorage : m_pStorages)
            {
                NN_RESULT_DO(pStorage->Flush());
            }
            NN_RESULT_SUCCESS;
        }
        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            int64_t totalSize = 0;
            for (auto pStorage : m_pStorages)
            {
                int64_t size = 0;
                NN_RESULT_DO(pStorage->GetSize(&size));
                totalSize += size;
            }

            *outValue = totalSize;
            NN_RESULT_SUCCESS;
        }

    private:
        std::vector<IStorage*> m_pStorages;
    };

    Result CalculateFileHash(char* pHash, size_t hashSize, IFileSystem* pFs, const char* path, int64_t hashTargetOffset, int64_t hashTargetSize)
    {
        std::unique_ptr<IFile> file;
        NN_RESULT_DO(pFs->OpenFile(&file, path, OpenMode_Read));

        const size_t BufferSize = 512;
        char buffer[BufferSize];
        NN_ASSERT(hashTargetSize <= BufferSize);

        size_t readSize;
        NN_RESULT_DO(file->Read(&readSize, hashTargetOffset, buffer, static_cast<size_t>(hashTargetSize), ReadOption()));
        NN_ASSERT(static_cast<int64_t>(readSize) == hashTargetSize);

        crypto::GenerateSha256Hash(pHash, hashSize, buffer, static_cast<size_t>(hashTargetSize));

        NN_RESULT_SUCCESS;
    }

    bool GetXmlNodeChild(::pugi::xml_node* pOutValue, const ::pugi::xml_node& node, const char* name) NN_NOEXCEPT
    {
        const ::pugi::xml_node child = node.child(name);

        if (child.empty())
        {
            return false;
        }

        *pOutValue = child;

        return true;
    }

    typedef bool (*MatchContentFunc)(const char* typeName);

    bool IsMatchNormalContentsList(const char* typeName)
    {
        // Normal に引き出す Type のリスト(XML に記述)
        const char* const NormalContents[] = { Control, Meta };
        for(size_t i = 0; i < sizeof(NormalContents) / sizeof(char*); i++)
        {
            if(nn::util::Strncmp(typeName, NormalContents[i], ContentFileMaxLength) == 0)
            {
                return true;
            }
        }
        return false;
    }

    bool IsMatchNormalNcaList(const char* ncaName)
    {
        for(int64_t i = 0; i < g_NormalContentsFileNum; i++)
        {
            if( nn::util::Strncmp(ncaName, g_NormalContentsFileList[i], ContentFileMaxLength) == 0)
            {
                return true;
            }
        }
        return false;
    }

    uint64_t CharToInt1Byte(const char* pIn)
    {
        char buf[3];
        memcpy(buf, pIn, 2);
        buf[2] = '\0';
        uint32_t output;
        sscanf(buf, "%x", &output);
        return output;
    }

    //! 与えられた inBuffer を UpdatePartition.xml として解釈して <Hash> の値を取得する
    //! UpdatePartition.xml に <Hash> がない場合は ABORT する
    bool CanGetHashFromUpdatePartitionXml(CardInfo* pCardInfo, char* inBuffer, size_t inBufferSize)
    {
        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // UpdatePartition の解釈
        if(GetXmlNodeChild(&rootNode, document, "UpdatePartition"))
        {
            auto childNode = rootNode.child("Hash");
            if (childNode.empty())
            {
                return false;
            }
            {
                auto hash = std::strtoull(childNode.child_value(), nullptr, 16);
                for (int i = 0; i < sizeof(pCardInfo->uppHashForLotCheck); i++)
                {
                    pCardInfo->uppHashForLotCheck[7 - i] = static_cast<char>((hash >> (8 * i)) & 0xFF);
                }
                DEVMENUCOMMAND_LOG("UPP HASH    : 0x");
                for (int i = 0; i < sizeof(pCardInfo->uppHashForLotCheck); i++)
                {
                    DEVMENUCOMMAND_LOG("%02x", 0xFF & pCardInfo->uppHashForLotCheck[i]);
                }
                DEVMENUCOMMAND_LOG("\n");
            }
            return true;
        }
        else
        {
            NN_ABORT("Hash not found in UpdatePartition.xml in Update\n");
        }
    }

    //! 与えられた inBuffer を アプリの cnmt.xml として解釈して Type が SystemUpdate なら CUP バージョンを取得する
    //! cnmt.xml に <ContentMeta> がない場合に ABORT する
    bool CanGetCupVersionAndIdFromContentMeta(CardInfo* pCardInfo, char* inBuffer, size_t inBufferSize)
    {
        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // ContentMeta の解釈
        if(GetXmlNodeChild(&rootNode, document, ContentMeta))
        {
            auto childNode = rootNode.child(Type);
            if(nn::util::Strncmp(childNode.child_value(), SystemUpdate, nn::util::Strnlen(SystemUpdate, ContentMetaXmlElementLengthMax)) != 0)
            {
                return false;
            }
            childNode = rootNode.child(Version);
            sscanf(childNode.child_value(), "%d", &pCardInfo->cupVersion);
            childNode = rootNode.child(Id);
            pCardInfo->cupId = 0;
            static const size_t CupIdByteNum = 8;
            for(size_t i = 0; i < CupIdByteNum; i++)
            {
                // 1Byte ずつ sscanf
                pCardInfo->cupId += (CharToInt1Byte(reinterpret_cast<const char*>(childNode.child_value()) + 2 + i * 2) & 0xFF) << (CupIdByteNum - 1 - i) * 8;
            }
            DEVMENUCOMMAND_LOG("\n--- CUP Parameter ---\n");
            DEVMENUCOMMAND_LOG("CUP Version : 0x%x\n", pCardInfo->cupVersion);
            DEVMENUCOMMAND_LOG("CUP ID      : 0x%016llx\n", pCardInfo->cupId);
            return true;
        }
        else
        {
            NN_ABORT("Content Meta not found in cnmt.xml in Update\n");
        }
    }

    //! コンテンツメタに記述されたコンテンツのうち、matchFunc でフィルタしたものの ID を fileList に登録する。
    //! fileList[*pListIndex] から昇順に登録し、登録した数だけ pListIndex を増加させる
    //! cnmt.xml に <ContentMeta> がない場合に ABORT する
    //! cnmt.xml の Type が Application, AddOnContent または Patch でない場合に ABORT する
    void SetMatchedFileName(int64_t* pListIndex, char (*fileList)[ContentFileMaxLength], char* inBuffer, size_t inBufferSize, MatchContentFunc matchFunc)
    {
        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // ContentMeta の解釈
        if(GetXmlNodeChild(&rootNode, document, ContentMeta))
        {
            int64_t listNum = *pListIndex;
            auto childNode = rootNode.child(Type);
            if(nn::util::Strncmp(childNode.child_value(), Patch, nn::util::Strnlen(Patch, ContentMetaXmlElementLengthMax)) == 0 ||
               nn::util::Strncmp(childNode.child_value(), AddOnContent, nn::util::Strnlen(AddOnContent, ContentMetaXmlElementLengthMax)) == 0)
            {
                return;
            }
            else if(nn::util::Strncmp(childNode.child_value(), Application, nn::util::Strnlen(Application, ContentMetaXmlElementLengthMax)) != 0)
            {
                NN_ABORT("error : Target nsp's type (%s) is not supported.\n", childNode.child_value());
            }
            auto itr = rootNode.begin();
            while(itr != rootNode.end())
            {
                // Content を見つける
                if(nn::util::Strncmp(itr->name(), Content, ContentFileMaxLength) == 0)
                {
                    // Type 名が List に入っていたら nca 名前をコピー
                    childNode = itr->child(Type);
                    if(matchFunc(childNode.child_value()))
                    {
                        childNode = itr->child(Id);
                        nn::util::Strlcpy(fileList[listNum], childNode.child_value(), ContentFileMaxLength);
                        childNode = itr->child(Type);
                        if(nn::util::Strncmp(childNode.child_value(), Meta, nn::util::Strnlen(Meta, ContentMetaXmlElementLengthMax)) == 0)
                        {
                            strncat(fileList[listNum], ".cnmt.nca", 1 + 4 + 1 + 3 + 1);
                        }
                        else
                        {
                            strncat(fileList[listNum], ".nca", 1 + 3 + 1);
                        }
                        listNum++;
                    }
                }
                itr++;
            }
            *pListIndex = listNum;
        }
        else
        {
            NN_ABORT("error : ContentMeta not found in cnmt.xml.");
        }
    }

    //! 与えられた inBuffer をアプリの cnmt.xml として解釈して g_NormalContentsFileList に、normal 領域にコピーする nca を登録する
    void SetNormalAreaFileName(int64_t* pListIndex, char* inBuffer, size_t inBufferSize)
    {
        return SetMatchedFileName(pListIndex, g_NormalContentsFileList, inBuffer, inBufferSize, IsMatchNormalContentsList);
    }

    //! 与えられた IFileSystem 以下の name の ファイルを読み込み、pOutReadSize, pOutBuffer に結果を出力
    Result ReadFileFromIFileSystem(size_t* pOutReadSize, char* pOutBuffer, size_t bufferSize, IFileSystem* pFs, const DirectoryEntry& entry, int64_t offset = 0)
    {
        auto nameBufferSize = util::Strnlen(entry.name, EntryNameLengthMax) + 2; // '/' と '\0' の分増やす
        std::unique_ptr<char[]> nameBuffer(new char[nameBufferSize]);
        nn::util::SNPrintf(nameBuffer.get(), nameBufferSize, "/%s", entry.name);
        std::unique_ptr<IFile> pFile;
        NN_RESULT_DO(pFs->OpenFile(&pFile, nameBuffer.get(), OpenMode_Read));
        NN_RESULT_DO(pFile->Read(pOutReadSize, offset, pOutBuffer, bufferSize, ReadOption()));
        NN_RESULT_SUCCESS;
    }

    //! 与えられた inBuffer を programinfo.xml として解析して、ToolVersion を取得する
    //! programinfo.xml に内容が追加・修正された場合はここをいじる
    void GetToolVersionFromProgramInfo(ProgramInfoXml* pOutProgramInfoXml, const char* inBuffer, size_t inBufferSize)
    {
        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // ProgramInfo の解釈
        if(GetXmlNodeChild(&rootNode, document, ProgramInfo))
        {
            auto childNode = rootNode.child(ToolVersion);
            nn::util::Strlcpy(pOutProgramInfoXml->toolVersion, childNode.child_value(), ToolVersionMaxLength);
        }
        else
        {
            NN_ABORT("Program Info not found in programinfo.xml in Update\n");
        }
    }

    //! 与えられた inBuffer を cardspec.xml として解析して、CardSize 及び ClockRate を取得する
    //! cardspec.xml に内容が追加・修正された場合はここをいじる
    void GetCardSizeAndSpeedFromCardSpec(CardInfo* pCardInfo, char* inBuffer, size_t inBufferSize)
    {
        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // ContentMeta の解釈
        if(GetXmlNodeChild(&rootNode, document, CardSpec))
        {
            auto childNode = rootNode.child(Size);
            sscanf(childNode.child_value(), "%d", &pCardInfo->cardSize);
            // Cardspec.xml に書かれている CardSize が設定可能な値であることを確認
            if(!IsGameCardSizeRight(pCardInfo->cardSize))
            {
                NN_ABORT("GameCardSize in cardspec.xml is wrong\n");
            }
            childNode = rootNode.child(ClockRate);
            sscanf(childNode.child_value(), "%d", &pCardInfo->cardClockRate);
            // Cardspec.xml に書かれている ClockRate が設定可能な値であることを確認
            if(!IsGameCardClockRateRight(pCardInfo->cardClockRate))
            {
                NN_ABORT("ClockRate in cardspec.xml is wrong\n");
            }
            // Cardspec.xml に書かれている ClockRate と CardSize が設定可能な組み合わせであることを確認
            else if((pCardInfo->cardClockRate == nn::fs::GameCardClockRate::ClockRate50MHz) && (pCardInfo->cardSize < nn::fs::GameCardSize::Size8GB))
            {
                NN_ABORT("Card Speed is %d MHz <=> Card Size is %d GB. Mismatch.\n", static_cast<int8_t>(pCardInfo->cardClockRate), static_cast<int8_t>(pCardInfo->cardSize));
            }
        }
        else
        {
            NN_ABORT("Content Meta not found in cnmt.xml in Update\n");
        }
    }

    //! 与えられた inBuffer を cnmt.xml として解析して、Type を調べる
    nn::ncm::ContentMetaType GetTypeFromContentMeta(char* inBuffer, size_t inBufferSize)
    {
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(inBuffer, inBufferSize);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        if(GetXmlNodeChild(&rootNode, document, ContentMeta))
        {
            auto childNode = rootNode.child(Type);
            if (nn::util::Strncmp(childNode.child_value(), Application, nn::util::Strnlen(Application, ContentMetaXmlElementLengthMax)) == 0)
            {
                return nn::ncm::ContentMetaType::Application;
            }
            else if (nn::util::Strncmp(childNode.child_value(), Patch, nn::util::Strnlen(Patch, ContentMetaXmlElementLengthMax)) == 0)
            {
                return nn::ncm::ContentMetaType::Patch;
            }
            else if (nn::util::Strncmp(childNode.child_value(), AddOnContent, nn::util::Strnlen(AddOnContent, ContentMetaXmlElementLengthMax)) == 0)
            {
                return nn::ncm::ContentMetaType::AddOnContent;
            }
        }
        return nn::ncm::ContentMetaType::Unknown;
    }

    template <class FuncT>
    Result SearchEntryImpl(DirectoryEntry* pDirEntry, IDirectory* pDir, FuncT func) NN_NOEXCEPT
    {
        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry entry;
            int64_t readNum = 0;
            NN_RESULT_DO(pDir->Read(&readNum, &entry, 1));

            if( readNum == 0 )
            {
                // ここにはこないはず
                NN_ABORT("error : can't find .cnmt.xml \n");
            }

            bool isFound;
            NN_RESULT_DO(func(&isFound, entry));
            if (isFound)
            {
                *pDirEntry = entry;
                NN_RESULT_SUCCESS;
            }
        }
    }

    Result SearchEntry(DirectoryEntry* pDirEntry, IDirectory* pDir, const char* ext) NN_NOEXCEPT
    {
        NN_RESULT_DO(SearchEntryImpl(pDirEntry, pDir, [&ext](bool* outValue, const DirectoryEntry& entry) NN_NOEXCEPT
        {
            auto fileEx = strstr(entry.name, ".");
            if(fileEx == nullptr)
            {
                NN_ABORT("error : file extention is wrong!\n");
            }
            // xml ファイルを探す
            *outValue = nn::util::Strncmp(fileEx, ext, ContentFileMaxLength) == 0;
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

    Result SearchFileEntry(DirectoryEntry* pDirEntry, IDirectory* pDir, const char* fileName) NN_NOEXCEPT
    {
        NN_RESULT_DO(SearchEntryImpl(pDirEntry, pDir, [&fileName](bool* outValue, const DirectoryEntry& entry) NN_NOEXCEPT
        {
            *outValue = nn::util::Strncmp(entry.name, fileName, ContentFileMaxLength) == 0;
            NN_RESULT_SUCCESS;
        }));
        NN_RESULT_SUCCESS;
    }

    //! pFs のルート から ".cnmt.xml" ファイルを探し、中身を解析する。
    //! ノーマル領域に含めるコンテンツを抜き出し、g_NormalContentsFileList にファイル名を入れる
    //! cnmt.xml が存在しない場合に ABORT する
    // TODO: cnmt.xml が複数含まれる場合の対応
    Result SetNormalContentFromNsp(IFileSystem* pFs)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));

        int64_t listNum = 0;

        // .cnmt.xml ファイルを探す
        DirectoryEntry dirEntry;
        NN_RESULT_DO(SearchEntry(&dirEntry, directory.get(), ".cnmt.xml"));

        // .cnmt.xml を Open
        size_t size = 0;
        size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
        std::unique_ptr<char[]> buffer(new char[bufferSize]);
        NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));

        // Normal エリアに入れるコンテンツのリストを作成
        SetNormalAreaFileName(&listNum, buffer.get(), size);
        g_NormalContentsFileNum = listNum;
        NN_RESULT_SUCCESS;
    }

    //! 与えられた DirectoryEntry を FileEntryForConstruct にプッシュする
    Result PushToEntry(std::vector<FileEntryForConstruct>* pOutArray, IFileSystem* pFs, DirectoryEntry& dirEntry, int64_t* pTotalOffset)
    {
        FileEntryForConstruct entry;
        nn::util::Strlcpy(entry.name, dirEntry.name, sizeof(entry.name));
        entry.offset = *pTotalOffset;
        entry.size = dirEntry.fileSize;
        entry.hashTargetOffset = 0;
        entry.hashTargetSize = static_cast<uint32_t>(std::min<uint64_t>(entry.size, 512)); // TORIAEZU: 対象ファイル (nca, tik, ...) にかかわらず先頭 512 バイト
        NN_RESULT_DO(CalculateFileHash(entry.hash, sizeof(entry.hash), pFs, (std::string("/") + entry.name).c_str(), entry.hashTargetOffset, entry.hashTargetSize));
        *pTotalOffset += nn::util::align_up(dirEntry.fileSize, Sha256PartitionFileSystemMeta::FileDataAlignmentSize);
        pOutArray->push_back(entry);
        NN_RESULT_SUCCESS;
    }

    Result CheckExternalKeyFromNsp(IFileSystem* pFs)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));
        nn::ncm::ContentMetaType contentMetaType = nn::ncm::ContentMetaType::Unknown;

        bool hasTicket = false;
        bool isInvalidToolVersion = false;

        while (NN_STATIC_CONDITION(true))
        {
            DirectoryEntry dirEntry;
            int64_t readNum = 0;
            NN_RESULT_DO(directory->Read(&readNum, &dirEntry, 1));

            if ( readNum == 0 )
            {
                break;
            }

            char* fileEx = strstr(dirEntry.name, ".");
            if (fileEx == nullptr)
            {
                NN_ABORT("error : file extention is wrong!\n");
            }

            if (nn::util::Strncmp(fileEx, ".cnmt.xml", nn::util::Strnlen(dirEntry.name, ContentFileMaxLength)) == 0)
            {
                // .cnmt.xml を Open
                size_t size = 0;
                size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                std::unique_ptr<char[]> buffer(new char[bufferSize]);
                NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                contentMetaType = GetTypeFromContentMeta(buffer.get(), size);
            }
            else if (nn::util::Strncmp(fileEx, ".tik", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) == 0 ||
                nn::util::Strncmp(fileEx, ".cert", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) == 0)
            {
                hasTicket = true;
            }
            else if (nn::util::Strncmp(fileEx, ".programinfo.xml", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) == 0)
            {
                // ToolVersion を取得してチェックを行う
                ProgramInfoXml programInfoXml;
                size_t size = 0;
                size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                std::unique_ptr<char[]> buffer(new char[bufferSize]);
                NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                GetToolVersionFromProgramInfo(&programInfoXml, buffer.get(), size);

                // ToolVersion が 0.xx 以外
                if (nn::util::Strncmp(programInfoXml.toolVersion, "0_", 2) != 0)
                {
                    isInvalidToolVersion = true;
                }
            }
        }

        // アプリケーション且つ ToolVersion が 0.xx 以外でチケットがある場合はエラー
        if (contentMetaType == nn::ncm::ContentMetaType::Application && isInvalidToolVersion && hasTicket)
        {
            NN_RESULT_THROW(devmenuUtil::ResultInvalidExternalKey());
        }

        NN_RESULT_SUCCESS;
    }

    Result CheckTypeAndGetInfoFromNsp(bool *isMatch, CardInfo* outCardInfo, IFileSystem* pFs, nn::ncm::ContentMetaType contentMetaType)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));

        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry dirEntry;
            int64_t readNum = 0;
            NN_RESULT_DO(directory->Read(&readNum, &dirEntry, 1));

            if( readNum == 0 )
            {
                break;
            }

            // --size が未設定かつファイル名が cardspec.xml なら解析して CardSize と ClockRate を設定する
            {
                char* fileEx = strstr(dirEntry.name, ".");
                if(fileEx == nullptr)
                {
                    NN_ABORT("error : file extention is wrong!\n");
                }

                if((static_cast<uint32_t>(outCardInfo->cardSize) == 0) && (nn::util::Strncmp(dirEntry.name, "cardspec.xml", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax) + 1) == 0))
                {
                    size_t size = 0;
                    size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                    std::unique_ptr<char[]> buffer(new char[bufferSize]);
                    NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                    GetCardSizeAndSpeedFromCardSpec(outCardInfo, buffer.get(), size);
                }
                if(nn::util::Strncmp(fileEx, ".cnmt.xml", ContentFileMaxLength) == 0)
                {
                    // .cnmt.xml を Open
                    size_t size = 0;
                    size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                    std::unique_ptr<char[]> buffer(new char[bufferSize]);
                    NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                    *isMatch = contentMetaType == GetTypeFromContentMeta(buffer.get(), size);
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result GetCupInfoInfoFromNsp(CardInfo* outCardInfo, IFileSystem* pFs)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));

        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry dirEntry;
            int64_t readNum = 0;
            NN_RESULT_DO(directory->Read(&readNum, &dirEntry, 1));

            if( readNum == 0 )
            {
                break;
            }

            // Update パーティション
            // cnmt.xml は解析して SystemUpdate なら CupVersion と CupID を取得
            // UpdatePartition.xml は解析して Hash を取得
            {
                char* fileEx = strstr(dirEntry.name, ".");
                if(fileEx == nullptr)
                {
                    NN_ABORT("error : file extention is wrong!\n");
                }
                if(nn::util::Strncmp(fileEx, ".cnmt.xml", ContentFileMaxLength) == 0)
                {
                    // .cnmt.xml を Open
                    size_t size = 0;
                    size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                    std::unique_ptr<char[]> buffer(new char[bufferSize]);
                    NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                    CanGetCupVersionAndIdFromContentMeta(outCardInfo, buffer.get(), size);
                }
                if(nn::util::Strncmp(dirEntry.name, "UpdatePartition.xml", ContentFileMaxLength) == 0)
                {
                    // Open
                    size_t size = 0;
                    size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
                    std::unique_ptr<char[]> buffer(new char[bufferSize]);
                    NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, pFs, dirEntry));
                    CanGetHashFromUpdatePartitionXml(outCardInfo, buffer.get(), size);
                }
            }
        }
        NN_RESULT_SUCCESS;
    }

    //! pFs のルート 1 階層のみ列挙し、Sha256PartitionFileSystemMeta::FileEntryForConstruct のリストを作成する。(*.nca, *.tik, *.cert 以外はリストに入れない)
    Result CreateEntryListFromNsp(int64_t *currentOffset, std::vector<FileEntryForConstruct>* pOutArray, int64_t* pOutTotalFileSize, IFileSystem* pFs, GameCardPartition gameCardPartition, bool isAoc)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));

        bool hasTicket = false;

        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry dirEntry;
            int64_t readNum = 0;
            NN_RESULT_DO(directory->Read(&readNum, &dirEntry, 1));

            if( readNum == 0 )
            {
                *pOutTotalFileSize = nn::util::align_up(*currentOffset, Sha256PartitionFileSystemMeta::FileDataAlignmentSize);
                break;
            }

            // Nomral パーティションはノーマルに入れるリストにあるものだけ入れる
            if( (gameCardPartition == GameCardPartition_Normal) && (!IsMatchNormalNcaList(dirEntry.name)) )
            {
                // リストになければスルー
                continue;
            }

            // *.nca じゃなかったらスルー
            char* fileEx = strrchr(dirEntry.name, static_cast<int>('.'));
            if(fileEx == nullptr)
            {
                NN_ABORT("error : file extention is wrong!\n");
            }
            if(nn::util::Strncmp(fileEx, ".nca", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) != 0 &&
               nn::util::Strncmp(fileEx, ".tik", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) != 0 &&
               nn::util::Strncmp(fileEx, ".cert", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) != 0)
            {
                continue;
            }

            if(nn::util::Strncmp(fileEx, ".tik", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) == 0 ||
               nn::util::Strncmp(fileEx, ".cert", nn::util::Strnlen(dirEntry.name, EntryNameLengthMax)) == 0)
            {
                hasTicket = true;
            }

            PushToEntry(pOutArray, pFs, dirEntry, currentOffset);
        }

        // AddOnContent でチケットが入っているとエラー
        if (isAoc && hasTicket)
        {
            DEVMENUCOMMAND_LOG("error : Can't write AddOnContent nsp that has ticket to gamecard. Please use convert-aoc or --no-ticket option of AuthoringTool to create the nsp.\n");
            return fs::ResultPathNotFound();
        }

        NN_RESULT_SUCCESS;
    }

    //------------------------------------------------------------------------------
    // ゲームカードへの書き込みスレッド
    //------------------------------------------------------------------------------

    //! 書き込みスレッド
    void RunWriteThread(void* pWritePageInfo)
    {
        int64_t offset = *reinterpret_cast<int64_t*>(pWritePageInfo);
        nn::os::SignalEvent(&g_WriteStartEvent);
#ifndef NN_BUILD_CONFIG_OS_WIN
        g_WriteResult = nn::fs::WriteToGameCard(offset, g_pWriteBuffer, WorkBufferSize);
#else
        g_WriteResult = g_XciStorage->Write(offset, g_pWriteBuffer, WorkBufferSize);
#endif
        nn::os::SignalEvent(&g_WriteFinishEvent);
    }

    //! 書き込みスレッド開始処理
    void InitializeWriteThread(int64_t* pHardwareOffset)
    {
        g_IsWriting = true;
        nn::os::InitializeEvent(&g_WriteStartEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&g_WriteFinishEvent, false, nn::os::EventClearMode_AutoClear);
        nn::Result result = nn::os::CreateThread( &g_WriteThread, RunWriteThread, pHardwareOffset, g_WriteThreadStack, ThreadStackSize, ThreadPriority );
        if ( !result.IsSuccess() )
        {
            NN_ABORT("failed to create a thread\n");
        }
        nn::os::SetThreadNamePointer(&g_WriteThread, g_ThreadName);
        nn::os::StartThread( &g_WriteThread );
    }

    //! 書き込みスレッド終了処理
    void FinalizeWriteThread()
    {
        nn::os::WaitThread( &g_WriteThread );
        nn::os::DestroyThread( &g_WriteThread );
        nn::os::FinalizeEvent(&g_WriteStartEvent);
        nn::os::FinalizeEvent(&g_WriteFinishEvent);
        g_IsWriting = false;
    }

    //! ゲームカードへの書き込み完了待ち。前の書き込みの result が返ってくる
    Result WaitForWriteFinish()
    {
        if(g_IsWriting)
        {
            nn::os::WaitEvent(&g_WriteFinishEvent);
            FinalizeWriteThread();
            return g_WriteResult;
        }
        NN_RESULT_SUCCESS;
    }

    //! ゲームカードへの書き込み用のスレッドを作成し、 readBuffer の内容をゲームカードに書き込む
    Result FlushToGameCard(int64_t* pHardwareOffset, char* readBuffer, size_t copySize, WriteProgress& writeProgress)
    {
#ifndef NN_BUILD_CONFIG_OS_WIN
        // TODO : Sleep Disble 対応後削除
        // 無操作状態の解除のための通知
        nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::TouchScreen::Mask);
#endif
        NN_RESULT_DO(WaitForWriteFinish());
        size_t readBufferOffset = *pHardwareOffset % WorkBufferSize; // 1 ループ目以外は 0 になる
        size_t writeSize        = copySize - readBufferOffset;
        memcpy(g_pWriteBuffer, readBuffer + readBufferOffset, writeSize);
        InitializeWriteThread(pHardwareOffset);
        writeProgress.AddWrittenSize(static_cast<int64_t>(writeSize));
        nn::os::WaitEvent(&g_WriteStartEvent);
        *pHardwareOffset += static_cast<int64_t>(writeSize);
        NN_RESULT_SUCCESS;
    }

    //! pSrcStorage の全域をゲームカードに書き込む
    Result CopyStorage(IStorage* pSrcStorage, int64_t* pHardwareOffset, int64_t* pReadBufferRestSize, char* readBuffer, WriteProgress& writeProgress)
    {
        int64_t restSize = 0;
        int64_t offset = 0;
        NN_RESULT_DO(pSrcStorage->GetSize(&restSize));
        while(restSize > 0)// 1 ファイル当たりのループ
        {
            size_t readSize = static_cast<size_t>(std::min<int64_t>(restSize, *pReadBufferRestSize));
            // ファイルから読み込んだ内容を readBuffer の現在位置にコピー
            NN_RESULT_DO(pSrcStorage->Read(offset, readBuffer + (WorkBufferSize - *pReadBufferRestSize), readSize));
            offset   += readSize;
            restSize -= readSize;
            *pReadBufferRestSize -= readSize;
            if(*pReadBufferRestSize == 0)
            {
                NN_RESULT_DO(FlushToGameCard(pHardwareOffset, readBuffer, WorkBufferSize, writeProgress));
                *pReadBufferRestSize = WorkBufferSize;
            }
        }
        NN_RESULT_SUCCESS;
    }

    //! header を ゲームカードに書き込む
    Result CopyHeader(int64_t* pHardwareOffset, int64_t* pReadBufferRestSize, PartitionContents& partition, char* readBuffer, WriteProgress& writeProgress)
    {
        int64_t restSize = partition.headerSize;
        int64_t offset = 0;
        while(restSize > 0)// 1 ファイル当たりのループ
        {
            size_t readSize = static_cast<size_t>(std::min<int64_t>(restSize, *pReadBufferRestSize));
            // ファイルから読み込んだ内容を readBuffer の現在位置にコピー
            memcpy(readBuffer + (WorkBufferSize - *pReadBufferRestSize), partition.headerBuffer.get(), readSize);
            offset   += readSize;
            restSize -= readSize;
            *pReadBufferRestSize -= readSize;
            if(*pReadBufferRestSize == 0)
            {
                NN_RESULT_DO(FlushToGameCard(pHardwareOffset, readBuffer, WorkBufferSize, writeProgress));
                *pReadBufferRestSize = WorkBufferSize;
            }
        }
        NN_RESULT_SUCCESS;
    }

    //! padding を ゲームカードに書き込む
    Result CopyPadding(int64_t* pHardwareOffset, int64_t* pReadBufferRestSize, size_t size, char* readBuffer, WriteProgress& writeProgress)
    {
        int64_t restSize = size;
        int64_t offset = 0;
        while(restSize > 0)// 1 ファイル当たりのループ
        {
            size_t readSize = static_cast<size_t>(std::min<int64_t>(restSize, *pReadBufferRestSize));
            // ファイルから読み込んだ内容を readBuffer の現在位置にコピー
            memset(readBuffer + (WorkBufferSize - *pReadBufferRestSize), 0x00, readSize);
            offset   += readSize;
            restSize -= readSize;
            *pReadBufferRestSize -= readSize;
            if(*pReadBufferRestSize == 0)
            {
                NN_RESULT_DO(FlushToGameCard(pHardwareOffset, readBuffer, WorkBufferSize, writeProgress));
                *pReadBufferRestSize = WorkBufferSize;
            }
        }
        NN_RESULT_SUCCESS;
    }


    //! 必要なら padding を ゲームカードに書き込む
    Result SetPaddingIfNeeded(int64_t* pHardwareOffset, int64_t* pReadBufferRestSize, char* readBuffer, WriteProgress& writeProgress, Sha256PartitionFileSystemMeta& headerFs, int currentIndex)
    {
        // padding 処理
        if(headerFs.GetEntryCount() - 1 > currentIndex) // 次のエントリーがある
        {
            PartitionEntry currentPartitionEntry;
            PartitionEntry nextPartitionEntry;
            memcpy(&currentPartitionEntry, headerFs.GetEntry(currentIndex), sizeof(PartitionEntry));
            memcpy(&nextPartitionEntry, headerFs.GetEntry(currentIndex + 1), sizeof(PartitionEntry));
            if(currentPartitionEntry.offset + currentPartitionEntry.size != nextPartitionEntry.offset)
            {
                NN_ABORT_UNLESS(nextPartitionEntry.offset > currentPartitionEntry.offset + currentPartitionEntry.size);
                size_t paddingSize = static_cast<size_t>(nextPartitionEntry.offset - (currentPartitionEntry.offset + currentPartitionEntry.size));
                NN_RESULT_DO(CopyPadding(pHardwareOffset, pReadBufferRestSize, paddingSize, readBuffer, writeProgress));
            }
        }
        NN_RESULT_SUCCESS;
    }


    //! pDstFs のルート 1 階層のみ列挙し、pSrcFs の同名のエントリのファイルデータで pDstFs 側のファイルデータを書き換える
    Result FillFileData(IFileSystem* pDstFs, IFileSystem** pSrcFs, int srcFsCount, PartitionContents& partition, WriteProgress& writeProgress, nn::fssrv::MemoryResourceFromStandardAllocator* pAllocator)
    {
        std::unique_ptr<IDirectory> directory;
        NN_RESULT_DO(pDstFs->OpenDirectory(&directory, "/", OpenDirectoryMode_File));

        std::unique_ptr<char[]> readBuffer(new char[WorkBufferSize]);
        static_assert(nn::dd::DeviceAddressSpaceMemoryRegionAlignment >= nn::gc::GcPageSize, "");
        std::unique_ptr<char[]> writeBuffer(new char[WorkBufferSize + nn::dd::DeviceAddressSpaceMemoryRegionAlignment - 1]);
        g_pWriteBuffer = reinterpret_cast<char*>(nn::util::align_up(
            reinterpret_cast<uintptr_t>(writeBuffer.get()), nn::dd::DeviceAddressSpaceMemoryRegionAlignment));
        int64_t hardwareOffset       = partition.hardwareOffset;
        // hardwareOffset のアラインのために、 1 ループ目はオフセットを持たせる
        int64_t readBufferRestSize   = WorkBufferSize - hardwareOffset % WorkBufferSize;

        NN_RESULT_DO(CopyHeader(&hardwareOffset, &readBufferRestSize, partition, readBuffer.get(), writeProgress));

        MemoryStorage memoryStorage(partition.headerBuffer.get(), partition.headerSize);
        Sha256PartitionFileSystemMeta headerFs;
        char hash[crypto::Sha256Generator::HashSize];
        crypto::GenerateSha256Hash(hash, sizeof(hash), partition.headerBuffer.get(), partition.headerSize);

        NN_RESULT_DO(headerFs.Initialize(&memoryStorage, pAllocator, hash, sizeof(hash)));

        while(NN_STATIC_CONDITION(true))
        {
            DirectoryEntry dirEntry;
            int64_t readNum = 0;
            NN_RESULT_DO(directory->Read(&readNum, &dirEntry, 1));

            if( readNum == 0 )
            {
                // まだゲームカードにかけていない余りがあれば書き込む
                if(readBufferRestSize != static_cast<int64_t>(WorkBufferSize))
                {
                    NN_RESULT_DO(FlushToGameCard(&hardwareOffset, readBuffer.get(), WorkBufferSize - static_cast<size_t>(readBufferRestSize), writeProgress));
                }
                NN_RESULT_DO(WaitForWriteFinish());
                NN_RESULT_SUCCESS;
            }

            std::unique_ptr<IFile> srcFile;
            Result result = fs::ResultPathNotFound();
            for (int i = 0; i < srcFsCount; i++)
            {
                result = pSrcFs[i]->OpenFile(&srcFile, (std::string() + "/" + dirEntry.name).c_str(), OpenMode_Read);
                if (result.IsSuccess())
                {
                    break;
                }
            }
            if (result.IsFailure())
            {
                return result;
            }
            FileStorage srcStorage(srcFile.get());
            NN_RESULT_DO(CopyStorage(&srcStorage, &hardwareOffset, &readBufferRestSize, readBuffer.get(), writeProgress));
            NN_RESULT_DO(SetPaddingIfNeeded(&hardwareOffset, &readBufferRestSize, readBuffer.get(), writeProgress, headerFs, headerFs.GetEntryIndex(dirEntry.name)));
        }
    }

    //! pFs の内容を Sha256PartitionFs に固めなおしたヘッダを生成する
    Result CreateHeaderFromEntryList(std::unique_ptr<char[]>* pOutHeaderBuffer, size_t* pOutHeaderSize, int64_t* pOutBodySize, std::vector<FileEntryForConstruct> entryArray, int64_t totalFileSize)
    {
        size_t headerSize;

        FileEntryForConstruct* pEntryArray = !entryArray.empty() ? &entryArray[0]
                                                                               : nullptr;
        NN_RESULT_DO(Sha256PartitionFileSystemMeta::QueryMetaDataSize(&headerSize, pEntryArray, static_cast<int>(entryArray.size())));
        NN_ASSERT(is_aligned(headerSize, Sha256PartitionFileSystemMeta::FileDataAlignmentSize));

        std::unique_ptr<char[]> headerBuffer(new char[headerSize]);
        NN_RESULT_THROW_UNLESS(headerBuffer != nullptr, ResultAllocationMemoryFailed());

        NN_RESULT_DO(Sha256PartitionFileSystemMeta::ConstructMetaData(headerBuffer.get(), headerSize, pEntryArray, static_cast<int>(entryArray.size())));

        *pOutHeaderBuffer = std::move(headerBuffer);
        *pOutBodySize = align_up(totalFileSize, Sha256PartitionFileSystemMeta::FileDataAlignmentSize);
        *pOutHeaderSize = headerSize;

        NN_RESULT_SUCCESS;
    }

    //! 与えられた Partition の Header を生成する
    Result CreateHeader(std::unique_ptr<char[]>* pOutHeaderBuffer, size_t* pOutHeaderSize, int64_t* pOutBodySize, IFileSystem** pFs, int fsCount, CardInfo* pCardInfo, GameCardPartition gameCardPartition)
    {
        NN_ASSERT(pFs != nullptr);

        int64_t totalFileSize = 0;
        std::vector<FileEntryForConstruct> entryArray;

        int64_t currentOffset = 0;

        if(gameCardPartition == GameCardPartition_Update)
        {
            if(pFs[0] != nullptr)
            {
                NN_RESULT_DO(GetCupInfoInfoFromNsp(pCardInfo, pFs[0]));
                NN_RESULT_DO(CreateEntryListFromNsp(&currentOffset, &entryArray, &totalFileSize, pFs[0], gameCardPartition, false));
            }
        }
        else if(gameCardPartition == GameCardPartition_Normal)
        {
            for (int i = 0; i < fsCount; i++)
            {
                if(pFs[i] == nullptr)
                {
                    continue;
                }

                NN_RESULT_DO(SetNormalContentFromNsp(pFs[i]));

                bool isTypeMatch = false;
                CardInfo tmpCardInfoByNsp;
                NN_RESULT_DO(CheckTypeAndGetInfoFromNsp(&isTypeMatch, &tmpCardInfoByNsp, pFs[i], nn::ncm::ContentMetaType::Application));
                if (!isTypeMatch)
                {
                    continue;
                }

                NN_RESULT_DO(CreateEntryListFromNsp(&currentOffset, &entryArray, &totalFileSize, pFs[i], gameCardPartition, false));
            }
        }
        else
        {
            CardInfo tmpCardInfo;
            int appCount = 0;

            // アプリ・追加コンテンツ・パッチの順に格納
            nn::ncm::ContentMetaType types[] =
            {
                nn::ncm::ContentMetaType::Application,
                nn::ncm::ContentMetaType::AddOnContent,
                nn::ncm::ContentMetaType::Patch
            };

            for (auto& type : types)
            {
                for (int i = 0; i < fsCount; i++)
                {
                    if(pFs[i] == nullptr)
                    {
                        continue;
                    }

                    bool isTypeMatch = false;
                    {
                        CardInfo tmpCardInfoByNsp;
                        NN_RESULT_DO(CheckTypeAndGetInfoFromNsp(&isTypeMatch, &tmpCardInfoByNsp, pFs[i], type));
                        if (!isTypeMatch)
                        {
                            continue;
                        }
                        tmpCardInfo.cardSize = std::max(tmpCardInfo.cardSize, tmpCardInfoByNsp.cardSize);
                        tmpCardInfo.cardClockRate = std::max(tmpCardInfo.cardClockRate, tmpCardInfoByNsp.cardClockRate);
                    }

                    NN_RESULT_DO(CreateEntryListFromNsp(&currentOffset, &entryArray, &totalFileSize, pFs[i], gameCardPartition, type == nn::ncm::ContentMetaType::AddOnContent));

                    if (type == nn::ncm::ContentMetaType::Application)
                    {
                        appCount++;
                    }
                }
            }

            if(appCount == 0)
            {
                DEVMENUCOMMAND_LOG("error: At least one application should be included.\n");
                return fs::ResultPathNotFound();
            }

            if (pCardInfo->cardSize < tmpCardInfo.cardSize)
            {
                if (static_cast<uint8_t>(pCardInfo->cardSize) != 0)
                {
                    DEVMENUCOMMAND_LOG("warning: nsp indicates %d as card size so used it though --size indicates %d.\n", tmpCardInfo.cardSize, pCardInfo->cardSize);
                }
                pCardInfo->cardSize = tmpCardInfo.cardSize;
            }

            if (pCardInfo->cardClockRate < tmpCardInfo.cardClockRate)
            {
                if (static_cast<uint8_t>(pCardInfo->cardClockRate) != 0)
                {
                    DEVMENUCOMMAND_LOG("warning: nsp indicates %d as card clock rate so used it though --clock-rate indicates %d.\n", tmpCardInfo.cardClockRate, pCardInfo->cardClockRate);
                }
                pCardInfo->cardClockRate = tmpCardInfo.cardClockRate;
            }
        }

        NN_RESULT_DO(CreateHeaderFromEntryList(pOutHeaderBuffer, pOutHeaderSize, pOutBodySize, entryArray, totalFileSize));
        NN_RESULT_SUCCESS;
    }

    Result CreateHeader(std::unique_ptr<char[]>* pOutHeaderBuffer, size_t* pOutHeaderSize, int64_t* pOutBodySize, IFileSystem* pFs, CardInfo* pCardInfo, GameCardPartition gameCardPartition)
    {
        IFileSystem* ppFs[] = { pFs };
        return CreateHeader(pOutHeaderBuffer, pOutHeaderSize, pOutBodySize, ppFs, 1, pCardInfo, gameCardPartition);
    }

    struct RootSha256PartitionFileSystemInfo
    {
    public:
        static const int HashSize = 32;

    public:
        int64_t headerOffset;
        int64_t headerSize;
        char    headerHash[HashSize];
    };

    int64_t RoundupRomAreaStartPageAddress(int64_t value)
    {
        const int64_t RomAreaStartByteAlignment = 0x40 * 512;
        return (value + (RomAreaStartByteAlignment - 1)) & ~(RomAreaStartByteAlignment - 1);
    }

    int64_t RoundupTotalPageSize(int64_t value)
    {
        const int64_t totalPageSizeByteAlignment = 512;
        return (value + (totalPageSizeByteAlignment - 1)) & ~(totalPageSizeByteAlignment - 1);
    }

    void SetDevCardParam(nn::gc::detail::DevCardParameter* pOutDevCardParam, nn::fs::GameCardSize cardSize, uint64_t normalAreaSize)
    {
        pOutDevCardParam->cardId1.makerCode                = nn::gc::detail::MakerCodeForCardId1_Lapis;
        pOutDevCardParam->cardId1.memoryType               = nn::gc::detail::MemoryType_T1RomFast;
        pOutDevCardParam->cardId2.cardSecurityNumber       = nn::gc::detail::CardSecurityNumber_2;
        pOutDevCardParam->romAreaStartAddr                 = static_cast<uint32_t>(normalAreaSize / (512 * 64));
        pOutDevCardParam->backupAreaStartAddr              = 0xFFFFFFFF;
        pOutDevCardParam->nandSize                         = nn::gc::detail::DevCardNandSize_32GB; // Read Only Rarameter のはず
        switch(cardSize)
        {
        case nn::fs::GameCardSize::Size1GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_1GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_1GB;
            }
            break;
        case nn::fs::GameCardSize::Size2GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_2GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_2GB;
            }
            break;
        case nn::fs::GameCardSize::Size4GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_4GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_4GB;
            }
            break;
        case nn::fs::GameCardSize::Size8GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_8GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_8GB;
            }
            break;
        case nn::fs::GameCardSize::Size16GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_16GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_16GB;
            }
            break;
        case nn::fs::GameCardSize::Size32GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_32GB;
                pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_32GB;
            }
            break;
        default:
            pOutDevCardParam->cardId1.memoryCapacity           = nn::gc::detail::MemoryCapacity_1GB;
            pOutDevCardParam->romSize                          = nn::gc::detail::DevCardRomSize_1GB;
        }
        static const uint32_t ReserveAreaUnitSize = nn::gc::detail::AvailableSizeBase / ( 8 * 1024 * 1024 );
        uint32_t reserveParamValue = static_cast<uint32_t>(cardSize) * ReserveAreaUnitSize;
        // ReservedArea の書き込みパラメータへの反映
        for(size_t i = 0 ; i < 3; i++)
        {
            pOutDevCardParam->reserveAreaStartAddr[i] = (reserveParamValue >> (i * 8))  & 0xFF;
        }
    }

    //! 全 Partition の Header を生成する
    Result CreatePartitionHeader(PartitionContents* pUpdaterPartition, PartitionContents* pNormalPartition, PartitionContents* pSecurePartition, IFileSystem* pSrcUpdateFs, IFileSystem** pSrcNormalFs, int srcNormalFsCount, IFileSystem** pSrcSecureFs, int srcSecureFsCount, CardInfo* pCardInfo)
    {
        NN_RESULT_DO(CreateHeader(&(pUpdaterPartition->headerBuffer), &(pUpdaterPartition->headerSize), &(pUpdaterPartition->bodySize), pSrcUpdateFs, pCardInfo, GameCardPartition_Update));
        NN_RESULT_DO(CreateHeader(&(pNormalPartition->headerBuffer), &(pNormalPartition->headerSize), &(pNormalPartition->bodySize), pSrcNormalFs, srcNormalFsCount, pCardInfo, GameCardPartition_Normal));
        NN_RESULT_DO(CreateHeader(&(pSecurePartition->headerBuffer), &(pSecurePartition->headerSize), &(pSecurePartition->bodySize), pSrcSecureFs, srcSecureFsCount, pCardInfo, GameCardPartition_Secure));
        NN_RESULT_SUCCESS;
    }

    //! RootHeader を生成する
    Result CreateRootHeader(PartitionContents* pRootHeaderPartition, PartitionContents& updaterPartition, PartitionContents& normalPartition, PartitionContents& securePartition, int64_t SecureOffsetFromRootBody)
    {
        // 暫定の固定セキュアエリアオフセット
        const int64_t HashTargetOffset = 0; // 3 パーティションいずれもヘッダのみ検証

        std::vector<FileEntryForConstruct> rootEntryArray;
        FileEntryForConstruct updateEntry = { 0,                         static_cast<uint64_t>(updaterPartition.headerSize + updaterPartition.bodySize), HashTargetOffset, static_cast<uint32_t>(updaterPartition.headerSize), {0}, "update"};
        crypto::GenerateSha256Hash(updateEntry.hash, sizeof(updateEntry.hash), updaterPartition.headerBuffer.get(), static_cast<size_t>(updaterPartition.headerSize));
        rootEntryArray.push_back(updateEntry);

        const auto NormalOffsetFromRootBody = static_cast<uint64_t>(updaterPartition.headerSize + updaterPartition.bodySize);
        FileEntryForConstruct normalEntry = { NormalOffsetFromRootBody,  static_cast<uint64_t>(normalPartition.headerSize + normalPartition.bodySize), HashTargetOffset, static_cast<uint32_t>(normalPartition.headerSize), {0}, "normal"};
        crypto::GenerateSha256Hash(normalEntry.hash, sizeof(normalEntry.hash), normalPartition.headerBuffer.get(), static_cast<size_t>(normalPartition.headerSize));
        rootEntryArray.push_back(normalEntry);

        // (gap)

        FileEntryForConstruct secureEntry = { static_cast<uint64_t>(SecureOffsetFromRootBody),  static_cast<uint64_t>(securePartition.headerSize + securePartition.bodySize), HashTargetOffset, static_cast<uint32_t>(securePartition.headerSize), {0}, "secure"};
        crypto::GenerateSha256Hash(secureEntry.hash, sizeof(secureEntry.hash), securePartition.headerBuffer.get(), static_cast<size_t>(securePartition.headerSize));
        rootEntryArray.push_back(secureEntry);


        pRootHeaderPartition->bodySize = align_up(SecureOffsetFromRootBody + securePartition.headerSize + securePartition.bodySize, Sha256PartitionFileSystemMeta::FileDataAlignmentSize);

        NN_RESULT_DO(Sha256PartitionFileSystemMeta::QueryMetaDataSize(&pRootHeaderPartition->headerSize, &rootEntryArray[0], static_cast<int>(rootEntryArray.size())));
        NN_ASSERT(is_aligned(pRootHeaderPartition->headerSize, Sha256PartitionFileSystemMeta::FileDataAlignmentSize));

        NN_ASSERT(pRootHeaderPartition->headerSize == ExpectedRootHeaderSize);
        pRootHeaderPartition->headerBuffer.reset(new char[pRootHeaderPartition->headerSize]);
        NN_RESULT_DO(Sha256PartitionFileSystemMeta::ConstructMetaData(pRootHeaderPartition->headerBuffer.get(), pRootHeaderPartition->headerSize, &rootEntryArray[0], static_cast<int>(rootEntryArray.size())));
        NN_RESULT_SUCCESS;
    }

    //! pDstStorage に KeyArea/CardHeader を除く xci を構築し、CardHeader に埋め込む用の RootSha256PartitionFileSystemInfo を返す
    Result WriteXci(
            RootSha256PartitionFileSystemInfo* pOutRootPartitionFsInfo,
            IStorage* pDstStorage,
            IFileSystem* pSrcUpdateFs,
            IFileSystem** pSrcNormalFs,
            int srcNormalFsCount,
            IFileSystem** pSrcSecureFs,
            int srcSecureFsCount,
            CardInfo* pCardInfo,
            bool isFastParam
    )
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_UNUSED(isFastParam);
#endif
        PartitionContents updaterPartition;
        PartitionContents normalPartition;
        PartitionContents securePartition;
        // nsp からヘッダを生成 * 3
        // upp の指定がない場合はヘッダのみを生成する
        NN_RESULT_DO(CreatePartitionHeader(&updaterPartition, &normalPartition, &securePartition, pSrcUpdateFs, pSrcNormalFs, srcNormalFsCount, pSrcSecureFs, srcSecureFsCount, pCardInfo));
        /*
        DEVMENUCOMMAND_LOG("\n+------------------------------------+\n");
        DEVMENUCOMMAND_LOG("|        |   Header   |     Body     |\n");
        DEVMENUCOMMAND_LOG("| Update |    %4d    |  %11lld |\n", updaterPartition.headerSize, updaterPartition.bodySize);
        DEVMENUCOMMAND_LOG("| Normal |    %4d    |  %11lld |\n", normalPartition.headerSize, normalPartition.bodySize);
        DEVMENUCOMMAND_LOG("| Secure |    %4d    |  %11lld |\n", securePartition.headerSize, securePartition.bodySize);
        DEVMENUCOMMAND_LOG("+------------------------------------+\n");
        */

        g_WriteProgress.SetTotalSize(FsPartitionHeaderOffset + ExpectedRootHeaderSize + updaterPartition.headerSize + updaterPartition.bodySize + normalPartition.headerSize + normalPartition.bodySize + securePartition.headerSize + securePartition.bodySize);
        NN_RESULT_DO(pCardInfo->SetTotalSize(g_WriteProgress.totalSize));

        DEVMENUCOMMAND_LOG("\n---- Card Parameter ----\n");
        DEVMENUCOMMAND_LOG("CardSize     : %d GB\n", static_cast<int>(pCardInfo->cardSize));
        DEVMENUCOMMAND_LOG("WriteSize    : %lld Byte\n", pCardInfo->totalSize);
        DEVMENUCOMMAND_LOG("\n---- Write Start ----\n");

        int64_t normalAreaSize = RoundupRomAreaStartPageAddress(g_WriteProgress.totalSize - (securePartition.headerSize + securePartition.bodySize));
        pCardInfo->normalAreaSize = normalAreaSize;

#ifndef NN_BUILD_CONFIG_OS_WIN
        nn::gc::detail::DevCardParameter param;
        memset(&param, 0x00, sizeof(nn::gc::detail::DevCardParameter));
        SetDevCardParam(&param, pCardInfo->cardSize, static_cast<uint64_t>(normalAreaSize));
        if(!isFastParam) // --fast がついてない場合パラメータによる遅延設定を行う(ロンチ時点の市場最遅付近エミュレート : SPRD-3803)
        {
            // 1st Latency : 毎回 630us wait
            param.waitCycle1ForRead[0] = 0x3f; //63 * 10us
            param.speedChangeEmurateWaitCycle1ForRead[0] = 0x3f; //63 * 10us
            // 2nd Latency : 15KB Read ごとに 30us wait
            param.speedChangeEmurateWaitCycle2FrecencyForRead = 0x03; // 3 * 10 ページ(15KB) に一回遅延
            param.speedChangeEmurateWaitCycle2ForRead[0] = 0xee; // 750(0x2ee) * 40ns(2Cycle単位) = 30us  wait
            param.speedChangeEmurateWaitCycle2ForRead[1] = 0x02; // 同上
        }
        NN_RESULT_DO(nn::fs::EraseAndWriteParamDirectly(reinterpret_cast<char*>(&param), sizeof(nn::gc::detail::DevCardParameter)));
#endif
        // 3 パーティションを固めたルートヘッダを生成

        PartitionContents rootHeader;
        NN_RESULT_DO(CreateRootHeader(&rootHeader, updaterPartition, normalPartition, securePartition, normalAreaSize - FsPartitionHeaderOffset - ExpectedRootHeaderSize));

        // pDstStorage 上に各ヘッダバッファを当てはめる

        SubStorage xciBodyStorage(pDstStorage, XciHeaderSize, rootHeader.headerSize + rootHeader.bodySize);

        MemoryStorage rootHeaderBufferStorage(rootHeader.headerBuffer.get(), rootHeader.headerSize);

        MemoryStorage updateHeaderBufferStorage(updaterPartition.headerBuffer.get(), updaterPartition.headerSize);
        SubStorage updateBodyStorage(&xciBodyStorage, rootHeader.headerSize + updaterPartition.headerSize, updaterPartition.bodySize);

        MemoryStorage normalHeaderBufferStorage(normalPartition.headerBuffer.get(), normalPartition.headerSize);
        SubStorage normalBodyStorage(&xciBodyStorage, rootHeader.headerSize + updaterPartition.headerSize + updaterPartition.bodySize + normalPartition.headerSize, normalPartition.bodySize);

        int64_t secureAreaOffsetFromXciBody = normalAreaSize - FsPartitionHeaderOffset;
        int64_t gapSize = secureAreaOffsetFromXciBody - (rootHeader.headerSize + updaterPartition.headerSize + updaterPartition.bodySize + normalPartition.headerSize + normalPartition.bodySize);
        PlaceHolderStorage PlaceHolderStorage(gapSize);

        MemoryStorage secureHeaderBufferStorage(securePartition.headerBuffer.get(), securePartition.headerSize);
        SubStorage secureBodyStorage(&xciBodyStorage, secureAreaOffsetFromXciBody + securePartition.headerSize, securePartition.bodySize);

        // 連結したストレージ
        std::vector<IStorage*> storagePtrArray;
        storagePtrArray.push_back(&rootHeaderBufferStorage);
        storagePtrArray.push_back(&updateHeaderBufferStorage);
        storagePtrArray.push_back(&updateBodyStorage);
        storagePtrArray.push_back(&normalHeaderBufferStorage);
        storagePtrArray.push_back(&normalBodyStorage);
        storagePtrArray.push_back(&PlaceHolderStorage);
        storagePtrArray.push_back(&secureHeaderBufferStorage);
        storagePtrArray.push_back(&secureBodyStorage);
        CatStorage vxciBodyStorage(storagePtrArray);

        // 3 パーティションを開く
        Sha256PartitionFileSystem rootPartitionFs;
        rootPartitionFs.Initialize(&vxciBodyStorage);

        std::unique_ptr<IFile> updateFile;
        NN_RESULT_DO(rootPartitionFs.OpenFile(&updateFile, "/update", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
        FileStorage updateFileStorage(updateFile.get());
        Sha256PartitionFileSystem updateFs;
        NN_RESULT_DO(updateFs.Initialize(&updateFileStorage));

        std::unique_ptr<IFile> normalFile;
        NN_RESULT_DO(rootPartitionFs.OpenFile(&normalFile, "/normal", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
        FileStorage normalFileStorage(normalFile.get());
        Sha256PartitionFileSystem normalFs;
        NN_RESULT_DO(normalFs.Initialize(&normalFileStorage));

        std::unique_ptr<IFile> secureFile;
        NN_RESULT_DO(rootPartitionFs.OpenFile(&secureFile, "/secure", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
        FileStorage secureFileStorage(secureFile.get());
        Sha256PartitionFileSystem secureFs;
        NN_RESULT_DO(secureFs.Initialize(&secureFileStorage));

        std::unique_ptr<char[]> workBuffer(new char[WorkBufferSizeFillFileData]);
        nn::mem::StandardAllocator bufferAllocator(workBuffer.get(), WorkBufferSizeFillFileData);
        nn::fssrv::MemoryResourceFromStandardAllocator allocator(&bufferAllocator);

        // 3 パーティション内のファイルデータを埋める
        updaterPartition.hardwareOffset = XciHeaderSize + rootHeader.headerSize;
        {
            IFileSystem* ppFs[] = { pSrcUpdateFs };
            NN_RESULT_DO(FillFileData(&updateFs, ppFs, 1, updaterPartition, g_WriteProgress, &allocator));
        }
        normalPartition.hardwareOffset  = updaterPartition.hardwareOffset + updaterPartition.headerSize + updaterPartition.bodySize;
        NN_RESULT_DO(FillFileData(&normalFs, pSrcNormalFs, srcNormalFsCount, normalPartition, g_WriteProgress, &allocator));
        securePartition.hardwareOffset  = KeyAreaSize + normalAreaSize;
        NN_RESULT_DO(FillFileData(&secureFs, pSrcSecureFs, srcSecureFsCount, securePartition, g_WriteProgress, &allocator));
        DEVMENUCOMMAND_LOG("Write rest sectors.\n");
        // RootHeader
        int64_t rootHeaderOffset = XciHeaderSize;
        NN_RESULT_DO(pDstStorage->Write(rootHeaderOffset, rootHeader.headerBuffer.get(), rootHeader.headerSize));

        // root info を返す
        pOutRootPartitionFsInfo->headerOffset = FsPartitionHeaderOffset; // KeyArea のサイズを含まない
        pOutRootPartitionFsInfo->headerSize   = rootHeader.headerSize;
        crypto::GenerateSha256Hash(pOutRootPartitionFsInfo->headerHash, sizeof(pOutRootPartitionFsInfo->headerHash), rootHeader.headerBuffer.get(), rootHeader.headerSize);
        NN_RESULT_SUCCESS;
    }


    //! 開発版の鍵で暗号化と署名付与を行う
    void SignAndEncryptCardHeaderDev(char* pCardHeader, size_t cardHeaderSize)
    {

        {
            const unsigned char EncryptionKeyDev[] = {0xCB, 0xA7, 0xB8, 0x75, 0xEB, 0x67, 0x05, 0xFB, 0x46, 0x0A, 0x33, 0xFD, 0x34, 0x09, 0x13, 0xB4};

            const int IvOffset = 0x120;
            const int EncryptedAreaOffset = 0x190;
            const int EncryptedAreaSize = 0x200 - EncryptedAreaOffset;

            char iv[16] = {0};
            memset(iv, 0xAA, sizeof(iv)); // fixed IV for dev
            memcpy(pCardHeader + IvOffset, iv, sizeof(iv));

            nn::crypto::EncryptAes128Cbc(
                    pCardHeader + EncryptedAreaOffset, EncryptedAreaSize,
                    EncryptionKeyDev, sizeof(EncryptionKeyDev),
                    iv, sizeof(iv),
                    pCardHeader + EncryptedAreaOffset, EncryptedAreaSize
            );
        }

        // 開発用の鍵をベタ持ち
        const unsigned char ModulusDev[] = {
                0xc8, 0x65, 0x8d, 0x9d, 0x15, 0xf4, 0xcc, 0x35, 0x7d, 0x3c, 0x7b, 0xbf, 0xa3, 0x7d, 0xa9, 0xfe,
                0x93, 0xd9, 0x3a, 0x64, 0x7c, 0x12, 0x81, 0xb8, 0xa7, 0x6d, 0xe6, 0x76, 0xa5, 0x9f, 0x95, 0xb1,
                0x0b, 0xc5, 0x93, 0x9f, 0x48, 0xe9, 0x4f, 0x3d, 0xd1, 0x94, 0x0f, 0x78, 0x70, 0x5a, 0x2c, 0x82,
                0x6c, 0xe9, 0xb0, 0xa7, 0x6c, 0xea, 0xb5, 0xc1, 0x20, 0xd0, 0x2a, 0x29, 0x42, 0xa6, 0x33, 0x70,
                0x75, 0x53, 0x3e, 0x88, 0x4a, 0xef, 0x35, 0x0e, 0x79, 0xe4, 0xb0, 0x0f, 0x90, 0xa2, 0xac, 0xf8,
                0x31, 0x02, 0xa3, 0x8e, 0x99, 0x7e, 0xf4, 0x72, 0x5a, 0x0b, 0xe8, 0x23, 0x4e, 0x87, 0xfb, 0x2f,
                0x22, 0x22, 0x57, 0xf6, 0xe1, 0x43, 0xfd, 0x11, 0xda, 0x2d, 0xe6, 0x25, 0x96, 0x4c, 0x6b, 0x3b,
                0x54, 0x0c, 0x22, 0x8c, 0xb5, 0x82, 0xdb, 0x49, 0x5c, 0xb0, 0x36, 0x13, 0x31, 0x6f, 0x1a, 0xff,
                0xa5, 0x1f, 0x70, 0x15, 0xac, 0xda, 0xf5, 0xd6, 0xe5, 0x71, 0x2f, 0x47, 0x43, 0xab, 0x00, 0x03,
                0xce, 0x9c, 0x70, 0xeb, 0x58, 0x6c, 0xe1, 0x3f, 0xc8, 0xd7, 0x43, 0xda, 0x34, 0xdd, 0x23, 0x76,
                0xe3, 0x39, 0xb6, 0x8e, 0x5d, 0x63, 0xd6, 0xdd, 0x42, 0x5b, 0xb4, 0x58, 0xcf, 0x2d, 0x47, 0x61,
                0x2f, 0x3f, 0xc3, 0x20, 0xf5, 0xd6, 0xdb, 0xfd, 0x75, 0xcb, 0x06, 0xbc, 0x94, 0x4e, 0xe5, 0x3d,
                0xc8, 0x70, 0xc6, 0xcb, 0xb9, 0xe0, 0x9b, 0x0f, 0x32, 0xa4, 0xc3, 0xca, 0x46, 0x8c, 0x44, 0x2d,
                0x2e, 0x71, 0xc8, 0xf0, 0x51, 0x17, 0x94, 0x5d, 0x40, 0xe2, 0x31, 0x9a, 0x24, 0x9f, 0x7c, 0xc5,
                0xdc, 0xb9, 0xb4, 0x43, 0x23, 0x70, 0xf7, 0x73, 0x0a, 0x5a, 0x6b, 0x8d, 0x9c, 0x76, 0xb1, 0x23,
                0x49, 0x35, 0x4e, 0x3e, 0x92, 0x22, 0xdf, 0xbb, 0x5e, 0xf4, 0x9e, 0x98, 0xcf, 0x51, 0xbe, 0xdf
        };

        const unsigned char PrivateExponentDev[] = {
                0x8a, 0xe8, 0x7a, 0x05, 0x4d, 0x56, 0xe6, 0xc6, 0x9d, 0x0c, 0xf3, 0xea, 0x9e, 0xf5, 0x58, 0x84,
                0xb3, 0x0f, 0x7c, 0xbe, 0x72, 0x92, 0x91, 0x80, 0xe1, 0xed, 0xf8, 0x36, 0x7e, 0x0f, 0xe4, 0xbf,
                0x58, 0x9d, 0x97, 0xfa, 0x6d, 0x9b, 0xeb, 0xe5, 0xba, 0x04, 0x9b, 0xc4, 0xac, 0x7c, 0x95, 0x25,
                0xc3, 0x41, 0x29, 0xc6, 0xa4, 0x64, 0xfd, 0x9f, 0xe5, 0x92, 0xe5, 0xe6, 0xbb, 0xe0, 0x60, 0xff,
                0xaf, 0x14, 0x08, 0xe0, 0x88, 0xb0, 0xcf, 0x62, 0x3d, 0x74, 0x1b, 0xe6, 0x6e, 0x27, 0x95, 0x26,
                0x65, 0x32, 0xa3, 0x15, 0x11, 0xa9, 0x42, 0xe0, 0x91, 0x32, 0x77, 0xb3, 0x21, 0x98, 0x78, 0xb0,
                0xef, 0x44, 0x47, 0x1d, 0x46, 0x80, 0xce, 0x47, 0x35, 0xa9, 0x63, 0x42, 0xe2, 0xb0, 0x0e, 0x14,
                0x8d, 0xe9, 0xeb, 0x79, 0x05, 0xc8, 0xb3, 0x20, 0x63, 0x33, 0x72, 0x08, 0x53, 0xd7, 0xe4, 0x29,
                0xde, 0x5f, 0x28, 0xac, 0xf3, 0xba, 0x36, 0x14, 0x34, 0x09, 0x59, 0x29, 0xdb, 0xc0, 0xcb, 0x67,
                0xe8, 0x32, 0x64, 0x89, 0xd1, 0xd4, 0x39, 0x97, 0xe4, 0x97, 0xc5, 0xd6, 0x26, 0xe0, 0x23, 0x10,
                0xa8, 0x0e, 0x03, 0x6f, 0x4e, 0x0f, 0x52, 0x39, 0x3b, 0xa6, 0x73, 0x14, 0xe7, 0x5e, 0xe8, 0x95,
                0x08, 0xfb, 0xc2, 0x21, 0x6a, 0x38, 0x92, 0x84, 0x7a, 0xc9, 0x33, 0xdb, 0xb7, 0xd3, 0xe5, 0x70,
                0x1c, 0x6e, 0x6c, 0xce, 0x93, 0xd6, 0x82, 0xb7, 0x41, 0x9a, 0x76, 0xa6, 0x1f, 0xc4, 0xee, 0x29,
                0xc8, 0x4f, 0xaa, 0x3a, 0x0d, 0xd1, 0x8e, 0x7d, 0xd7, 0xba, 0x43, 0xeb, 0x7d, 0x9c, 0x94, 0x9a,
                0x2c, 0x62, 0x9a, 0x1a, 0xf8, 0x0f, 0xdd, 0x93, 0xab, 0xd5, 0x5c, 0xa0, 0xd4, 0x03, 0x47, 0x96,
                0xd9, 0xa4, 0xb2, 0xc9, 0x90, 0xaa, 0xfa, 0x62, 0x9d, 0xdf, 0x35, 0x66, 0xe7, 0x31, 0x47, 0x61
        };

        const char pub[] = {0x01, 0x00, 0x01};

        char* pSign = pCardHeader;
        char* pSignTarget = pCardHeader + crypto::Rsa2048Pkcs1Sha256Signer::SignatureSize;
        const int SignatureSize = crypto::Rsa2048Pkcs1Sha256Signer::SignatureSize;

        auto succeeded = crypto::SignRsa2048Pkcs1Sha256(
                pSign, SignatureSize,
                ModulusDev, sizeof(ModulusDev),
                PrivateExponentDev, sizeof(PrivateExponentDev),
                pSignTarget, cardHeaderSize - SignatureSize);
        NN_ABORT_UNLESS(succeeded == true);

        {
            auto verified = crypto::VerifyRsa2048Pkcs1Sha256(
                    pCardHeader, SignatureSize,
                    ModulusDev, sizeof(ModulusDev),
                    pub, sizeof(pub),
                    pCardHeader + SignatureSize, SignatureSize
            );
            NN_ABORT_UNLESS(verified == true);
        }

    }

    void SetFixedValueToCardHeader(char* pOutBuffer)
    {
        // 開発用のカードヘッダ固定値をベタ持ち
        const size_t InitialDataLength        = 64;
        const size_t TitleKeyLength           = 16;
        const size_t InitialDataHashLength    = 32;
        const size_t OffsetInitialData        = 0;
        const size_t OffsetItitleKey          = 0x200;
        const size_t OffsetInitialDataHash    = 0x1160;

        const unsigned char InitialData[InitialDataLength] =
        {
            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
            0x5E, 0x22, 0x4E, 0x28, 0x1E, 0xDD, 0xFC, 0x53, 0x36, 0x95, 0x0C, 0xA7, 0x5E, 0x5F, 0x4A, 0x8C,
            0x7A, 0x9E, 0x0A, 0x37, 0x77, 0x7E, 0x2E, 0x35, 0xEB, 0xEE, 0x9D, 0xD9, 0xA4, 0x01, 0xFD, 0x07,
            0xF1, 0x10, 0x00, 0x35, 0xBB, 0x24, 0xA8, 0xD2, 0x60, 0x04, 0xE0, 0xE2, 0x00, 0x00, 0x00, 0x00
        };
        const unsigned char TitleKey[TitleKeyLength] =
        {
            0x84, 0x17, 0xFA, 0xB6, 0x82, 0x9A, 0x3D, 0xC9, 0x2D, 0x53, 0x4F, 0xC4, 0xFC, 0x23, 0x45, 0x7B
        };
        memset(pOutBuffer , 0x00, KeyAreaSize + CardHeaderSize);
        memcpy(pOutBuffer + OffsetInitialData, InitialData, InitialDataLength);
        memcpy(pOutBuffer + OffsetItitleKey,   TitleKey,    TitleKeyLength);
        crypto::GenerateSha256Hash(pOutBuffer + OffsetInitialDataHash, InitialDataHashLength, pOutBuffer, 512);

        // Magic Code("HEAD") の設定
        const size_t MagicCodePos = 0x1100;
        pOutBuffer[MagicCodePos + 0]  = 0x48;//"H"
        pOutBuffer[MagicCodePos + 1]  = 0x45;//"E"
        pOutBuffer[MagicCodePos + 2]  = 0x41;//"A"
        pOutBuffer[MagicCodePos + 3]  = 0x44;//"D"

        // BackupAreaStartPageAddress の設定
        const size_t BackupAreaPos = 0x1108;
        const char BackupAreaByteNum = 4;
        const uint32_t BackupArea = 0xFFFFFFFF;
        memcpy(pOutBuffer + BackupAreaPos, &BackupArea, BackupAreaByteNum);

        // TitleKey Index の設定
        const size_t TitleKeyIndexPos   = 0x110c;
        const size_t TitleKeyIndexValue = 0x01;
        pOutBuffer[TitleKeyIndexPos]    = TitleKeyIndexValue;

        // packageId の設定
        const size_t PackageIdPos     = 0x1110;
        const char PackageIdByteNum = 8;
        for(char i = 0; i < PackageIdByteNum ; i++)
        {
            pOutBuffer[PackageIdPos + i] = PackageIdByteNum - 1 - i;
        }

        // SEL_SEC の設定
        const size_t SelSecPos   = 0x1180;
        const size_t SelSecValue = 0x01;
        pOutBuffer[SelSecPos]    = SelSecValue;

        // SEL_T1_KEY の設定
        const size_t SelT1KeyPos   = 0x1184;
        const size_t SelT1KeyValue = 0x02;
        pOutBuffer[SelT1KeyPos]    = SelT1KeyValue;

        // ACC_CTRL1 のデフォルト値設定
        const size_t AccCtrl1Pos     = 0x1198;
        const size_t AccCtrl1ByteNum = 4;
        const size_t AccCtrl1Value   = 0x00A10010;
        memcpy(pOutBuffer + AccCtrl1Pos, &AccCtrl1Value, AccCtrl1ByteNum);

        // WAIT1_TIME_READ の設定
        const size_t LatencySettingPos = 0x119c;
        const size_t LatencySettingByteNum = 4;
        const size_t LatencySetting = 5000;//(200us/40ns)
        for(size_t i = 0; i < LatencySettingByteNum; i++)
        {
            pOutBuffer[LatencySettingPos + i] = 0xFF & (LatencySetting >> 8 * i);
        }

    }

    Result CreateKeyAreaAndCardHeader(char* pOutBuffer, size_t pOutBufferSize, CardInfo& cardInfo,  int64_t headerOffset, int64_t headerSize, const char* headerHashBuffer, size_t headerHashBufferSize, bool isAutoBoot, bool isHistoryErase, bool isRepairTool)
    {
        // ゲームカード用の鍵領域とカードヘッダを生成
        const size_t  HeaderHashSize        = 32;
        const size_t  OffsetHeaderOffset    = 0x130;
        const size_t  OffsetHeaderSize      = 0x138;
        const size_t  OffsetHeaderHash      = 0x140;
        const size_t  OffsetAttribute       = 0x10F;

        NN_ABORT_UNLESS(pOutBufferSize >= CardHeaderSize);
        NN_ABORT_UNLESS(headerHashBufferSize >= HeaderHashSize);
        SetFixedValueToCardHeader(pOutBuffer);

        // Rom 容量の設定
        const size_t RomCapacityPos = 0x110D;
        unsigned char sizeValue = 0;
        switch(cardInfo.cardSize)
        {
            case nn::fs::GameCardSize::Size1GB  : sizeValue = 0xFA; break;
            case nn::fs::GameCardSize::Size2GB  : sizeValue = 0xF8; break;
            case nn::fs::GameCardSize::Size4GB  : sizeValue = 0xF0; break;
            case nn::fs::GameCardSize::Size8GB  : sizeValue = 0xE0; break;
            case nn::fs::GameCardSize::Size16GB : sizeValue = 0xE1; break;
            case nn::fs::GameCardSize::Size32GB : sizeValue = 0xE2; break;
            default : NN_ABORT("Card Size is invalid. card Size is now %d\n", static_cast<uint32_t>(cardInfo.cardSize));
        }
        *(reinterpret_cast<unsigned char*>(pOutBuffer) + RomCapacityPos) = sizeValue;

        // ACC_CTRL1 の設定
        const size_t SpeedControlPos = 0x1198;
        const char   SpeedLowBit     = 0x01;
        if(cardInfo.cardClockRate == nn::fs::GameCardClockRate::ClockRate25MHz)
        {
            pOutBuffer[SpeedControlPos] |= SpeedLowBit;
        }

        // CUP バージョンの設定
        const size_t CupVersionPos     = 0x11B0;
        const size_t CupVersionByteNum = 4;
        for(size_t i = 0; i < CupVersionByteNum; i++)
        {
            pOutBuffer[CupVersionPos + i] = 0xFF & (cardInfo.cupVersion >> 8 * i);
        }

        // CUP ID の設定
        const size_t CupIdPos     = 0x11C0;
        const size_t CupIdByteNum = 8;
        for(size_t i = 0; i < CupIdByteNum; i++)
        {
            pOutBuffer[CupIdPos + i] = 0xFF & (cardInfo.cupId >> 8 * i);
        }

        // UPP Hash  の設定
        const size_t UppHashPos  = 0x11B8;
        const size_t UppHashByteNum = 8;
        for(size_t i = 0; i < UppHashByteNum; i++)
        {
            pOutBuffer[UppHashPos + i] = cardInfo.uppHashForLotCheck[i];
        }

        // PartitionFs ヘッダの設定
        memcpy(pOutBuffer + KeyAreaSize + OffsetHeaderOffset, &headerOffset,     sizeof(headerOffset));
        memcpy(pOutBuffer + KeyAreaSize + OffsetHeaderSize,   &headerSize,       sizeof(headerSize));
        memcpy(pOutBuffer + KeyAreaSize + OffsetHeaderHash,    headerHashBuffer, HeaderHashSize);

        // CardHeader の flags 設定
        const char   AutoBootBit     = 0x01;
        const char   HistoryEraseBit = 0x02;
        const char   RepairToolBit   = 0x04;
        // 自動起動フラグ
        if(isAutoBoot)
        {
            *(pOutBuffer + KeyAreaSize + OffsetAttribute) |= AutoBootBit;
        }
        // 履歴削除フラグ
        if(isHistoryErase)
        {
            *(pOutBuffer + KeyAreaSize + OffsetAttribute) |= HistoryEraseBit;
        }
        // 修理ツールフラグ
        if(isRepairTool)
        {
            *(pOutBuffer + KeyAreaSize + OffsetAttribute) |= RepairToolBit;
        }

        // ノーマルエリアサイズの設定
        const size_t RomAreaStartPageAddressPos = 0x1104;
        const size_t LimAreaPos                 = 0x118C;

        uint32_t normalAreaPageSize = static_cast<uint32_t>(cardInfo.normalAreaSize / nn::gc::GcPageSize);
        *(reinterpret_cast<uint32_t*>(pOutBuffer + RomAreaStartPageAddressPos)) = normalAreaPageSize;
        *(reinterpret_cast<uint32_t*>(pOutBuffer + LimAreaPos))                 = normalAreaPageSize;

        // 有効データの最終ページアドレスの設定
        const size_t ValidLastPageAddressPos = 0x1118;
        uint32_t validAreaPageSize  = static_cast<uint32_t>(RoundupTotalPageSize(cardInfo.totalSize) / nn::gc::GcPageSize) - 1;
        *(reinterpret_cast<uint32_t*>(pOutBuffer + ValidLastPageAddressPos)) = validAreaPageSize;
        /*
        DEVMENUCOMMAND_LOG("CardHeader \n");
        for(int i = 0; i < 512/16; i++)
        {
            for(int j = 0; j < 16; j++)
            {
                DEVMENUCOMMAND_LOG("%02x ", *(pOutBuffer  + KeyAreaSize + i * 16 + j));
            }
            DEVMENUCOMMAND_LOG("\n");
        }
        */
        // 暗号化・署名
        SignAndEncryptCardHeaderDev(pOutBuffer + KeyAreaSize, CardHeaderSize);

        NN_RESULT_SUCCESS;
    }

    using SearchEntryFunction = Result(*)(DirectoryEntry*, IDirectory*, const char*);
    Result ReadFileFromNspImpl( size_t* pOutSize, std::unique_ptr<char[]>* pOutBuffer, const char* pNspFilePath, const char* findString, SearchEntryFunction searchEntry) NN_NOEXCEPT
    {
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        nn::fs::FileHandle srcFileHandle = {};
        std::unique_ptr<nn::fssystem::PartitionFileSystem> srcNspSecureFs;
        std::unique_ptr<nn::fs::FileHandleStorage> srcFileHandleStorage;

        NN_RESULT_DO(OpenFile(&srcFileHandle, pNspFilePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile( srcFileHandle );
        };

        srcFileHandleStorage.reset(new nn::fs::FileHandleStorage( srcFileHandle ));
        srcNspSecureFs.reset(new nn::fssystem::PartitionFileSystem());
        NN_RESULT_DO(srcNspSecureFs->Initialize(srcFileHandleStorage.get()));

        std::unique_ptr<nn::fs::fsa::IDirectory> directory;
        NN_RESULT_DO(srcNspSecureFs->OpenDirectory(&directory, "/", nn::fs::OpenDirectoryMode_File));

        // .cnmt.xml ファイルを探す
        nn::fs::DirectoryEntry dirEntry;
        NN_RESULT_DO(searchEntry(&dirEntry, directory.get(), findString));

        size_t size = 0;
        size_t bufferSize = static_cast<size_t>(dirEntry.fileSize);
        std::unique_ptr<char[]> buffer(new char[bufferSize]);
        NN_RESULT_DO(ReadFileFromIFileSystem(&size, buffer.get(), bufferSize, srcNspSecureFs.get(), dirEntry));

        *pOutSize = size;
        *pOutBuffer = std::move(buffer);
#else
        NN_UNUSED( pOutSize );
        NN_UNUSED( pOutBuffer );
        NN_UNUSED( pNspFilePath );
        NN_UNUSED( findString );
        NN_UNUSED( searchEntry );
#endif
        NN_RESULT_SUCCESS;
    }

    Result ReadFileFromNsp( size_t* pOutSize, std::unique_ptr<char[]>* pOutBuffer, const char* pNspFilePath, const char* pFileName ) NN_NOEXCEPT
    {
        return ReadFileFromNspImpl(pOutSize, pOutBuffer, pNspFilePath, pFileName, SearchFileEntry);
    }

    Result ReadFileWithExtensionFromNsp( size_t* pOutSize, std::unique_ptr<char[]>* pOutBuffer, const char* pNspFilePath, const char* pExtension ) NN_NOEXCEPT
    {
        return ReadFileFromNspImpl(pOutSize, pOutBuffer, pNspFilePath, pExtension, SearchEntry);
    }

    void ReadContentMetaInfo(ContentMetaInfo* outValue, const ::pugi::xml_node& node)
    {
        ContentMetaInfo meta{};
        auto idNode = node.child(Id);
        meta.id = static_cast<Bit64>(std::strtoul(idNode.child_value(), nullptr, 16));

        auto versionNode = node.child(Version);
        meta.version = static_cast<uint32_t>(std::strtoul(versionNode.child_value(), nullptr, 16));

        auto digestNode = node.child(Digest);
        util::Strlcpy(meta.digest, digestNode.child_value(), static_cast<int>(NN_ARRAY_SIZE(meta.digest)));

        *outValue = meta;
    }

    void ReadContentMetaInfoList(std::vector<ContentMetaInfo>* pOutList, const ::pugi::xml_node& rootNode)
    {
        const util::string_view contentMetaName(ContentMeta);
        std::for_each(rootNode.begin(), rootNode.end(), [&contentMetaName, &pOutList](const ::pugi::xml_node& node) NN_NOEXCEPT
        {
            const util::string_view nodeName(node.name());
            if (contentMetaName == nodeName)
            {
                ContentMetaInfo meta{};
                ReadContentMetaInfo(&meta, node);
                pOutList->push_back(meta);
            }
        });
    }

    Result ReadContentMetaInfoFromNsp(ContentMetaInfo* outValue, const char* nspPath)
    {
        std::unique_ptr<char[]> buffer;
        size_t bufferSize;
        NN_RESULT_DO(ReadFileWithExtensionFromNsp(&bufferSize, &buffer, nspPath, ".cnmt.xml"));

        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(buffer.get(), bufferSize);
        NN_ABORT_UNLESS(result.status == ::pugi::xml_parse_status::status_ok, "pugi load failure. (status=%d)\n", result.status);

        ReadContentMetaInfo(outValue, document.child(ContentMeta));

        NN_RESULT_SUCCESS;
    }

    void InitializeCardInfo(CardInfo* pOutCardInfo, nn::fs::GameCardSize cardSize, nn::fs::GameCardClockRate cardClockRate)
    {
        // cardSize が指定されている
        if((static_cast<uint8_t>(cardSize) != 0))
        {
            pOutCardInfo->cardSize  = cardSize;
            pOutCardInfo->cardClockRate = (static_cast<uint32_t>(cardSize) > 4) ? nn::fs::GameCardClockRate::ClockRate50MHz : nn::fs::GameCardClockRate::ClockRate25MHz;
        }

        // cardClockRate が指定されている
        if (static_cast<uint8_t>(cardClockRate) != 0)
        {
            if (static_cast<uint8_t>(cardClockRate) <= static_cast<uint8_t>(pOutCardInfo->cardClockRate))
            {
                pOutCardInfo->cardClockRate = cardClockRate;
            }
            else
            {
                DEVMENUCOMMAND_LOG("%d is specified as CardClockRate but it is invalid. Ignored it and set %d.\n", static_cast<uint8_t>(cardClockRate), static_cast<uint8_t>(pOutCardInfo->cardClockRate));
            }
        }
    }
}

    //! 与えられた GameCardSize が正規の値かどうかをチェックする
    bool IsGameCardSizeRight(nn::fs::GameCardSize gameCardSize) NN_NOEXCEPT
    {
        if( gameCardSize == nn::fs::GameCardSize::Size1GB ||
                gameCardSize == nn::fs::GameCardSize::Size2GB ||
                gameCardSize == nn::fs::GameCardSize::Size4GB ||
                gameCardSize == nn::fs::GameCardSize::Size8GB ||
                gameCardSize == nn::fs::GameCardSize::Size16GB ||
                gameCardSize == nn::fs::GameCardSize::Size32GB)
        {
            return true;
        }
        return false;
    }

    //! 与えられた GameCardClockRate が正規の値かどうかをチェックする
    bool IsGameCardClockRateRight(nn::fs::GameCardClockRate gameCardClockRate) NN_NOEXCEPT
    {
        if( gameCardClockRate == nn::fs::GameCardClockRate::ClockRate25MHz ||
            gameCardClockRate == nn::fs::GameCardClockRate::ClockRate50MHz )
        {
            return true;
        }
        return false;
    }

    bool IsGameCardInsertedWithinTime(size_t timeout)
    {
        size_t counter = 0;
        while(!nn::fs::IsGameCardInserted())
        {
            if((timeout != 0) && (++counter > timeout))
            {
                // Timeout 以上の時間待ちでアプリ正常終了
                DEVMENUCOMMAND_LOG("error : timeout. Could not detect GameCard for %zd seconds.\n", timeout);
                return false;
            }
            DEVMENUCOMMAND_LOG("Wait for Inserting GameCard..\n");
#ifdef NN_BUILD_CONFIG_OS_HORIZON
            nn::idle::ReportUserIsActive();
#endif
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        return true;
    }

    Result EraseGameCard(size_t timeout) NN_NOEXCEPT
    {
        nn::ns::GameCardStopper stopper;
        nn::ns::GetGameCardStopper(&stopper);
        if(!IsGameCardInsertedWithinTime(timeout))
        {
            // カード未挿入のため終了
            NN_RESULT_SUCCESS;
        }

        GameCardHandle cardHandle = 0; // TODO: GetGameCardHandle() で取得できるよう修正
        std::unique_ptr<nn::fs::IStorage> dstXciStorage;
        NN_RESULT_DO(OpenGameCardPartition(&dstXciStorage, cardHandle, GameCardPartitionRaw::RootWriteOnly));

        const nn::fs::GameCardSize ErasedGameCardSize = nn::fs::GameCardSize::Size1GB;
        const uint64_t ErasedNormalAreaSize = 0x80000 * 512;
        DEVMENUCOMMAND_LOG("Erase gamecard.\n");
        return nn::fs::EraseGameCard(ErasedGameCardSize, ErasedNormalAreaSize);
    }

    Result WriteFromNsp(const char (*srcRomNspPaths)[MaxFilePathLength], int srcRomNspCount, nn::fs::GameCardSize cardSize, nn::fs::GameCardClockRate cardClockRate, const char* srcUpdateNspPath, bool isAutoBoot, bool isHistoryErase, bool isRepairTool, size_t timeout, bool isFastparam, const char* dstXciPath) NN_NOEXCEPT
    {
        ::pugi::set_memory_management_functions(AllocateTest, DeallocateTest);

#ifndef NN_BUILD_CONFIG_OS_WIN
        nn::ns::GameCardStopper stopper;
        nn::ns::GetGameCardStopper(&stopper);
#endif

        CardInfo cardInfo;
        InitializeCardInfo(&cardInfo, cardSize, cardClockRate);

#ifdef NN_BUILD_CONFIG_OS_WIN
        // dst
        if (!dstXciPath || dstXciPath[0] == '\0')
        {
            NN_ABORT("Please specify output file path.");
        }
        NN_UNUSED(timeout);
        const int64_t StorageSize = 512 * 1024 * 1024;
        NN_RESULT_TRY(CreateFile(dstXciPath, StorageSize))
            NN_RESULT_CATCH(ResultPathAlreadyExists)
            {
                // through
            }
        NN_RESULT_END_TRY;
        FileHandle dstHandle;
        NN_RESULT_DO(OpenFile(&dstHandle, dstXciPath, OpenMode_Write | OpenMode_Read | OpenMode_AllowAppend));
        NN_RESULT_DO(SetFileSize(dstHandle, StorageSize));
        std::shared_ptr<nn::fs::IStorage> dstXciStorage(new FileHandleStorage(dstHandle));
#else
        // カード挿入待ち
        if(!IsGameCardInsertedWithinTime(timeout))
        {
            // カード未挿入のため終了
            NN_RESULT_THROW(nn::fs::ResultGameCardDevCardUnexpectedFailure());
        }
        // dst
        GameCardHandle cardHandle = 0;
        // NN_RESULT_DO(GetGameCardHandle(&cardHandle));
        std::unique_ptr<nn::fs::IStorage> dstXciStorage;
        NN_RESULT_DO(OpenGameCardPartition(&dstXciStorage, cardHandle, GameCardPartitionRaw::RootWriteOnly));
#endif

        // TODO: サイズアライン合わせによって上書きが発生してしまっていないかチェック
        std::shared_ptr<IStorage> dstAlignedXciStorage(new SizeAlignmentWriteOnlyStorage<nn::gc::GcPageSize>(std::move(dstXciStorage)));
#ifdef NN_BUILD_CONFIG_OS_WIN
        g_XciStorage = dstAlignedXciStorage;
#endif

        std::unique_ptr<FileHandle[]> srcHandle(new FileHandle[srcRomNspCount]);
        for (int i = 0; i < srcRomNspCount; i++)
        {
            srcHandle[i].handle = nullptr;
        }
        std::unique_ptr<IFileSystem*> pSrcSecureFs(new IFileSystem*[srcRomNspCount]);
        std::unique_ptr<IFileSystem*> pSrcNormalFs(new IFileSystem*[srcRomNspCount]);

        {
            // src(secure, normal)
            // TODO: normal 領域は未使用なので削除
            std::unique_ptr<PartitionFileSystem[]> srcSecureFs(new PartitionFileSystem[srcRomNspCount]);
            std::unique_ptr<PartitionFileSystem[]> srcNormalFs(new PartitionFileSystem[srcRomNspCount]);
            NN_UTIL_SCOPE_EXIT
            {
                for (int i = 0; i < srcRomNspCount; i++)
                {
                    if (srcHandle[i].handle != nullptr)
                    {
                        CloseFile(srcHandle[i]);
                    }
                }
            };
            for (int i = 0; i < srcRomNspCount; i++)
            {
                NN_RESULT_DO(OpenFile(&srcHandle[i], srcRomNspPaths[i], OpenMode_Read));
                std::shared_ptr<FileHandleStorage> srcNspStorage = std::make_shared<FileHandleStorage>(srcHandle[i]);

                NN_RESULT_DO(srcSecureFs[i].Initialize(srcNspStorage));
                NN_RESULT_DO(srcNormalFs[i].Initialize(srcNspStorage));

                (pSrcSecureFs.get())[i] = &srcSecureFs[i];
                (pSrcNormalFs.get())[i] = &srcNormalFs[i];

                NN_RESULT_DO(CheckExternalKeyFromNsp((pSrcNormalFs.get())[i]));
            }

            // upp(update)
            FileHandle srcUpdateHandle = { nullptr };
            std::unique_ptr<PartitionFileSystem> srcUpdateFs;
            std::unique_ptr<FileHandleStorage> srcUpdateNspStorage;
            NN_UTIL_SCOPE_EXIT
            {
                if (srcUpdateHandle.handle != nullptr)
                {
                    CloseFile( srcUpdateHandle );
                }
            };
            if (srcUpdateNspPath[0] != '\0')
            {
                NN_RESULT_DO(OpenFile(&srcUpdateHandle, srcUpdateNspPath, OpenMode_Read));
                srcUpdateNspStorage.reset(new FileHandleStorage(srcUpdateHandle));
                srcUpdateFs.reset(new PartitionFileSystem());
                NN_RESULT_DO(srcUpdateFs->Initialize(srcUpdateNspStorage.get()));
            }

            // xci 書き出し
            RootSha256PartitionFileSystemInfo rootInfo;
            NN_RESULT_DO(WriteXci(&rootInfo, dstAlignedXciStorage.get(), srcUpdateFs.get(), pSrcNormalFs.get(), srcRomNspCount, pSrcSecureFs.get(), srcRomNspCount, &cardInfo, isFastparam));

            // Key Area + Card header のバッファ作成
            std::unique_ptr<char[]> headerBuffer(new char[KeyAreaSize + CardHeaderSize]);
            NN_RESULT_DO(CreateKeyAreaAndCardHeader(headerBuffer.get(), KeyAreaSize + CardHeaderSize, cardInfo, rootInfo.headerOffset, rootInfo.headerSize, rootInfo.headerHash, sizeof(rootInfo.headerHash), isAutoBoot, isHistoryErase, isRepairTool));

            // KeyArea + CardHeader 書き出し
            NN_RESULT_DO(dstAlignedXciStorage->Write(0, headerBuffer.get(), KeyAreaSize + CardHeaderSize));
            g_WriteProgress.writtenSize = g_WriteProgress.totalSize;
            g_WriteProgress.PrintProgress();
        }

        dstAlignedXciStorage.reset();
        dstXciStorage.reset();

#ifdef NN_BUILD_CONFIG_OS_WIN
        NN_RESULT_DO(FlushFile(dstHandle));
        CloseFile(dstHandle);
#endif

        DEVMENUCOMMAND_LOG("Finished\n");
        NN_RESULT_SUCCESS;
    }

    Result WriteFromNsp(const char (*srcRomNspPaths)[MaxFilePathLength], int srcRomNspCount, nn::fs::GameCardSize cardSize, nn::fs::GameCardClockRate cardClockRate, const char* srcUpdateNspPath, bool isAutoBoot, bool isHistoryErase, bool isRepairTool, size_t timeout, bool isFastParam) NN_NOEXCEPT
    {
        g_WriteProgress.Initialize();
        return WriteFromNsp(srcRomNspPaths, srcRomNspCount, cardSize, cardClockRate, srcUpdateNspPath, isAutoBoot, isHistoryErase, isRepairTool, timeout, isFastParam, nullptr);
    }

    void GetGameCardWriteProgress(GameCardWriteProgress* pOutProgress) NN_NOEXCEPT
    {
        // 値を取るだけなら mutex するほどでもない
        pOutProgress->currentSize = g_WriteProgress.writtenSize;
        pOutProgress->totalSize   = g_WriteProgress.totalSize;
    }

    /**
    * @brief       指定したファイルが指定したコンテンツタイプかチェックする
    *
    * @param[out]  pOutValue        チェック結果(true:コンテンツタイプが一致  false:コンテンツタイプが不一致)
    * @param[in]   pNspFilePath     nspファイルパス
    * @param[in]   contentMetaType  コンテンツタイプ（アプリケーション、パッチ、AddOnContent）
    * @return      チェックの成否を Result 値で返します。
    */
    Result CheckContentMeta( bool* pOutValue, const char* pNspFilePath, nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT
    {
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        // xml ファイルの読み込み
        size_t size = 0;
        std::unique_ptr<char[]> buffer;
        NN_RESULT_DO(ReadFileWithExtensionFromNsp(&size, &buffer, pNspFilePath, ".cnmt.xml"));

        // xml の中身解析
        ::pugi::xml_document document;
        ::pugi::xml_parse_result result = document.load_buffer(buffer.get(), size);
        if (result.status != ::pugi::xml_parse_status::status_ok)
        {
            NN_ABORT("pugi load failure\n");
        }
        auto rootNode = ::pugi::xml_node();
        // ContentMeta の解釈
        if( GetXmlNodeChild( &rootNode, document, ContentMeta ) )
        {
            auto childNode = rootNode.child( Type );
            const char* type;
            if ( contentMetaType == nn::ncm::ContentMetaType::Application )
            {
                type = Application;
            }
            else if ( contentMetaType == nn::ncm::ContentMetaType::Patch )
            {
                type = Patch;
            }
            else if ( contentMetaType == nn::ncm::ContentMetaType::AddOnContent )
            {
                type = AddOnContent;
            }
            else
            {
                NN_ABORT( "Invalid argument\n" );
            }

            if ( nn::util::Strncmp( childNode.child_value(), type, nn::util::Strnlen( type, ContentMetaXmlElementLengthMax ) ) == 0 )
            {
                *pOutValue = true;
            }
            else
            {
                *pOutValue = false;
            }
        }
        else
        {
            NN_ABORT("Content Meta not found in cnmt.xml\n");
        }
#else
        NN_UNUSED( pOutValue );
        NN_UNUSED( pNspFilePath );
        NN_UNUSED( contentMetaType );
#endif
        NN_RESULT_SUCCESS;
    }

    MultiApplicationCardIndicator::MultiApplicationCardIndicator(const char* nspPath) NN_NOEXCEPT : m_NspPath(nspPath), m_ApplicationList(), m_PatchList() {}

    Result MultiApplicationCardIndicator::Initialize()
    {
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        // Multi-ApplicationCard の情報を取得
        std::unique_ptr<char[]> macXmlBuffer;
        size_t macXmlSize;
        NN_RESULT_DO(ReadFileFromNsp(&macXmlSize, &macXmlBuffer, m_NspPath, "multiapplicationcard.xml"));

        ::pugi::xml_document macXml;
        ::pugi::xml_parse_result result = macXml.load_buffer(macXmlBuffer.get(), macXmlSize);
        NN_ABORT_UNLESS(result.status == ::pugi::xml_parse_status::status_ok, "pugi load failure. (status=%d)\n", result.status);

        auto rootNode = macXml.child(MultiApplicationCard);
        ReadContentMetaInfoList(&m_ApplicationList, rootNode.child(Application));
        ReadContentMetaInfoList(&m_PatchList, rootNode.child(Patch));

        // ゲームカードの情報を取得
        std::unique_ptr<char[]> cardSpecBuffer;
        size_t cardSpecSize;
        NN_RESULT_DO(ReadFileFromNsp(&cardSpecSize, &cardSpecBuffer, m_NspPath, "cardspec.xml"));

        CardInfo cardInfo;
        GetCardSizeAndSpeedFromCardSpec(&cardInfo, cardSpecBuffer.get(), cardSpecSize);
        m_CardSize = cardInfo.cardSize;
        m_CardClockRate = cardInfo.cardClockRate;

#endif // !defined ( NN_BUILD_CONFIG_OS_WIN )

        NN_RESULT_SUCCESS;
    }

    bool MultiApplicationCardIndicator::Verify(const char (*pathList)[MaxFilePathLength], int numPath)
    {
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        if (numPath != m_ApplicationList.size() + m_PatchList.size())
        {
            DEVMENUCOMMAND_LOG("Please specify nsp files listed on the multi-application card indicator nsp\n");
            return false;
        }

        std::vector<ContentMetaInfo> metaList;
        for (int i = 0; i < numPath; i++)
        {
            ContentMetaInfo meta;
            auto result = ReadContentMetaInfoFromNsp(&meta, pathList[i]);
            if (result.IsFailure())
            {
                DEVMENUCOMMAND_LOG("Cannot read nsp file. (result=0x%08x)\n", result.GetInnerValueForDebug());
                return false;
            }
            metaList.push_back(meta);
        }

        int appCount = 0;
        auto error = std::find_if(metaList.begin(), metaList.end(), [this, &appCount](const ContentMetaInfo& meta)
        {
            auto appFound = (appCount < m_ApplicationList.size()) ? m_ApplicationList[appCount] == meta : false;
            auto patchFound = std::find(m_PatchList.begin(), m_PatchList.end(), meta) != m_PatchList.end();

            // 見つからない、もしくは、アプリケーションとパッチの両方に同じ情報が登録されているとエラー
            if (appFound == patchFound)
            {
                return true;
            }

            if (appFound)
            {
                appCount++;
            }

            return false;
        });

        if (error != metaList.end())
        {
            DEVMENUCOMMAND_LOG("Please specify nsp files in the order that the multi-application card indicator nsp specifies.\n");
            return false;
        }

        std::sort(metaList.begin(), metaList.end());
        auto removedIterator = std::unique(metaList.begin(), metaList.end());

        if (removedIterator != metaList.end())
        {
            DEVMENUCOMMAND_LOG("Duplicated content is found.\n");
            return false;
        }
#else
        NN_UNUSED(pathList);
        NN_UNUSED(numPath);
#endif

        return true;
    }
