﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <string>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_StorageId.h"

using namespace nn;

namespace
{
    const int64_t MaxDirectoryNameLength = 300;
    const size_t ReadBufferSize = 16 * 1024 * 1024;

    const char SdCardMountName[] = "sdcard";
    const char BuiltInUserMountName[] = "user";
    const char BuiltInSystemMountName[] = "system";

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " filesystem list-all bissystem\n"
        "       " DEVMENUCOMMAND_NAME " filesystem list-all bisuser\n"
        "       " DEVMENUCOMMAND_NAME " filesystem list-all sdcard\n"
        "       " DEVMENUCOMMAND_NAME " filesystem delete-file <file_path> [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " filesystem delete-directory <dir_path> [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " filesystem create-file <file_path> --src <src_file_path> [-s <sdcard|builtin|auto>]\n"
        "       " DEVMENUCOMMAND_NAME " filesystem create-directory <dir_path> [-s <sdcard|builtin|auto>]\n"
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    // directoryPath の合計ファイル数とファイルサイズを計算する
    Result CalculateDirectorySize(int64_t* outDirectoryFileSize, int* outDirectoryFileCount, const char* directoryPath)
    {
        fs::DirectoryHandle directoryHandleForCalculation;

        NN_RESULT_DO(fs::OpenDirectory(&directoryHandleForCalculation, directoryPath, fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseDirectory(directoryHandleForCalculation);
        };

        int64_t entryCount = 1;
        fs::DirectoryEntry entry;

        *outDirectoryFileSize = 0;
        *outDirectoryFileCount = 0;

        while (entryCount)
        {
            NN_RESULT_DO(fs::ReadDirectory(&entryCount, &entry, directoryHandleForCalculation, 1));

            if (entry.directoryEntryType == fs::DirectoryEntryType_File)
            {
                *outDirectoryFileSize += entry.fileSize;
            }
            (*outDirectoryFileCount)++;
        }
        (*outDirectoryFileCount)--;

        NN_RESULT_SUCCESS;
    }

    // directoryPath の以下のファイルを全て表示する
    // directoryDepth は directoryPath のルートからの階層の深さ(ディレクトリの表示の階層を決めるため)
    Result PrintList(const char* directoryPath, int directoryDepth)
    {
        fs::DirectoryHandle directoryHandle;
        NN_RESULT_DO(fs::OpenDirectory(&directoryHandle, directoryPath, fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseDirectory(directoryHandle);
        };

        int64_t entryCount = 1;
        fs::DirectoryEntry entry;

        while (entryCount)
        {
            NN_RESULT_DO(fs::ReadDirectory(&entryCount, &entry, directoryHandle, 1));

            if (entryCount)
            {
                for (int i = 0; i < directoryDepth; i++)
                {
                    DEVMENUCOMMAND_LOG("  ");
                }
            }

            if (entry.directoryEntryType == fs::DirectoryEntryType_File)
            {

                DEVMENUCOMMAND_LOG("%50s %12lld\n", entry.name, entry.fileSize);
            }

            else if (entry.directoryEntryType == fs::DirectoryEntryType_Directory)
            {
                int64_t directoryFileSize = 0;
                int directoryFileCount = 0;

                char childDirectoryPath[MaxDirectoryNameLength];
                sprintf(childDirectoryPath, "%s%s/", directoryPath, entry.name);

                // 子ディレクトリのファイルの合計サイズ、数を求める
                CalculateDirectorySize(&directoryFileSize, &directoryFileCount, childDirectoryPath);

                DEVMENUCOMMAND_LOG("%s/", entry.name);
                for (size_t i = 0; i < 37 - std::string(entry.name).length(); i++)
                {
                    DEVMENUCOMMAND_LOG(" ");
                }
                DEVMENUCOMMAND_LOG("%14d %12lld\n", directoryFileCount, directoryFileSize);

                // 子ディレクトリに対して再帰的に関数を呼び出す
                PrintList(childDirectoryPath, directoryDepth + 1);
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result ListAllCommand(bool* outValue, const Option& option)
    {
        // 有効なコマンドでない時
        if (!option.HasTarget())
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // bissystem:/ と bisuser:/ の場合
        if (option.HasTarget())
        {
            std::string command = std::string(option.GetTarget());

            // bissystem や bisuser だけでも動くように ":/" を追記
            if (command == "bissystem" || command == "bisuser" || command == "sdcard")
            {
                command += ":/";
            }

            //コマンドの最後が '/' でなければ '/' を追記
            if (command.at(command.size() - 1) != '/')
            {
                command += "/";
            }

            if (command.substr(0, 9) == std::string("bissystem"))
            {
                NN_RESULT_DO(fs::MountBis("system", fs::BisPartitionId::System));
                NN_UTIL_SCOPE_EXIT
                {
                    fs::Unmount("system");
                };

                char directory[MaxDirectoryNameLength];
                sprintf(directory, "system%s", command.substr(command.find_first_of(':')).c_str());

                PrintList(directory, 0);
            }

            else if (command.substr(0, 7) == std::string("bisuser"))
            {
                NN_RESULT_DO(fs::MountBis("user", fs::BisPartitionId::User));
                NN_UTIL_SCOPE_EXIT
                {
                    fs::Unmount("user");
                };

                char directory[MaxDirectoryNameLength];
                sprintf(directory, "user%s", command.substr(command.find_first_of(':')).c_str());

                PrintList(directory, 0);
            }

            else if (command.substr(0, 6) == std::string("sdcard"))
            {
                NN_RESULT_DO(fs::MountSdCardForDebug("sdcard"));
                NN_UTIL_SCOPE_EXIT
                {
                    fs::Unmount("sdcard");
                };

                char directory[MaxDirectoryNameLength];
                sprintf(directory, "sdcard%s", command.substr(command.find_first_of(':')).c_str());

                PrintList(directory, 0);
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result MountTargetStorage(const char** outMountName, ncm::StorageId storageId) NN_NOEXCEPT
    {
        switch (storageId)
        {
        case ncm::StorageId::SdCard:
            NN_RESULT_DO(fs::MountSdCardForDebug(SdCardMountName));
            *outMountName = SdCardMountName;
            break;
        case ncm::StorageId::BuildInSystem:
            NN_RESULT_DO(fs::MountBis(BuiltInSystemMountName, fs::BisPartitionId::System));
            *outMountName = BuiltInSystemMountName;
            break;
        case ncm::StorageId::BuildInUser:
        case ncm::StorageId::Any:
            NN_RESULT_DO(fs::MountBis(BuiltInUserMountName, fs::BisPartitionId::User));
            *outMountName = BuiltInUserMountName;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_SUCCESS;
    }

    util::optional<ncm::StorageId> GetFileSystemCommandStorage(const Option& option) NN_NOEXCEPT
    {
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("builtin");
        if (!storage && option.HasKey("-s"))
        {
            storage = ToSystemStorage(option.GetValue("-s"));
        }
        return storage;
    }

    Result DeleteFile(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        auto storage = GetFileSystemCommandStorage(option);
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        const char* mountName = nullptr;
        NN_RESULT_DO(MountTargetStorage(&mountName, *storage));
        NN_UTIL_SCOPE_EXIT { fs::Unmount(mountName); };

        std::string path = std::string(mountName) + std::string(":") + std::string(option.GetTarget());
        NN_RESULT_DO(fs::DeleteFile(path.c_str()));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteDirectory(bool* outValue, const Option& option)
    {
        *outValue = false;

        auto storage = GetFileSystemCommandStorage(option);
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        const char* mountName = nullptr;
        NN_RESULT_DO(MountTargetStorage(&mountName, *storage));
        NN_UTIL_SCOPE_EXIT { fs::Unmount(mountName); };

        std::string path = std::string(mountName) + std::string(":") + std::string(option.GetTarget());
        NN_RESULT_DO(fs::DeleteDirectoryRecursively(path.c_str()));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CreateFile(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        auto storage = GetFileSystemCommandStorage(option);
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        if (!option.HasKey("--src"))
        {
            NN_LOG("You need --src option for 'create-file' sub command.\n");
            NN_RESULT_SUCCESS;
        }

        auto srcFilePath = option.GetValue("--src");
        if (!devmenuUtil::IsAbsolutePath(srcFilePath))
        {
            NN_LOG("'%s' is not an absolute path.\n", srcFilePath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        fs::FileHandle srcFile;
        NN_RESULT_DO(fs::OpenFile(&srcFile, srcFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(srcFile); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, srcFile));

        const char* mountName = nullptr;
        NN_RESULT_DO(MountTargetStorage(&mountName, *storage));
        NN_UTIL_SCOPE_EXIT { fs::Unmount(mountName); };

        std::string path = std::string(mountName) + std::string(":") + std::string(option.GetTarget());
        NN_RESULT_TRY(fs::DeleteFile(path.c_str()))
            NN_RESULT_CATCH(fs::ResultPathNotFound) {}
        NN_RESULT_END_TRY;
        NN_RESULT_DO(fs::CreateFile(path.c_str(), fileSize));

        fs::FileHandle dstFile;
        NN_RESULT_DO(fs::OpenFile(&dstFile, path.c_str(), fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(dstFile); };

        std::unique_ptr<char[]> buffer(new char[ReadBufferSize]);
        int64_t offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            size_t readSize = offset + static_cast<int64_t>(ReadBufferSize) > fileSize ? static_cast<size_t>(fileSize - offset) : ReadBufferSize;
            NN_RESULT_DO(fs::ReadFile(srcFile, offset, buffer.get(), readSize));
            NN_RESULT_DO(fs::WriteFile(dstFile, offset, buffer.get(), readSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
            offset += readSize;
            if (offset >= fileSize)
            {
                break;
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CreateDirectoryImpl(std::string path) NN_NOEXCEPT
    {
        auto index = path.rfind("/");
        if (index == std::string::npos)
        {
            NN_RESULT_SUCCESS;
        }
        if (!(index > 0 && path.at(index - 1) == ':'))
        {
            auto parentDir = path.substr(0, index);
            NN_RESULT_DO(CreateDirectoryImpl(parentDir));
        }

        NN_RESULT_TRY(fs::CreateDirectory(path.c_str()))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists) {}
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result CreateDirectory(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        auto storage = GetFileSystemCommandStorage(option);
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }

        const char* mountName = nullptr;
        NN_RESULT_DO(MountTargetStorage(&mountName, *storage));
        NN_UTIL_SCOPE_EXIT { fs::Unmount(mountName); };

        std::string path = std::string(mountName) + std::string(":") + std::string(option.GetTarget());
        NN_RESULT_DO(CreateDirectoryImpl(path));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "list-all",               ListAllCommand },
        { "delete-file",            DeleteFile },
        { "delete-directory",       DeleteDirectory },
        { "create-file",            CreateFile },
        { "create-directory",       CreateDirectory },
    };
}

Result FileSystemCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu filesystem command. See '" DEVMENUCOMMAND_NAME " filesystem --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
