﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>
#include <vector>
#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/am/service/am_SystemProgramIds.h>
#include <nn/apm/apm_Api.h>
#include <nn/apm/apm_System.h>
#include <nn/bconfig/bconfig_Api.h>
#include <nn/ec/ec_ShopServiceAccessorForDebug.h>
#include <nn/fatal/fatal_Api.h>
#include <nn/fatal/fatal_ApiPrivate.h>
#include <nn/fs.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/fs/fs_DebugPrivate.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/idle/idle_SystemApi.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_PreInstallApi.h>
#include <nn/ns/ns_PseudoDeviceIdApi.h>
#include <nn/ns/ns_VulnerabilityApi.h>
#include <nn/ns/srv/ns_VulnerabilityManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/omm/omm_Result.h>
#include <nn/omm/omm_Api.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_DebugCommand.h"
#include "DevMenuCommand_Eula.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Language.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_StorageId.h"
#include "DevMenuCommand_StrToUll.h"

#include "DevMenuCommand_DebugCommandData.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    #define MEMORY_MODE_OPTION_STRING "auto|4GB|6GB|4GB-internal|6GB-internal|4GB-sdk"
#else
    #define MEMORY_MODE_OPTION_STRING "auto|4GB|6GB"
#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " debug get-operation-mode-policy\n"
        "       " DEVMENUCOMMAND_NAME " debug set-operation-mode-policy <auto|handheld|console>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-performance-mode-policy\n"
        "       " DEVMENUCOMMAND_NAME " debug set-performance-mode-policy <auto|normal|boost>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-current-performance-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug get-cpu-overclock-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-cpu-overclock\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-cpu-overclock\n"
        "       " DEVMENUCOMMAND_NAME " debug get-sleep-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug set-sleep-mode <default|pseudo>\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " debug set-sleep-pending-span <milliseconds>\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug get-idle-time-detect-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug set-idle-time-detect-mode <on|off>\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-fatal-screen\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-fatal-screen\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " debug throw-fatal <result_value> [--no-error-report]\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-fatal-extra-info\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-fatal-extra-info\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-force-maintenance-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-force-maintenance-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-applet-development\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-applet-development\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug switch-menu <devmenu|homemenu>\n"
        "       " DEVMENUCOMMAND_NAME " debug switch-shop <devshop|shop>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-sd-card-logging-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-sd-card-logging\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-sd-card-logging\n"
        "       " DEVMENUCOMMAND_NAME " debug needs-update-vulnerability\n"
        "       " DEVMENUCOMMAND_NAME " debug set-needs-update-vulnerability-policy <default|force_true|force_false>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-safe-system-version\n"
        "       " DEVMENUCOMMAND_NAME " debug update-safe-system-version --id <system_update_id> --version <version>\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        // DevMenuCommand でも利用可能だが、非公開機能のため DevMenuCommandSystem でのみ表示する
        "       " DEVMENUCOMMAND_NAME " debug enable-field-testing\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-field-testing\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug set-memory-mode <" MEMORY_MODE_OPTION_STRING ">\n"
        "       " DEVMENUCOMMAND_NAME " debug get-memory-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-fill-memory\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-fill-memory\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-user-exception-handler\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-user-exception-handler\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-red-screen-at-system-error\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-red-screen-at-system-error\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-pmu\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-pmu\n"
        "       " DEVMENUCOMMAND_NAME " debug set-initial-tick-value <7days|14days|30days|90days|hex>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-initial-tick-value\n"
        "       " DEVMENUCOMMAND_NAME " debug get-nifm-testing-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-nifm-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-nifm-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-errorviewer-auto-close\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-errorviewer-auto-close\n"
        "       " DEVMENUCOMMAND_NAME " debug get-gpu-resource-limitation-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-gpu-resource-limitation\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-gpu-resource-limitation\n"
        "       " DEVMENUCOMMAND_NAME " debug get-background-download-stress-testing-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-background-download-stress-testing\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-background-download-stress-testing\n"
        "       " DEVMENUCOMMAND_NAME " debug get-background-download-stress-testing-info [-r]\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " debug get-background-apply-delta-stress-testing-info [-r]\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug set-background-download-stress-testing-storage <sdcard|builtin|system>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-jit-debug-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-jit-debug\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-jit-debug\n"
        "       " DEVMENUCOMMAND_NAME " debug get-cpu-crash-automatic-dump-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-cpu-crash-automatic-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-cpu-crash-automatic-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug get-cpu-crash-automatic-full-dump-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-cpu-crash-automatic-full-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-cpu-crash-automatic-full-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug set-screenshot-target <screenshot|physical>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-screenshot-target\n"
        "       " DEVMENUCOMMAND_NAME " debug get-gpu-crash-automatic-dump-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-gpu-crash-automatic-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-gpu-crash-automatic-dump\n"
        "       " DEVMENUCOMMAND_NAME " debug get-terminate-application-on-gpu-error-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-terminate-application-on-gpu-error\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-terminate-application-on-gpu-error\n"
        "       " DEVMENUCOMMAND_NAME " debug set-fs-access-log-mode <log|sdcard|log+sdcard|off>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-fs-access-log-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug set-fs-speed-emulation-mode <faster|slower|random|off>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-fs-speed-emulation-mode\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " debug set-firmware-debug-settings <dev|prod>\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug get-battery-draining-performance-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-battery-draining-performance\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-battery-draining-performance\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-cec\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-cec\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " debug get-not-locked-nfp-access-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-not-locked-nfp-access\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-not-locked-nfp-access\n"
        "       " DEVMENUCOMMAND_NAME " debug get-nfp-play-report-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-nfp-play-report\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-nfp-play-report\n"
        "       " DEVMENUCOMMAND_NAME " debug get-system-play-report-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-system-play-report\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-system-play-report\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-mount-sdcard\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-mount-sdcard\n"
        "       " DEVMENUCOMMAND_NAME " debug set-boolean-fwdbg --name <name> --key <key> <true/false>\n"
        "       " DEVMENUCOMMAND_NAME " debug set-string-fwdbg --name <name> --key <key> <value>\n"
        "       " DEVMENUCOMMAND_NAME " debug set-integer-fwdbg --name <name> --key <key> <integer value>\n"
        "       " DEVMENUCOMMAND_NAME " debug set-hexadecimal-fwdbg --name <name> --key <key> <hexadecimal value>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-rid-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-rid-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-rid-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug set-initial-launch-settings --completed <true|false> --has-timestamp <true|false> [--timestamp-by-elapsed <seconds>]\n"
        "       " DEVMENUCOMMAND_NAME " debug get-initial-launch-settings\n"
        "       " DEVMENUCOMMAND_NAME " debug set-device-language <language>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-device-language\n"
        "       " DEVMENUCOMMAND_NAME " debug set-prepurchase-response-mode <normal|delay|still_unavailable>\n"
        "       " DEVMENUCOMMAND_NAME " debug get-prepurchase-response-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-nim-error-simulate --url <url-prefix> --result <hex-result> [--error-rate <0-10000>]\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-nim-error-simulate\n"
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        "       " DEVMENUCOMMAND_NAME " debug enable-debug-response-simulate --dynamic-rights\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-debug-response-simulate --dynamic-rights\n"
        "       " DEVMENUCOMMAND_NAME " debug register-debug-response-simulate --server <OwnedConsumableServiceItem|Catalog|DynamicRights> --url-path <url-path-prefix> [--response <response-body> | --response-file <absolute-file-path-of-response-body>] [--result-inner-value <hex-result-inner-value>] [--rate <0-10000>]\n"
        "       " DEVMENUCOMMAND_NAME " debug clear-debug-response-simulate\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug abort [<result_value>]\n"
        "       " DEVMENUCOMMAND_NAME " debug set-cleanup-sdcard-result-value <result_value>\n"
        "       " DEVMENUCOMMAND_NAME " debug set-blacklist <application_id> [--version <version_value>]\n"
        "       " DEVMENUCOMMAND_NAME " debug get-resume-all-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-resume-all\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-resume-all\n"
#endif
        "       " DEVMENUCOMMAND_NAME " debug get-exhibition-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-exhibition-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-exhibition-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug get-no-vsync-capability-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-no-vsync-capability\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-no-vsync-capability\n"
        "       " DEVMENUCOMMAND_NAME " debug ease-nro-restriction\n"
        "       " DEVMENUCOMMAND_NAME " debug reset-pseudo-device-id\n"
        "       " DEVMENUCOMMAND_NAME " debug get-gpu-timeout-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-gpu-timeout\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-gpu-timeout\n"
        "       " DEVMENUCOMMAND_NAME " debug get-in-store-demo-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-in-store-demo-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-in-store-demo-mode\n"
        "       " DEVMENUCOMMAND_NAME " debug get-hdcp-authentication-failed-emulation-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-hdcp-authentication-failed-emulation\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-hdcp-authentication-failed-emulation\n"
        "       " DEVMENUCOMMAND_NAME " debug get-graphics-firmware-memory-margin-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-graphics-firmware-memory-margin\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-graphics-firmware-memory-margin\n"
        "       " DEVMENUCOMMAND_NAME " debug copy --source <abosolute_path> --destination <abosolute_path> [--force] [--skip-error-file]\n"
        "       " DEVMENUCOMMAND_NAME " debug get-cleanup-application-cache-storage-enabled\n"
        "       " DEVMENUCOMMAND_NAME " debug enable-cleanup-application-cache-storage\n"
        "       " DEVMENUCOMMAND_NAME " debug disable-cleanup-application-cache-storage\n"
        "       " DEVMENUCOMMAND_NAME " debug set-host-romfs-redirection-target <builtin|off>\n"
        ""; // 終端

    nn::bconfig::BootConfig bootConfigData;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    bool IsValidOperationModeString(const char* modeString)
    {
        if (std::string(modeString) == "auto") return true;
        if (std::string(modeString) == "handheld") return true;
        if (std::string(modeString) == "console") return true;

        return false;
    }

    omm::OperationModePolicy GetOperationModePolicyByString(const char* policy) NN_NOEXCEPT
    {
        if (std::strncmp(policy, "auto", sizeof("auto")) == 0) return omm::OperationModePolicy::Auto;
        if (std::strncmp(policy, "handheld", sizeof("handheld")) == 0) return omm::OperationModePolicy::Handheld;
        if (std::strncmp(policy, "console", sizeof("console")) == 0) return omm::OperationModePolicy::Console;

        return omm::OperationModePolicy::Auto;
    }

    Result SetFirmwareDebugSettingsValueImpl(bool *outValue, const void* pVal, size_t valSize, const char* name, const char* key, bool needsReboot)
    {
        settings::fwdbg::SetSettingsItemValue(name, key, pVal, valSize);
        if (needsReboot)
        {
            NN_LOG("Please restart your target to apply this change.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetStringFirmwareDebugSettings(bool *outValue, const char* str, const char* name, const char* key, bool needsReboot)
    {
        auto strLength = util::Strnlen( str, 512 );
        return SetFirmwareDebugSettingsValueImpl(outValue, str, strLength + 1, name, key, needsReboot);
    }

    template <class T>
    Result SetFirmwareDebugSettingsImpl(bool *outValue, T value, const char* name, const char* key, bool needsReboot)
    {
        return SetFirmwareDebugSettingsValueImpl(outValue, &value, sizeof( value ), name, key, needsReboot);
    }

    Result SetBooleanFirmwareDebugSetting(bool *outValue, bool isEnabled, const char* name, const char* key, bool needsReboot)
    {
        return SetFirmwareDebugSettingsImpl(outValue, isEnabled, name, key, needsReboot);
    }

    Result EnableBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key, bool needsReboot = false)
    {
        return SetBooleanFirmwareDebugSetting(outValue, true, name, key, needsReboot);
    }

    Result DisableBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key, bool needsReboot = false)
    {
        return SetBooleanFirmwareDebugSetting(outValue, false, name, key, needsReboot);
    }

    Result GetBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key)
    {
        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), name, key);

        if( sizeof(isEnabled) == size )
        {
            NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }

    Result GetOperationModePolicy(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        char modeString[16];
        settings::fwdbg::GetSettingsItemValue(modeString, sizeof(modeString), "omm", "operation_mode_policy");

        NN_LOG("%s\n", modeString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetOperationModePolicy(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a policy. Set one of [auto|handheld|console].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (!IsValidOperationModeString(modeString))
        {
            NN_LOG("%s is not valid policy. Set one of [auto|handheld|console].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        settings::fwdbg::SetSettingsItemValue("omm", "operation_mode_policy", modeString, std::strlen(modeString) + 1);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        omm::Initialize();
        NN_UTIL_SCOPE_EXIT{ omm::Finalize(); };
        auto policy = GetOperationModePolicyByString(modeString);
        NN_RESULT_DO(omm::SetOperationModePolicy(policy));
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    void ApmLoadAndApplySettings()
    {
        // 即時反映
        nn::apm::InitializeForSystem();
        nn::apm::LoadAndApplySettings();
        nn::apm::FinalizeForSystem();
    }
#endif

#ifdef NN_BUILD_CONFIG_SPEC_NX
    const char* GetPerformanceModeString(apm::PerformanceMode mode)
    {
        switch (mode)
        {
        case apm::PerformanceMode_Normal: return "normal";
        case apm::PerformanceMode_Boost: return "boost";
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
#endif

    bool IsValidPerformanceModeString(const char* modeString)
    {
        if (std::string(modeString) == "auto") return true;
        if (std::string(modeString) == "normal") return true;
        if (std::string(modeString) == "boost") return true;

        return false;
    }

    Result GetPerformanceModePolicy(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        char modeString[16];
        settings::fwdbg::GetSettingsItemValue(modeString, sizeof(modeString), "apm", "performance_mode_policy");

        NN_LOG("%s\n", modeString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetPerformanceModePolicy(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a policy. Set one of [auto|normal|boost].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (!IsValidPerformanceModeString(modeString))
        {
            NN_LOG("%s is not valid policy. Set one of [auto|normal|boost].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        settings::fwdbg::SetSettingsItemValue("apm", "performance_mode_policy", modeString, std::strlen(modeString) + 1);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        ApmLoadAndApplySettings();
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetCurrentPerformanceMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

#ifdef NN_BUILD_CONFIG_SPEC_NX
        // TODO: oe::GetPerformanceMode() であるべき
        auto performanceModeString = GetPerformanceModeString(apm::GetPerformanceMode());
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(performanceModeString);
#endif
        NN_LOG("%s\n", performanceModeString);
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetCpuOverclockEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "apm", "sdev_cpu_overclock_enabled");
    }

    Result EnableCpuOverclock(bool* outValue, const Option&)
    {
        NN_RESULT_DO(EnableBooleanFirmwareDebugSetting(outValue, "apm", "sdev_cpu_overclock_enabled", true));
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        ApmLoadAndApplySettings();
#endif
        NN_RESULT_SUCCESS;
    }

    Result DisableCpuOverclock(bool* outValue, const Option&)
    {
        NN_RESULT_DO(DisableBooleanFirmwareDebugSetting(outValue, "apm", "sdev_cpu_overclock_enabled", true));
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        ApmLoadAndApplySettings();
#endif
        NN_RESULT_SUCCESS;
    }

    bool GetEnterSleepFlagByModeString(bool* pOutEnterSleepFlag, const char* modeString)
    {
        if ( std::string(modeString) == "default" )
        {
            *pOutEnterSleepFlag = true;
            return true;
        }
        if ( std::string(modeString) == "pseudo" )
        {
            *pOutEnterSleepFlag = false;
            return true;
        }
        return false;
    }

    const char* GetSleepModeString(bool enterSleepFlag)
    {
        return enterSleepFlag ? "default" : "pseudo";
    }

    Result GetSleepMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool enterSleepFlag;
        settings::fwdbg::GetSettingsItemValue(&enterSleepFlag, sizeof(enterSleepFlag), "systemsleep", "enter_sleep");

        auto sleepModeString = GetSleepModeString(enterSleepFlag);
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(sleepModeString);
#endif
        NN_LOG("%s\n", sleepModeString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSleepMode(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        bool enterSleepFlag = true;
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a sleep mode. Set one of [default|pseudo].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if ( !GetEnterSleepFlagByModeString(&enterSleepFlag, modeString) )
        {
            NN_LOG("%s is not valid sleep mode. Set one of [default|pseudo].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        settings::fwdbg::SetSettingsItemValue("systemsleep", "enter_sleep", &enterSleepFlag, sizeof(enterSleepFlag));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool GetIdleTimeDetectModeByModeString(bool* pOutIdleTimeDetectMode, const char* modeString)
    {
        if ( std::string(modeString) == "on" )
        {
            *pOutIdleTimeDetectMode = true;
            return true;
        }
        if ( std::string(modeString) == "off" )
        {
            *pOutIdleTimeDetectMode = false;
            return true;
        }
        return false;
    }

    const char* GetIdleTimeDetectString(bool idleTimeDetectMode)
    {
        return idleTimeDetectMode ? "on" : "off";
    }

    Result GetIdleTimeDetectMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isAutoSleepDisabled;
        settings::fwdbg::GetSettingsItemValue(&isAutoSleepDisabled, sizeof(isAutoSleepDisabled), "systemsleep", "disable_auto_sleep");
        bool isIdleTimeDetectEnabled = (isAutoSleepDisabled != true); // 反転
        auto idleTimeDetectString = GetIdleTimeDetectString(isIdleTimeDetectEnabled);
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(idleTimeDetectString);
#endif
        NN_LOG("%s\n", idleTimeDetectString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetIdleTimeDetectMode(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        bool isIdleTimeDetectEnabled = true;
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a idle time detect mode. Set one of [on|off].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if ( !GetIdleTimeDetectModeByModeString(&isIdleTimeDetectEnabled, modeString) )
        {
            NN_LOG("%s is not valid idle time detect mode. Set one of [on|off].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        bool isAutoSleepDisabled = (isIdleTimeDetectEnabled != true); // 反転
        settings::fwdbg::SetSettingsItemValue("systemsleep", "disable_auto_sleep", &isAutoSleepDisabled, sizeof(isAutoSleepDisabled));

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        // 即時反映
        nn::idle::InitializeForSystem();
        nn::idle::LoadAndApplySettings();
        nn::idle::FinalizeForSystem();
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableFatalScreen(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "fatal", "transition_to_fatal");
    }

    Result DisableFatalScreen(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "fatal", "transition_to_fatal");
    }

    int GetMemoryModeIndex(const char* modeString)
    {
        if (std::string(modeString) == "auto") return 0x00;
        if (std::string(modeString) == "4GB")  return 0x01;
        if (std::string(modeString) == "6GB")  return 0x11;
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        if (std::string(modeString) == "4GB-internal") return 0x02;
        if (std::string(modeString) == "6GB-internal") return 0x12;
        if (std::string(modeString) == "4GB-sdk")      return 0x03;
#endif

        return -1;
    }

    const char* GetMemoryModeString(int mode)
    {
        switch (mode)
        {
        case 0x00:
            return "auto";
        case 0x01:
            return "4GB";
        case 0x11:
            return "6GB";
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        case 0x02:
            return "4GB-internal";
        case 0x12:
            return "6GB-internal";
        case 0x03:
            return "4GB-sdk";
#endif
        default:
            break;
        }
        return "unknown mode";
    }

    Result SetMemoryMode(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a memory mode. Set one of [" MEMORY_MODE_OPTION_STRING "].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto index = GetMemoryModeIndex(modeString);
        if (index  < 0)
        {
            NN_LOG("%s is not valid memory mode. Set one of [" MEMORY_MODE_OPTION_STRING "].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetMemoryMode(&bootConfigData, static_cast<Bit8>(index));
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetMemoryMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        Bit8 mode = nn::bconfig::GetMemoryMode(&bootConfigData);
        auto memoryModeString = GetMemoryModeString(mode);
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(memoryModeString);
#endif
        NN_LOG("Current memory mode: %s\n", memoryModeString);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableFillMemory(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetEnableNonZeroFillMemory(&bootConfigData, true);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableFillMemory(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetEnableNonZeroFillMemory(&bootConfigData, false);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableUserExceptionHandler(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetEnableUserExceptionHandler(&bootConfigData, true);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableUserExceptionHandler(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetEnableUserExceptionHandler(&bootConfigData, false);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableRedScreenAtSystemError(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetCallShowErrorOnPanic(&bootConfigData, true);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableRedScreenAtSystemError(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetCallShowErrorOnPanic(&bootConfigData, false);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnablePerformanceMonitoringUnit(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetPerformanceMonitoringUnit(&bootConfigData, true);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisablePerformanceMonitoringUnit(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetPerformanceMonitoringUnit(&bootConfigData, false);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetInitialTickValue(bool* outValue, const Option& option)
    {
        Bit64 value;

        auto modeString = option.GetTarget();
        if (modeString[0] == '0' && modeString[1] == 'x')
        {
            value = strtoll(modeString, nullptr, 16);
            double days = value / 19200000.0 / 3600 / 24;
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
            NN_UNUSED(days);
#endif
            NN_LOG("Set Initial Tick Value: 0x%llX (~%g days)\n", value, days);
        }
        else if ( std::string(modeString) == "0" )
        {
            value = 0;
        }
        else if ( std::string(modeString) == "7days" )
        {
            value = 0x00000A8FAAAC0000ull; // 7 days
        }
        else if ( std::string(modeString) == "14days" )
        {
            value = 0x0000151F55580000ull; // 14 days
        }
        else if ( std::string(modeString) == "30days" )
        {
            value = 0x00002D4324980000ull; // 30 days
        }
        else if ( std::string(modeString) == "90days" )
        {
            value = 0x000087C96DC80000ull; // 90 days
        }
        else
        {
            NN_LOG("You must specify an initial tick value. Set one of [0|7days|14days|30days|90days] or hex value (0x12345678abcd).\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::bconfig::SetInitialTscValue(&bootConfigData, value);
        nn::bconfig::SaveBootConfig(&bootConfigData);

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetInitialTickValue(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::bconfig::LoadBootConfig(&bootConfigData);
        nn::Bit64 value = nn::bconfig::GetInitialTscValue(&bootConfigData);
        double days = value / 19200000.0 / 3600 / 24;
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(days);
#endif
        NN_LOG("Initial Tick Value: 0x%llX (~%g days)\n", value, days);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result ThrowFatal(bool* outValue, const Option& option)
    {
        unsigned long resultValue = 0;
        auto resultString = option.GetTarget();
        if (std::string(resultString) != "")
        {
            resultValue = std::strtoul(resultString, nullptr, 16);
        }

        bool noErrorReport = option.HasKey("--no-error-report");

        auto result = nn::result::detail::ConstructResult(resultValue);

        if (noErrorReport)
        {
            fatal::ThrowFatalWithoutErrorReport(result);
        }
        else
        {
            fatal::ThrowFatal(result);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableForceMaintenanceMode(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "boot", "force_maintenance");
    }

    Result DisableForceMaintenanceMode(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "boot", "force_maintenance");
    }

    Result EnableAppletDevelopment(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "am.debug", "dev_function");
    }

    Result DisableAppletDevelopment(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "am.debug", "dev_function");
    }

    bool ConvertToNeedsUpdateVulnerabilityPolicy(nn::ns::srv::VulnerabilityManager::NeedsUpdatePolicy* out, const char* policyString)
    {
        using NeedsUpdatePolicy = nn::ns::srv::VulnerabilityManager::NeedsUpdatePolicy;
        if (std::string(policyString) == "default") { *out = NeedsUpdatePolicy::Default; return true; }
        if (std::string(policyString) == "force_true") { *out = NeedsUpdatePolicy::ForceTrue; return true; }
        if (std::string(policyString) == "force_false") { *out = NeedsUpdatePolicy::ForceFalse; return true; }
        return false;
    }

    Result SetNeedsUpdateVulnerabilityPolicy(bool* outValue, const Option& option)
    {
        nn::ns::srv::VulnerabilityManager::NeedsUpdatePolicy policy;
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a policy. Set one of [default|force_true|force_false].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (!ConvertToNeedsUpdateVulnerabilityPolicy(&policy, modeString))
        {
            NN_LOG("%s is not valid policy. Set one of [default|force_true|force_false].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        settings::fwdbg::SetSettingsItemValue("vulnerability", "needs_update_vulnerability_policy", &policy, sizeof(policy));
        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result NeedsUpdateVulnerabilityCommand(bool* outValue, const Option&)
    {
        auto result = nn::ns::NeedsUpdateVulnerability();
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(result);
#endif
        NN_LOG("%s\n", result ? "True" : "False");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetSafeSystemVersion(bool* outValue, const Option&)
    {
        ncm::ContentMetaKey key;
        nn::ns::GetSafeSystemVersion(&key);
        NN_LOG("0x%016llx, %d\n", key.id, key.version);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UpdateSafeSystemVersion(bool* outValue, const Option& option)
    {
        if (!option.HasKey("--id"))
        {
            NN_LOG("You must specify system update id with '--id' option.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto idString = option.GetValue("--id");
        Bit64 id = 0;
        if (std::string(idString) != "")
        {
            id = std::strtoul(idString, nullptr, 16);
        }

        if (!option.HasKey("--version"))
        {
            NN_LOG("You must specify system update version with '--version' option.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto versionString = option.GetValue("--version");
        uint32_t version = 0;
        if (std::string(versionString) != "")
        {
            version = std::strtoul(versionString, nullptr, 10);
        }

        ns::UpdateSafeSystemVersionForDebug({ id }, version);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableMountSdCard(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "ns.sdcard", "mount_sdcard", true);
    }

    Result DisableMountSdCard(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "ns.sdcard", "mount_sdcard", true);
    }
    Result SetBooleanFwdbg(bool* outValue, const Option& option)
    {
        auto name = option.GetValue("--name");
        if (name == nullptr)
        {
            NN_LOG("--name option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto key = option.GetValue("--key");
        if (key == nullptr)
        {
            NN_LOG("--key option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto valueString = std::string(option.GetTarget());
        bool value;
        if (valueString == "true")
        {
            value = true;
        }
        else if (valueString == "false")
        {
            value = false;
        }
        else
        {
            NN_LOG("value is required and should be true or false");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        settings::fwdbg::SetSettingsItemValue(name, key, &value, sizeof(value));
        NN_LOG("%s %s is set to %s\n", name, key, valueString.c_str());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result SetStringFwdbg(bool* outValue, const Option& option)
    {
        auto name = option.GetValue("--name");
        if (name == nullptr)
        {
            NN_LOG("--name option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto key = option.GetValue("--key");
        if (key == nullptr)
        {
            NN_LOG("--key option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto value = std::string(option.GetTarget());

        settings::fwdbg::SetSettingsItemValue(name, key, value.c_str(), value.size() + 1);
        NN_LOG("%s %s is set to %s\n", name, key, value.c_str());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result SetIntegerFwdbg(bool* outValue, const Option& option)
    {
        auto name = option.GetValue("--name");
        if (name == nullptr)
        {
            NN_LOG("--name option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto key = option.GetValue("--key");
        if (key == nullptr)
        {
            NN_LOG("--key option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto value = static_cast<int>(STR_TO_ULL(option.GetTarget(), nullptr, 10));

        settings::fwdbg::SetSettingsItemValue(name, key, &value, sizeof(value));
        NN_LOG("%s %s is set to %d\n", name, key, value);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result SetHexadecimalFwdbg(bool* outValue, const Option& option)
    {
        auto name = option.GetValue("--name");
        if (name == nullptr)
        {
            NN_LOG("--name option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto key = option.GetValue("--key");
        if (key == nullptr)
        {
            NN_LOG("--key option is required");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto value = static_cast<uint32_t>(STR_TO_ULL(option.GetTarget(), nullptr, 16));

        settings::fwdbg::SetSettingsItemValue(name, key, &value, sizeof(value));
        NN_LOG("%s %s is set to 0x%x\n", name, key, value);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result SwitchMenu(bool* outValue, const Option& option)
    {
        ncm::ProgramId menuId = { NN_TOOL_DEVMENUCOMMAND_PROGRAM_ID_OF_DEVMENU };
        ncm::ProgramId overlayId = { NN_TOOL_DEVMENUCOMMAND_PROGRAM_ID_OF_DEVOVERLAYDISP };
        auto menuString = option.GetTarget();
        bool isMenuOnly = option.HasKey( "--menuonly" ); // Menu のみ入れ替え
        bool isOaOnly = option.HasKey( "--overlayonly" ); // OA のみ入れ替え
        bool hasEitherKey = isMenuOnly || isOaOnly;

        // 両方指定したら終了
        if ( isMenuOnly && isOaOnly )
        {
            NN_LOG( "You can't specify both args at a time.\n" );
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (std::string(menuString) == "devmenu")
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
            NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
            AcceptEulaInDevVersion(); // DevMenu へ切り替え時、自動的に EULA 同意
        }
        else if (std::string(menuString) == "homemenu")
        {
            menuId = nn::am::service::ProgramId_SystemAppletMenu;
            overlayId = nn::am::service::ProgramId_OverlayApplet;

            // Home Menu が入っていなかったら終了
            if ( !devmenuUtil::IsProgramInstalled( menuId ) )
            {
                NN_LOG( "You didn't install Home Menu.\n" );
                NN_LOG( "Process is canceled.\n" );
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            RejectEulaOfDevVersion(); // HomeMenu 切り替え時、 DevMenu で自動的に同意した EULA はリジェクト
        }
        else if (std::string(menuString) == "")
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " debug switch-menu <devmenu|homemenu>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LOG("%s is not valid menu. Set one of [devmenu|homemenu].\n", menuString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if ( !hasEitherKey || isMenuOnly )
        {
            char menuIdStr[32];
            util::SNPrintf( menuIdStr, sizeof(menuIdStr), "0x%016llx", menuId.value );
            settings::fwdbg::SetSettingsItemValue( "ns.applet", "system_applet_id", menuIdStr, util::Strnlen(menuIdStr, sizeof(menuIdStr)) + 1 );
        }
        if ( !hasEitherKey || isOaOnly )
        {
            char overlayIdStr[32];
            util::SNPrintf( overlayIdStr, sizeof(overlayIdStr), "0x%016llx", overlayId.value );
            settings::fwdbg::SetSettingsItemValue( "ns.applet", "overlay_applet_id", overlayIdStr, util::Strnlen(overlayIdStr, sizeof(overlayIdStr)) + 1 );
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SwitchShop(bool* outValue, const Option& option)
    {
        ncm::ProgramId shopId = nn::am::service::ProgramId_LibraryAppletDummyShop;
        auto shopString = option.GetTarget();

        if (std::string(shopString) == "devshop")
        {
            // do nothing
        }
        else if (std::string(shopString) == "shop")
        {
            shopId = nn::am::service::ProgramId_LibraryAppletShop;

            // ShopN が入っていなかったら終了
            if ( !devmenuUtil::IsProgramInstalled( shopId ) )
            {
                NN_LOG( "You didn't install shop.\n" );
                NN_LOG( "Process is canceled.\n" );
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else if (std::string(shopString) == "")
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " debug switch-shop <devshop|shop>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LOG("%s is not valid shop. Set one of [devshop|shop].\n", shopString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char shopIdStr[32];
        util::SNPrintf( shopIdStr, sizeof(shopIdStr), "0x%016llx", shopId.value );
        settings::fwdbg::SetSettingsItemValue( "ns.applet", "shop_applet_id", shopIdStr, util::Strnlen(shopIdStr, sizeof(shopIdStr)) + 1 );

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetAssumedSystemAppletVersion(bool* outValue, const Option& option)
    {
        auto assumedSystemAppletVersionString = option.GetTarget();
        if (std::strlen(assumedSystemAppletVersionString) == 0)
        {
            NN_LOG("Set assumed-system-applet-version integer\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto assumedSystemAppletVersion = std::strtol(assumedSystemAppletVersionString, nullptr, 10);
        settings::fwdbg::SetSettingsItemValue("systemupdate", "assumed_system_applet_version", &assumedSystemAppletVersion, sizeof(assumedSystemAppletVersion));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetAssumedSystemAppletVersion(bool* outValue, const Option&)
    {
        int assumedSystemAppletVersion = 0;
        auto keySize = settings::fwdbg::GetSettingsItemValue(&assumedSystemAppletVersion, sizeof(assumedSystemAppletVersion), "systemupdate", "assumed_system_applet_version");

        if (!(sizeof(assumedSystemAppletVersion) == keySize))
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_LOG("Assumed System Applet Version is %d\n", assumedSystemAppletVersion);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetNifmTestingModeEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "nifm", "is_communication_control_enabled_for_test");
    }

    Result EnableNifmTestingMode(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "nifm", "is_communication_control_enabled_for_test");
    }

    Result DisableNifmTestingMode(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "nifm", "is_communication_control_enabled_for_test");
    }

    Result GetSdCardLoggingEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "lm", "enable_sd_card_logging");
    }

    Result EnableSdCardLogging(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "lm", "enable_sd_card_logging", true);
    }

    Result DisableSdCardLogging(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "lm", "enable_sd_card_logging", true);
    }

    Result EnableFatalExtraInfoCore(bool* outValue, bool showNeedRebootMessage)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "fatal", "show_extra_info", showNeedRebootMessage);
    }

    Result DisableFatalExtraInfoCore(bool* outValue, bool showNeedRebootMessage)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "fatal", "show_extra_info", showNeedRebootMessage);
    }

    Result EnableFieldTesting(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "systemconfig", "field_testing", true);
    }

    Result DisableFieldTesting(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "systemconfig", "field_testing", true);
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result SetSleepPendingSpan(bool* outValue, const Option& option)
    {
        auto sleepPendingMsString = option.GetTarget();
        if (std::strlen(sleepPendingMsString) == 0)
        {
            NN_LOG("Set milliseconds integer\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int sleepPendingMs = std::strtol(sleepPendingMsString, nullptr, 10);
        settings::fwdbg::SetSettingsItemValue("systemsleep", "sleep_pending_time_ms", &sleepPendingMs, sizeof(sleepPendingMs));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableFatalExtraInfo(bool* outValue, const Option&)
    {
        return EnableFatalExtraInfoCore(outValue, true);
    }

    Result DisableFatalExtraInfo(bool* outValue, const Option&)
    {
        return DisableFatalExtraInfoCore(outValue, true);
    }
#endif

    Result EnableErrorViewerAutoClose(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "err", "applet_auto_close");
    }

    Result DisableErrorViewerAutoClose(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "err", "applet_auto_close");
    }

    Result GetGpuResourceLimitationEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "am.gpu", "gpu_scheduling_enabled");
    }

    Result EnableGpuResourceLimitation(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "am.gpu", "gpu_scheduling_enabled", true);
    }

    Result DisableGpuResourceLimitation(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "am.gpu", "gpu_scheduling_enabled", true);
    }

    Result GetBackgroundDownloadStressTestingEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "systemupdate", "enable_background_download_stress_testing");
    }

    Result EnableBackgroundDownloadStressTesting(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "systemupdate", "enable_background_download_stress_testing");
    }

    Result DisableBackgroundDownloadStressTesting(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "systemupdate", "enable_background_download_stress_testing");
    }

    Result GetBackgroundDownloadStressTestingInfo(bool* outValue, const Option& option)
    {
        *outValue = false;
        bool doRepeat = option.HasKey("-r");

        const char* StatusStrings[] =
        {
            "NotRunning",
            "Waiting",
            "Downloading",
            "Error",
        };
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(StatusStrings);
#endif

        bool first = true;
        do
        {
            nn::nim::BackgroundDownloadStressTaskInfo info;
            NN_RESULT_DO(nn::ns::GetBackgroundDownloadStressTaskInfo(&info));
            if (first)
            {
                NN_LOG("BackgroundDownloadStressTest: %s\n", info.isEnabled ? "Enabled" : "Disabled");
                first = false;
            }

            if (info.isEnabled)
            {
                auto lastResult = info.GetLastResult();
                NN_LOG("Status      : %s\n", StatusStrings[static_cast<int>(info.state)]);
                if (!lastResult.IsSuccess())
                {
                    NN_LOG("ErrorResult : module=%d, description=%d, result=0x%08x\n", lastResult.GetModule(), lastResult.GetDescription(), lastResult.GetInnerValueForDebug());
                }
                else
                {
                    NN_LOG("Throughput  : %.3f (KB/sec)\n", info.throughput);
                }
            }
            else
            {
                break;
            }

            NN_LOG("---\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        } while (doRepeat);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetBackgroundDownloadStressTestingStorage(bool* outValue, const Option& option)
    {
        auto storage = ToInstallStorage(option.GetTarget());
        if (!storage)
        {
            storage = ToSystemStorage(option.GetTarget());
        }
        NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());
        if (*storage == ncm::StorageId::Any)
        {
            storage = ncm::StorageId::BuiltInSystem;
        }

        int storageValue = static_cast<int>(*storage);
        NN_RESULT_DO(SetFirmwareDebugSettingsImpl(outValue, storageValue, "systemupdate", "background_download_stress_testing_storage", true));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result GetBackgroundApplyDeltaStressTestingInfo(bool* outValue, const Option& option)
    {
        *outValue = false;
        bool doRepeat = option.HasKey("-r");

        const char* StatusStrings[] =
        {
            "NotRunning",
            "Waiting",
            "Applying",
            "Error",
        };
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(StatusStrings);
#endif

        bool first = true;
        do
        {
            nn::nim::BackgroundDownloadStressTaskInfo info;
            NN_RESULT_DO(nn::ns::GetBackgroundApplyDeltaStressTaskInfo(&info));
            if (first)
            {
                NN_LOG("BackgroundApplyDeltaStressTest: %s\n", info.isEnabled ? "Enabled" : "Disabled");
                NN_LOG("Storage     : %s\n", devmenuUtil::GetStorageIdString(info.storageId));
                first = false;
            }

            if (info.isEnabled)
            {
                auto lastResult = info.GetLastResult();
                NN_LOG("Status      : %s\n", StatusStrings[static_cast<int>(info.state)]);
                if (!lastResult.IsSuccess())
                {
                    NN_LOG("ErrorResult : module=%d, description=%d, result=0x%08x\n", lastResult.GetModule(), lastResult.GetDescription(), lastResult.GetInnerValueForDebug());
                    break;
                }
                else
                {
                    NN_LOG("Throughput  : %.3f (KB/sec)\n", info.throughput);
                }
            }
            else
            {
                break;
            }

            NN_LOG("---\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        } while (doRepeat);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result FillNandFreeSpace(bool* outValue, const Option& option)
    {
        auto marginOption = option.GetValue("--margin");
        size_t margin = marginOption ? std::strtoul(marginOption, nullptr, 10) * 1024 : 64 * 1024;

        int64_t freeSpaceSize;
        NN_RESULT_DO(nn::ns::GetFreeSpaceSize(&freeSpaceSize, nn::ncm::StorageId::BuildInUser));
        auto paddingSize = std::max<int64_t>(freeSpaceSize - margin, 0);
        NN_LOG("Filling nand free space...\n");
        NN_RESULT_TRY(nn::fs::CreatePaddingFile(paddingSize))
            NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough)
            {
            }
        NN_RESULT_END_TRY

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char SdCardPaddingDirectoryPath[] = "sdcard:/padding";

    Result FillSdCardFreeSpace(bool* outValue, const Option& option)
    {
        auto marginOption = option.GetValue("--margin");
        int64_t margin = marginOption ? std::strtoll(marginOption, nullptr, 10) * 1024 : 64 * 1024;

        NN_RESULT_DO(nn::fs::MountSdCardForDebug(SdCardMountName));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(SdCardMountName); };

        nn::fs::DirectoryEntryType type;
        if (nn::fs::GetEntryType(&type, SdCardPaddingDirectoryPath).IsSuccess())
        {
            NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(SdCardPaddingDirectoryPath));
        }

        NN_LOG("Filling sdcard free space...\n");
        NN_RESULT_DO(nn::fs::CreateDirectory(SdCardPaddingDirectoryPath));

        int count = 0;
        while (NN_STATIC_CONDITION(true))
        {
            const int64_t MaxFileSize = 2LL * 1024 * 1024 * 1024;

            int64_t freeSpaceSize;
            NN_RESULT_DO(nn::fs::GetFreeSpaceSize(&freeSpaceSize, "sdcard:/"));
            if (freeSpaceSize <= margin)
            {
                break;
            }

            auto paddingSize = std::min<int64_t>(freeSpaceSize - margin, MaxFileSize);
            char path[256];
            util::TSNPrintf(path, sizeof(path), "%s/%02x", SdCardPaddingDirectoryPath, count++);
            NN_RESULT_DO(nn::fs::CreateFile(path, paddingSize));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetJitDebugEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "jit_debug", "enable_jit_debug");
    }

    Result EmptyNandFreeSpace(bool* outValue, const Option&)
    {
        NN_LOG("Emptying nand free space...\n");
        NN_RESULT_DO(nn::fs::DeleteAllPaddingFiles());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EmptySdCardFreeSpace(bool* outValue, const Option&)
    {
        NN_RESULT_DO(nn::fs::MountSdCardForDebug(SdCardMountName));
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(SdCardMountName); };

        NN_LOG("Emptying sdcard free space...\n");
        NN_RESULT_TRY(nn::fs::DeleteDirectoryRecursively(SdCardPaddingDirectoryPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound) {}
        NN_RESULT_END_TRY
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableJitDebug(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "jit_debug", "enable_jit_debug");
    }

    Result DisableJitDebug(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "jit_debug", "enable_jit_debug");
    }

    Result GetCpuCrashAutomaticDumpEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "auto_dump");
    }

    Result EnableCpuCrashAutomaticDump(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "auto_dump");
    }

    Result DisableCpuCrashAutomaticDump(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "auto_dump");
    }

    Result GetCpuCrashAutomaticFullDumpEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "full_dump");
    }

    Result EnableCpuCrashAutomaticFullDump(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "full_dump");
    }

    Result DisableCpuCrashAutomaticFullDump(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "snap_shot_dump", "full_dump");
    }

    Result SetScreenShotTarget(bool* outValue, const Option& option)
    {
        if(!option.HasTarget())
        {
            NN_LOG("<value> is required\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto target = option.GetTarget();
        char value[64] = {};
        size_t valueLength = std::min(static_cast<size_t>(nn::util::Strlcpy(value, target, static_cast<int>(sizeof(value)))) + 1, sizeof(value));
        settings::fwdbg::SetSettingsItemValue("capsrv", "screenshot_layerstack", value, valueLength);
        NN_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetScreenShotTarget(bool* outValue, const Option&)
    {
        char value[64] = {};
        settings::fwdbg::GetSettingsItemValue(value, sizeof(value), "capsrv", "screenshot_layerstack");
        value[sizeof(value) - 1] = 0;
        NN_LOG("%s\n", value);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetGpuCrashAutomaticDumpEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "gpu_core_dump", "auto_dump");
    }

    Result EnableGpuCrashAutomaticDump(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "gpu_core_dump", "auto_dump");
    }

    Result DisableGpuCrashAutomaticDump(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "gpu_core_dump", "auto_dump");
    }

    Result GetTerminateApplicationOnGpuErrorEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "am.debug", "gpu_error_kill_app");
    }

    Result EnableTerminateApplicationOnGpuError(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "am.debug", "gpu_error_kill_app");
    }

    Result DisableTerminateApplicationOnGpuError(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "am.debug", "gpu_error_kill_app");
    }

    bool ParseFsAccessLogModeString(uint32_t* outValue, std::string&& mode)
    {
        if ( mode == "log" )
        {
            *outValue = nn::fs::AccessLogMode_Log;
        }
        else if ( mode == "sdcard" )
        {
            *outValue = nn::fs::AccessLogMode_SdCard;
        }
        else if ( mode == "off" )
        {
            *outValue = nn::fs::AccessLogMode_Off;
        }
        else
        {
            return false;
        }
        return true;
    }

    Result SetFsAccessLogMode(bool* outValue, const Option& option)
    {
        uint32_t accessLogMode = nn::fs::AccessLogMode_Off;
        const char* modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a fs access log mode. Set one of [log|sdcard|log+sdcard|off].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        const char* begin = modeString;
        const char separator = '+';
        const char* separatePosition = strchr(begin, separator);
        while ( separatePosition != nullptr )
        {
            uint32_t parsedAccessLogMode = nn::fs::AccessLogMode_Off;
            if ( !ParseFsAccessLogModeString(&parsedAccessLogMode, std::string(begin, separatePosition)) )
            {
                NN_LOG("%s is not valid fs access log mode. Set one of [log|sdcard|log+sdcard|off].\n", modeString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            accessLogMode |= parsedAccessLogMode;
            begin = separatePosition + 1;
            separatePosition = strchr(begin, separator);
        }

        {
            uint32_t parsedAccessLogMode = nn::fs::AccessLogMode_Off;
            if ( !ParseFsAccessLogModeString(&parsedAccessLogMode, std::string(begin)) )
            {
                NN_LOG("%s is not valid fs access log mode. Set one of [log|sdcard|log+sdcard|off].\n", modeString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            accessLogMode |= parsedAccessLogMode;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SetGlobalAccessLogMode(accessLogMode));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetFsAccessLogMode(bool* outValue, const Option&)
    {
        uint32_t accessLogMode = nn::fs::AccessLogMode_Off;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetGlobalAccessLogMode(&accessLogMode));
        switch ( accessLogMode )
        {
        case nn::fs::AccessLogMode_Off:
            NN_LOG("off\n");
            break;
        case nn::fs::AccessLogMode_Log:
            NN_LOG("log\n");
            break;
        case nn::fs::AccessLogMode_SdCard:
            NN_LOG("sdcard\n");
            break;
        case nn::fs::AccessLogMode_Log|nn::fs::AccessLogMode_SdCard:
            NN_LOG("log+sdcard\n");
            break;
        default:
            NN_LOG("Unknown\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool ParseFsSpeedEmulationModeString(nn::fs::SpeedEmulationMode* outValue, std::string&& mode)
    {
        if ( mode == "faster" )
        {
            *outValue = nn::fs::SpeedEmulationMode::Faster;
        }
        else if ( mode == "slower" )
        {
            *outValue = nn::fs::SpeedEmulationMode::Slower;
        }
        else if ( mode == "random" )
        {
            *outValue = nn::fs::SpeedEmulationMode::Random;
        }
        else if ( mode == "off" )
        {
            *outValue = nn::fs::SpeedEmulationMode::None;
        }
        else
        {
            return false;
        }
        return true;
    }

    Result SetFsSpeedEmulationMode(bool* outValue, const Option& option)
    {
        nn::fs::SpeedEmulationMode speedEmulationMode = nn::fs::SpeedEmulationMode::None;
        const char* modeString = option.GetTarget();
        if ( std::string(modeString) == "" )
        {
            NN_LOG("You must specify a fs speed emulation mode. Set one of [faster|slower|random|off].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if ( !ParseFsSpeedEmulationModeString(&speedEmulationMode, std::string(modeString)) )
        {
            NN_LOG("%s is not valid fs speed emulation mode. Set one of [faster|slower|random|off].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(speedEmulationMode));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetFsSpeedEmulationMode(bool* outValue, const Option&)
    {
        nn::fs::SpeedEmulationMode speedEmulationMode = nn::fs::SpeedEmulationMode::None;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetSpeedEmulationMode(&speedEmulationMode));
        switch ( speedEmulationMode )
        {
        case nn::fs::SpeedEmulationMode::None:
            NN_LOG("off\n");
            break;
        case nn::fs::SpeedEmulationMode::Faster:
            NN_LOG("faster\n");
            break;
        case nn::fs::SpeedEmulationMode::Slower:
            NN_LOG("slower\n");
            break;
        case nn::fs::SpeedEmulationMode::Random:
            NN_LOG("random\n");
            break;
        default:
            NN_LOG("Unknown\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    nifm::RequirementPreset ParsePreset(const char* preset)
    {
        if (std::string(preset) == "") return nifm::RequirementPreset_InternetGeneric;
        if (std::string(preset) == "best-effort") return nifm::RequirementPreset_InternetBestEffort;
        if (std::string(preset) == "local") return nifm::RequirementPreset_LocalGeneric;
        if (std::string(preset) == "applet") return nifm::RequirementPreset_InternetForApplet;
        if (std::string(preset) == "net-connect") return nifm::RequirementPreset_InternetForNetConnectApplet;
        if (std::string(preset) == "web-auth") return nifm::RequirementPreset_InternetForWifiWebAuthApplet;
        if (std::string(preset) == "local-applet") return nifm::RequirementPreset_LocalForApplet;
        if (std::string(preset) == "system") return nifm::RequirementPreset_InternetForSystemProcess;
        if (std::string(preset) == "system-persistent") return nifm::RequirementPreset_InternetForSystemProcessPersistent;
        if (std::string(preset) == "system-continuous") return nifm::RequirementPreset_InternetForSystemProcessContinuous;
        return nifm::RequirementPreset_Invalid;
    }

    Result RequestNetworkConnection(bool* outValue, const Option& option)
    {
        auto presetString = option.GetTarget();
        auto preset = ParsePreset(presetString);
        auto waitSpanString = option.GetValue("--wait-span");
        TimeSpan waitSpan = waitSpanString ? TimeSpan::FromMilliSeconds(std::strtoul(waitSpanString, nullptr, 10)) :
                                             TimeSpan::FromMilliSeconds(10000);
        auto requestSpanString = option.GetValue("--request-span");
        TimeSpan requestSpan = requestSpanString ? TimeSpan::FromMilliSeconds(std::strtoul(requestSpanString, nullptr, 10)) :
                                                   TimeSpan::FromMilliSeconds(10000);

        if (preset == nifm::RequirementPreset_Invalid)
        {
            NN_LOG("%s is unknown requirement preset\n", presetString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
        nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(connection.GetRequestHandle(), preset));
        connection.SubmitRequest();
        if (!connection.GetSystemEvent().TimedWait(waitSpan))
        {
            NN_LOG("Waiting request timed out\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!connection.IsAvailable())
        {
            NN_LOG("Network connection is not available\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_LOG("Network connection for preset %s is available. Holding %lld ms...\n", presetString, requestSpan.GetMilliSeconds());
        os::SleepThread(requestSpan);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    nn::Result InstallSystemProgram(const char* path)
    {
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        static const size_t BufferSize = 16 * 1024 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        nn::ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, nn::ncm::StorageId::BuildInSystem, buffer.get(), BufferSize));

        NN_RESULT_DO(task.Prepare());
        NN_RESULT_DO(devmenuUtil::WaitInstallTaskExecute(&task));
        NN_RESULT_DO(task.Commit());

        NN_RESULT_SUCCESS;
    }

    nn::Result SetFirmwareDebugSettings(bool* outValue, const Option& option)
    {
        const nn::ncm::SystemDataId fwdbgHbDataId = { 0x010000000000B1B8 };
        const nn::ncm::SystemDataId fwdbgUsbDataId = { 0x010000000000B1B9 };
        const nn::ncm::SystemDataId fwdbgProdDataId = { 0x010000000000B1BA };

        auto fwdbgString = option.GetTarget();
        nn::ncm::SystemDataId dataId = fwdbgProdDataId;

        if (std::string(fwdbgString) == "dev")
        {
            nn::settings::factory::ConfigurationId1 ConfigurationId1 = {};
            nn::settings::factory::GetConfigurationId1(&ConfigurationId1);
            if (strncmp("SDEV", ConfigurationId1.string, 4) == 0)
            {
                dataId = fwdbgHbDataId;
            }
            else
            {
                dataId = fwdbgUsbDataId;
            }
        }
        else if(std::string(fwdbgString) == "prod")
        {
            //do nothing
        }
        else if (std::string(fwdbgString) == "")
        {
            NN_LOG("usage: " DEVMENUCOMMAND_NAME " debug set-firmware-debug-settings <dev|prod>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LOG("%s is invalid argument. Set one of [dev/prod].\n", fwdbgString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        size_t mountSystemDataCacheBufferSize = 0;
        NN_RESULT_DO(nn::fs::QueryMountSystemDataCacheSize(&mountSystemDataCacheBufferSize, dataId));

        std::unique_ptr<char[]> buffer(new char[mountSystemDataCacheBufferSize]);
        NN_RESULT_DO(nn::fs::MountSystemData("SystemData", dataId, buffer.get(), mountSystemDataCacheBufferSize));
        NN_UTIL_SCOPE_EXIT { nn::fs::Unmount("SystemData"); };

        {
            nn::fs::DirectoryHandle directory;
            NN_RESULT_DO(nn::fs::OpenDirectory(&directory, "SystemData:/", nn::fs::OpenDirectoryMode_File));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directory); };

            while (NN_STATIC_CONDITION(true))
            {
                nn::fs::DirectoryEntry entry;
                int64_t count;
                NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, directory, 1));
                if (count == 0)
                {
                    break;
                }

                if (devmenuUtil::IsNspFile(entry.name))
                {
                    std::string path = std::string("SystemData:/") + entry.name;
                    InstallSystemProgram(path.c_str());
               }
            }

        }

        NN_LOG("Please restart your target to apply this change.\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result GetBatteryDrainingPerformanceEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "apm", "battery_draining_enabled");
    }

    Result EnableBatteryDrainingPerformance(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "apm", "battery_draining_enabled", true);
    }

    Result DisableBatteryDrainingPerformance(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "apm", "battery_draining_enabled", true);
    }

    Result EnableCec(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::settings::system::TvSettings    settings;
        nn::settings::system::GetTvSettings(&settings);
        settings.flags.Set<nn::settings::system::TvFlag::AllowsCec>();
        nn::settings::system::SetTvSettings(settings);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableCec(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nn::settings::system::TvSettings    settings;
        nn::settings::system::GetTvSettings(&settings);
        settings.flags.Reset<nn::settings::system::TvFlag::AllowsCec>();
        nn::settings::system::SetTvSettings(settings);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnablePcm(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "pcm", "enable");
    }

    Result DisablePcm(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "pcm", "enable");
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result GetNotLockedNfpAccessEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "nfp", "not_locked_tag");
    }

    Result EnableNotLockedNfpAccess(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "nfp", "not_locked_tag", false);
    }

    Result DisableNotLockedNfpAccess(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "nfp", "not_locked_tag", false);
    }

    Result GetNfpPlayReportEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "nfp", "play_report");
    }

    Result EnableNfpPlayReport(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "nfp", "play_report", false);
    }

    Result DisableNfpPlayReport(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "nfp", "play_report", false);
    }

    Result GetSystemPlayReportEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "systemreport", "enabled");
    }

    Result EnableSystemPlayReport(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "systemreport", "enabled", true);
    }

    Result DisableSystemPlayReport(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "systemreport", "enabled", true);
    }

    Result SetInitialLaunchSettings(bool* outValue, const Option& option)
    {
        auto isCompleted = devmenuUtil::GetBooleanOptionValue(option, "--completed");
        if( !isCompleted )
        {
            NN_LOG("--completed is required and should be true or false.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto hasTimeStamp = devmenuUtil::GetBooleanOptionValue(option, "--has-timestamp");
        if( !hasTimeStamp )
        {
            NN_LOG("--has-timestamp is required and should be true or false.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize()); };
        time::SteadyClockTimePoint timePoint;
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

        // 経過時間によるタイムスタンプ指定。
        auto timeStampStr = option.GetValue("--timestamp-by-elapsed");
        if( timeStampStr != nullptr )
        {
            timePoint -= TimeSpan::FromSeconds(static_cast<int64_t>(std::strtoul(timeStampStr, nullptr, 10)));
        }

        settings::system::InitialLaunchSettings initialSettings;
        initialSettings.flags.Set<settings::system::InitialLaunchFlag::IsCompleted>(*isCompleted);
        initialSettings.flags.Set<settings::system::InitialLaunchFlag::HasTimeStamp>(*hasTimeStamp);
        initialSettings.timeStamp = timePoint;

        settings::system::SetInitialLaunchSettings(initialSettings);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetInitialLaunchSettings(bool* outValue, const Option&)
    {
        settings::system::InitialLaunchSettings initialSettings;
        settings::system::GetInitialLaunchSettings(&initialSettings);

        NN_LOG("Initial Launch : %s.\n", initialSettings.flags.Test<settings::system::InitialLaunchFlag::IsCompleted>() ? "Completed" : "NOT completed");
        if( initialSettings.flags.Test<settings::system::InitialLaunchFlag::HasTimeStamp>() )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForMenu());
            NN_UTIL_SCOPE_EXIT{ NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize()); };

            time::SteadyClockTimePoint currentTimePoint;
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));
            int64_t elapsedSeconds;
            if(time::GetSpanBetween(&elapsedSeconds, initialSettings.timeStamp, currentTimePoint).IsSuccess() )
            {
                NN_LOG("Timestamp      : %lld (%lld seconds before).\n", initialSettings.timeStamp.value, elapsedSeconds);
            }
            else
            {
                NN_LOG("Timestamp      : %lld (uncomparable with the current steady clock time).\n");
            }
        }
        else
        {
            NN_LOG("Timestamp      : NOT stored.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetDeviceLanguage(bool* outValue, const Option& option)
    {
        auto language = devmenuUtil::language::ConvertStringToLanguage(option.GetTarget());
        settings::system::SetLanguageCode(settings::LanguageCode::Make(language));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetDeviceLanguage(bool* outValue, const Option&)
    {
        settings::LanguageCode code;
        settings::GetLanguageCode(&code);
        NN_LOG("Language: %s\n", devmenuUtil::language::ConvertLanguageCodeToString(code));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetPrepurchaseResponseMode(bool* outValue, const Option& option)
    {
        std::string string = std::string(option.GetTarget());

        if (string == "normal")
        {
            bool flag = false;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_delay_response", &flag, sizeof(flag));
            flag = false;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_still_unavailable_response", &flag, sizeof(flag));
        }
        else if (string == "delay")
        {
            bool flag = true;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_delay_response", &flag, sizeof(flag));
            flag = false;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_still_unavailable_response", &flag, sizeof(flag));
        }
        else if (string == "still_unavailable")
        {
            bool flag = false;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_delay_response", &flag, sizeof(flag));
            flag = true;
            settings::fwdbg::SetSettingsItemValue("nim.install", "simulate_eci_still_unavailable_response", &flag, sizeof(flag));
        }
        else
        {
            NN_LOG("You must specify a prepurchase response mode. Set one of [normal|delay|still_unavailable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetPrepurchaseResponseMode(bool* outValue, const Option&)
    {
        bool delayFlag;
        settings::fwdbg::GetSettingsItemValue(&delayFlag, sizeof(delayFlag), "nim.install", "simulate_eci_delay_response");
        bool stillUnavailableFlag;
        settings::fwdbg::GetSettingsItemValue(&stillUnavailableFlag, sizeof(stillUnavailableFlag), "nim.install", "simulate_eci_still_unavailable_response");

        if (delayFlag)
        {
            NN_LOG("delay\n");
        }
        else if (stillUnavailableFlag)
        {
            NN_LOG("still_unavailable\n");
        }
        else
        {
            NN_LOG("normal\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result EnableNimErrorSimulate(bool* outValue, const Option& option)
    {
        auto url = option.GetValue("--url");
        if (url == nullptr)
        {
            NN_LOG("--url option is required\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto urlLength = util::Strnlen(url, 256);
        if (urlLength == 256)
        {
            NN_LOG("--url is too long\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto result = option.GetValue("--result");
        if (result == nullptr)
        {
            NN_LOG("--result option is required\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto rate = option.GetValue("--error-rate");
        if (rate == nullptr)
        {
            rate = "10000";
        }

        auto resultValue = static_cast<int32_t>(STR_TO_ULL(result, nullptr, 16));
        auto rateValue = static_cast<int32_t>(STR_TO_ULL(rate, nullptr, 10));

        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "error_url_prefix", url, urlLength + 1);
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "error_rate", &rateValue, sizeof(rateValue));
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "error_result", &resultValue, sizeof(resultValue));

        bool enabled = true;
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_error_simulate", &enabled, sizeof(enabled));

        nim::ReloadErrorSimulation();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableNimErrorSimulate(bool* outValue, const Option&)
    {
        bool enabled = false;
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_error_simulate", &enabled, sizeof(enabled));
        nim::ReloadErrorSimulation();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    //! @brief  ec::DebugForShopServiceAccessor デバッグ支援機能が提供するデバッグレスポンス機能を用いたレスポンスシミュレーションに関する機能群です。
    namespace NimDebugResponseSimulation
    {
        class ServerProfile
        {
        private:
            const char*                 m_pAlias;
            const Bit16                 m_AliasLength;
            const ec::ShopService::Type m_Type;

        public:
            explicit ServerProfile(const char* pName, ec::ShopService::Type type_) NN_NOEXCEPT
                : m_pAlias(pName), m_AliasLength(static_cast<Bit16>(std::strlen(pName))), m_Type(type_) {}

            NN_FORCEINLINE int GetAliasLength() const NN_NOEXCEPT
            {
                return static_cast<int>(m_AliasLength);
            }

            NN_FORCEINLINE const char* GetAliasName() const NN_NOEXCEPT
            {
                return m_pAlias;
            }

            NN_FORCEINLINE ec::ShopService::Type GetType() const NN_NOEXCEPT
            {
                return m_Type;
            }
        };

        const ServerProfile* QueryServerProfileFromName(const char* pQueryName) NN_NOEXCEPT
        {
            int sourceLength;
            if (nullptr != pQueryName && 0 < (sourceLength = util::Strnlen(pQueryName, 48)))
            {
                NN_FUNCTION_LOCAL_STATIC(const ServerProfile, s_Servers, [] =
                    {
                        ServerProfile("OwnedConsumableServiceItem", ec::ShopService::Type_OwnedConsumableServiceItem),
                        ServerProfile("Catalog", ec::ShopService::Type_Catalog),
                        ServerProfile("DynamicRights", ec::ShopService::Type_Catalog),
                    }
                );
                for (const auto& profile : s_Servers)
                {
                    if (profile.GetAliasLength() == sourceLength && 0 == ::nn::util::Strnicmp(profile.GetAliasName(), pQueryName, sourceLength))
                    {
                        return &profile;
                    }
                }
            }
            return nullptr;
        }

        //! @brief      nim エラーシミュレーション機能可用性のマスター設定を変更します。
        //! @details    本 API で変更された可用性状態は、enable-nim-error-simulate / disable-nim-error-simulate と排他関係にあります。@n
        //!             enable-nim-error-simulate / disable-nim-error-simulate と本APIの設定状態はデバイス再起動によって相互に可用性が反映されます。@n
        //!             デバイス再起動を経由しなければ、相互の可用性状態は、明示的な各コマンドの呼び出しが行われるまで反映されません。
        Result SetAvailability(bool availability) NN_NOEXCEPT
        {
            settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_error_simulate", &availability, sizeof(availability));
            NN_RESULT_SUCCESS;
        }

        //! @brief      nim エラーシミュレーション機能可用性の DynamicRights 依存設定を変更します。
        Result SetAvailabilityForDynamicRights(bool availability) NN_NOEXCEPT
        {
            settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_simulate_dynamic_rights", &availability, sizeof(availability));
            NN_RESULT_SUCCESS;
        }

        Result Enable(bool* outValue, const Option& option) NN_NOEXCEPT
        {
            ec::DebugForShopServiceAccessor::ScopedSession session;
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::Initialize(&session));

            NN_RESULT_DO(SetAvailability(true));
            if (option.HasKey("--dynamic-rights"))
            {
                NN_RESULT_DO(SetAvailabilityForDynamicRights(true));
            }

            ec::DebugForShopServiceAccessor::RefreshAvailability();
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        Result Disable(bool* outValue, const Option& option) NN_NOEXCEPT
        {
            ec::DebugForShopServiceAccessor::ScopedSession session;
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::Initialize(&session));

            NN_RESULT_DO(SetAvailability(false));
            if (option.HasKey("--dynamic-rights"))
            {
                NN_RESULT_DO(SetAvailabilityForDynamicRights(false));
            }

            ec::DebugForShopServiceAccessor::RefreshAvailability();
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        Result Register(bool* outValue, const Option& option) NN_NOEXCEPT
        {
            const auto pServerName = option.GetValue("--server");

            // 必須要件チェック
            bool validRequired = true;
            const auto pServerProfile = QueryServerProfileFromName(pServerName);
            if (nullptr == pServerProfile)
            {
                DEVMENUCOMMAND_LOG("`--server` option is required.\n");
                validRequired = false;
            }
            const auto pUrlPath = option.GetValue("--url-path");
            if (nullptr == pUrlPath)
            {
                DEVMENUCOMMAND_LOG("`--url-path` option is required.\n");
                validRequired = false;
            }
            constexpr int limitLength = 255;
            const auto urlLength = util::Strnlen(pUrlPath, limitLength);
            if (urlLength == limitLength)
            {
                DEVMENUCOMMAND_LOG("`--url-path` is too long. Please specify under length of %d.\n", limitLength);
                validRequired = false;
            }
            const auto pBodyFilePath = option.GetValue("--response-file");
            if (nullptr != pBodyFilePath && !devmenuUtil::IsAbsolutePath(pBodyFilePath))
            {
                DEVMENUCOMMAND_LOG("'%s' is not an absolute path in host.\n", pBodyFilePath);
                validRequired = false;
            }
            if (!validRequired)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            // デバッグオプション
            const auto isVerbose = option.HasKey("--verbose");

            // デフォルト値チェック
            Result resultValue = ResultSuccess();
            const auto pExpectResult = option.GetValue("--result-inner-value");
            if (nullptr != pExpectResult)
            {
                resultValue = result::detail::ConstructResult(static_cast<uint32_t>(STR_TO_ULL(pExpectResult, nullptr, 16)));
            }
            uint32_t rateValue = 10000;
            const auto pHappenedRate = option.GetValue("--rate");
            if (nullptr != pHappenedRate)
            {
                rateValue = static_cast<uint32_t>(STR_TO_ULL(pHappenedRate, nullptr, 10));
            }

            // ゼロ長レスポンスはデバッグAPIがサポートしていないので最短の長さ 1 ( '\0' のみ配列 ) をデフォルト。
            size_t actualBodySize = 1;
            const char* pActualBody = "";
            std::unique_ptr<char[]> body;

            size_t rawBodySize = 0;
            const auto pRawBody = option.GetValue("--response");
            if (nullptr != pRawBody && (rawBodySize = std::strlen(pRawBody)) > 0)
            {
                // コマンドライン上の有効な文字列経由の指定。
                pActualBody = pRawBody;
                actualBodySize = rawBodySize;
            }
            else if (nullptr != pBodyFilePath)
            {
                // ファイル経由のレスポンスボディ読み込み。
                fs::FileHandle file;
                NN_RESULT_DO(fs::OpenFile(&file, pBodyFilePath, fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT{fs::CloseFile(file);};

                int64_t fileSize;
                NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

                if (fileSize > 0)
                {
                    actualBodySize = static_cast<size_t>(fileSize);
                    std::unique_ptr<char[]> buffer(new char[actualBodySize + 1]);
                    NN_RESULT_DO(fs::ReadFile(file, 0, buffer.get(), actualBodySize));
                    buffer[actualBodySize] = '\0';  // ログ出力用.
                    body = std::move(buffer);
                    pActualBody = body.get();
                }
            }

            // 登録処理本体。
            ec::DebugForShopServiceAccessor::ScopedSession session;
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::Initialize(&session));
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::RegisterResponse(pServerProfile->GetType(), pUrlPath, pActualBody, actualBodySize, resultValue, rateValue));

            // 結果ログ出力。
            if (isVerbose)
            {
                // レスポンスボディがバイナリの場合は途中の '\0' で出力が途切れる可能性があります。
                DEVMENUCOMMAND_LOG("[DevMenuCommand] Requested values {\n    url-path(%s), result(%lu-%lu), rate(%lu)\n    response(%zu byte)=>\n%s\n}\n"
                    , pUrlPath, resultValue.GetModule()
                    , resultValue.GetDescription()
                    , rateValue
                    , actualBodySize, pActualBody
                );
            }
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        Result Clear(bool* outValue, const Option& option) NN_NOEXCEPT
        {
            ec::DebugForShopServiceAccessor::ScopedSession session;
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::Initialize(&session));
            NN_RESULT_DO(ec::DebugForShopServiceAccessor::ClearResponse());
            *outValue = true;
            NN_UNUSED(option);
            NN_RESULT_SUCCESS;
        }

    }   // ~NimDebugResponseSimulation
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

    Result SetSslTestInit(bool* outValue, const Option& option)
    {
        static const char* const  FalseValue = "false";
        static const char* const  TrueValue  = "true";
        static const size_t       FalseLen   = strlen(FalseValue);
        static const size_t       TrueLen    = strlen(TrueValue);

        bool isInitForTestEnabled = false;

        auto pIsInitForTestEnabled = option.GetValue("--initialize_for_test");
        if (pIsInitForTestEnabled == nullptr)
        {
            pIsInitForTestEnabled = FalseValue;
        }

        if(strncmp(FalseValue, pIsInitForTestEnabled, FalseLen) == 0)
        {
            isInitForTestEnabled = false;
        }
        else if(strncmp(TrueValue, pIsInitForTestEnabled, TrueLen) == 0)
        {
            isInitForTestEnabled = true;
        }
        else
        {
            NN_LOG("Invalid value for --initialize_for_test. Valid values: 'true', 'false'\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        settings::fwdbg::SetSettingsItemValue("ssl.test", "initialize_for_test", &isInitForTestEnabled, sizeof(isInitForTestEnabled));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CallAbort(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto resultString = option.GetTarget();
        if( std::string(resultString) != "" )
        {
            auto resultValue = std::strtoul(resultString, nullptr, 16);
            auto result = nn::result::detail::ConstructResult(resultValue);
            if( !result.IsFailure() )
            {
                DEVMENUCOMMAND_LOG("<result_value> must not be 0.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        else
        {
            NN_ABORT();
        }
        NN_RESULT_SUCCESS;
    }
#endif

    Result GetEulaState(bool* outValue, const Option&)
    {
        settings::system::RegionCode regionCode;
        settings::system::GetRegionCode(&regionCode);

        std::unique_ptr<settings::system::EulaVersion[]> pEulaVers(new settings::system::EulaVersion[settings::system::EulaVersionCountMax]);
        int count = settings::system::GetEulaVersions(pEulaVers.get(), settings::system::EulaVersionCountMax);

        for (int i = 0; i < count; ++i)
        {
            settings::system::EulaVersion& eulaVer = pEulaVers[i];
            if (eulaVer.regionCode == regionCode)
            {
                uint16_t major = eulaVer.version >> 16;
                uint16_t minor = 0x0000FFFF & eulaVer.version;
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
                NN_UNUSED(major);
                NN_UNUSED(minor);
#endif
                NN_LOG("EULA has been already accepted. (v%d.%d)\n", major, minor);
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        NN_LOG("EULA is not accepted.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetExhibitionModeEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "devmenu", "enable_exhibition_mode");
    }


    Result EnableExhibitionMode(bool* outValue, const Option&)
    {
        NN_RESULT_DO(DisableFatalExtraInfoCore(outValue, false));
        return EnableBooleanFirmwareDebugSetting(outValue, "devmenu", "enable_exhibition_mode", true);
    }

    Result DisableExhibitionMode(bool* outValue, const Option&)
    {
        NN_RESULT_DO(EnableFatalExtraInfoCore(outValue, false));
        return DisableBooleanFirmwareDebugSetting(outValue, "devmenu", "enable_exhibition_mode", true);
    }

    Result GetNoVsyncCapabilityModeEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "nv", "nvn_no_vsync_capability");

        if (!(sizeof(isEnabled) == size))
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableNoVsyncCapability(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "nv", "nvn_no_vsync_capability");
    }

    Result DisableNoVsyncCapability(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "nv", "nvn_no_vsync_capability");
    }

    Result EaseNroRestriction(bool* outValue, const Option& option)
    {
        // fwdbg の ease_nro_restriction の値を書き換える
        bool isEaseNroRestriction;
        auto valueString = std::string(option.GetTarget());
        if (valueString == "true")
        {
            isEaseNroRestriction = true;
        }
        else if (valueString == "false")
        {
            isEaseNroRestriction = false;
        }
        else
        {
            NN_LOG("value is required and should be true or false");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        settings::fwdbg::SetSettingsItemValue("ro", "ease_nro_restriction", &isEaseNroRestriction, sizeof(isEaseNroRestriction));
        NN_LOG("ro ease_nro_restriction is set to %s\n", valueString.c_str());

        // 設定に成功したので、true を返す
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ResetPseudoDeviceId(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        NN_RESULT_TRY(ns::ResetSystemSeedForPseudoDeviceId());
        NN_RESULT_CATCH(ns::ResultFailedToReadFirmwareDebugKey)
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(ns::ResultNotPermittedOnProduction)
        {
            NN_LOG("Curret Mode is PROD MODE. Therefore you can't reset Pseudo Device Id.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_END_TRY;

        NN_LOG("Resetting Pseudo Device ID has been done.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetGpuTimeoutModeEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool isEnabled( false );
        char val[2] = {};
        settings::fwdbg::GetSettingsItemValue(val, sizeof(val), "nv", "gpu_timeout_ms_max");
        if( util::Strnlen(val, sizeof(val)) == 0 )
        {
            isEnabled = true;
        }
        NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableGpuTimeout(bool* outValue, const Option&)
    {
        const char* pVal = "";    // "" : ドライバでデフォルト値が設定される。HR20.2 以前 8sec, 以降 1sec
        return SetStringFirmwareDebugSettings( outValue, pVal, "nv", "gpu_timeout_ms_max", true );
    }

    Result DisableGpuTimeout(bool* outValue, const Option&)
    {
        const char* pVal = "0";
        return SetStringFirmwareDebugSettings( outValue, pVal, "nv", "gpu_timeout_ms_max", true );
    }

    Result GetRetailInteractiveDisplayModeEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        NN_LOG("%s\n", settings::system::GetQuestFlag() ? "enabled" : "disabled");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableRetailInteractiveDisplayMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        settings::system::SetQuestFlag(true);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableRetailInteractiveDisplayMode(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        settings::system::SetQuestFlag(false);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetHdcpAuthenticationFailedEmulationEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        bool isEnabled( false );

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        omm::Initialize();
        NN_UTIL_SCOPE_EXIT{ omm::Finalize(); };
        isEnabled = omm::GetHdcpAuthenticationFailedEmulationEnabled();
#endif
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        NN_UNUSED(isEnabled);
#endif
        NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetInStoreDemoModeEnabled(bool* outValue, const Option& option)
    {
        return GetRetailInteractiveDisplayModeEnabled(outValue, option);
    }

    Result EnableInStoreDemoMode(bool* outValue, const Option& option)
    {
        // HOME メニュー への切り替え
        char* argv[] = {
            const_cast<char*>(""), const_cast<char*>("debug"),
            const_cast<char*>("switch-menu"), const_cast<char*>("homemenu") };

        SwitchMenu(outValue,Option(4, argv));
        if (false == *outValue)
        {
            NN_RESULT_SUCCESS;
        }

        // 初回起動フラグのチェック
        settings::system::InitialLaunchSettings initialLaunchSettings;
        settings::system::GetInitialLaunchSettings(&initialLaunchSettings);
        const auto isInitialLaunchCompleted = initialLaunchSettings.flags.Test<settings::system::InitialLaunchFlag::IsCompleted>();

        // RetailInteractiveDisplayMenu がインストール済みかのチェック
        const nn::ncm::ProgramId retailInteractiveDisplaydMenuId = {NN_TOOL_DEVMENUCOMMAND_PROGRAM_ID_OF_RIDMENU};
        const auto isRetailInteractiveDisplayMenuInstalled = devmenuUtil::IsProgramInstalled(retailInteractiveDisplaydMenuId, nn::ncm::StorageId::BuiltInUser);

        if (!isInitialLaunchCompleted)
        {
            NN_LOG("-----------------------------------------------------------------------------------------------------------------\n");
            NN_LOG("Initial launched menu has been swithed to Home Menu but its initial launch settings is not completed.\n");
            NN_LOG("Complete the settings and then execute the command again.\n");
        }

        if (!isRetailInteractiveDisplayMenuInstalled)
        {
            NN_LOG("-----------------------------------------------------------------------------------------------------------------\n");
            NN_LOG("In-store demo menu is not installed.\n");
            NN_LOG("Install the menu and then execute the command again.\n");
        }

        if (true == (*outValue = isInitialLaunchCompleted && isRetailInteractiveDisplayMenuInstalled))
        {
            // In-Store Demo Mode の有効化
            EnableRetailInteractiveDisplayMode(outValue, option);
            NN_LOG("-----------------------------------------------------------------------------------------------------------------\n");
            NN_LOG("In-store demo mode has been enabled. Reboot the device, and the menu will be launched automatically.\n");
        }

        NN_LOG("-----------------------------------------------------------------------------------------------------------------\n");

        NN_RESULT_SUCCESS;
    }

    Result DisableInStoreDemoMode(bool* outValue, const Option& option)
    {
        return DisableRetailInteractiveDisplayMode(outValue, option);
    }

    Result EnableHdcpAuthenticationFailedEmulation(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        omm::Initialize();
        NN_UTIL_SCOPE_EXIT{ omm::Finalize(); };
        NN_RESULT_TRY(omm::SetHdcpAuthenticationFailedEmulation(true));
        NN_RESULT_CATCH(omm::ResultNotPermittedOnProduction)
        {
            NN_LOG("Curret Mode is PROD MODE. Therefore you can't enable Hdcp Authentication Emulation.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_END_TRY;
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableHdcpAuthenticationFailedEmulation(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        omm::Initialize();
        NN_UTIL_SCOPE_EXIT{ omm::Finalize(); };
        NN_RESULT_DO(omm::SetHdcpAuthenticationFailedEmulation(false));
#endif

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetGraphicsFirmwareMemoryMarginEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "nv", "nv_graphics_firmware_memory_margin");

        if (!(sizeof(isEnabled) == size))
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableGraphicsFirmwareMemoryMargin(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "nv", "nv_graphics_firmware_memory_margin");
    }

    Result DisableGraphicsFirmwareMemoryMargin(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "nv", "nv_graphics_firmware_memory_margin");
    }

    template <typename ResultType>
    void EraseErrorFileInfo(std::vector<std::unique_ptr<devmenuUtil::ErrorFileInfo>>* pErrorFileInfoList)
    {
        auto remove = std::remove_if(pErrorFileInfoList->begin(), pErrorFileInfoList->end(), [](std::unique_ptr<devmenuUtil::ErrorFileInfo >& info)->bool
        {
            return ResultType().Includes(info->result);
        });

        pErrorFileInfoList->erase(remove, pErrorFileInfoList->end());
    }

    template <typename ResultType>
    bool ShowErrorFile(const std::vector<std::unique_ptr<devmenuUtil::ErrorFileInfo>>& errorFileInfoList, const char* errorMessage)
    {
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
        // DevMenu の Release ビルド時の warning 対策
        NN_UNUSED(errorMessage);
#endif

        bool isErrorMessageDisplayed = false;

        for (auto& info : errorFileInfoList)
        {
            if (ResultType().Includes(info->result))
            {
                if (!isErrorMessageDisplayed)
                {
                    // 最初のファイル表示前にエラー内容を表示
                    NN_LOG(errorMessage);
                    isErrorMessageDisplayed = true;
                }
                NN_LOG("  %s\n", info->path);
            }
        }

        return isErrorMessageDisplayed;
    }

    Result Copy(bool* outValue, const Option& option)
    {
        *outValue = false;

        if (!option.HasKey("--destination"))
        {
            NN_LOG("Please specify destination path.\n");
            NN_RESULT_SUCCESS;
        }

        if (!option.HasKey("--source"))
        {
            NN_LOG("Please specify source path.\n");
            NN_RESULT_SUCCESS;
        }

        bool isSdCardMounted = true;
        NN_RESULT_TRY(nn::fs::MountSdCardForDebug(SdCardMountName))
            NN_RESULT_CATCH(nn::fs::ResultPortSdCardNoDevice)
            {
                isSdCardMounted = false;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            if (isSdCardMounted)
            {
                nn::fs::Unmount(SdCardMountName);
            }
        };

        std::vector<std::unique_ptr<devmenuUtil::ErrorFileInfo>> errorFileInfoList;
        auto force = option.HasKey("--force");
        auto skipsErrorFile = option.HasKey("--skip-error-file");

        auto result = devmenuUtil::Copy(&errorFileInfoList, option.GetValue("--destination"), option.GetValue("--source"), force, skipsErrorFile);

        if (force)
        {
            // --force オプションが指定されている場合は ResultPathAlreadyExists のエラーを無視する為にリストから削除
            EraseErrorFileInfo<nn::fs::ResultPathAlreadyExists>(&errorFileInfoList);
        }

        bool isSuccess = true;
        // ResultTargetLocked のエラーの表示
        if (ShowErrorFile<nn::fs::ResultTargetLocked>(errorFileInfoList, "Failed to copy the following file or directory. Target file or directory is locked.\n"))
        {
            EraseErrorFileInfo<nn::fs::ResultTargetLocked>(&errorFileInfoList);
            isSuccess = false;
        }

        // ResultPathAlreadyExists のエラーの表示
        if (ShowErrorFile<nn::fs::ResultPathAlreadyExists>(errorFileInfoList, "Failed to copy the following file or directory. Destination path is already exist.\n"))
        {
            EraseErrorFileInfo<nn::fs::ResultPathAlreadyExists>(&errorFileInfoList);
            isSuccess = false;
        }

        bool isErrorMessageDisplayed = false;
        // エラーメッセージを表示していないファイルを表示
        for (auto& info : errorFileInfoList)
        {
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
            // DevMenu の Release ビルド時の warning 対策
            NN_UNUSED(info);
#endif
            if (!isErrorMessageDisplayed)
            {
                NN_LOG("Failed to copy the following file or directory.\n");
                isErrorMessageDisplayed = true;
            }
            NN_LOG("  %s, result => 0x%08x.\n", info->path, info->result.GetInnerValueForDebug());
            isSuccess = false;
        }

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                // ファイルをディレクトリで上書きしようとすると、force でも AlreadyExists になる。
                // NN_ASSERT(!option.HasKey("--force"));
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::fs::ResultTargetLocked)
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::fs::ResultNotMounted)
            {
                NN_LOG("Specified volume is not mounted.\n");
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = isSuccess;
        NN_RESULT_SUCCESS;
    }

    Result GetCleanupApplicationCacheStorageEnabled(bool* outValue, const Option&) NN_NOEXCEPT
    {
        return GetBooleanFirmwareDebugSetting(outValue, "ns.application", "cleanup_cache_storage_enabled");
    }

    Result EnableCleanupApplicationCacheStorage(bool* outValue, const Option&) NN_NOEXCEPT
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "ns.application", "cleanup_cache_storage_enabled", true);
    }

    Result DisableCleanupApplicationCacheStorage(bool* outValue, const Option&) NN_NOEXCEPT
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "ns.application", "cleanup_cache_storage_enabled", true);
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result SetCleanupSdCardResultValue(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto resultString = option.GetTarget();
        if (!resultString || strnlen(resultString, 1) == 0)
        {
            DEVMENUCOMMAND_LOG("<result_value> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto resultValue = static_cast<uint32_t>(std::strtoul(resultString, nullptr, 16));
        NN_RESULT_DO(SetFirmwareDebugSettingsImpl(outValue, resultValue, "ns.sdcard", "cleanup_sdcard_result_value", true));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetBlackList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        uint32_t version = option.HasKey("--version") ? static_cast<uint32_t>(std::strtoull(option.GetValue("--version"), NULL, 10)) : UINT32_MAX;
        // 2 回呼び出すので、１つ目の時にはログを表示しないようにする
        char appIdStr[20];
        util::TSNPrintf(appIdStr, sizeof(appIdStr), "0x%016llx", appId.value);
        NN_RESULT_DO(SetStringFirmwareDebugSettings(outValue, appIdStr, "ns.application", "application_id_on_blacklist", false));
        NN_RESULT_DO(SetFirmwareDebugSettingsImpl(outValue, version, "ns.application", "application_version_on_blacklist", true));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetResumeAllEnabled(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "devmenu", "enable_resume_all");

        if (!(sizeof(isEnabled) == size))
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result EnableResumeAll(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "devmenu", "enable_resume_all");
    }

    Result DisableResumeAll(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "devmenu", "enable_resume_all");
    }

    Result EnableDynamicRightsCheck(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(EnableBooleanFirmwareDebugSetting(outValue, "ns.application", "debug_elicense_import"));
        auto licenseType = option.HasKey("--debug-license-type") ? static_cast<int>(STR_TO_ULL(option.GetValue("--debug-license-type"), nullptr, 10)) : 0;
        NN_RESULT_DO(SetFirmwareDebugSettingsImpl(outValue, licenseType, "ns.application", "debug_license_type", true));
        NN_RESULT_SUCCESS;
    }

    Result DisableDynamicRightsCheck(bool* outValue, const Option&)
    {
        NN_RESULT_DO(DisableBooleanFirmwareDebugSetting(outValue, "ns.application", "debug_elicense_import"));
        NN_RESULT_DO(SetFirmwareDebugSettingsImpl(outValue, 0, "ns.application", "debug_license_type", true));
        NN_RESULT_SUCCESS;
    }

    Result CleanupPreInstall(bool* outValue, const Option& option)
    {
        *outValue = false;

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ns::AsyncResult async;
        if (option.HasKey("--all"))
        {
            NN_RESULT_DO(ns::RequestCleanupAllPreInstalledApplications(&async));
        }
        else
        {
            NN_RESULT_DO(ns::RequestCleanupPreInstalledApplication(&async, devmenuUtil::GetApplicationIdTarget(option)));
        }
        NN_RESULT_DO(async.Get());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    Result SetHostRomfsRedirectionTarget(bool* outValue, const Option& option)
    {
        nn::ncm::StorageId storageId;
        auto storageString = option.GetTarget();
        if (std::string(storageString) == "builtin")
        {
            storageId = nn::ncm::StorageId::BuiltInUser;
        }
        else if (std::string(storageString) == "off")
        {
            storageId = nn::ncm::StorageId::None;
        }
        else
        {
            NN_LOG("%s is not valid redirection target. Set one of [builtin|off].\n", storageString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SetDataStorageRedirectTarget(storageId));
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "get-operation-mode-policy",                      GetOperationModePolicy                      },
        { "set-operation-mode-policy",                      SetOperationModePolicy                      },
        { "get-performance-mode-policy",                    GetPerformanceModePolicy                    },
        { "set-performance-mode-policy",                    SetPerformanceModePolicy                    },
        { "get-current-performance-mode",                   GetCurrentPerformanceMode                   },
        { "get-cpu-overclock-enabled",                      GetCpuOverclockEnabled                      },
        { "enable-cpu-overclock",                           EnableCpuOverclock                          },
        { "disable-cpu-overclock",                          DisableCpuOverclock                         },
        { "get-sleep-mode",                                 GetSleepMode                                },
        { "set-sleep-mode",                                 SetSleepMode                                },
        { "get-idle-time-detect-mode",                      GetIdleTimeDetectMode                       },
        { "set-idle-time-detect-mode",                      SetIdleTimeDetectMode                       },
        { "switch-menu",                                    SwitchMenu                                  },
        { "switch-shop",                                    SwitchShop                                  },
        { "set-assumed-system-applet-version",              SetAssumedSystemAppletVersion               },
        { "get-assumed-system-applet-version",              GetAssumedSystemAppletVersion               },
        { "enable-fatal-screen",                            EnableFatalScreen                           },
        { "disable-fatal-screen",                           DisableFatalScreen                          },
        { "set-memory-mode",                                SetMemoryMode                               },
        { "get-memory-mode",                                GetMemoryMode                               },
        { "enable-fill-memory",                             EnableFillMemory                            },
        { "disable-fill-memory",                            DisableFillMemory                           },
        { "enable-user-exception-handler",                  EnableUserExceptionHandler                  },
        { "disable-user-exception-handler",                 DisableUserExceptionHandler                 },
        { "enable-red-screen-at-system-error",              EnableRedScreenAtSystemError                },
        { "disable-red-screen-at-system-error",             DisableRedScreenAtSystemError               },
        { "enable-pmu",                                     EnablePerformanceMonitoringUnit             },
        { "disable-pmu",                                    DisablePerformanceMonitoringUnit            },
        { "set-initial-tick-value",                         SetInitialTickValue                         },
        { "get-initial-tick-value",                         GetInitialTickValue                         },
        { "get-nifm-testing-mode-enabled",                  GetNifmTestingModeEnabled                   },
        { "enable-nifm-testing-mode",                       EnableNifmTestingMode                       },
        { "disable-nifm-testing-mode",                      DisableNifmTestingMode                      },
        { "enable-errorviewer-auto-close",                  EnableErrorViewerAutoClose                  },
        { "disable-errorviewer-auto-close",                 DisableErrorViewerAutoClose                 },
        { "get-gpu-resource-limitation-enabled",            GetGpuResourceLimitationEnabled             },
        { "enable-gpu-resource-limitation",                 EnableGpuResourceLimitation                 },
        { "disable-gpu-resource-limitation",                DisableGpuResourceLimitation                },
        { "get-background-download-stress-testing-enabled", GetBackgroundDownloadStressTestingEnabled   },
        { "enable-background-download-stress-testing",      EnableBackgroundDownloadStressTesting       },
        { "disable-background-download-stress-testing",     DisableBackgroundDownloadStressTesting      },
        { "get-background-download-stress-testing-info",    GetBackgroundDownloadStressTestingInfo      },
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "get-background-apply-delta-stress-testing-info", GetBackgroundApplyDeltaStressTestingInfo    },
#endif
        { "set-background-download-stress-testing-storage", SetBackgroundDownloadStressTestingStorage   },
        { "fill-nand-free-space",                           FillNandFreeSpace                           },
        { "empty-nand-free-space",                          EmptyNandFreeSpace                          },
        { "fill-sdcard-free-space",                         FillSdCardFreeSpace },
        { "empty-sdcard-free-space",                        EmptySdCardFreeSpace },
        { "get-jit-debug-enabled",                          GetJitDebugEnabled                          },
        { "enable-jit-debug",                               EnableJitDebug                              },
        { "disable-jit-debug",                              DisableJitDebug                             },
        { "get-cpu-crash-automatic-dump-enabled",           GetCpuCrashAutomaticDumpEnabled             },
        { "enable-cpu-crash-automatic-dump",                EnableCpuCrashAutomaticDump                 },
        { "disable-cpu-crash-automatic-dump",               DisableCpuCrashAutomaticDump                },
        { "get-cpu-crash-automatic-full-dump-enabled",      GetCpuCrashAutomaticFullDumpEnabled         },
        { "enable-cpu-crash-automatic-full-dump",           EnableCpuCrashAutomaticFullDump             },
        { "disable-cpu-crash-automatic-full-dump",          DisableCpuCrashAutomaticFullDump            },
        { "get-sd-card-logging-enabled",                    GetSdCardLoggingEnabled                     },
        { "enable-sd-card-logging",                         EnableSdCardLogging                         },
        { "disable-sd-card-logging",                        DisableSdCardLogging                        },
        { "set-screenshot-target",                          SetScreenShotTarget                         },
        { "get-screenshot-target",                          GetScreenShotTarget                         },
        { "get-gpu-crash-automatic-dump-enabled",           GetGpuCrashAutomaticDumpEnabled             },
        { "enable-gpu-crash-automatic-dump",                EnableGpuCrashAutomaticDump                 },
        { "disable-gpu-crash-automatic-dump",               DisableGpuCrashAutomaticDump                },
        { "get-terminate-application-on-gpu-error-enabled", GetTerminateApplicationOnGpuErrorEnabled    },
        { "enable-terminate-application-on-gpu-error",      EnableTerminateApplicationOnGpuError        },
        { "disable-terminate-application-on-gpu-error",     DisableTerminateApplicationOnGpuError       },
        { "set-fs-access-log-mode",                         SetFsAccessLogMode                          },
        { "get-fs-access-log-mode",                         GetFsAccessLogMode                          },
        { "set-fs-speed-emulation-mode",                    SetFsSpeedEmulationMode                     },
        { "get-fs-speed-emulation-mode",                    GetFsSpeedEmulationMode                     },
        { "request-network-connection",                     RequestNetworkConnection                    },
        { "enable-field-testing",                           EnableFieldTesting                          },
        { "disable-field-testing",                          DisableFieldTesting                         },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "enable-force-maintenance-mode",                  EnableForceMaintenanceMode                  },
        { "disable-force-maintenance-mode",                 DisableForceMaintenanceMode                 },
        { "enable-applet-development",                      EnableAppletDevelopment                     },
        { "disable-applet-development",                     DisableAppletDevelopment                    },
        { "throw-fatal",                                    ThrowFatal                                  },
        { "set-sleep-pending-span",                         SetSleepPendingSpan                         },
        { "enable-fatal-extra-info",                        EnableFatalExtraInfo                        },
        { "disable-fatal-extra-info",                       DisableFatalExtraInfo                       },
        { "set-firmware-debug-settings",                    SetFirmwareDebugSettings                    },
        { "needs-update-vulnerability",                     NeedsUpdateVulnerabilityCommand             },
        { "set-needs-update-vulnerability-policy",          SetNeedsUpdateVulnerabilityPolicy           },
        { "update-safe-system-version",                     UpdateSafeSystemVersion                     },
        { "get-safe-system-version",                        GetSafeSystemVersion                        },
        { "enable-mount-sdcard",                            EnableMountSdCard                           },
        { "disable-mount-sdcard",                           DisableMountSdCard                          },
        { "set-boolean-fwdbg",                              SetBooleanFwdbg                             },
        { "set-string-fwdbg",                               SetStringFwdbg                              },
        { "set-integer-fwdbg",                              SetIntegerFwdbg                             },
        { "set-hexadecimal-fwdbg",                          SetHexadecimalFwdbg                         },
#endif
        { "get-battery-draining-performance-enabled",       GetBatteryDrainingPerformanceEnabled        },
        { "enable-battery-draining-performance",            EnableBatteryDrainingPerformance            },
        { "disable-battery-draining-performance",           DisableBatteryDrainingPerformance           },
        { "enable-cec",                                     EnableCec                                   },
        { "disable-cec",                                    DisableCec                                  },
        { "enable-pcm",                                     EnablePcm                                   },
        { "disable-pcm",                                    DisablePcm                                  },
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "get-not-locked-nfp-access-enabled",              GetNotLockedNfpAccessEnabled                },
        { "enable-not-locked-nfp-access",                   EnableNotLockedNfpAccess                    },
        { "disable-not-locked-nfp-access",                  DisableNotLockedNfpAccess                   },
        { "get-nfp-play-report-enabled",                    GetNfpPlayReportEnabled                     },
        { "enable-nfp-play-report",                         EnableNfpPlayReport                         },
        { "disable-nfp-play-report",                        DisableNfpPlayReport                        },
        { "get-system-play-report-enabled",                 GetSystemPlayReportEnabled                  },
        { "enable-system-play-report",                      EnableSystemPlayReport                      },
        { "disable-system-play-report",                     DisableSystemPlayReport                     },
        { "get-rid-mode-enabled",                           GetRetailInteractiveDisplayModeEnabled      },
        { "enable-rid-mode",                                EnableRetailInteractiveDisplayMode          },
        { "disable-rid-mode",                               DisableRetailInteractiveDisplayMode         },
        { "set-initial-launch-settings",                    SetInitialLaunchSettings                    },
        { "get-initial-launch-settings",                    GetInitialLaunchSettings                    },
        { "set-device-language",                            SetDeviceLanguage                           },
        { "get-device-language",                            GetDeviceLanguage                           },
        { "set-prepurchase-response-mode",                  SetPrepurchaseResponseMode                  },
        { "get-prepurchase-response-mode",                  GetPrepurchaseResponseMode                  },
        { "enable-nim-error-simulate",                      EnableNimErrorSimulate                      },
        { "disable-nim-error-simulate",                     DisableNimErrorSimulate                     },
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        { "enable-debug-response-simulate",                 NimDebugResponseSimulation::Enable          },
        { "disable-debug-response-simulate",                NimDebugResponseSimulation::Disable         },
        { "register-debug-response-simulate",               NimDebugResponseSimulation::Register        },
        { "clear-debug-response-simulate",                  NimDebugResponseSimulation::Clear           },
#endif
        { "set-ssl-test-init",                              SetSslTestInit                              },
        { "abort",                                          CallAbort                                   },
        { "set-cleanup-sdcard-result-value",                SetCleanupSdCardResultValue                 },
        { "set-blacklist",                                  SetBlackList                                },
        { "get-resume-all-enabled",                         GetResumeAllEnabled                         },
        { "enable-resume-all",                              EnableResumeAll                             },
        { "disable-resume-all",                             DisableResumeAll                            },
        { "enable-dynamic-rights-check",                    EnableDynamicRightsCheck                    },
        { "disable-dynamic-rights-check",                   DisableDynamicRightsCheck                   },
        { "cleanup-preinstall",                             CleanupPreInstall                           },
#endif
        { "get-exhibition-mode-enabled",                    GetExhibitionModeEnabled                    },
        { "enable-exhibition-mode",                         EnableExhibitionMode                        },
        { "disable-exhibition-mode",                        DisableExhibitionMode                       },
        { "get-eula-state",                                 GetEulaState                                },
        { "get-no-vsync-capability-enabled",                GetNoVsyncCapabilityModeEnabled             },
        { "enable-no-vsync-capability",                     EnableNoVsyncCapability                     },
        { "disable-no-vsync-capability",                    DisableNoVsyncCapability                    },
        { "ease-nro-restriction",                           EaseNroRestriction                          },
        { "reset-pseudo-device-id",                         ResetPseudoDeviceId                         },
        { "get-gpu-timeout-enabled",                        GetGpuTimeoutModeEnabled                    },
        { "enable-gpu-timeout",                             EnableGpuTimeout                            },
        { "disable-gpu-timeout",                            DisableGpuTimeout                           },
        { "get-in-store-demo-mode-enabled",                 GetInStoreDemoModeEnabled                   },
        { "enable-in-store-demo-mode",                      EnableInStoreDemoMode                       },
        { "disable-in-store-demo-mode",                     DisableInStoreDemoMode                      },
        { "get-hdcp-authentication-failed-emulation-enabled", GetHdcpAuthenticationFailedEmulationEnabled },
        { "enable-hdcp-authentication-failed-emulation",    EnableHdcpAuthenticationFailedEmulation     },
        { "disable-hdcp-authentication-failed-emulation",   DisableHdcpAuthenticationFailedEmulation    },
        { "get-graphics-firmware-memory-margin-enabled",    GetGraphicsFirmwareMemoryMarginEnabled      },
        { "enable-graphics-firmware-memory-margin",         EnableGraphicsFirmwareMemoryMargin          },
        { "disable-graphics-firmware-memory-margin",        DisableGraphicsFirmwareMemoryMargin         },
        { "copy",                                           Copy                                        },
        { "get-cleanup-application-cache-storage-enabled",  GetCleanupApplicationCacheStorageEnabled    },
        { "enable-cleanup-application-cache-storage",       EnableCleanupApplicationCacheStorage        },
        { "disable-cleanup-application-cache-storage",      DisableCleanupApplicationCacheStorage       },
        { "set-host-romfs-redirection-target",              SetHostRomfsRedirectionTarget               },

    };

}   // namespace

//------------------------------------------------------------------------------------------------

namespace {

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    const auto IsDevMenuCommandSystem = true;
#else
    const auto IsDevMenuCommandSystem = false;
#endif

void ShowHelpMessage() NN_NOEXCEPT
{
    NN_LOG(HelpMessage);
#if !(defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM))
    // TORIAEZU: DevMenu 系では関連オブジェクトのリンクがされないため、一旦使用しない。
    for (auto&& e : nn::devmenucommand::debug::GetDebugSubCommands())
    {
        if (nn::devmenucommand::NeedsHelpMessage(e, IsDevMenuCommandSystem))
        {
            NN_LOG("       " DEVMENUCOMMAND_NAME " debug %s %s\n", e.name.c_str(), e.argumentHelpMessage.c_str());
        }
    }
#endif
}

}

Result DebugCommand(bool* outValue, const Option& option)
{
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
    NN_UNUSED(HelpMessage);
#endif
    if (!option.HasSubCommand())
    {
        ShowHelpMessage();
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        ShowHelpMessage();
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const auto& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }
#if !(defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM))
    // TORIAEZU: DevMenu 系では関連オブジェクトのリンクがされないため、一旦使用しない。
    for (auto&& e : nn::devmenucommand::debug::GetDebugSubCommands())
    {
        if (e.name == option.GetSubCommand())
        {
            if (nn::devmenucommand::IsAvailable(e, IsDevMenuCommandSystem))
            {
                return e.function(outValue, option);
            }
        }
    }
#endif
    NN_UNUSED(IsDevMenuCommandSystem);

    NN_LOG("'%s' is not a DevMenu debug command. See '" DEVMENUCOMMAND_NAME " debug --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
